# ubuntuone.syncdaemon.tests.test_action_queue - action queue tests
#
# Author: Tim Cole <tim.cole@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" ActionQueue tests """

from ubuntuone.syncdaemon.action_queue import (
    ActionQueue, ActionQueueCommand, RequestQueue, WaitForCondition)
from twisted.internet import defer
from twisted.internet.defer import inlineCallbacks
from twisted.trial.unittest import TestCase as TwistedTestCase


class AppendCommand(ActionQueueCommand):
    """Simple command which appends a value to a list."""

    def __init__(self, request_queue, append_to, value):
        """Initialize, remembering list and value."""
        super(AppendCommand, self).__init__(request_queue)
        self.append_to = append_to
        self.value = value

    def _run(self):
        """Do the actual append."""
        self.append_to.append(self.value)
        return defer.succeed(None)

    def __repr__(self):
        """Helper when debugging."""
        return "<AppendCommand: %s>" % (self.value,)
    __str__ = __repr__


class FakeEventQueue(object):
    """Fake event queue."""

    def __init__(self):
        """Initialize fake event queue instance."""
        self.events = []
        self._listeners = []

    def subscribe(self, listener):
        """Minimum to deal with listener."""
        self._listeners.append(listener)

    def push(self, name, *args, **kw):
        """Push event onto queue."""
        self.events.append((name, args, kw))


class FakeActionQueue(object):
    """Fake action queue."""

    def __init__(self, event_queue):
        """Initialize fake action queue instance."""
        self.event_queue = event_queue


class CommandTests(TwistedTestCase):
    """Isolated tests of the action queue command running infrastructure."""

    @inlineCallbacks
    def setUp(self):
        """Set up mocked event queue and so on."""
        yield TwistedTestCase.setUp(self)
        self.event_queue = FakeEventQueue()
        self.action_queue = ActionQueue(self.event_queue, main=None,
                                        host="bogus", port=1234,
                                        dns_srv=False)
        self.request_queue = RequestQueue("TEST_QUEUE",
                                          self.action_queue)

    def test_simple_commands(self):
        """Test command execution."""
        out = []
        AppendCommand(self.request_queue,
                      append_to=out, value="a").queue()
        AppendCommand(self.request_queue,
                      append_to=out, value="b").queue()
        self.request_queue.run()
        self.request_queue.run()
        self.assertEqual(["a", "b"], out)

    def test_blocked_command_commutative(self):
        """Test blocked command execution in a commutative queue."""
        self.request_queue.commutative = True
        out = []
        cmds = [AppendCommand(self.request_queue, append_to=out, value=v) \
                for v in ("a", "b", "c")]
        for cmd in cmds:
            cmd.queue()
        cmds[1].is_runnable = False
        self.request_queue.run()
        self.request_queue.run()
        self.assertEqual(["a", "c"], out)
        self.request_queue.run()
        self.assertEqual(["a", "c"], out)
        cmds[1].is_runnable = True
        self.request_queue.check_conditions()
        self.request_queue.run()
        self.assertEqual(["a", "c", "b"], out)

    def test_blocked_command_noncommutative(self):
        """Test blocked command execution in a non-commutative queue."""
        out = []
        cmds = [AppendCommand(self.request_queue, append_to=out, value=v) \
                for v in ("a", "b", "c")]
        for cmd in cmds:
            cmd.queue()
        cmds[1].is_runnable = False
        self.request_queue.run()
        self.request_queue.run()
        self.assertEqual(["a"], out)
        cmds[1].is_runnable = True
        self.request_queue.check_conditions()
        self.request_queue.run()
        self.assertEqual(["a", "b"], out)
        self.request_queue.run()
        self.assertEqual(["a", "b", "c"], out)

    def test_predicate_satisfaction(self):
        """Test that waits really wait."""
        fixture = [False]
        cmd = WaitForCondition(self.request_queue,
                               condition=lambda: fixture[0])
        cmd.queue()
        d = self.request_queue.run()
        d.addCallback(lambda _: fixture.append('x'))

        self.request_queue.check_conditions()
        self.assertEqual(1, len(fixture))
        self.assertEqual(0, len(self.request_queue.waiting))

        fixture[0] = True
        self.request_queue.check_conditions()
        self.assertEqual(2, len(fixture))
        self.assertEqual(0, len(self.request_queue.waiting))
