/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "syncdaemon-metadata.h"

G_DEFINE_TYPE(SyncdaemonMetadata, syncdaemon_metadata, G_TYPE_OBJECT)

struct _SyncdaemonMetadataPrivate {
	gchar *path;
	gchar *local_hash;
	gchar *server_hash;
	gchar *share_id;
	gchar *node_id;
	gboolean is_synced;
	gboolean is_dir;
};

static void
syncdaemon_metadata_finalize (GObject *object)
{
	SyncdaemonMetadata *metadata = SYNCDAEMON_METADATA (object);

	if (metadata->priv != NULL) {
		if (metadata->priv->path != NULL)
			g_free (metadata->priv->path);

		if (metadata->priv->local_hash != NULL)
			g_free (metadata->priv->local_hash);

		if (metadata->priv->server_hash != NULL)
			g_free (metadata->priv->server_hash);

		if (metadata->priv->share_id != NULL)
			g_free (metadata->priv->share_id);

		if (metadata->priv->node_id != NULL)
			g_free (metadata->priv->node_id);

		g_free (metadata->priv);
	}

	G_OBJECT_CLASS (syncdaemon_metadata_parent_class)->finalize (object);
}

static void
syncdaemon_metadata_class_init (SyncdaemonMetadataClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_metadata_finalize;
}

static void
syncdaemon_metadata_init (SyncdaemonMetadata *metadata)
{
	metadata->priv = g_new0 (SyncdaemonMetadataPrivate, 1);
}

/**
 * syncdaemon_metadata_new:
 */
SyncdaemonMetadata *
syncdaemon_metadata_new (void)
{
	return g_object_new (SYNCDAEMON_TYPE_METADATA, NULL);
}

/**
 * syncdaemon_metadata_new_from_hash_table:
 */
SyncdaemonMetadata *
syncdaemon_metadata_new_from_hash_table (GHashTable *hash)
{
	SyncdaemonMetadata *metadata;

	metadata = g_object_new (SYNCDAEMON_TYPE_METADATA, NULL);
	if (hash != NULL) {
		syncdaemon_metadata_set_path (metadata, g_hash_table_lookup (hash, "path"));
		syncdaemon_metadata_set_local_hash (metadata, g_hash_table_lookup (hash, "local_hash"));
		syncdaemon_metadata_set_server_hash (metadata, g_hash_table_lookup (hash, "server_hash"));
		syncdaemon_metadata_set_share_id (metadata, g_hash_table_lookup (hash, "share_id"));
		syncdaemon_metadata_set_node_id (metadata, g_hash_table_lookup (hash, "node_id"));
		syncdaemon_metadata_set_is_synced (
			metadata,
			g_strcmp0 (g_hash_table_lookup (hash, "quick_tree_synced"), "synced") == 0);
		metadata->priv->is_dir = g_strcmp0 (g_hash_table_lookup (hash, "is_dir"), "True") == 0;
		/* FIXME: we ignore the other values in the hash table since none of our
		   clients use them for now */
	}

	return metadata;
}

/**
 * syncdaemon_metadata_get_path:
 */
const gchar *
syncdaemon_metadata_get_path (SyncdaemonMetadata *metadata)
{
	g_return_val_if_fail (SYNCDAEMON_IS_METADATA (metadata), NULL);

	return (const gchar *) metadata->priv->path;
}

/**
 * syncdaemon_metadata_set_path:
 */
void
syncdaemon_metadata_set_path (SyncdaemonMetadata *metadata, const gchar *path)
{
	g_return_if_fail (SYNCDAEMON_IS_METADATA (metadata));

	if (metadata->priv->path != NULL)
		g_free (metadata->priv->path);

	metadata->priv->path = g_strdup (path);
}

/**
 * syncdaemon_metadata_get_local_hash:
 */
const gchar *
syncdaemon_metadata_get_local_hash (SyncdaemonMetadata *metadata)
{
	g_return_val_if_fail (SYNCDAEMON_IS_METADATA (metadata), NULL);

	return (const gchar *) metadata->priv->local_hash;
}

/**
 * syncdaemon_metadata_set_local_hash:
 */
void
syncdaemon_metadata_set_local_hash (SyncdaemonMetadata *metadata, const gchar *local_hash)
{
	g_return_if_fail (SYNCDAEMON_IS_METADATA (metadata));

	if (metadata->priv->local_hash != NULL)
		g_free (metadata->priv->local_hash);

	metadata->priv->local_hash = g_strdup (local_hash);
}

/**
 * syncdaemon_metadata_get_server_hash:
 */
const gchar *
syncdaemon_metadata_get_server_hash (SyncdaemonMetadata *metadata)
{
	g_return_val_if_fail (SYNCDAEMON_IS_METADATA (metadata), NULL);

	return (const gchar *) metadata->priv->server_hash;
}

/**
 * syncdaemon_metadata_set_server_hash:
 */
void
syncdaemon_metadata_set_server_hash (SyncdaemonMetadata *metadata, const gchar *server_hash)
{
	g_return_if_fail (SYNCDAEMON_IS_METADATA (metadata));

	if (metadata->priv->server_hash != NULL)
		g_free (metadata->priv->server_hash);

	metadata->priv->server_hash = g_strdup (server_hash);
}

/**
 * syncdaemon_metadata_get_share_id:
 */
const gchar *
syncdaemon_metadata_get_share_id (SyncdaemonMetadata *metadata)
{
	g_return_val_if_fail (SYNCDAEMON_IS_METADATA (metadata), NULL);

	return (const gchar *) metadata->priv->share_id;
}

/**
 * syncdaemon_metadata_set_share_id:
 */
void
syncdaemon_metadata_set_share_id (SyncdaemonMetadata *metadata, const gchar *share_id)
{
	g_return_if_fail (SYNCDAEMON_IS_METADATA (metadata));

	if (metadata->priv->share_id != NULL)
		g_free (metadata->priv->share_id);

	metadata->priv->share_id = g_strdup (share_id);
}

/**
 * syncdaemon_metadata_get_node_id:
 */
const gchar *
syncdaemon_metadata_get_node_id (SyncdaemonMetadata *metadata)
{
	g_return_val_if_fail (SYNCDAEMON_IS_METADATA (metadata), NULL);

	return (const gchar *) metadata->priv->node_id;
}

/**
 * syncdaemon_metadata_set_node_id:
 */
void
syncdaemon_metadata_set_node_id (SyncdaemonMetadata *metadata, const gchar *node_id)
{
	g_return_if_fail (SYNCDAEMON_IS_METADATA (metadata));

	if (metadata->priv->node_id != NULL)
		g_free (metadata->priv->node_id);

	metadata->priv->node_id = g_strdup (node_id);
}

/**
 * syncdaemon_metadata_get_is_synced:
 */
gboolean
syncdaemon_metadata_get_is_synced (SyncdaemonMetadata *metadata)
{
	gchar *local, *server;
	gboolean is_updated;

	g_return_val_if_fail (SYNCDAEMON_IS_METADATA (metadata), FALSE);

	/* If it's a dir, we have the 'quick_tree_synced' value */
	if (metadata->priv->is_dir)
		return metadata->priv->is_synced;

	local = metadata->priv->local_hash;
	server = metadata->priv->server_hash;

	is_updated = local && server && *local && *server &&
		g_strcmp0 (local, server) == 0;

        return is_updated;
}

/**
 * syncdaemon_metadata_set_is_synced:
 */
void
syncdaemon_metadata_set_is_synced (SyncdaemonMetadata *metadata, gboolean is_synced)
{
	g_return_if_fail (SYNCDAEMON_IS_METADATA (metadata));

        metadata->priv->is_synced = is_synced;
}

