/*
 * ubuntuone-nautilus.c - Nautilus extensions for Ubuntu One
 *
 * Authors: Tim Cole <tim.cole@canonical.com>
 *          Rodney Dawes <rodney.dawes@canonical.com>
 *          Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2009-2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdlib.h>
#include <glib/gi18n-lib.h>
#include <libnautilus-extension/nautilus-extension-types.h>
#include <libnautilus-extension/nautilus-file-info.h>
#include <libnautilus-extension/nautilus-info-provider.h>
#include <libnautilus-extension/nautilus-menu-provider.h>
#include <libnautilus-extension/nautilus-location-widget-provider.h>
#include <libsyncdaemon/libsyncdaemon.h>
#include "ubuntuone-nautilus.h"
#include "location-widget.h"
#include "share-dialog.h"

/* We need to do this explicitly because older versions of nautilus
 * don't seem to do it for us
 */
#include <gtk/gtk.h>

#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

/* The header is generated from ubuntuone-marshallers.list */
#include "ubuntuone-marshallers.h"

/* This is used when a change is pending */
#define UPDATE_PENDING "pending"

static void ubuntuone_nautilus_finalize(GObject * object);

static void ubuntuone_nautilus_register_type (GTypeModule * module);

/* Utility functions */
static void ubuntuone_nautilus_reset_emblem (UbuntuOneNautilus * uon,
					     const char * path);

static void ubuntuone_nautilus_observed_file_unref (gpointer user_data,
                                                    GObject *where_the_object_was);

static void ubuntuone_nautilus_observed_file_foreach_unref (gpointer key,
							    gpointer value,
							    gpointer user_data);

/* DBus signal and async method call handlers */
static void ubuntuone_nautilus_update_meta (SyncdaemonFilesystemInterface *interface,
					    gboolean success,
					    SyncdaemonMetadata *metadata,
					    gpointer user_data);
static void ubuntuone_nautilus_daemon_ready (SyncdaemonDaemon *daemon, gpointer user_data);
static void ubuntuone_nautilus_upload_started (SyncdaemonDaemon *daemon,
					       gchar * path,
					       gpointer user_data);
static void ubuntuone_nautilus_upload_finished (SyncdaemonDaemon *daemon,
						 gchar * path,
						SyncdaemonTransferInfo * info,
						 gpointer user_data);
static void ubuntuone_nautilus_download_started (SyncdaemonDaemon *daemon,
						 gchar * path,
						 gpointer user_data);
static void ubuntuone_nautilus_download_finished (SyncdaemonDaemon *daemon,
						  gchar * path,
						  SyncdaemonTransferInfo * info,
						  gpointer user_data);
static void ubuntuone_nautilus_udf_created (SyncdaemonDaemon *daemon,
					    gboolean success,
					    SyncdaemonFolderInfo *folder_info,
					    gpointer user_data);
static void ubuntuone_nautilus_udf_deleted (SyncdaemonDaemon *daemon,
					    gboolean success,
					    SyncdaemonFolderInfo *folder_info,
					    gpointer user_data);
static void ubuntuone_nautilus_file_published (SyncdaemonDaemon *daemon,
					       gboolean success,
					       SyncdaemonFileInfo *finfo,
					       gpointer user_data);
static void ubuntuone_nautilus_got_public_files (SyncdaemonDaemon *daemon,
                                                 gboolean success,
						 GSList *files,
                                                 gpointer user_data);
static void ubuntuone_nautilus_share_created (SyncdaemonDaemon *daemon,
					      gboolean success,
					      SyncdaemonShareInfo *share_info,
					      gpointer user_data);
static void ubuntuone_nautilus_share_deleted (SyncdaemonDaemon *daemon,
					      gboolean success,
					      SyncdaemonShareInfo *share_info,
					      gpointer user_data);


static GObjectClass * parent_class = NULL;

static void ubuntuone_nautilus_reset_emblem (UbuntuOneNautilus * uon,
					     const char * path) {

  GHashTableIter iter;
  gchar *key;
  NautilusFileInfo *file;

  /* Reset emblems from all files in the specified path */
  g_hash_table_iter_init (&iter, uon->observed);
  while (g_hash_table_iter_next (&iter, (gpointer *) &key, (gpointer *) &file)) {
    if (g_str_has_prefix (key, path) || g_strcmp0 (key, path) == 0) {
      nautilus_file_info_invalidate_extension_info (file);
    }
  }
}

/* Called when NautilusFileInfo is unreferenced */
static void ubuntuone_nautilus_observed_file_unref (gpointer user_data,
                                                    GObject *where_the_object_was) {
  UbuntuOneNautilus * uon;
  NautilusFileInfo * file;
  gchar * path = NULL;

  uon  = UBUNTUONE_NAUTILUS(user_data);
  file = NAUTILUS_FILE_INFO(where_the_object_was);
  path = g_filename_from_uri (nautilus_file_info_get_uri (file), NULL, NULL);

  if (g_hash_table_lookup (uon->observed, path ))
    g_hash_table_remove (uon->observed, path);

}

/* Cleanup routine for weak reference callbacks */
static void ubuntuone_nautilus_observed_file_foreach_unref (gpointer key,
							    gpointer value,
							    gpointer user_data) {
  UbuntuOneNautilus * uon;
  NautilusFileInfo * file;

  uon = UBUNTUONE_NAUTILUS(user_data);
  file = NAUTILUS_FILE_INFO(value);

  g_object_weak_unref (G_OBJECT(file),
		       (GWeakNotify)ubuntuone_nautilus_observed_file_unref,
		       uon);

}

/* 
 * Check if the passed folder is inside a folder shared TO the user.
 *
 * Added to fix bug #712674
 */
gboolean
ubuntuone_is_inside_shares (UbuntuOneNautilus *uon, const gchar *path)
{
  gboolean is_shared_to_me = FALSE;
  gchar *resolved_path;
  gchar *prefix = g_build_filename (g_get_user_data_dir (), "ubuntuone", "shares", G_DIR_SEPARATOR_S, NULL);

  /* 
   *  Have to use realpath because path contains a symlink like
   * ~/Ubuntu One/Shared With Me -> ~/.local/share/ubuntuone/shares
   * 
   * This also claims ~/Ubuntu One/Shared with Me/foo is
   * in a share even if it's not true. that's intentional since
   * those files are never uploaded, and thus it makes no sense
   * to publish them.
   */
  resolved_path = realpath(path, NULL);
  if (g_str_has_prefix (resolved_path, prefix)) {
    is_shared_to_me = TRUE;
  }
  free (resolved_path);
  g_free (prefix);
  return is_shared_to_me;
}

/* Are we in an Ubuntu One managed (root or udf) directory */
gboolean ubuntuone_is_storagefs (UbuntuOneNautilus * uon,
				 const char * path,
				 gboolean * is_root) {
  gboolean managed = FALSE;
  gchar * dirpath;
  gchar * shared;
  GSList * udfs, * l;
  SyncdaemonFoldersInterface *interface;

  *is_root = FALSE;

  if (!uon->managed)
    return FALSE;

  if (!path)
    return FALSE;

  /* Check the Ubuntu One directory */
  if (strcmp (path, uon->managed) == 0) {
    *is_root = TRUE;
    return TRUE;
  }

  shared = g_build_filename (uon->managed, "Shared With Me", NULL);
  if (strncmp (path, shared, strlen (shared)) == 0) {
    managed = FALSE;
    goto fsdone;
  }

  dirpath = g_strdup_printf ("%s/", uon->managed);
  if (strncmp (path, dirpath, strlen (dirpath)) == 0) {
    managed = TRUE;
    goto fsdone;
  }

  g_free (dirpath);

  /* Check for UDFs */
  interface = SYNCDAEMON_FOLDERS_INTERFACE (syncdaemon_daemon_get_folders_interface (uon->syncdaemon));
  if (interface != NULL) {
    udfs = syncdaemon_folders_interface_get_folders (SYNCDAEMON_FOLDERS_INTERFACE (interface));
    for (l = udfs; l != NULL && l->data != NULL; l = l->next) {
      const char *tmp_path;

      tmp_path = syncdaemon_folder_info_get_path (SYNCDAEMON_FOLDER_INFO (l->data));
      if (!tmp_path)
        continue;

      if (strcmp (path, tmp_path) == 0) {
	 managed = TRUE;
	 *is_root = TRUE;
	 break;
       }

       dirpath = g_strdup_printf ("%s/", (char *) tmp_path);
       if (strncmp (path, dirpath, strlen (dirpath)) == 0) {
	 managed = TRUE;
	 g_free (dirpath);
	 break;
       }

       g_free (dirpath);
    }
    g_slist_free (udfs);
  }

 fsdone:
  g_free (shared);
  return managed;
}

static void
ubuntuone_nautilus_add_observed (UbuntuOneNautilus * uon,
				 gchar * path,
				 NautilusFileInfo * file)
{
  g_object_weak_ref (G_OBJECT(file),
		     (GWeakNotify) ubuntuone_nautilus_observed_file_unref,
		     uon);
  g_hash_table_insert (uon->observed, g_strdup (path), file);
}

/* Update file info provider */
static NautilusOperationResult
ubuntuone_nautilus_update_file_info (NautilusInfoProvider * provider,
				     NautilusFileInfo * file,
				     GClosure * update_complete,
				     NautilusOperationHandle ** handle)
{
  UbuntuOneNautilus * uon;
  gboolean is_root;
  char * path = NULL;

  uon = UBUNTUONE_NAUTILUS(provider);

  /* If syncdaemon is not ready, do nothing */
  if (!syncdaemon_daemon_is_ready (uon->syncdaemon))
    return NAUTILUS_OPERATION_COMPLETE;

  path = g_filename_from_uri (nautilus_file_info_get_uri (file), NULL, NULL);
  if (path == NULL)
    return NAUTILUS_OPERATION_COMPLETE;

  /* Always add it to the observed hash table, so that we can update emblems */
  ubuntuone_nautilus_add_observed (uon, path, file);

  if (!g_hash_table_lookup (uon->updated, path) &&
      !g_hash_table_lookup (uon->needsupdating, path) &&
      ubuntuone_is_storagefs (uon, path, &is_root)) {
    SyncdaemonFilesystemInterface *interface;

    /* Add the unsynchronized emblem anyway, and update later */
    nautilus_file_info_add_emblem (file, "ubuntuone-updating");

    interface = (SyncdaemonFilesystemInterface *) syncdaemon_daemon_get_filesystem_interface (uon->syncdaemon);
    if (interface != NULL) {
      syncdaemon_filesystem_interface_get_metadata_async (interface,
							  path,
							  nautilus_file_info_is_directory (file),
							  (SyncdaemonGotMetadataFunc) ubuntuone_nautilus_update_meta,
							  uon);
      goto updating_meta;
    }
  }

  if (g_hash_table_lookup (uon->uploads, path) ||
      g_hash_table_lookup (uon->downloads, path))
    nautilus_file_info_add_emblem (file, "ubuntuone-updating");
  else if (g_hash_table_lookup (uon->updated, path))
    nautilus_file_info_add_emblem (file, "ubuntuone-synchronized");
  else if (g_hash_table_lookup (uon->needsupdating, path))
    nautilus_file_info_add_emblem (file, "ubuntuone-updating");

  if (g_hash_table_lookup (uon->udfs, path))
    nautilus_file_info_add_emblem (file, "ubuntuone-enabled");

  if (g_hash_table_lookup (uon->public, path))
    nautilus_file_info_add_emblem (file, "ubuntuone-public");

  if (g_hash_table_lookup (uon->shares, path))
    nautilus_file_info_add_emblem (file, "shared");

 updating_meta:
  g_free (path);

  return NAUTILUS_OPERATION_COMPLETE;
}

static void ubuntuone_nautilus_info_provider_iface_init (NautilusInfoProviderIface * iface) {
  iface->update_file_info = ubuntuone_nautilus_update_file_info;
}

static gboolean
show_location (void)
{
  GConfClient *conf_client;
  GConfValue * show_location_value;
  gboolean show_location;

  conf_client = gconf_client_get_default ();
  show_location_value = gconf_client_get (conf_client, EXPANDER_SHOWN_KEY, NULL);
  g_object_unref (conf_client);

  if (show_location_value == NULL) {
    show_location = TRUE;
  } else {
    show_location = gconf_value_get_bool (show_location_value);
  }
  gconf_value_free (show_location_value);

  return show_location;
}

static void toggle_location() {
  GConfClient *conf_client;

  conf_client = gconf_client_get_default ();
  gconf_client_set_bool (conf_client, EXPANDER_SHOWN_KEY, !show_location(), NULL);
}

/* LocationWidget provider */
static GtkWidget * ubuntuone_nautilus_get_location_widget(NautilusLocationWidgetProvider * provider,
							  const char * uri,
							  GtkWidget * parent) {
  UbuntuOneNautilus * uon;
  gchar *path;
  GtkWidget *location;

  /* if the bar is disabled, do nothing */
  if (!show_location())
    return NULL;

  uon = UBUNTUONE_NAUTILUS (provider);

  /* If syncdaemon is not ready, do nothing */
  if (!syncdaemon_daemon_is_ready (uon->syncdaemon))
    return NULL;

  path = g_filename_from_uri (uri, NULL, NULL);

  if (!path)
    return NULL;

  location = location_widget_new (uon, path);

  g_free (path);

  return location;
}

static void ubuntuone_nautilus_bar_provider_iface_init (NautilusLocationWidgetProviderIface * iface) {
  iface->get_widget = ubuntuone_nautilus_get_location_widget;
}

static void __cb_data_free (struct _CBData * data, gboolean destroy) {
  if (!data)
    return;

  data->uon = NULL;
  data->parent = NULL;
  data->make_public = FALSE;

  if (data->path) {
    g_free (data->path);
    data->path = NULL;
  }

  if (destroy) {
    g_free (data);
    data = NULL;
  }

}

/* Menu callbacks */
static void ubuntuone_nautilus_public_meta (SyncdaemonFilesystemInterface *interface,
					    gboolean success,
					    SyncdaemonMetadata *metadata,
					    gpointer user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  GError * error = NULL;
  const gchar * share_id, * node_id;
  SyncdaemonInterface *public;

  if (!success) {
    g_warning ("ERROR: getting metadata for public file");
    return;
  }

  share_id = syncdaemon_metadata_get_share_id (metadata);
  node_id = syncdaemon_metadata_get_node_id (metadata);

  public = syncdaemon_daemon_get_publicfiles_interface (data->uon->syncdaemon);
  if (public != NULL) {
    syncdaemon_publicfiles_interface_change_public_access (SYNCDAEMON_PUBLICFILES_INTERFACE (public),
							   share_id, node_id, data->make_public);
  }
}

static void ubuntuone_nautilus_unsubscribe_folder (NautilusMenuItem *item,
						   gpointer user_data) {
  SyncdaemonInterface *interface;
  struct _CBData * data = (struct _CBData *) user_data;

  /* Perform the removal of this folder */
  interface = syncdaemon_daemon_get_folders_interface (data->uon->syncdaemon);
  if (interface != NULL) {
    SyncdaemonFolderInfo *folder_info;

    folder_info = syncdaemon_folders_interface_get_info (SYNCDAEMON_FOLDERS_INTERFACE (interface),
                                                         data->path);
    if (folder_info != NULL) {
      if (ubuntuone_nautilus_check_shares_and_public_files (data->uon, folder_info, data->parent)) {
        syncdaemon_folders_interface_delete (SYNCDAEMON_FOLDERS_INTERFACE (interface),
                                             syncdaemon_folder_info_get_volume_id (folder_info));
      }
      g_object_unref (G_OBJECT (folder_info));
    }
  }
}

static void ubuntuone_nautilus_subscribe_folder (NautilusMenuItem *item,
						 gpointer user_data) {
  SyncdaemonInterface *interface;
  struct _CBData * data = (struct _CBData *) user_data;

  /* Perform the addition of this folder */
  interface = syncdaemon_daemon_get_folders_interface (data->uon->syncdaemon);
  if (interface != NULL) {
    /* If there is no user authenticated, make Syncdaemon do so */
    if (!syncdaemon_authentication_has_credentials (syncdaemon_daemon_get_authentication (data->uon->syncdaemon)))
      syncdaemon_daemon_connect (data->uon->syncdaemon);
    syncdaemon_folders_interface_create (SYNCDAEMON_FOLDERS_INTERFACE (interface),
                                         data->path);
  }
}

static void ubuntuone_nautilus_copy_public_url (NautilusMenuItem * item,
						gpointer user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  gchar * url;

  url = g_hash_table_lookup (data->uon->public, data->path);
  gtk_clipboard_set_text (gtk_clipboard_get(GDK_SELECTION_CLIPBOARD),
			  url, strlen (url));
  gtk_clipboard_store (gtk_clipboard_get(GDK_SELECTION_CLIPBOARD));
}

static void ubuntuone_nautilus_toggle_publicity (NautilusMenuItem * item,
						 gpointer user_data) {
  SyncdaemonFilesystemInterface *interface;
  struct _CBData * data = (struct _CBData *) user_data;

  interface = (SyncdaemonFilesystemInterface *) syncdaemon_daemon_get_filesystem_interface (data->uon->syncdaemon);
  if (interface != NULL) {
    /* we know this will not be a directory (so no need for _and_quick_tree_synced) */
    syncdaemon_filesystem_interface_get_metadata_async (interface, data->path, FALSE,
							(SyncdaemonGotMetadataFunc) ubuntuone_nautilus_public_meta,
							data);
  }

  g_hash_table_replace (data->uon->public, g_strdup (data->path), UPDATE_PENDING);
  ubuntuone_nautilus_reset_emblem (data->uon, data->path);
}

static void ubuntuone_nautilus_share_folder (NautilusMenuItem * item,
					     gpointer * user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  GtkWidget * dialog;

  dialog = share_dialog_new (data->parent, data->uon, data->path);
  gtk_widget_show (dialog);
}

static void ubuntuone_nautilus_unshare_folder (NautilusMenuItem * item,
					       gpointer * user_data) {
  struct _CBData * data = (struct _CBData *) user_data;
  gchar * share_id;
  SyncdaemonSharesInterface *interface;

  interface = (SyncdaemonSharesInterface *) syncdaemon_daemon_get_shares_interface (data->uon->syncdaemon);
  if (interface != NULL) {
    share_id = g_hash_table_lookup (data->uon->shares, data->path);
    syncdaemon_shares_interface_delete (interface, share_id);
  }
}

/* Menu provider */
static GList * ubuntuone_nautilus_get_menu_items (NautilusMenuProvider * provider,
						  GtkWidget * window,
						  GList * files) {
  UbuntuOneNautilus * uon;
  NautilusFileInfo * file;
  NautilusMenu * submenu;
  NautilusMenuItem * root_item;
  GList * items = NULL;
  gchar * path;
  gchar * item;
  gchar * homedir_path;
  gboolean is_managed, is_root, is_udf, is_public, is_shared, is_pending;
  gboolean is_shared_to_me, is_inhome, is_dir, is_regular;

  uon = UBUNTUONE_NAUTILUS (provider);

  /* If syncdaemon is not ready, do nothing */
  if (!syncdaemon_daemon_is_ready (uon->syncdaemon))
    return NULL;

  is_managed = is_root = is_udf = is_public = is_shared = is_pending = FALSE;
  is_shared_to_me = is_inhome = is_dir = is_regular = FALSE;

  if (g_list_length (files) != 1)
    return NULL;

  file = g_list_nth_data (files, 0);
  path = g_filename_from_uri (nautilus_file_info_get_uri (file), NULL, NULL);

  if (path == NULL)
    goto done;

  if (ubuntuone_is_storagefs (uon, path, &is_root))
    is_managed = TRUE;

  if (ubuntuone_is_inside_shares (uon, path))
    is_shared_to_me = TRUE;

  homedir_path = g_strdup_printf ("%s/", g_get_home_dir());
  if (strncmp (path, homedir_path, strlen (homedir_path)) == 0)
	is_inhome = TRUE;
  g_free (homedir_path);

  if ((item = g_hash_table_lookup (uon->udfs, path)) != NULL) {
    is_udf = TRUE;
	if (strcmp (item, UPDATE_PENDING) == 0)
	  is_pending = TRUE;
  } else if ((item = g_hash_table_lookup (uon->public, path)) != NULL) {
    is_public = TRUE;
	if (strcmp (item, UPDATE_PENDING) == 0)
	  is_pending = TRUE;
  }
  if (g_hash_table_lookup (uon->shares, path) != NULL)
    is_shared = TRUE;

  is_dir = nautilus_file_info_is_directory (file);
  is_regular = nautilus_file_info_get_file_type (file) == G_FILE_TYPE_REGULAR;

  /* Create new structure only if it does not already exist */
  if (uon->cb_data)
    __cb_data_free (uon->cb_data, FALSE);
  else
    uon->cb_data = g_new0 (struct _CBData, 1);

  uon->cb_data->uon = uon;
  uon->cb_data->parent = window;
  uon->cb_data->path = g_strdup (path);

  root_item = nautilus_menu_item_new ("ubuntuone",
                                      _("_Ubuntu One"),
                                      _("Ubuntu One options"),
                                      "ubuntuone");
  submenu = nautilus_menu_new ();
  nautilus_menu_item_set_submenu (root_item, submenu);

  items = g_list_append (items, root_item);

  /* Share/unshare */
  {
    NautilusMenuItem * item;

    if ((is_managed || is_udf) && !is_root && is_dir) {

      item = nautilus_menu_item_new ("ubuntuone-share",
                                     _("_Share..."),
                                     _("Share this folder on Ubuntu One"),
                                     "ubuntuone");
          if (is_pending)
            g_object_set (item, "sensitive", FALSE, NULL);

      g_signal_connect (item, "activate",
                        G_CALLBACK (ubuntuone_nautilus_share_folder),
                        uon->cb_data);
    } else {
      /* the different tooltips will probably do us no good */
      if (is_root) {
        item = nautilus_menu_item_new ("ubuntuone-noshare-root",
                                       _("_Share..."),
                                       _("Sorry, you can't share the root of a Ubuntu One volume"),
                                       "ubuntuone");
      } else if (!(is_managed || is_udf)) {
        item = nautilus_menu_item_new ("ubuntuone-noshare-unmanaged",
                                       _("_Share..."),
                                       _("Sorry, you can't share folders not managed by Ubuntu One"),
                                       "ubuntuone");
      } else {
        item = nautilus_menu_item_new ("ubuntuone-noshare-unmanaged",
                                       _("_Share..."),
                                       _("Sorry, you can only share folders"),
                                       "ubuntuone");
      }
      g_object_set (item, "sensitive", FALSE, NULL);
    }

    nautilus_menu_append_item (submenu, item);
  }

  if ((is_managed && is_shared) && !is_root && is_dir) {
    NautilusMenuItem * item;

    item = nautilus_menu_item_new ("ubuntuone-unshare",
				   _("Stop _Sharing"),
				   _("Stop sharing this folder on Ubuntu One"),
				   "ubuntuone");
	if (is_pending)
	  g_object_set (item, "sensitive", FALSE, NULL);

    g_signal_connect (item, "activate",
		      G_CALLBACK (ubuntuone_nautilus_unshare_folder),
		      uon->cb_data);
    nautilus_menu_append_item (submenu, item);
  }

  {
    /* UDF logic
     *
     * XXX: clean this up and separate the logic out and reuse this
     * and locationbar somewhere (libsd?)
     */
    NautilusMenuItem * item = NULL;
    if (is_dir && is_inhome) {
      /* UDFs could be happening */
      if (is_managed) {
	if (strcmp (path, uon->managed) == 0) {
	  /* the Ubuntu One directory, no UDFs */
          item = nautilus_menu_item_new ("ubuntuone-no-disable-u1",
                                         _("Stop Synchronizing This _Folder"),
                                         _("Sorry, you can't stop synchronizing ~/Ubuntu One"),
                                         "ubuntuone");
          g_object_set (item, "sensitive", FALSE, NULL);
	} else if (is_root) {
	  /* the root of a UDF: disabling possible */

	  item = nautilus_menu_item_new ("ubuntuone-disable-udf",
					 _("Stop Synchronizing This _Folder"),
					 _("Stop synchronizing this folder with Ubuntu One"),
					 "ubuntuone");

	  g_signal_connect (item, "activate",
			    G_CALLBACK (ubuntuone_nautilus_unsubscribe_folder),
			    uon->cb_data);
	}
      } else {
	/* unmanaged */

	item = nautilus_menu_item_new ("ubuntuone-enable-udf",
				       _("Synchronize This _Folder"),
				       _("Start synchronizing this folder with Ubuntu One"),
				       "ubuntuone");

	g_signal_connect (item, "activate",
			  G_CALLBACK (ubuntuone_nautilus_subscribe_folder),
			  uon->cb_data);
      }
    } else {
      if (is_dir) {
        item = nautilus_menu_item_new ("ubuntuone-no-disable-u1",
				       _("Synchronize This _Folder"),
				       _("Sorry, you can only synchronize folders within your home folder"),
                                       "ubuntuone");
      } else {
        item = nautilus_menu_item_new ("ubuntuone-no-disable-u1",
				       _("Synchronize This _Folder"),
				       _("Sorry, you can only synchronize folders"),
                                       "ubuntuone");
      }
      g_object_set (item, "sensitive", FALSE, NULL);
    }
    if (!item) {
	item = nautilus_menu_item_new ("ubuntuone-no-udf-operation-possible",
				       _("Synchronize This _Folder"),
				       _("Synchronization not possible for this folder"),
				       "ubuntuone");
	g_object_set (item, "sensitive", FALSE, NULL);
    }
    nautilus_menu_append_item (submenu, item);
  }

  /* public files */
  {
    NautilusMenuItem * item = NULL, * urlitem = NULL;

    if (!is_shared_to_me && is_managed && is_regular) {

      if (is_public) {
        urlitem = nautilus_menu_item_new ("ubuntuone-geturl",
                                          _("Copy Web _Link"),
                                          _("Copy the Ubuntu One public URL for this file to the clipboard."),
                                          "ubuntuone");
        if (is_pending)
          g_object_set (urlitem, "sensitive", FALSE, NULL);

        g_signal_connect (urlitem, "activate",
                          G_CALLBACK (ubuntuone_nautilus_copy_public_url),
                          uon->cb_data);
        item = nautilus_menu_item_new ("ubuntuone-unpublish",
                                       _("Stop _Publishing"),
                                       _("No longer share this file with everyone via Ubuntu One."),
                                       "ubuntuone");
        if (is_pending)
          g_object_set (item, "sensitive", FALSE, NULL);

        uon->cb_data->make_public = FALSE;
      } else {
        item = nautilus_menu_item_new ("ubuntuone-publish",
                                       _("_Publish"),
                                       _("Make this file available to anyone via Ubuntu One."),
                                       "ubuntuone");
        uon->cb_data->make_public = TRUE;
      }
      g_signal_connect (item, "activate",
                        G_CALLBACK (ubuntuone_nautilus_toggle_publicity),
                        uon->cb_data);
    }
    if (!urlitem) {
      urlitem = nautilus_menu_item_new ("ubuntuone-geturl",
                                        _("Copy Web _Link"),
                                        _("Sorry, no public URL for this file via Ubuntu One."),
                                        "ubuntuone");
      g_object_set (urlitem, "sensitive", FALSE, NULL);
    }
    if (!item) {
      item = nautilus_menu_item_new ("ubuntuone-publish",
                                     _("_Publish"),
                                     _("Sorry, unable to publish via Ubuntu One."),
                                     "ubuntuone");
      g_object_set (item, "sensitive", FALSE, NULL);
    }

    nautilus_menu_append_item (submenu, item);
    nautilus_menu_append_item (submenu, urlitem);
  }

  /* location bar enable/disable */
  {
    NautilusMenuItem * item;
    if (show_location()) {
      item = nautilus_menu_item_new ("ubuntuone-location-hide",
                                     _("Hide _Ribbon"),
                                     _("Do not show the Ubuntu One ribbon"
                                       " in selected folders"),
                                     "ubuntuone");
    } else {
      item = nautilus_menu_item_new ("ubuntuone-location-show",
                                     _("Show _Ribbon in Some Folders"),
                                     _("Show the Ubuntu One ribbon"
                                       " in selected folders"),
                                     "ubuntuone");
    }
    g_signal_connect (item, "activate",
                      G_CALLBACK (toggle_location),
                      uon->cb_data);
    nautilus_menu_append_item (submenu, item);
  }

 done:
  g_free (path);

  return items;
}

static GList * ubuntuone_nautilus_get_bg_menu_items (NautilusMenuProvider * provider,
						     GtkWidget * window,
						     NautilusFileInfo * folder) {
  GList * files = NULL;
  GList * items = NULL;

  files = g_list_prepend (files, folder);
  items = ubuntuone_nautilus_get_menu_items (provider, window, files);
  files = g_list_remove (files, folder);
  g_list_free (files);

  return items;
}

static void ubuntuone_nautilus_menu_provider_iface_init (NautilusMenuProviderIface * iface) {
  iface->get_file_items = ubuntuone_nautilus_get_menu_items;
  iface->get_background_items = ubuntuone_nautilus_get_bg_menu_items;
}

/* GType and nautilus module stuff */
static GType un_type = 0;

static void ubuntuone_nautilus_instance_init (UbuntuOneNautilus * uon) {
  SyncdaemonSharesInterface *interface;

  uon->connected = FALSE;
  uon->uploads = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->downloads = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->shares = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
  uon->updated = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->needsupdating = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->observed = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->udfs = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  uon->public = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);

  uon->syncdaemon = syncdaemon_daemon_new ();
  g_signal_connect (G_OBJECT (uon->syncdaemon), "ready",
		    G_CALLBACK (ubuntuone_nautilus_daemon_ready), uon);

  /* Default to ~/Ubuntu One for now, as it's all we really support */
  uon->managed = g_build_filename (g_get_home_dir (), "Ubuntu One", NULL);
  uon->gotroot = FALSE;
  uon->gotudfs = FALSE;
  uon->gotpubs = FALSE;

  /* Connec to status-related signals */
  g_signal_connect (G_OBJECT (uon->syncdaemon), "upload_started",
		    G_CALLBACK (ubuntuone_nautilus_upload_started), uon);
  g_signal_connect (G_OBJECT (uon->syncdaemon), "upload_finished",
		    G_CALLBACK (ubuntuone_nautilus_upload_finished), uon);
  g_signal_connect (G_OBJECT (uon->syncdaemon), "download_started",
		    G_CALLBACK (ubuntuone_nautilus_download_started), uon);
  g_signal_connect (G_OBJECT (uon->syncdaemon), "download_finished",
		    G_CALLBACK (ubuntuone_nautilus_download_finished), uon);

  /* Connect to shares-related signals */
  g_signal_connect (G_OBJECT (uon->syncdaemon), "share_created",
		    G_CALLBACK (ubuntuone_nautilus_share_created), uon);
  g_signal_connect (G_OBJECT (uon->syncdaemon), "share_deleted",
		    G_CALLBACK (ubuntuone_nautilus_share_deleted), uon);

  /* Connect to folder-related signals */
  g_signal_connect (G_OBJECT (uon->syncdaemon), "folder_created",
		    G_CALLBACK (ubuntuone_nautilus_udf_created), uon);
  g_signal_connect (G_OBJECT (uon->syncdaemon), "folder_subscribed",
		    G_CALLBACK (ubuntuone_nautilus_udf_created), uon);
  g_signal_connect (G_OBJECT (uon->syncdaemon), "folder_deleted",
		    G_CALLBACK (ubuntuone_nautilus_udf_deleted), uon);
  g_signal_connect (G_OBJECT (uon->syncdaemon), "folder_unsubscribed",
		    G_CALLBACK (ubuntuone_nautilus_udf_deleted), uon);

  /* Connect to public files-related signals */
  g_signal_connect (G_OBJECT (uon->syncdaemon), "file_published",
		    G_CALLBACK (ubuntuone_nautilus_file_published), uon);
  g_signal_connect (G_OBJECT (uon->syncdaemon), "got_published_files",
		    G_CALLBACK (ubuntuone_nautilus_got_public_files), uon);

}

static void ubuntuone_nautilus_class_init (UbuntuOneNautilusClass * klass) {
  parent_class = g_type_class_peek_parent (klass);

  G_OBJECT_CLASS(klass)->finalize = ubuntuone_nautilus_finalize;
}

static void ubuntuone_nautilus_finalize(GObject * object) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS(object);

  __cb_data_free (uon->cb_data, TRUE);

  if (uon->syncdaemon)
    g_object_unref (uon->syncdaemon);

  g_hash_table_destroy (uon->uploads);
  uon->uploads = NULL;

  g_hash_table_destroy (uon->downloads);
  uon->downloads = NULL;

  g_hash_table_destroy (uon->shares);
  uon->shares = NULL;

  g_hash_table_destroy (uon->udfs);
  uon->udfs = NULL;

  g_hash_table_destroy (uon->public);
  uon->public = NULL;

  g_hash_table_destroy (uon->updated);
  uon->updated = NULL;

  g_hash_table_destroy (uon->needsupdating);
  uon->needsupdating = NULL;

  /* We need to remove all weak reference callbacks */
  g_hash_table_foreach (uon->observed, 
			(GHFunc)ubuntuone_nautilus_observed_file_foreach_unref,
			uon);

  g_hash_table_destroy (uon->observed);
  uon->observed = NULL;
}

GType ubuntuone_nautilus_get_type (void) {
  return un_type;
}

static void ubuntuone_nautilus_register_type (GTypeModule * module) {
  static const GTypeInfo info = {
    sizeof (UbuntuOneNautilusClass),
    (GBaseInitFunc) NULL,
    (GBaseFinalizeFunc) NULL,
    (GClassInitFunc) ubuntuone_nautilus_class_init,
    NULL,
    NULL,
    sizeof (UbuntuOneNautilus),
    0,
    (GInstanceInitFunc) ubuntuone_nautilus_instance_init,
  };

  static const GInterfaceInfo info_provider_iface_info = {
    (GInterfaceInitFunc) ubuntuone_nautilus_info_provider_iface_init,
    NULL,
    NULL
  };

  static const GInterfaceInfo bar_provider_iface_info = {
    (GInterfaceInitFunc) ubuntuone_nautilus_bar_provider_iface_init,
    NULL,
    NULL
  };

  static const GInterfaceInfo menu_provider_iface_info = {
    (GInterfaceInitFunc) ubuntuone_nautilus_menu_provider_iface_init,
    NULL,
    NULL
  };

  un_type = g_type_module_register_type (module, 
					 G_TYPE_OBJECT,
					 "UbuntuOneNautilus",
					 &info, 0);
  
  g_type_module_add_interface (module,
			       un_type,
			       NAUTILUS_TYPE_INFO_PROVIDER,
			       &info_provider_iface_info);

  g_type_module_add_interface (module,
			       un_type,
			       NAUTILUS_TYPE_LOCATION_WIDGET_PROVIDER,
			       &bar_provider_iface_info);

  /* XXX Need to sign a request via DBus */
  g_type_module_add_interface (module,
			       un_type,
			       NAUTILUS_TYPE_MENU_PROVIDER,
			       &menu_provider_iface_info);
}


/* DBus signal handlers and async method call handlers */
static void ubuntuone_nautilus_update_meta (SyncdaemonFilesystemInterface *interface,
					    gboolean success,
					    SyncdaemonMetadata *metadata,
					    gpointer user_data) {
  UbuntuOneNautilus * uon;
  GError * error = NULL;
  gchar * path;
  gboolean is_root;

  if (!success) {
    g_warning ("ERROR: Could not retrieve metadata");
    return;
  }

  uon = UBUNTUONE_NAUTILUS (user_data);
  path = g_strdup (syncdaemon_metadata_get_path (metadata));

  if (ubuntuone_is_storagefs (uon, path, &is_root)) {
    if (syncdaemon_metadata_get_is_synced (metadata)) {
      g_hash_table_replace (uon->updated, path, path);
      ubuntuone_nautilus_reset_emblem (uon, path);
    } else
      g_hash_table_replace (uon->needsupdating, path, path);
  } else {
    g_hash_table_remove (uon->updated, path);
    g_hash_table_remove (uon->needsupdating, path);
    ubuntuone_nautilus_reset_emblem (uon, path);

    g_free (path);
  }
}

static void
ubuntuone_nautilus_daemon_ready (SyncdaemonDaemon *daemon, gpointer user_data)
{
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gboolean is_online = FALSE;
  SyncdaemonInterface *interface;

  interface = syncdaemon_daemon_get_status_interface (daemon);
  if (interface != NULL) {
    SyncdaemonStatusInfo *status_info;

    status_info = syncdaemon_status_interface_get_current_status (SYNCDAEMON_STATUS_INTERFACE (interface));
    is_online = syncdaemon_status_info_get_online (status_info);
  }

  /* Get the root when we get a status change signal, if we haven't yet */
  if (!uon->gotroot) {
    uon->managed = g_strdup (syncdaemon_daemon_get_root_dir (uon->syncdaemon));
    if (uon->managed != NULL)
      uon->gotroot = TRUE;
  }

  /* Get the list of UDFs if we haven't already */
  if (!uon->gotudfs) {
    SyncdaemonInterface *interface;

    interface = syncdaemon_daemon_get_folders_interface (uon->syncdaemon);
    if (interface != NULL) {
	    GSList *folders, *l;

      folders = syncdaemon_folders_interface_get_folders (SYNCDAEMON_FOLDERS_INTERFACE (interface));
      for (l = folders; l != NULL; l = l->next) {
        SyncdaemonFolderInfo *folder_info = SYNCDAEMON_FOLDER_INFO (l->data);

	ubuntuone_nautilus_udf_created (uon->syncdaemon, TRUE, folder_info, uon);
	uon->gotudfs = TRUE;
      }

      g_slist_free (folders);
    }
  }

  /* Get the list of shared folders */
  interface = syncdaemon_daemon_get_shares_interface (uon->syncdaemon);
  if (SYNCDAEMON_IS_SHARES_INTERFACE (interface)) {
    GSList *shares;

    shares = syncdaemon_shares_interface_get_shared (SYNCDAEMON_SHARES_INTERFACE (interface));
    while (shares != NULL) {
      ubuntuone_nautilus_share_created (uon->syncdaemon, TRUE, SYNCDAEMON_SHARE_INFO (shares->data), uon);
      shares = g_slist_remove (shares, shares->data);
    }
  }

  /* Get the list of public files if we haven't already */
  if (is_online && !uon->gotpubs) {
    SyncdaemonInterface *public;

    public = syncdaemon_daemon_get_publicfiles_interface (uon->syncdaemon);
    if (public != NULL) {
      syncdaemon_publicfiles_interface_get_public_files (SYNCDAEMON_PUBLICFILES_INTERFACE (public));
    }
  }
}

static void ubuntuone_nautilus_upload_started (SyncdaemonDaemon *daemon,
					       gchar * path,
					       gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);

  if (!g_hash_table_lookup (uon->uploads, path)) {
    gchar *new_path = g_strdup (path);
    g_hash_table_insert (uon->uploads, new_path, new_path);
    ubuntuone_nautilus_reset_emblem (uon, path);
  }
}

static void ubuntuone_nautilus_upload_finished (SyncdaemonDaemon *daemon,
						gchar * path,
						SyncdaemonTransferInfo * info,
						gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * new_path;

  g_hash_table_remove (uon->uploads, path);
  g_hash_table_remove (uon->needsupdating, path);

  new_path = g_strdup (path);
  g_hash_table_replace (uon->updated, new_path, new_path);

  ubuntuone_nautilus_reset_emblem (uon, path);
}

static void ubuntuone_nautilus_download_started (SyncdaemonDaemon *daemon,
						 gchar * path,
						 gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);

  if (!g_hash_table_lookup (uon->downloads, path)) {
    gchar *new_path = g_strdup (path);
    g_hash_table_insert (uon->downloads, new_path, new_path);
    ubuntuone_nautilus_reset_emblem (uon, path);
  }
}

static void ubuntuone_nautilus_download_finished (SyncdaemonDaemon *daemon,
						  gchar * path,
						  SyncdaemonTransferInfo * info,
						  gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  gchar * new_path;

  g_hash_table_remove (uon->downloads, path);
  g_hash_table_remove (uon->needsupdating, path);

  new_path = g_strdup (path);
  g_hash_table_replace (uon->updated, new_path, new_path);

  ubuntuone_nautilus_reset_emblem (uon, path);
}

static void ubuntuone_nautilus_udf_created (SyncdaemonDaemon *daemon,
					    gboolean success,
					    SyncdaemonFolderInfo *folder_info,
					    gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  const gchar * path, * id;
  gboolean subscribed;

  if (!success)
      return;

  path = syncdaemon_folder_info_get_path (folder_info);
  id = syncdaemon_folder_info_get_volume_id (folder_info);
  subscribed = syncdaemon_folder_info_get_subscribed (folder_info);

  if (!path || !id || !subscribed)
      return;

  if (!g_hash_table_lookup (uon->udfs, path)) {
      GHashTableIter iter;
      gchar *key;
      NautilusFileInfo *file;

      g_hash_table_insert (uon->udfs, g_strdup (path), g_strdup (id));

      /* Update the emblems of the files on this new UDF */
      g_hash_table_iter_init (&iter, uon->observed);
      while (g_hash_table_iter_next (&iter, (gpointer *) &key, (gpointer *) file)) {
          if (g_str_has_prefix (key, path))
              nautilus_file_info_add_emblem (file, "ubuntuone-updating");
      }
  }
}

static gboolean
remove_by_path_in_hash (gpointer key, gpointer value, gpointer user_data)
{
  if (g_str_has_prefix ((const gchar *) key, (const gchar *) user_data) ||
      g_strcmp0 ((const gchar *) key, (const gchar *) user_data) == 0)
    return TRUE;

  return FALSE;
}

static void
ubuntuone_nautilus_udf_deleted (SyncdaemonDaemon *daemon,
				gboolean success,
				SyncdaemonFolderInfo *folder_info,
				gpointer user_data)
{
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  const gchar * path;

  path = syncdaemon_folder_info_get_path (folder_info);
  if (path != NULL && success) {
    g_hash_table_remove (uon->udfs, path);
    ubuntuone_nautilus_reset_emblem (uon, path);

    /* Now remove the files from the status hash tables */
    g_hash_table_foreach_remove (uon->updated, (GHRFunc) remove_by_path_in_hash, (gpointer) path);
    g_hash_table_foreach_remove (uon->needsupdating, (GHRFunc) remove_by_path_in_hash, (gpointer) path);
  }
}

static void ubuntuone_nautilus_file_published (SyncdaemonDaemon *daemon,
					       gboolean success,
					       SyncdaemonFileInfo *finfo,
					       gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  const gchar * path, * url;
  gboolean is_public;

  if (!success)
    return;

  path = syncdaemon_file_info_get_path (finfo);
  url = syncdaemon_file_info_get_public_url (finfo);
  is_public = syncdaemon_file_info_get_is_public (finfo);

  if (!is_public && g_hash_table_lookup (uon->public, path))
    g_hash_table_remove (uon->public, path);

  if (is_public)
    g_hash_table_replace (uon->public, g_strdup (path), g_strdup (url));

  ubuntuone_nautilus_reset_emblem (uon, path);
}

static void ubuntuone_nautilus_got_public_files (SyncdaemonDaemon *daemon,
                                                 gboolean success,
                                                 GSList *files,
                                                 gpointer user_data) {
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  int i;

  if (success) {
    GSList *l;

    uon->gotpubs = TRUE;
    g_hash_table_remove_all (uon->public);
    for (l = files; l != NULL; l = l->next) {
      SyncdaemonFileInfo *finfo = SYNCDAEMON_FILE_INFO (l->data);

      g_hash_table_insert (uon->public,
			   g_strdup (syncdaemon_file_info_get_path (finfo)),
			   g_strdup (syncdaemon_file_info_get_public_url (finfo)));
      ubuntuone_nautilus_reset_emblem (uon, syncdaemon_file_info_get_path (finfo));
    }
  }
}

static void
ubuntuone_nautilus_share_created (SyncdaemonDaemon *daemon,
				  gboolean success,
				  SyncdaemonShareInfo *share_info,
				  gpointer user_data)
{
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  const gchar * path;

  path = syncdaemon_share_info_get_path (share_info);
  if (success) {
    const gchar * id;

    id = syncdaemon_share_info_get_volume_id (share_info);
    if (!g_hash_table_lookup (uon->shares, path)) {
      g_hash_table_insert (uon->shares, g_strdup (path), g_strdup (id));
      ubuntuone_nautilus_reset_emblem (uon, path);
    }
  } else {
    ubuntuone_show_error_dialog (uon, _("Error creating share."),
				 _("There was an error sharing the folder '%s'"),
				 path);
  }
}

static void
ubuntuone_nautilus_share_deleted (SyncdaemonDaemon *daemon,
				  gboolean success,
				  SyncdaemonShareInfo *share_info,
				  gpointer user_data)
{
  UbuntuOneNautilus * uon = UBUNTUONE_NAUTILUS (user_data);
  const gchar * path;

  path = syncdaemon_share_info_get_path (share_info);
  if (success) {
    g_hash_table_remove (uon->shares, path);
    ubuntuone_nautilus_reset_emblem (uon, path);
  } else {
    if (path != NULL) {
      ubuntuone_show_error_dialog (uon, _("Error deleting share."),
				   _("There was an error removing the share for the folder '%s'"),
				   path);
    } else {
      ubuntuone_show_error_dialog (uon, _("Error deleting share."),
				   _("There was an error removing the share"));
    }
  }
}

/* Required Nautilus module handling methods */
void nautilus_module_initialize (GTypeModule * module) {
#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  ubuntuone_nautilus_register_type (module);
}

void nautilus_module_shutdown (void) {
}


void nautilus_module_list_types (const GType ** types,
				 int * num_types) {
  static GType type_list[1];
  
  type_list[0] = UBUNTUONE_TYPE_NAUTILUS;

  *types = type_list;
  *num_types = 1;
}
