/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "config.h"
#ifdef HAVE_GDBUS
#else
#include <dbus/dbus-glib.h>
#endif
#include "syncdaemon-filesystem-interface.h"

G_DEFINE_TYPE(SyncdaemonFilesystemInterface, syncdaemon_filesystem_interface, SYNCDAEMON_TYPE_INTERFACE)

struct _SyncdaemonFilesystemInterfacePrivate {
	GObject *proxy;
};

static void
syncdaemon_filesystem_interface_finalize (GObject *object)
{
	SyncdaemonFilesystemInterface *interface = SYNCDAEMON_FILESYSTEM_INTERFACE (object);

	if (interface->priv != NULL) {
		g_free (interface->priv);
	}

	G_OBJECT_CLASS (syncdaemon_filesystem_interface_parent_class)->finalize (object);
}

static void
syncdaemon_filesystem_interface_class_init (SyncdaemonFilesystemInterfaceClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_filesystem_interface_finalize;
}

static void
syncdaemon_filesystem_interface_init (SyncdaemonFilesystemInterface *interface)
{
	interface->priv = g_new0 (SyncdaemonFilesystemInterfacePrivate, 1);

	/* Setup DBus proxy */
	interface->priv->proxy = syncdaemon_interface_setup_proxy (SYNCDAEMON_INTERFACE (interface),
								   "com.ubuntuone.SyncDaemon",
								   "/filesystem", "com.ubuntuone.SyncDaemon.FileSystem");
}

/**
 * syncdaemon_filesystem_interface_new:
 */
SyncdaemonFilesystemInterface *
syncdaemon_filesystem_interface_new (SyncdaemonDaemon *daemon)
{
	g_return_val_if_fail (SYNCDAEMON_IS_DAEMON (daemon), NULL);

	return g_object_new (SYNCDAEMON_TYPE_FILESYSTEM_INTERFACE, "daemon", daemon, NULL);
}

/**
 * syncdaemon_filesystem_interface_get_metadata:
 */
SyncdaemonMetadata *
syncdaemon_filesystem_interface_get_metadata (SyncdaemonFilesystemInterface *interface,
					      const gchar *path,
                                              gboolean with_subtree_sync_check)
{
	GError *error = NULL;
	GHashTable *hash;
        const char *method;

	g_return_val_if_fail (SYNCDAEMON_IS_FILESYSTEM_INTERFACE (interface), NULL);

        if (with_subtree_sync_check)
                method = "get_metadata_and_quick_tree_synced";
        else
                method = "get_metadata";

	if (!dbus_g_proxy_call (DBUS_G_PROXY (interface->priv->proxy), method, &error,
			       G_TYPE_STRING, path,
			       G_TYPE_INVALID,
			       dbus_g_type_get_map ("GHashTable",
						    G_TYPE_STRING,
						    G_TYPE_STRING), &hash,
			       G_TYPE_INVALID)) {
		g_warning ("Failed calling %s: %s", method, error->message);
		g_error_free (error);

		return NULL;
	}

	return syncdaemon_metadata_new_from_hash_table (hash);
}

typedef struct {
	SyncdaemonFilesystemInterface *interface;
	SyncdaemonGotMetadataFunc callback;
	gpointer user_data;
} GotMetadataData;

static void
got_metadata_cb (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer user_data)
{
	GHashTable *hash;
	GError *error = NULL;
	GotMetadataData *gmd = (GotMetadataData *) user_data;

	if (dbus_g_proxy_end_call (proxy, call_id, &error,
				   dbus_g_type_get_map ("GHashTable",
							G_TYPE_STRING,
							G_TYPE_STRING), &hash,
				   G_TYPE_INVALID)) {
		SyncdaemonMetadata *metadata;

		metadata = syncdaemon_metadata_new_from_hash_table (hash);
		gmd->callback (gmd->interface, TRUE, metadata, gmd->user_data);

		/* Free memory */
		g_object_unref (G_OBJECT (metadata));
		g_hash_table_destroy (hash);
	} else {
		g_warning ("Error getting metadata: %s", error->message);
		g_error_free (error);
		gmd->callback (gmd->interface, FALSE, NULL, gmd->user_data);
	}

	g_free (gmd);
}

/**
 * syncdaemon_filesystem_interface_get_metadata_async:
 */
void
syncdaemon_filesystem_interface_get_metadata_async (SyncdaemonFilesystemInterface *interface,
						    const gchar *path,
                                                    gboolean with_subtree_sync_check,
						    SyncdaemonGotMetadataFunc callback,
						    gpointer user_data)
{
	GotMetadataData *gmd;
        const gchar *method;

	g_return_if_fail (SYNCDAEMON_IS_FILESYSTEM_INTERFACE (interface));

        if (with_subtree_sync_check)
                method = "get_metadata_and_quick_tree_synced";
        else
                method = "get_metadata";

	gmd = g_new0 (GotMetadataData, 1);
	gmd->interface = interface;
	gmd->callback = callback;
	gmd->user_data = user_data;

	dbus_g_proxy_begin_call (DBUS_G_PROXY (interface->priv->proxy),
                                 method,
				 got_metadata_cb, gmd, NULL,
				 G_TYPE_STRING, path,
				 G_TYPE_INVALID);
}
