/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "syncdaemon-file-info.h"

G_DEFINE_TYPE(SyncdaemonFileInfo, syncdaemon_file_info, G_TYPE_OBJECT)

struct _SyncdaemonFileInfoPrivate {
	gchar *volume_id;
	gchar *node_id;
	gchar *public_url;
	gchar *path;
	gboolean is_public;
};

static void
syncdaemon_file_info_finalize (GObject *object)
{
	SyncdaemonFileInfo *finfo = SYNCDAEMON_FILE_INFO (object);

	if (finfo->priv != NULL) {
		if (finfo->priv->volume_id != NULL)
			g_free (finfo->priv->volume_id);

		if (finfo->priv->node_id != NULL)
			g_free (finfo->priv->node_id);

		if (finfo->priv->public_url != NULL)
			g_free (finfo->priv->public_url);

		if (finfo->priv->path != NULL)
			g_free (finfo->priv->path);

		g_free (finfo->priv);
	}

	G_OBJECT_CLASS (syncdaemon_file_info_parent_class)->finalize (object);
}

static void
syncdaemon_file_info_class_init (SyncdaemonFileInfoClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_file_info_finalize;
}

static void
syncdaemon_file_info_init (SyncdaemonFileInfo *finfo)
{
	finfo->priv = g_new0 (SyncdaemonFileInfoPrivate, 1);
}

/**
 * syncdaemon_file_info_new:
 */
SyncdaemonFileInfo *
syncdaemon_file_info_new (void)
{
	return g_object_new (SYNCDAEMON_TYPE_FILE_INFO, NULL);
}

/**
 * syncdaemon_file_info_new_from_hash_table:
 */
SyncdaemonFileInfo *
syncdaemon_file_info_new_from_hash_table (GHashTable *hash)
{
	SyncdaemonFileInfo *finfo;

	finfo = g_object_new (SYNCDAEMON_TYPE_FILE_INFO, NULL);
	if (hash != NULL) {
		syncdaemon_file_info_set_volume_id (finfo, g_hash_table_lookup (hash, "volume_id"));
		syncdaemon_file_info_set_node_id (finfo, g_hash_table_lookup (hash, "node_id"));
		syncdaemon_file_info_set_public_url (finfo, g_hash_table_lookup (hash, "public_url"));
		syncdaemon_file_info_set_path (finfo, g_hash_table_lookup (hash, "path"));
		syncdaemon_file_info_set_is_public (
			finfo,
			g_strcmp0 (g_hash_table_lookup (hash, "is_public"), "True") == 0);			    
	}

	return finfo;
}

/**
 * syncdaemon_file_info_get_volume_id:
 */
const gchar *
syncdaemon_file_info_get_volume_id (SyncdaemonFileInfo *finfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo), NULL);

	return (const gchar *) finfo->priv->volume_id;
}

/**
 * syncdaemon_file_info_set_volume_id:
 */
void
syncdaemon_file_info_set_volume_id (SyncdaemonFileInfo *finfo, const gchar *volume_id)
{
	g_return_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo));

	if (finfo->priv->volume_id != NULL)
		g_free (finfo->priv->volume_id);

	finfo->priv->volume_id = g_strdup (volume_id);
}

/**
 * syncdaemon_file_info_get_node_id:
 */
const gchar *
syncdaemon_file_info_get_node_id (SyncdaemonFileInfo *finfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo), NULL);

	return (const gchar *) finfo->priv->node_id;
}

/**
 * syncdaemon_file_info_set_node_id:
 */
void
syncdaemon_file_info_set_node_id (SyncdaemonFileInfo *finfo, const gchar *node_id)
{
	g_return_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo));

	if (finfo->priv->node_id != NULL)
		g_free (finfo->priv->node_id);

	finfo->priv->node_id = g_strdup (node_id);
}

/**
 * syncdaemon_file_info_get_public_url:
 */
const gchar *
syncdaemon_file_info_get_public_url (SyncdaemonFileInfo *finfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo), NULL);

	return (const gchar *) finfo->priv->public_url;
}

/**
 * syncdaemon_file_info_set_public_url:
 */
void
syncdaemon_file_info_set_public_url (SyncdaemonFileInfo *finfo, const gchar *public_url)
{
	g_return_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo));

	if (finfo->priv->public_url != NULL)
		g_free (finfo->priv->public_url);

	finfo->priv->public_url = g_strdup (public_url);
}

/**
 * syncdaemon_file_info_get_path:
 */
const gchar *
syncdaemon_file_info_get_path (SyncdaemonFileInfo *finfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo), NULL);

	return (const gchar *) finfo->priv->path;
}

/**
 * syncdaemon_file_info_set_path:
 */
void
syncdaemon_file_info_set_path (SyncdaemonFileInfo *finfo, const gchar *path)
{
	g_return_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo));

	if (finfo->priv->path != NULL)
		g_free (finfo->priv->path);

	finfo->priv->path = g_strdup (path);
}

/**
 * syncdaemon_file_info_get_is_public:
 */
gboolean
syncdaemon_file_info_get_is_public (SyncdaemonFileInfo *finfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo), FALSE);

	return finfo->priv->is_public;
}

/**
 * syncdaemon_file_info_set_is_public:
 */
void
syncdaemon_file_info_set_is_public (SyncdaemonFileInfo *finfo, gboolean is_public)
{
	g_return_if_fail (SYNCDAEMON_IS_FILE_INFO (finfo));

	finfo->priv->is_public = is_public;
}
