#
# Authors: Manuel de la Pena <manuel@canonical.com>
#          Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Test the filesystem notifications on windows."""

import os
import tempfile
import thread
import time
import itertools

from twisted.internet import defer

from contrib.testing.testcase import BaseTwistedTestCase
from ubuntuone.platform.windows import os_helper
from ubuntuone.platform.windows.pyinotify import (
    ProcessEvent,
    IN_OPEN,
    IN_CLOSE_WRITE
)
from ubuntuone.platform.windows import filesystem_notifications
from ubuntuone.platform.windows.filesystem_notifications import (
    FilesystemMonitor,
    NotifyProcessor,
    Watch,
    WatchManager,
    FILE_NOTIFY_CHANGE_FILE_NAME,
    FILE_NOTIFY_CHANGE_DIR_NAME,
    FILE_NOTIFY_CHANGE_ATTRIBUTES,
    FILE_NOTIFY_CHANGE_SIZE,
    FILE_NOTIFY_CHANGE_LAST_WRITE,
    FILE_NOTIFY_CHANGE_SECURITY,
    FILE_NOTIFY_CHANGE_LAST_ACCESS,
)


class FakeException(Exception):
    """A fake Exception used in tests."""


class TestCaseHandler(ProcessEvent):
    """ProcessEvent used for test cases."""

    thread_id = None

    def my_init(self, main_thread_id=None, number_events=None, **kwargs):
        """Init the event notifier."""
        self.processed_events = []
        self.main_thread_id = main_thread_id
        self.deferred = defer.Deferred()
        assert number_events is not None
        self.number_events = number_events

    def append_event(self, event):
        self.processed_events.append(event)
        if len(self.processed_events) == self.number_events:
            self.deferred.callback(self.processed_events)

    def process_IN_CREATE(self, event):
        """Process the event and add it to the list."""
        self.append_event(event)
        self._verify_thread_id()

    def process_IN_DELETE(self, event):
        """Process the event and add it to the list."""
        self.append_event(event)
        self._verify_thread_id()

    def process_default(self, event):
        """Process the event and add it to the list."""
        self.append_event(event)
        self._verify_thread_id()

    def _verify_thread_id(self):
        """Verify that the event was processed in the correct thread."""
        if self.main_thread_id:
            assert self.main_thread_id == thread.get_ident()


class TestWatch(BaseTwistedTestCase):
    """Test the watch so that it returns the same events as pyinotify."""

    timeout = 5

    def setUp(self):
        """Set infor for the tests."""
        super(TestWatch, self).setUp()
        self.basedir = self.mktemp('test_root')
        self.mask = FILE_NOTIFY_CHANGE_FILE_NAME | \
            FILE_NOTIFY_CHANGE_DIR_NAME | \
            FILE_NOTIFY_CHANGE_ATTRIBUTES | \
            FILE_NOTIFY_CHANGE_SIZE | \
            FILE_NOTIFY_CHANGE_LAST_WRITE | \
            FILE_NOTIFY_CHANGE_SECURITY | \
            FILE_NOTIFY_CHANGE_LAST_ACCESS


    @defer.inlineCallbacks
    def _perform_operations(self, path, mask, auto_add, actions, number_events):
        """Perform the file operations and returns the recorded events."""
        handler = TestCaseHandler(number_events=number_events)
        manager = WatchManager(handler)
        yield manager.add_watch(os_helper.get_windows_valid_path(path), mask,
                          auto_add=auto_add)
        # execution the actions
        actions()
        # process the recorded events
        ret = yield handler.deferred
        self.addCleanup(manager.stop)
        defer.returnValue(ret)

    def _perform_timed_operations(self, path, mask, auto_add, actions,
                                  time_out):
        """Perform the file operations and returns the recorded events."""
        manager = WatchManager()
        manager.add_watch(os_helper.get_windows_valid_path(path), mask,
                          auto_add=auto_add)
        # execution the actions
        actions()
        # process the recorded events
        time.sleep(time_out)
        events = self.handler.processed_events
        return events

    @defer.inlineCallbacks
    def test_file_create(self):
        """Test that the correct event is returned on a file create."""
        file_name = os.path.join(self.basedir, 'test_file_create')

        def create_file():
            """Action used for the test."""
            # simply create a new file
            fd = open(file_name, 'w')
            fd.flush()
            os.fsync(fd)
            fd.close()

        events = yield self._perform_operations(self.basedir, self.mask, False,
            create_file, 1)
        event = events[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x100, event.mask)
        self.assertEqual('IN_CREATE', event.maskname)
        self.assertEqual(os.path.split(file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, file_name), event.pathname)
        self.assertEqual(0, event.wd)

    @defer.inlineCallbacks
    def test_dir_create(self):
        """Test that the correct event is returned on a dir creation."""
        dir_name = os.path.join(self.basedir, 'test_dir_create')

        def create_dir():
            """Action for the test."""
            os.mkdir(dir_name)

        events = yield self._perform_operations(self.basedir, self.mask, False,
            create_dir, 1)
        event = events[0]
        self.assertTrue(event.dir)
        self.assertEqual(0x40000100, event.mask)
        self.assertEqual('IN_CREATE|IN_ISDIR', event.maskname)
        self.assertEqual(os.path.split(dir_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, dir_name), event.pathname)
        self.assertEqual(0, event.wd)

    @defer.inlineCallbacks
    def test_file_remove(self):
        """Test that the correct event is raised when a file is removed."""
        file_name = os.path.join(self.basedir, 'test_file_remove')
        # create the file before recording
        open(file_name, 'w').close()

        def remove_file():
            """Action for the test."""
            os.remove(file_name)

        events = yield self._perform_operations(self.basedir, self.mask, False,
                                                remove_file, 1)
        event = events[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x200, event.mask)
        self.assertEqual('IN_DELETE', event.maskname)
        self.assertEqual(os.path.split(file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, file_name), event.pathname)
        self.assertEqual(0, event.wd)

    @defer.inlineCallbacks
    def test_dir_remove(self):
        """Test that the correct event is raised when a dir is removed."""
        dir_name = os.path.join(self.basedir, 'test_dir_remove')
        # create the dir before recording
        os.mkdir(dir_name)

        def remove_dir():
            """Action for the test."""
            os.rmdir(dir_name)

        events = yield self._perform_operations(self.basedir, self.mask, False,
                                                remove_dir, 1)
        event = events[0]
        self.assertTrue(event.dir)
        self.assertEqual(0x40000200, event.mask)
        self.assertEqual('IN_DELETE|IN_ISDIR', event.maskname)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, dir_name), event.pathname)
        self.assertEqual(0, event.wd)

    @defer.inlineCallbacks
    def test_file_write(self):
        """Test that the correct event is raised when a file is written."""
        file_name = os.path.join(self.basedir, 'test_file_write')
        # create the file before recording
        fd = open(file_name, 'w')
        def write_file():
            """Action for the test."""
            fd.write('test')
            fd.close()

        events = yield self._perform_operations(self.basedir, self.mask, False,
                                                write_file, 1)
        event = events[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x2, event.mask)
        self.assertEqual('IN_MODIFY', event.maskname)
        self.assertEqual(os.path.split(file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, file_name), event.pathname)
        self.assertEqual(0, event.wd)
        # clean behind us by removeing the file
        os.remove(file_name)

    @defer.inlineCallbacks
    def test_file_moved_to_watched_dir_same_watcher(self):
        """Test that the correct event is raised when a file is moved."""
        from_file_name = os.path.join(self.basedir,
            'test_file_moved_to_watched_dir_same_watcher')
        to_file_name = os.path.join(self.basedir,
            'test_file_moved_to_watched_dir_same_watcher_2')
        open(from_file_name, 'w').close()
        #create file before recording

        def move_file():
            """Action for the test."""
            os.rename(from_file_name, to_file_name)

        events = yield self._perform_operations(self.basedir, self.mask,
                                                False, move_file, 2)
        move_from_event = events[0]
        move_to_event = events[1]
        # first test the move from
        self.assertFalse(move_from_event.dir)
        self.assertEqual(0x40, move_from_event.mask)
        self.assertEqual('IN_MOVED_FROM', move_from_event.maskname)
        self.assertEqual(os.path.split(from_file_name)[1], move_from_event.name)
        self.assertEqual('.', move_from_event.path)
        self.assertEqual(os.path.join(self.basedir, from_file_name),
            move_from_event.pathname)
        self.assertEqual(0, move_from_event.wd)
        # test the move to
        self.assertFalse(move_to_event.dir)
        self.assertEqual(0x80, move_to_event.mask)
        self.assertEqual('IN_MOVED_TO', move_to_event.maskname)
        self.assertEqual(os.path.split(to_file_name)[1], move_to_event.name)
        self.assertEqual('.', move_to_event.path)
        self.assertEqual(os.path.join(self.basedir, to_file_name),
            move_to_event.pathname)
        self.assertEqual(os.path.split(from_file_name)[1],
            move_to_event.src_pathname)
        self.assertEqual(0, move_to_event.wd)
        # assert that both cookies are the same
        self.assertEqual(move_from_event.cookie, move_to_event.cookie)

    @defer.inlineCallbacks
    def test_file_moved_to_not_watched_dir(self):
        """Test that the correct event is raised when a file is moved."""
        from_file_name = os.path.join(self.basedir,
            'test_file_moved_to_not_watched_dir')
        open(from_file_name, 'w').close()

        def move_file():
            """Action for the test."""
            os.rename(from_file_name, os.path.join(tempfile.mkdtemp(),
                'test_file_moved_to_not_watched_dir'))

        # while on linux we will have to do some sort of magic like facundo
        # did, on windows we will get a deleted event which is much more
        # cleaner, first time ever windows is nicer!
        events = yield self._perform_operations(self.basedir, self.mask, False,
                                                move_file, 1)
        event = events[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x200, event.mask)
        self.assertEqual('IN_DELETE', event.maskname)
        self.assertEqual(os.path.split(from_file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, from_file_name), event.pathname)
        self.assertEqual(0, event.wd)

    @defer.inlineCallbacks
    def test_file_move_from_not_watched_dir(self):
        """Test that the correct event is raised when a file is moved."""
        from_file_name = os.path.join(tempfile.mkdtemp(),
            'test_file_move_from_not_watched_dir')
        to_file_name = os.path.join(self.basedir,
            'test_file_move_from_not_watched_dir')
        # create file before we record
        open(from_file_name, 'w').close()

        def move_files():
            """Action for the test."""
            os.rename(from_file_name, to_file_name)

        # while on linux we have to do some magic operations like facundo did
        # on windows we have a created file, hurray!
        events = yield self._perform_operations(self.basedir, self.mask, False,
                                                move_files, 1)
        event = events[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x100, event.mask)
        self.assertEqual('IN_CREATE', event.maskname)
        self.assertEqual(os.path.split(to_file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, to_file_name),
            event.pathname)
        self.assertEqual(0, event.wd)

    @defer.inlineCallbacks
    def test_dir_moved_to_watched_dir_same_watcher(self):
        """Test that the correct event is raised when a dir is moved."""
        from_dir_name = os.path.join(self.basedir,
            'test_dir_moved_to_watched_dir_same_watcher')
        to_dir_name = os.path.join(self.basedir,
            'test_dir_moved_to_watched_dir_same_watcher_2')
        os.mkdir(from_dir_name)

        def move_file():
            """Action for the test."""
            os.rename(from_dir_name, to_dir_name)

        events = yield self._perform_operations(self.basedir,
            self.mask, False, move_file, 2)
        move_from_event = events[0]
        move_to_event = events[1]
        # first test the move from
        self.assertTrue(move_from_event.dir)
        self.assertEqual(0x40000040, move_from_event.mask)
        self.assertEqual('IN_MOVED_FROM|IN_ISDIR', move_from_event.maskname)
        self.assertEqual(os.path.split(from_dir_name)[1], move_from_event.name)
        self.assertEqual('.', move_from_event.path)
        self.assertEqual(os.path.join(self.basedir, from_dir_name),
            move_from_event.pathname)
        self.assertEqual(0, move_from_event.wd)
        # test the move to
        self.assertTrue(move_to_event.dir)
        self.assertEqual(0x40000080, move_to_event.mask)
        self.assertEqual('IN_MOVED_TO|IN_ISDIR', move_to_event.maskname)
        self.assertEqual(os.path.split(to_dir_name)[1], move_to_event.name)
        self.assertEqual('.', move_to_event.path)
        self.assertEqual(os.path.join(self.basedir, to_dir_name),
            move_to_event.pathname)
        self.assertEqual(os.path.split(from_dir_name)[1], move_to_event.src_pathname)
        self.assertEqual(0, move_to_event.wd)
        # assert that both cookies are the same
        self.assertEqual(move_from_event.cookie, move_to_event.cookie)

    @defer.inlineCallbacks
    def test_dir_moved_to_not_watched_dir(self):
        """Test that the correct event is raised when a file is moved."""
        dir_name = os.path.join(self.basedir,
            'test_dir_moved_to_not_watched_dir')
        os.mkdir(dir_name)

        def move_dir():
            """Action for the test."""
            os.rename(dir_name, os.path.join(tempfile.mkdtemp(),
                'test_dir_moved_to_not_watched_dir'))

        # on windows a move to outside a watched dir translates to a remove
        events = yield self._perform_operations(self.basedir, self.mask, False,
                                                move_dir, 1)
        event = events[0]
        self.assertTrue(event.dir)
        self.assertEqual(0x40000200, event.mask)
        self.assertEqual('IN_DELETE|IN_ISDIR', event.maskname)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, dir_name), event.pathname)
        self.assertEqual(0, event.wd)

    @defer.inlineCallbacks
    def test_dir_move_from_not_watched_dir(self):
        """Test that the correct event is raised when a file is moved."""
        from_dir_name = os.path.join(tempfile.mkdtemp(),
            'test_dir_move_from_not_watched_dir')
        to_dir_name = os.path.join(self.basedir,
            'test_dir_move_from_not_watched_dir')
        # create file before we record
        os.mkdir(from_dir_name)

        def move_dir():
            """Action for the test."""
            os.rename(from_dir_name, to_dir_name)

        events = yield self._perform_operations(self.basedir, self.mask, False,
                                                move_dir, 1)
        event = events[0]
        self.assertTrue(event.dir)
        self.assertEqual(0x40000100, event.mask)
        self.assertEqual('IN_CREATE|IN_ISDIR', event.maskname)
        self.assertEqual(os.path.split(from_dir_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, to_dir_name), event.pathname)
        self.assertEqual(0, event.wd)

    def test_exclude_filter(self):
        """Test that the exclude filter works as expectd."""
        handler = TestCaseHandler(number_events=0)
        manager = WatchManager(handler)
        # add a watch that will always exclude all actions
        manager.add_watch(os_helper.get_windows_valid_path(self.basedir),
                          self.mask, auto_add=True,
                          exclude_filter=lambda x: True )
        # execution the actions
        file_name = os.path.join(self.basedir, 'test_file_create')
        open(file_name, 'w').close()
        # give some time for the system to get the events
        time.sleep(1)
        self.assertEqual(0, len(handler.processed_events))
    test_exclude_filter.skip = "we must rethink this test."

    def test_open_dir_muted(self):
        """Test that the opening of dirs is ignored."""
        dir_name = os.path.join(tempfile.mkdtemp(), 'test_dir_open')
        # create file before we record
        os.mkdir(dir_name)

        def open_dir():
            """Action for the test."""
            os.startfile(dir_name)

        events = self._perform_timed_operations(self.basedir, self.mask, False,
            open_dir, 2)
        self.assertEqual(0, len(events))
    test_open_dir_muted.skip = "we must rethink this test."

    def test_ignore_path(self):
        """Test that events from a path are ignored."""
        events = []

        def fake_processor(event):
            """Memorize the processed events."""
            events.append(event)

        path = u'\\\\?\\C:\\path'  # a valid windows path
        child = u'child'
        watch = Watch(1, path, None, True, fake_processor)
        watch.ignore_path(os.path.join(path, child))
        paths_to_ignore = []
        for file_name in 'abcdef':
            paths_to_ignore.append((1, os.path.join(child, file_name)))
        # ensure that the watch is watching
        watch._watching = True
        watch._process_events(paths_to_ignore)
        self.assertEqual(0, len(events),
                         'All events should have been ignored.')

    def test_not_ignore_path(self):
        """Test that we do get the events when they do not match."""
        events = []

        def fake_processor(event):
            """Memorize the processed events."""
            events.append(event)

        path = u'\\\\?\\C:\\path'  # a valid windows path
        child = u'child'
        watch = Watch(1, path, None, True, fake_processor)
        watch.ignore_path(os.path.join(path, child))
        paths_not_to_ignore = []
        for file_name in 'abcdef':
            paths_not_to_ignore.append((1, os.path.join(
                                                child + file_name, file_name)))
        # ensure that the watch is watching
        watch._watching = True
        watch._process_events(paths_not_to_ignore)
        self.assertEqual(len(paths_not_to_ignore), len(events),
                         'No events should have been ignored.')

    def test_mixed_ignore_path(self):
        """Test that we do get the correct events."""
        events = []

        def fake_processor(event):
            """Memorize the processed events."""
            events.append(event.pathname)

        child = u'child'
        path = u'\\\\?\\C:\\path\\'  # a valid windows path
        watch = Watch(1, path, None, True, fake_processor)
        watch.ignore_path(os.path.join(path, child))
        paths_not_to_ignore = []
        paths_to_ignore = []
        expected_events = []
        for file_name in 'abcdef':
            valid = os.path.join(child + file_name, file_name)
            paths_to_ignore.append((1, os.path.join(child, file_name)))
            paths_not_to_ignore.append((1, valid))
            expected_events.append(os.path.join('C:\\path', valid))
        # ensure that the watch is watching
        watch._watching = True
        watch._process_events(paths_not_to_ignore)
        self.assertEqual(len(paths_not_to_ignore), len(events),
                         'Wrong number of events ignored.')
        self.assertTrue(all([event in expected_events for event in events]),
                        'Paths ignored that should have not been ignored.')

    def test_undo_ignore_path_ignored(self):
        """Test that we do deal with events from and old ignored path."""
        events = []

        def fake_processor(event):
            """Memorize the processed events."""
            events.append(event)

        path = u'\\\\?\\C:\\path'  # a valid windows path
        child = u'child'
        watch = Watch(1, path, None, True, fake_processor)
        watch.ignore_path(os.path.join(path, child))
        watch.remove_ignored_path(os.path.join(path, child))
        paths_not_to_ignore = []
        for file_name in 'abcdef':
            paths_not_to_ignore.append((1, os.path.join(child, file_name)))
        # ensure that the watch is watching
        watch._watching = True
        watch._process_events(paths_not_to_ignore)
        self.assertEqual(len(paths_not_to_ignore), len(events),
                         'All events should have been accepted.')

    def test_undo_ignore_path_other_ignored(self):
        """Test that we can undo and the other path is ignored."""
        events = []

        def fake_processor(event):
            """Memorize the processed events."""
            events.append(event.pathname)

        path = u'\\\\?\\C:\\path'  # a valid windows path
        child_a = u'childa'
        child_b = u'childb'
        watch = Watch(1, path, None, True, fake_processor)
        watch.ignore_path(os.path.join(path, child_a))
        watch.ignore_path(os.path.join(path, child_b))
        watch.remove_ignored_path(os.path.join(path, child_a))
        paths_to_ignore = []
        paths_not_to_ignore = []
        expected_events = []
        for file_name in 'abcdef':
            paths_to_ignore.append((1, os.path.join(child_b, file_name)))
            valid = os.path.join(child_a, file_name)
            paths_not_to_ignore.append((1, valid))
            expected_events.append(os.path.join('C:\\path', valid))
        # ensure that the watch is watching
        watch._watching = True
        watch._process_events(paths_not_to_ignore)
        self.assertEqual(len(paths_not_to_ignore), len(events),
                         'All events should have been accepted.')
        self.assertTrue(all([event in expected_events for event in events]),
                        'Paths ignored that should have not been ignored.')

    @defer.inlineCallbacks
    def test_call_deferred_already_called(self):
        """Test that the function is not called."""
        method_args = []

        def fake_call(*args, **kwargs):
            """Execute the call."""
            method_args.append((args, kwargs),)

        path = u'\\\\?\\C:\\path'  # a valid windows path
        watch = Watch(1, path, None, True, None)
        yield watch._watch_started_deferred.callback(True)
        watch._call_deferred(fake_call, None)
        self.assertEqual(0, len(method_args))

    def test_call_deferred_not_called(self):
        """Test that is indeed called."""
        method_args = []

        def fake_call(*args, **kwargs):
            """Execute the call."""
            method_args.append((args, kwargs),)

        path = u'\\\\?\\C:\\path'  # a valid windows path
        watch = Watch(1, path, None, True, None)
        watch._call_deferred(fake_call, None)
        self.assertEqual(1, len(method_args))

    def test_started_property(self):
        """Test that the started property returns the started deferred."""
        path = u'\\\\?\\C:\\path'  # a valid windows path
        watch = Watch(1, path, None, True, None)
        self.assertEqual(watch.started, watch._watch_started_deferred)

    def test_stopped_property(self):
        """Test that the stopped property returns the stopped deferred."""
        path = u'\\\\?\\C:\\path'  # a valid windows path
        watch = Watch(1, path, None, True, None)
        self.assertEqual(watch.stopped, watch._watch_stopped_deferred)

    def random_error(self, *args):
        """Throw a fake exception."""
        raise FakeException()

    @defer.inlineCallbacks
    def test_start_watching_fails_early_in_thread(self):
        """An early failure inside the thread should errback the deferred."""
        test_path = self.mktemp("test_directory")
        self.patch(filesystem_notifications, "CreateFileW", self.random_error)
        watch = Watch(1, test_path, None, True, None)
        d = watch.start_watching()
        yield self.assertFailure(d, FakeException)

    @defer.inlineCallbacks
    def test_start_watching_fails_late_in_thread(self):
        """A late failure inside the thread should errback the deferred."""
        test_path = self.mktemp("test_directory")
        self.patch(filesystem_notifications, "ReadDirectoryChangesW",
                   self.random_error)
        watch = Watch(1, test_path, None, True, None)
        d = watch.start_watching()
        yield self.assertFailure(d, FakeException)

    @defer.inlineCallbacks
    def test_close_handle_is_called_on_error(self):
        """CloseHandle is called when there's an error in the watch thread."""
        test_path = self.mktemp("test_directory")
        close_called = []
        self.patch(filesystem_notifications, "CreateFileW", lambda *_: None)
        self.patch(filesystem_notifications, "CloseHandle",
                   close_called.append)
        self.patch(filesystem_notifications, "ReadDirectoryChangesW",
                   self.random_error)
        watch = Watch(1, test_path, self.mask, True, None)
        d = watch.start_watching()
        yield self.assertFailure(d, FakeException)
        self.assertEqual(len(close_called), 1)
        yield watch.stop_watching()

    @defer.inlineCallbacks
    def test_stop_watching_fired_when_watch_thread_finishes(self):
        """The deferred returned is fired when the watch thread finishes."""
        events = []
        test_path = self.mktemp("another_test_directory")
        fake_processor = events.append
        watch = Watch(1, test_path, self.mask, True, fake_processor)
        yield watch.start_watching()
        self.assertNotEqual(watch._watch_handle, None)
        yield watch.stop_watching()
        self.assertEqual(watch._watch_handle, None)


class TestWatchManager(BaseTwistedTestCase):
    """Test the watch manager."""

    def setUp(self):
        """Set each of the tests."""
        super(TestWatchManager, self).setUp()
        self.parent_path = u'\\\\?\\C:\\'  # a valid windows path
        self.path = self.parent_path + u'path'
        self.watch = Watch(1, self.path, None, True, None)
        self.manager = WatchManager(None)
        self.manager._wdm = {1: self.watch}

    @defer.inlineCallbacks
    def test_stop(self):
        """Test that the different watches are stopped."""
        self.was_called = False

        def fake_stop_watching(watch):
            """Fake stop watch."""
            self.was_called = True
            return defer.succeed(True)

        self.patch(Watch, "stop_watching", fake_stop_watching)
        yield self.manager.stop()
        self.assertTrue(self.was_called, 'The watch stop should be called.')

    @defer.inlineCallbacks
    def test_stop_multiple(self):
        """Test that stop is fired when *all* watches have stopped."""

        def fake_stop_watching(watch):
            """Another fake stop watch."""
            return watch.stopped

        self.patch(Watch, "stop_watching", fake_stop_watching)
        second_path = self.parent_path + u"second_path"
        second_watch = Watch(2, second_path, None, True, None)
        self.manager._wdm[2] = second_watch
        d = self.manager.stop()
        self.assertFalse(d.called, "Not fired before all watches end")
        self.watch.stopped.callback(None)
        self.assertFalse(d.called, "Not fired before all watches end")
        second_watch.stopped.callback(None)
        yield d
        self.assertTrue(d.called, "Fired after the watches ended")

    def test_get_present_watch(self):
        """Test that we can get a Watch using is wd."""
        self.assertEqual(self.watch, self.manager.get_watch(1))

    def test_get_missing_watch(self):
        """Test that we get an error when trying to get a missing wd."""
        self.assertRaises(KeyError, self.manager.get_watch, (1,))

    @defer.inlineCallbacks
    def test_delete_present_watch(self):
        """Test that we can remove a present watch."""
        self.was_called = False

        def stop_watching():
            """Fake stop watch."""
            self.was_called = True
            return defer.succeed(True)

        self.watch.stop_watching = stop_watching
        yield self.manager.del_watch(1)
        self.assertRaises(KeyError, self.manager.get_watch, (1,))

    def test_add_single_watch(self):
        """Test the addition of a new single watch."""
        self.was_called = False

        def fake_start_watching(*args):
            """Fake start watch."""
            self.was_called = True

        self.patch(Watch, "start_watching", fake_start_watching)
        self.manager._wdm = {}

        mask = 'bit_mask'
        auto_add = True
        self.manager.add_watch(self.path, mask, auto_add=auto_add)
        self.assertEqual(1, len(self.manager._wdm))
        self.assertTrue(self.was_called, 'The watch start was not called.')
        self.assertEqual(self.path + os.path.sep, self.manager._wdm[0]._path)
        self.assertEqual(mask, self.manager._wdm[0]._mask)
        self.assertEqual(auto_add, self.manager._wdm[0]._auto_add)

    def test_update_present_watch(self):
        """Test the update of a present watch."""
        mask = 'mask'
        self.assertRaises(NotImplementedError, self.manager.update_watch,
                                                                 1, mask)

    def test_get_watch_present_wd(self):
        """Test that the correct path is returned."""
        self.assertEqual(self.path + os.path.sep, self.manager.get_path(1))

    def test_get_watch_missing_wd(self):
        """Test that the correct path is returned."""
        self.manager._wdm = {}
        self.assertEqual(None, self.manager.get_path(1))

    def test_get_wd_exact_path(self):
        """Test the wd is returned when there is a watch for the path."""
        self.assertEqual(1, self.manager.get_wd(self.path))

    def test_get_wd_child_path(self):
        """Test the wd is returned when we have a child path."""
        child = os.path.join(self.path, 'child')
        self.assertEqual(1, self.manager.get_wd(child))

    def test_get_wd_unwatched(self):
        """A watch on an unwatched path returns None."""
        self.assertEqual(None, self.manager.get_wd(self.parent_path))

    @defer.inlineCallbacks
    def test_rm_present_wd(self):
        """Test the removal of a present watch."""

        def fake_stop_watching():
            """Fake stop watch."""
            return defer.succeed(True)

        self.patch(self.watch, "stop_watching", fake_stop_watching)
        yield self.manager.rm_watch(1)
        self.assertEqual(None, self.manager._wdm.get(1))

    def test_rm_root_path(self):
        """Test the removal of a root path."""
        events = []

        def fake_processor(event):
            """Memorize the processed events."""
            events.append(event.pathname)

        self.watch._processor = fake_processor
        self.manager.rm_path(self.path)
        self.assertEqual(self.watch, self.manager._wdm.get(1))
        self.watch._watching = True
        self.watch._process_events([(1, os.path.join(self.path, 'test'))])
        self.assertEqual(0, len(events))

    def test_rm_child_path(self):
        """Test the removal of a child path."""
        events = []

        def fake_processor(event):
            """Memorize the processed events."""
            events.append(event.pathname)

        self.watch._processor = fake_processor
        child = os.path.join(self.path, u'child')
        self.manager.rm_path(child)
        self.assertEqual(self.watch, self.manager._wdm[1])
        # assert that the correct event is ignored
        self.watch._watching = True
        self.watch._process_events([(1, os.path.join('child', 'test'))])
        self.assertEqual(0, len(events))
        # assert that other events are not ignored
        self.watch._process_events([(1, 'test')])
        self.assertEqual(1, len(events))


class TestWatchManagerAddWatches(BaseTwistedTestCase):
    """Test the watch manager."""
    timeout = 5

    def test_add_watch_twice(self):
        """Adding a watch twice succeeds when the watch is running."""
        self.patch(Watch, "start_watching", lambda self: self.started)
        manager = WatchManager(None)
        # no need to stop watching because start_watching is fake

        path = u'\\\\?\\C:\\test'  # a valid windows path
        mask = 'fake bit mask'
        d1 = manager.add_watch(path, mask)
        d2 = manager.add_watch(path, mask)

        self.assertFalse(d1.called, "Should not be called yet.")
        self.assertFalse(d2.called, "Should not be called yet.")

        manager._wdm.values()[0].started.callback(True)

        self.assertTrue(d1.called, "Should already be called.")
        self.assertTrue(d2.called, "Should already be called.")


class FakeEvent(object):
    """Fake event."""

    def __init__(self, wd=0, dir=True, name=None, path=None, pathname=None,
                 cookie=None):
        """Create fake event."""
        self.dir = dir
        self.wd = wd
        self.name = name
        self.path = path
        self.pathname = pathname
        self.cookie = cookie


class FakeLog(object):
    """A fake log that is used by the general processor."""

    def __init__(self):
        """Create the fake."""
        self.called_methods = []

    def info(self, *args):
        """Fake the info call."""
        self.called_methods.append(('info', args))


class FakeGeneralProcessor(object):
    """Fake implementation of the general processor."""

    def __init__(self):
        """Create the fake."""
        self.called_methods = []
        self.paths_to_return = []
        self.log = FakeLog()
        self.share_id = None
        self.ignore = False

    def rm_from_mute_filter(self, event, paths):
        """Fake rm_from_mute_filter."""
        self.called_methods.append(('rm_from_mute_filter', event, paths))

    def add_to_mute_filter(self, event, paths):
        """Fake add_to_move_filter."""
        self.called_methods.append(('add_to_mute_filter', event, paths))

    def is_ignored(self, path):
        """Fake is_ignored."""
        self.called_methods.append(('is_ignored', path))
        return self.ignore

    def push_event(self, event):
        """Fake push event."""
        self.called_methods.append(('push_event', event))

    def eq_push(self, event, path=None, path_to=None, path_from=None):
        """Fake event to push event."""
        self.called_methods.append(('eq_push', event, path, path_to,
                                    path_from))

    def get_paths_starting_with(self, fullpath, include_base=False):
        """Fake get_paths_starting_with."""
        self.called_methods.append(('get_paths_starting_with', fullpath,
                                    include_base))
        return self.paths_to_return

    def get_path_share_id(self, path):
        """Fake get_path_share_id."""
        self.called_methods.append(('get_path_share_id', path))
        return self.share_id

    def rm_watch(self, path):
        """Fake the remove watch."""
        self.called_methods.append(('rm_watch', path))

    def freeze_begin(self, path):
        """Fake freeze_begin"""
        self.called_methods.append(('freeze_begin', path))

    def freeze_rollback(self):
        """Fake rollback."""
        self.called_methods.append(('freeze_rollback',))

    def freeze_commit(self, path):
        """Fake freeze commit."""
        self.called_methods.append(('freeze_commit', path))


class TestNotifyProcessor(BaseTwistedTestCase):
    """Test the notify processor."""

    def setUp(self):
        """set up the diffeent tests."""
        super(TestNotifyProcessor, self).setUp()
        self.processor = NotifyProcessor(None)
        self.general = FakeGeneralProcessor()
        self.processor.general_processor = self.general

    def test_rm_from_mute_filter(self):
        """Test that we remove the event from the mute filter."""
        event = 'event'
        paths = 'paths'
        self.processor.rm_from_mute_filter(event, paths)
        self.assertEqual(1, len(self.general.called_methods))
        self.assertEqual('rm_from_mute_filter',
                         self.general.called_methods[0][0])
        self.assertEqual(event, self.general.called_methods[0][1])
        self.assertEqual(paths, self.general.called_methods[0][2])

    def test_add_to_mute_filter(self):
        """Test that we add the path to the mute filter."""
        event = 'event'
        paths = 'paths'
        self.processor.add_to_mute_filter(event, paths)
        self.assertEqual(1, len(self.general.called_methods))
        self.assertEqual('add_to_mute_filter',
                         self.general.called_methods[0][0])
        self.assertEqual(event, self.general.called_methods[0][1])
        self.assertEqual(paths, self.general.called_methods[0][2])

    def test_platform_is_ignored(self):
        """Test that we do indeed ignore the correct paths."""
        not_ignored = 'test'
        ignored = not_ignored + '.lnk'
        self.assertFalse(self.processor.platform_is_ignored(not_ignored),
                         'Only links should be ignored.')
        self.assertTrue(self.processor.platform_is_ignored(ignored),
                         'Links should be ignored.')

    def test_is_ignored(self):
        """Test that we do ensure that the path is ignored."""
        path = 'path'
        self.processor.is_ignored(path)
        self.assertEqual(1, len(self.general.called_methods))
        self.assertEqual('is_ignored',
                         self.general.called_methods[0][0])
        self.assertEqual(path, self.general.called_methods[0][1])

    def test_release_held_event(self):
        """Test that we do release the held event."""
        event = 'event'
        # set the held event to assert that is pushed
        self.processor.held_event = event
        self.processor.release_held_event()
        self.assertEqual('push_event',
                         self.general.called_methods[0][0])
        self.assertEqual(event, self.general.called_methods[0][1])

    def test_process_IN_MODIFY_dir(self):
        """Test that the modify works as exepcted with dirs."""
        event = FakeEvent(dir=True)
        self.processor.process_IN_MODIFY(event)
        # no method should be called
        self.assertEqual(0, len(self.general.called_methods))

    def test_process_IN_MODIFY_file(self):
        """Test that the modify works as expected with files."""
        event = FakeEvent(dir=False, wd=0, name='name',
                          path='path', pathname='pathname')
        self.processor.process_IN_MODIFY(event)
        # we should be getting two different method, and open and a close
        self.assertEqual(2, len(self.general.called_methods))
        self.assertEqual('push_event',
                         self.general.called_methods[0][0])
        self.assertEqual('push_event',
                         self.general.called_methods[1][0])
        self.assertEqual(event.dir, self.general.called_methods[0][1].dir)
        self.assertEqual(event.wd, self.general.called_methods[0][1].wd)
        self.assertEqual(event.name, self.general.called_methods[0][1].name)
        self.assertEqual(event.path, self.general.called_methods[0][1].path)
        self.assertEqual(event.pathname,
                         self.general.called_methods[0][1].pathname)
        self.assertEqual(IN_OPEN,
                         self.general.called_methods[0][1].mask)
        self.assertEqual(event.dir, self.general.called_methods[1][1].dir)
        self.assertEqual(event.wd, self.general.called_methods[1][1].wd)
        self.assertEqual(event.name, self.general.called_methods[1][1].name)
        self.assertEqual(event.path, self.general.called_methods[1][1].path)
        self.assertEqual(event.pathname,
                         self.general.called_methods[1][1].pathname)
        self.assertEqual(IN_CLOSE_WRITE,
                         self.general.called_methods[1][1].mask)

    def test_process_IN_MOVED_FROM(self):
        """Test that the in moved from works as expected."""
        event = FakeEvent(dir=False, wd=0, name='name',
                          path='path', pathname='pathname')
        self.processor.process_IN_MOVED_FROM(event)
        self.assertEqual(event, self.processor.held_event)

    def test_process_IN_MOVED_TO_dir(self):
        """Test that the in moved to works as expected."""
        event = FakeEvent(wd=0, dir=True, name='name', path='path',
                          pathname=os.path.join('test', 'pathname'),
                          cookie='cookie')
        held_event = FakeEvent(wd=0, dir=True, name='hname', path='hpath',
                               pathname=os.path.join('test', 'hpathname'),
                               cookie='cookie')
        self.general.share_id = 'my_share_id'
        self.processor.held_event = held_event
        self.processor.process_IN_MOVED_TO(event)
        self.assertEqual(5, len(self.general.called_methods))
        # assert that the ignores are called
        self.assertEqual('is_ignored', self.general.called_methods[0][0])
        self.assertEqual(held_event.pathname, self.general.called_methods[0][1])
        self.assertEqual('is_ignored', self.general.called_methods[1][0])
        self.assertEqual(event.pathname, self.general.called_methods[1][1])
        # assert that we do request the share_id
        self.assertEqual('get_path_share_id', self.general.called_methods[2][0])
        self.assertEqual(os.path.split(event.pathname)[0],
                         self.general.called_methods[2][1],
                         'Get the share_id for event')
        self.assertEqual('get_path_share_id', self.general.called_methods[3][0])
        self.assertEqual(os.path.split(held_event.pathname)[0],
                         self.general.called_methods[3][1],
                         'Get the share_id for held event.')

        self.assertEqual('eq_push', self.general.called_methods[4][0])
        self.assertEqual('FS_DIR_MOVE', self.general.called_methods[4][1])
        self.assertEqual(event.pathname, self.general.called_methods[4][3])
        self.assertEqual(held_event.pathname, self.general.called_methods[4][4])

    def test_process_IN_MOVED_TO_file(self):
        """Test that the in moved to works as expected."""
        event = FakeEvent(wd=0, dir=False, name='name', path='path',
                          pathname=os.path.join('test', 'pathname'),
                          cookie='cookie')
        held_event = FakeEvent(wd=0, dir=False, name='hname', path='hpath',
                               pathname=os.path.join('test', 'hpathname'),
                               cookie='cookie')
        self.general.share_id = 'my_share_id'
        self.processor.held_event = held_event
        self.processor.process_IN_MOVED_TO(event)
        self.assertEqual(5, len(self.general.called_methods))
        # assert that the ignores are called
        self.assertEqual('is_ignored', self.general.called_methods[0][0])
        self.assertEqual(held_event.pathname, self.general.called_methods[0][1])
        self.assertEqual('is_ignored', self.general.called_methods[1][0])
        self.assertEqual(event.pathname, self.general.called_methods[1][1])
        # assert that we do request the share_id
        self.assertEqual('get_path_share_id', self.general.called_methods[2][0])
        self.assertEqual(os.path.split(event.pathname)[0],
                         self.general.called_methods[2][1],
                         'Get the share_id for event')
        self.assertEqual('get_path_share_id', self.general.called_methods[3][0])
        self.assertEqual(os.path.split(held_event.pathname)[0],
                         self.general.called_methods[3][1],
                         'Get the share_id for held event.')

        self.assertEqual('eq_push', self.general.called_methods[4][0])
        self.assertEqual('FS_FILE_MOVE', self.general.called_methods[4][1])
        self.assertEqual(event.pathname, self.general.called_methods[4][3])
        self.assertEqual(held_event.pathname, self.general.called_methods[4][4])

    def test_fake_create_event_dir(self):
        """Test that the in moved to works as expected."""
        event = FakeEvent(wd=0, dir=True, name='name', path='path',
                          pathname='pathname')
        self.processor._fake_create_event(event)
        self.assertEqual(1, len(self.general.called_methods))
        self.assertEqual('eq_push', self.general.called_methods[0][0])
        self.assertEqual('FS_DIR_CREATE', self.general.called_methods[0][1])
        self.assertEqual(event.pathname, self.general.called_methods[0][2])

    def test_fake_create_event_file(self):
        """Test that the in moved to works as expected."""
        event = FakeEvent(wd=0, dir=False, name='name', path='path',
                          pathname='pathname')
        self.processor._fake_create_event(event)
        self.assertEqual(2, len(self.general.called_methods))
        self.assertEqual('eq_push', self.general.called_methods[0][0])
        self.assertEqual('FS_FILE_CREATE', self.general.called_methods[0][1])
        self.assertEqual(event.pathname, self.general.called_methods[0][2])
        self.assertEqual('eq_push', self.general.called_methods[1][0])
        self.assertEqual('FS_FILE_CLOSE_WRITE',
                         self.general.called_methods[1][1])
        self.assertEqual(event.pathname, self.general.called_methods[1][2])

    def test_fake_delete_create_event_dir(self):
        """Test that we do fake a delete and a later delete."""
        event = FakeEvent(wd=0, dir=True, name='name', path='path',
                          pathname='pathname')
        held_event = FakeEvent(wd=0, dir=True, name='hname', path='hpath',
                               pathname='hpathname')
        self.processor.held_event = held_event
        self.processor._fake_delete_create_event(event)
        self.assertEqual(2, len(self.general.called_methods))
        self.assertEqual('eq_push', self.general.called_methods[0][0])
        self.assertEqual('FS_DIR_DELETE', self.general.called_methods[0][1])
        self.assertEqual(held_event.pathname, self.general.called_methods[0][2])
        self.assertEqual('eq_push', self.general.called_methods[1][0])
        self.assertEqual('FS_DIR_CREATE', self.general.called_methods[1][1])
        self.assertEqual(event.pathname, self.general.called_methods[1][2])

    def test_fake_delete_create_event_file(self):
        """Test that we do fake a delete and a later delete."""
        event = FakeEvent(wd=0, dir=False, name='name', path='path',
                          pathname='pathname')
        held_event = FakeEvent(wd=0, dir=False, name='hname', path='hpath',
                               pathname='hpathname')
        self.processor.held_event = held_event
        self.processor._fake_delete_create_event(event)
        self.assertEqual(3, len(self.general.called_methods))
        self.assertEqual('eq_push', self.general.called_methods[0][0])
        self.assertEqual('FS_FILE_DELETE', self.general.called_methods[0][1])
        self.assertEqual(held_event.pathname, self.general.called_methods[0][2])
        self.assertEqual('eq_push', self.general.called_methods[1][0])
        self.assertEqual('FS_FILE_CREATE', self.general.called_methods[1][1])
        self.assertEqual(event.pathname, self.general.called_methods[1][2])
        self.assertEqual('eq_push', self.general.called_methods[2][0])
        self.assertEqual('FS_FILE_CLOSE_WRITE',
                         self.general.called_methods[2][1])
        self.assertEqual(event.pathname, self.general.called_methods[2][2])

    def test_process_default_no_held(self):
        """Test that the process default works as expected."""
        event = 'event'
        self.processor.process_default(event)
        self.assertEqual(1, len(self.general.called_methods))
        self.assertEqual('push_event',
                         self.general.called_methods[0][0])
        self.assertEqual(event,
                         self.general.called_methods[0][1])

    def test_process_default_with_held(self):
        """Test that the process default works as expected."""
        event = 'event'
        held_event = 'held_event'
        self.processor.held_event = held_event
        self.processor.process_default(event)
        self.assertEqual(2, len(self.general.called_methods))
        self.assertEqual('push_event',
                         self.general.called_methods[0][0])
        self.assertEqual(held_event,
                         self.general.called_methods[0][1])
        self.assertEqual('push_event',
                         self.general.called_methods[1][0])
        self.assertEqual(event,
                         self.general.called_methods[1][1])

    def test_handle_dir_delete_files(self):
        """Test that the handle dir delete works as expected."""
        path = 'path'
        present_files = 'abcde'
        # create files and dirs to be returned from the get paths
        for file_name in present_files:
            self.general.paths_to_return.append((file_name, False))
        self.processor.handle_dir_delete(path)
        # there are calls for, rm the wathc, get paths and then one per file
        self.assertEqual(len(present_files) + 2,
                         len(self.general.called_methods))
        rm_call = self.general.called_methods.pop(0)
        self.assertEqual('rm_watch', rm_call[0])
        self.assertEqual(path, rm_call[1])
        paths_call = self.general.called_methods.pop(0)
        self.assertEqual('get_paths_starting_with', paths_call[0])
        self.assertEqual(path, paths_call[1])
        self.assertFalse(paths_call[2])
        # we need to push the delete events in reverse order because we want
        # to delete children before we delete parents
        present_files = present_files[::-1]
        for i, called_method in enumerate(self.general.called_methods):
            self.assertEqual('eq_push', called_method[0])
            self.assertEqual('FS_FILE_DELETE', called_method[1])
            self.assertEqual(present_files[i], called_method[2])

    def test_handle_dir_delete_dirs(self):
        """Test that the handle dir delete works as expected."""
        path = 'path'
        present_files = 'abcde'
        # create files and dirs to be returned from the get paths
        for file_name in present_files:
            self.general.paths_to_return.append((file_name, True))
        self.processor.handle_dir_delete(path)
        # there are calls for, rm the wathc, get paths and then one per file
        self.assertEqual(2 * len(present_files) + 2,
                         len(self.general.called_methods))
        rm_call = self.general.called_methods.pop(0)
        self.assertEqual('rm_watch', rm_call[0])
        self.assertEqual(path, rm_call[1])
        paths_call = self.general.called_methods.pop(0)
        self.assertEqual('get_paths_starting_with', paths_call[0])
        self.assertEqual(path, paths_call[1])
        self.assertFalse(paths_call[2])
        # we need to push the delete events in reverse order because we want
        # to delete children before we delete parents
        present_files = present_files[::-1]

        # from http://docs.python.org/library/itertools.html#recipes
        def grouper(n, iterable, fillvalue=None):
            "grouper(3, 'ABCDEFG', 'x') --> ABC DEF Gxx"
            args = [iter(iterable)] * n
            return itertools.izip_longest(fillvalue=fillvalue, *args)

        for i, called_methods in enumerate(grouper(2,
                                           self.general.called_methods)):
            rm_call = called_methods[0]
            self.assertEqual('rm_watch', rm_call[0])
            self.assertEqual(present_files[i], rm_call[1])
            push_call = called_methods[1]
            self.assertEqual('eq_push', push_call[0])
            self.assertEqual('FS_DIR_DELETE', push_call[1])
            self.assertEqual(present_files[i], push_call[2])

    def test_freeze_begin(self):
        """Test that the freeze being works as expected."""
        path = 'path'
        self.processor.freeze_begin(path)
        self.assertEqual(1, len(self.general.called_methods))
        self.assertEqual('freeze_begin',
                         self.general.called_methods[0][0])
        self.assertEqual(path, self.general.called_methods[0][1])

    def test_freeze_rollback(self):
        """Test that the freeze rollback works as expected."""
        self.processor.freeze_rollback()
        self.assertEqual(1, len(self.general.called_methods))
        self.assertEqual('freeze_rollback',
                         self.general.called_methods[0][0])

    def test_freeze_commit(self):
        """Test that the freeze commit works as expected."""
        path = 'path'
        self.processor.freeze_commit(path)
        self.assertEqual(1, len(self.general.called_methods))
        self.assertEqual('freeze_commit',
                         self.general.called_methods[0][0])
        self.assertEqual(path, self.general.called_methods[0][1])


class FilesystemMonitorTestCase(BaseTwistedTestCase):
    """Tests for the FilesystemMonitor."""
    timeout = 5

    def test_add_watch_twice(self):
        """Check the deferred returned by a second add_watch."""
        self.patch(Watch, "start_watching", lambda self: self.started)
        monitor = FilesystemMonitor(None, None)
        # no need to stop watching because start_watching is fake

        parent_path = 'C:\\test'  # a valid windows path in utf-8 bytes
        child_path = parent_path + "\\child"
        d1 = monitor.add_watch(parent_path)
        d2 = monitor.add_watch(child_path)

        self.assertFalse(d1.called, "Should not be called yet.")
        self.assertFalse(d2.called, "Should not be called yet.")

        monitor._watch_manager._wdm.values()[0].started.callback(True)

        self.assertTrue(d1.called, "Should already be called.")
        self.assertTrue(d2.called, "Should already be called.")

    @defer.inlineCallbacks
    def test_add_watches_to_udf_ancestors(self):
        """Test that the ancestor watches are not added."""

        class FakeVolume(object):
            """A fake UDF."""

            def __init__(self, ancestors):
                """Create a new instance."""
                self.ancestors = ancestors

        ancestors = ['~', '~\\Pictures', '~\\Pictures\\Home',]
        volume = FakeVolume(ancestors)
        monitor = FilesystemMonitor(None, None)
        added = yield monitor.add_watches_to_udf_ancestors(volume)
        self.assertTrue(added, 'We should always return true.')
        # lets ensure that we never added the watches
        self.assertEqual(0, len(monitor._watch_manager._wdm.values()),
                         'No watches should have been added.')
