# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""tests for the syncdaemon tools module """
import os
import uuid

from dbus.exceptions import DBusException
from ubuntuone.syncdaemon import (
    action_queue,
    event_queue,
    states,
    volume_manager,
)

from ubuntuone.devtools.handlers import MementoHandler

from ubuntuone.platform.linux import dbus_interface, tools
from tests.platform.linux.test_dbus import (
    FakeCommand, DBusTwistedTestCase, AllEventsSender,
)

from twisted.internet import defer, reactor


class TestToolsBase(DBusTwistedTestCase):
    """ Base test case for SyncDaemonTool tests"""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(TestToolsBase, self).setUp()
        self.tool = tools.SyncDaemonTool(self.bus)
        self.home_dir = self.mktemp('ubuntuonehacker')
        old_home = os.environ['HOME']
        os.environ['HOME'] = self.home_dir
        self.addCleanup(os.environ.__setitem__, 'HOME', old_home)
        self.handler = MementoHandler()
        self.tool.log.addHandler(self.handler)
        yield self.main.wait_for_nirvana()

    @defer.inlineCallbacks
    def create_file(self, path):
        """ creates a test file in fsm """
        share_path = os.path.join(self.shares_dir, 'share_tools')
        share = volume_manager.Share(share_path, volume_id='tools_share_id')
        yield self.main.vm.add_share(share)
        self.fs_manager.create(path, "tools_share_id")
        self.fs_manager.set_node_id(path, "node_id")
        defer.returnValue(('tools_share_id', 'node_id'))


class TestToolsBasic(TestToolsBase):
    """Basic test of SyncDaemonTool."""

    def _create_share(self, accepted=True, subscribed=True):
        """Return a newly created Share."""
        share_path = os.path.join(self.shares_dir, 'share_name')
        share = volume_manager.Share(path=share_path, volume_id='share_vol_id')
        return share

    def _create_udf(self, volume_id, node_id, suggested_path, subscribed=True):
        """Create an UDF and returns it and the volume"""
        path = volume_manager.get_udf_path(suggested_path)
        udf = volume_manager.UDF(str(volume_id), str(node_id), suggested_path,
                                 path, subscribed)
        return udf

    def test_wait_connected(self):
        """ test wait_connected """
        self.action_q.connect()
        d = self.tool.wait_connected()
        # test callback, pylint: disable-msg=C0111
        def connected(result):
            self.assertEqual(True, result)
        d.addBoth(connected)
        return d

    def test_wait_no_more_events(self):
        """Test wait_no_more_events."""
        # subscribe the all event sender, as it's needed for this test
        self.event_q.subscribe(AllEventsSender(self.dbus_iface))
        d = self.tool.wait_no_more_events(last_event_interval=.1)

        self.event_q.push("FS_FILE_CREATE", path="somepath")

        def check(result):
            """Check result."""
            self.assertTrue(self.handler.check_debug("new event",
                                                     "FS_FILE_CREATE"))
            self.assertTrue(self.handler.check_debug("No more events"))
            self.assertEqual(True, result)

        d.addBoth(check)
        return d

    def test_all_downloads(self):
        """ test wait_all_downloads """
        d = self.tool.wait_all_downloads()

        # test callback, pylint: disable-msg=C0111
        def downloads(result):
            self.assertEqual(True, result)
        d.addBoth(downloads)
        return d

    def test_all_uploads(self):
        """ test wait_all_uploads """
        d = self.tool.wait_all_uploads()

        # test callback, pylint: disable-msg=C0111
        def uploads(result):
            self.assertEqual(True, result)
        d.addBoth(uploads)
        return d

    def test_wait_for_nirvana(self):
        """ test wait_for_nirvana """
        # setup States to be in Nirvana condition
        self.main.state_manager.state = states.StateManager.QUEUE_MANAGER
        self.main.state_manager.queues.state = states.QueueManager.IDLE

        # unsubscribe VM and States to not receive everything
        self.event_q.unsubscribe(self.main.vm)
        self.event_q.unsubscribe(self.main.state_manager)
        d = self.tool.wait_for_nirvana(last_event_interval=.1)

        # test callback, pylint: disable-msg=C0111
        def callback(result):
            self.assertEqual(True, result)
        d.addBoth(callback)

        # clear downloading
        reactor.callLater(0, self.action_q.connect)

        def fire_events():
            for event_name in event_queue.EVENTS.keys():
                args = event_queue.EVENTS[event_name]
                self.event_q.push(event_name, **dict((x, x) for x in args))

        # fire fake events to keep the deferred running
        reactor.callLater(0, fire_events)
        # 1 sec later, clear the download queue, and wait to reach nirvana
        d.addCallback(lambda _: self.event_q.subscribe(self.main.vm))
        return d

    def test_get_metadata(self):
        """ check that get_metadata works as expected """
        path = os.path.join(self.root_dir, "foo")
        self.fs_manager.create(path, "")
        self.fs_manager.set_node_id(path, "node_id")
        d = self.tool.get_metadata(path)
        # the callback, pylint: disable-msg=C0111
        def callback(result):
            self.assertEqual(path, str(result['path']))
            self.assertEqual('', str(result['share_id']))
            self.assertEqual('node_id', result['node_id'])
        d.addCallback(callback)
        return d

    def test_quit(self):
        """test the quit method."""
        # helper functions, we need to call quit but don't quit
        # pylint: disable-msg=C0111,W0601,W0602
        quitted = False
        def fake_quit():
            global quitted
            quitted = True
        self.main.quit = fake_quit
        d = self.tool.quit()
        # test callback, pylint: disable-msg=C0111
        def check(result):
            global quitted
            self.assertTrue(quitted)
        d.addBoth(check)
        return d

    @defer.inlineCallbacks
    def test_accept_share(self):
        """Test accept_share method"""
        share_path = os.path.join(self.main.shares_dir, 'share')
        yield self.main.vm.add_share(volume_manager.Share(path=share_path,
                                     volume_id='share_id', access_level='Read',
                                     accepted=False, node_id="node_id"))
        self.assertEqual(False, self.main.vm.shares['share_id'].accepted)
        result = yield self.tool.accept_share('share_id')

        self.assertEqual('Yes', result['answer'])
        self.assertEqual('share_id', result['volume_id'])
        self.assertEqual(True, self.main.vm.shares['share_id'].accepted)

    @defer.inlineCallbacks
    def test_reject_share(self):
        """Test the reject_share method"""
        share_path = os.path.join(self.main.shares_dir, 'share')
        yield self.main.vm.add_share(volume_manager.Share(path=share_path,
                                     volume_id='share_id', access_level='Read',
                                     accepted=False))
        self.assertEqual(False, self.main.vm.shares['share_id'].accepted)
        result = yield self.tool.reject_share('share_id')

        self.assertEqual('No', result['answer'])
        self.assertEqual('share_id', result['volume_id'])
        self.assertEqual(False, self.main.vm.shares['share_id'].accepted)

    def test_wait_for_signal(self):
        """Test wait_for_signal method"""
        d = self.tool.wait_for_signal('Foo', lambda _: True)
        def check(result):
            """do the assert"""
            self.assertEqual(True, result)
        d.addCallback(check)
        client = tools.DBusClient(self.bus, '/events',
                                  dbus_interface.DBUS_IFACE_EVENTS_NAME)
        client.send_signal('Foo', True)
        return d

    def test_wait_for_signal_failure(self):
        """Test (with error) wait_for_signal method"""
        def filter(value):
            """broken filter"""
            raise ValueError('DIE!!!!')
        d = self.tool.wait_for_signal('Foo', filter)
        def check(result):
            """do the assert"""
            self.assertEqual('DIE!!!!', result.getErrorMessage())
            self.assertEqual(ValueError, result.type)
        d.addErrback(check)
        client = tools.DBusClient(self.bus, '/events',
                                  dbus_interface.DBUS_IFACE_EVENTS_NAME)
        client.send_signal('Foo', True)
        return d

    def test_wait_for_signals_ok(self):
        """Test wait_for_signals method"""
        d = self.tool.wait_for_signals('FooOk','BarError')
        def check(result):
            """do the assert"""
            self.assertEqual(True, result[0])
        d.addCallback(check)
        client = tools.DBusClient(self.bus, '/folders',
                                  dbus_interface.DBUS_IFACE_FOLDERS_NAME)
        client.send_signal('FooOk', True)
        return d

    def test_wait_for_signals_error(self):
        """Test (with error) wait_for_signals method"""
        d = self.tool.wait_for_signals('FooOk', 'BarError')
        def handle_error(error):
            """do the assert"""
            errorname, errorargs = error.value.args
            self.assertEqual('BarError', errorname)
        d.addCallbacks(self.fail, handle_error)
        client = tools.DBusClient(self.bus, '/folders',
                                  dbus_interface.DBUS_IFACE_FOLDERS_NAME)
        client.send_signal('BarError', True)
        return d

    def test_connect(self):
        """Test the connect method"""
        self.assertEqual(self.main.state_manager.state,
                          states.StateManager.QUEUE_MANAGER)
        d = self.main.wait_for('SYS_USER_DISCONNECT')
        self.main.external.dbus_iface.disconnect()
        def connect(r):
            d = self.main.wait_for('SYS_USER_CONNECT')
            self.tool.connect()
            d.addCallbacks(lambda x: x, self.fail)
            return d
        d.addCallbacks(connect, self.fail)
        return d

    def test_disconnect(self):
        """Test the disconnect method"""
        self.assertEqual(self.main.state_manager.state,
                          states.StateManager.QUEUE_MANAGER)
        d = self.main.wait_for('SYS_USER_DISCONNECT')
        self.tool.disconnect()
        d.addCallbacks(self.assertFalse, self.fail)
        return d

    def test_get_status(self):
        """Test the status method"""
        d = self.tool.get_status()
        def handler(result):
            state = states.StateManager.QUEUE_MANAGER
            self.assertEqual(state.name, result['name'])
            self.assertEqual(state.description, result['description'])
            self.assertEqual(state.is_error, bool(result['is_error']))
            self.assertEqual(state.is_connected, bool(result['is_connected']))
            self.assertEqual(state.is_online, bool(result['is_online']))
        d.addCallbacks(handler, self.fail)
        return d

    @defer.inlineCallbacks
    def test_free_space(self):
        """Test SyncDaemonTool.waiting."""
        share_path = os.path.join(self.main.shares_dir, 'share')
        share = volume_manager.Share(path=share_path, volume_id='vol_id')
        yield self.main.vm.add_share(share)

        self.main.vm.update_free_space('vol_id', 12345)
        result = yield self.tool.free_space('vol_id')
        self.assertEqual(result, 12345)

    @defer.inlineCallbacks
    def test_waiting_simple(self):
        """Test SyncDaemonTool.waiting."""
        # inject the fake data
        c1 = FakeCommand("node_a_foo", "node_a_bar", path='path')
        c2 = FakeCommand("node_b_foo", "node_b_bar")
        c2.running = False
        self.action_q.queue.waiting.extend([c1, c2])
        result = yield self.tool.waiting()

        self.assertEqual(2, len(result))

        pl = dict(share_id='node_a_foo', node_id='node_a_bar',
                  other='', path='path', running='True')
        self.assertEqual(result[0], ('FakeCommand', str(id(c1)), pl))

        pl = dict(share_id='node_b_foo', node_id='node_b_bar',
                  other='', running='')
        self.assertEqual(result[1], ('FakeCommand', str(id(c2)), pl))

    @defer.inlineCallbacks
    def test_waiting_metadata(self):
        """Test SyncDaemonTool.waiting_metadata."""
        # inject the fake data
        self.action_q.queue.waiting.extend([
                FakeCommand("node_a_foo", "node_a_bar", path='path'),
                FakeCommand("node_b_foo", "node_b_bar")])
        result = yield self.tool.waiting_metadata()

        self.assertEqual(2, len(result))

        pl = dict(share_id='node_a_foo', node_id='node_a_bar',
                  other='', path='path', running='True')
        self.assertEqual(result[0], ('FakeCommand', pl))

        pl = dict(share_id='node_b_foo', node_id='node_b_bar',
                  other='', running='True')
        self.assertEqual(result[1], ('FakeCommand', pl))

    def test_waiting_content(self):
        """Test waiting_content."""

        class FakeContentCommand(FakeCommand, action_queue.Upload):
            """Fake command that goes in content queue."""
            def __init__(self, *args, **kwargs):
                FakeCommand.__init__(self, *args, **kwargs)

        # inject the fake data
        self.action_q.queue.waiting.extend([
                FakeContentCommand("", "node_id", path='/some/path'),
                FakeContentCommand("", "node_id_1", path='/other/path')])

        d = self.tool.waiting_content()
        def handler(result):
            node, node_1 = result
            self.assertEqual('/some/path', str(node['path']))
            self.assertEqual('/other/path', str(node_1['path']))
            self.assertEqual('', str(node['share']))
            self.assertEqual('', str(node_1['share']))
            self.assertEqual('node_id', str(node['node']))
            self.assertEqual('node_id_1', str(node_1['node']))
            self.assertTrue(result)
        d.addCallbacks(handler, self.fail)
        return d

    def test_start(self):
        """Test start method"""
        # first quit the currently running client
        d = self.tool.start()
        d.addCallbacks(lambda r: self.assertEqual(r, None), self.fail)
        return d

    def test_create_folder(self):
        """Test for Folders.create."""
        path = os.path.join(self.home_dir, u'ñoño')
        id = uuid.uuid4()
        node_id = uuid.uuid4()
        d = defer.Deferred()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_OK", volume_id=id,
                                   node_id=node_id, marker=marker)
        self.main.action_q.create_udf = create_udf

        d = self.tool.create_folder(path)
        return d

    @defer.inlineCallbacks
    def test_delete_folder(self):
        """Test for Folders.delete."""
        path = os.path.join(self.home_dir, u'ñoño').encode('utf-8')
        # patch AQ.delete_volume
        def delete_volume(volume_id, path):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id="folder_id")
        self.main.action_q.delete_volume = delete_volume

        # create sample folder
        suggested_path = os.path.join("~", u'ñoño').encode('utf-8')
        udf = volume_manager.UDF("folder_id", "node_id", suggested_path, path,
                                 subscribed=True)
        yield self.main.vm.add_udf(udf)

        d = self.tool.delete_folder("folder_id")
        yield d

    @defer.inlineCallbacks
    def test_subscribe_folder(self):
        """Test for Folders.subscribe and that it fires a dbus signal."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=False)
        yield self.main.vm.add_udf(udf)
        yield self.tool.subscribe_folder(udf.id)

        self.assertTrue(self.main.vm.udfs[udf.id].subscribed,
                        "UDF %s isn't subscribed" % udf.id)

    @defer.inlineCallbacks
    def test_unsubscribe_folder(self):
        """Test for Folders.unsubscribe."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=True)
        yield self.main.vm.add_udf(udf)
        yield self.tool.unsubscribe_folder(udf.id)

        self.assertFalse(self.main.vm.udfs[udf.id].subscribed,
                         "UDF %s is subscribed" % udf.id)

    @defer.inlineCallbacks
    def test_subscribe_share(self):
        """Test for Shares.subscribe."""
        share = self._create_share(accepted=True, subscribed=False)
        yield self.main.vm.add_share(share)
        yield self.tool.subscribe_share(share.volume_id)

        self.assertTrue(self.main.vm.shares[share.id].subscribed,
                        "share %s should be subscribed" % share)

    @defer.inlineCallbacks
    def test_unsubscribe_share(self):
        """Test for Shares.unsubscribe."""
        share = self._create_share(accepted=True, subscribed=False)
        yield self.main.vm.add_share(share)
        yield self.tool.unsubscribe_share(share.volume_id)

        self.assertFalse(self.main.vm.shares[share.id].subscribed,
                         "share %s should not be subscribed" % share)

    def test_change_public_access(self):
        """Test change_public_access."""
        node_id = str(uuid.uuid4())
        share_id = ""
        path = os.path.join(self.root_dir, "foo")
        self.fs_manager.create(path, "")
        self.fs_manager.set_node_id(path, node_id)

        def change_public_access(share_id, node_id, is_public):
            """Fake change_public_access"""
            self.main.event_q.push("AQ_CHANGE_PUBLIC_ACCESS_OK",
                                   share_id=share_id, node_id=node_id,
                                   is_public=True,
                                   public_url='http://example.com')
        self.main.action_q.change_public_access = change_public_access

        d = self.tool.change_public_access(path, True)
        def check(file_info):
            """Check the returned file info dictionary."""
            self.assertEqual(share_id, file_info['share_id'])
            self.assertEqual(node_id, file_info['node_id'])
            self.assertEqual('True', file_info['is_public'])
            self.assertEqual('http://example.com', file_info['public_url'])
        d.addCallback(check)
        return d

    @defer.inlineCallbacks
    def test_get_throttling_limits(self):
        """Test for get_throttling_limits."""
        limits = yield self.tool.get_throttling_limits()
        self.assertEqual(-1, limits[u'download'])
        self.assertEqual(-1, limits[u'upload'])

    @defer.inlineCallbacks
    def test_set_throttling_limits(self):
        """Test for set_throttling_limits."""
        yield self.tool.set_throttling_limits(10, 20)
        limits = yield self.tool.get_throttling_limits()
        self.assertEqual(10, limits[u'download'])
        self.assertEqual(20, limits[u'upload'])

    @defer.inlineCallbacks
    def test_is_throttling_enabled(self):
        """Test for is_throttling_enabled."""
        enabled = yield self.tool.is_throttling_enabled()
        self.assertFalse(enabled)
        self.main.action_q.enable_throttling()
        enabled = yield self.tool.is_throttling_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_enable_throttling(self):
        """Test for enable_throttling."""
        yield self.tool.enable_throttling(True)
        enabled = self.main.action_q.throttling_enabled
        self.assertTrue(enabled)
        yield self.tool.enable_throttling(False)
        enabled = self.main.action_q.throttling_enabled
        self.assertFalse(enabled)

    @defer.inlineCallbacks
    def test_is_files_sync_enabled(self):
        """Test for is_files_sync_enabled."""
        yield self.tool.enable_files_sync(False)
        enabled = yield self.tool.is_files_sync_enabled()
        self.assertFalse(enabled)

        yield self.tool.enable_files_sync(True)
        enabled = yield self.tool.is_files_sync_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_enable_files_sync(self):
        """Test for enable_files_sync."""
        yield self.tool.enable_files_sync(True)
        enabled = yield self.tool.is_files_sync_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_enable_files_sync_when_disabled(self):
        """Test for enable_files_sync."""
        called = []
        self.patch(self.tool, 'start', lambda: called.append('start'))
        yield self.tool.enable_files_sync(False)
        # be sure file sync is disabled

        yield self.tool.enable_files_sync(True)
        # assert that the services was started
        self.assertEqual(called, ['start'])

        config = tools.get_user_config()
        self.assertEqual(True, config.get_files_sync_enabled())

    @defer.inlineCallbacks
    def test_disable_files_sync(self):
        """Test for enable_files_sync."""
        yield self.tool.enable_files_sync(False)
        enabled = yield self.tool.is_files_sync_enabled()
        self.assertFalse(enabled)

    @defer.inlineCallbacks
    def test_disable_files_sync_when_disabled(self):
        """Test for enable_files_sync."""
        yield self.tool.enable_files_sync(False)

        yield self.tool.enable_files_sync(False)
        enabled = yield self.tool.is_files_sync_enabled()
        self.assertFalse(enabled)

    @defer.inlineCallbacks
    def test_disable_files_sync_when_enabled(self):
        """Test for enable_files_sync."""
        called = []
        self.patch(self.tool, 'quit', lambda: called.append('quit'))
        yield self.tool.enable_files_sync(True)
        # be sure file sync is enabled

        yield self.tool.enable_files_sync(False)
        # assert that the services was stopped
        self.assertEqual(called, ['quit'])

        config = tools.get_user_config()
        self.assertEqual(False, config.get_files_sync_enabled())

    @defer.inlineCallbacks
    def test_is_autoconnect_enabled(self):
        """Test for is_autoconnect_enabled."""
        yield self.tool.enable_autoconnect(False)
        enabled = yield self.tool.is_autoconnect_enabled()
        self.assertFalse(enabled)

        yield self.tool.enable_autoconnect(True)
        enabled = yield self.tool.is_autoconnect_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_enable_autoconnect(self):
        """Test for enable_autoconnect."""
        yield self.tool.enable_autoconnect(True)
        enabled = yield self.tool.is_autoconnect_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_disable_autoconnect(self):
        """Test for disable_autoconnect."""
        yield self.tool.enable_autoconnect(False)
        enabled = yield self.tool.is_autoconnect_enabled()
        self.assertFalse(enabled)

    @defer.inlineCallbacks
    def test_is_show_all_notifications_enabled(self):
        """Test for is_show_all_notifications_enabled."""
        yield self.tool.enable_show_all_notifications(False)
        enabled = yield self.tool.is_show_all_notifications_enabled()
        self.assertFalse(enabled)

        yield self.tool.enable_show_all_notifications(True)
        enabled = yield self.tool.is_show_all_notifications_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_enable_show_all_notifications(self):
        """Test for enable_show_all_notifications."""
        yield self.tool.enable_show_all_notifications(True)
        enabled = yield self.tool.is_show_all_notifications_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_disable_show_all_notifications(self):
        """Test for disable_show_all_notifications."""
        yield self.tool.enable_show_all_notifications(False)
        enabled = yield self.tool.is_show_all_notifications_enabled()
        self.assertFalse(enabled)

    @defer.inlineCallbacks
    def test_is_share_autosubscribe_enabled(self):
        """Test for is_share_autosubscribe_enabled."""
        yield self.tool.enable_share_autosubscribe(False)
        enabled = yield self.tool.is_share_autosubscribe_enabled()
        self.assertFalse(enabled)

        yield self.tool.enable_share_autosubscribe(True)
        enabled = yield self.tool.is_share_autosubscribe_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_enable_share_autosubscribe(self):
        """Test for enable_share_autosubscribe."""
        yield self.tool.enable_share_autosubscribe(True)
        enabled = yield self.tool.is_share_autosubscribe_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_disable_share_autosubscribe(self):
        """Test for disable_share_autosubscribe."""
        yield self.tool.enable_share_autosubscribe(False)
        enabled = yield self.tool.is_share_autosubscribe_enabled()
        self.assertFalse(enabled)

    @defer.inlineCallbacks
    def test_is_udf_autosubscribe_enabled(self):
        """Test for is_udf_autosubscribe_enabled."""
        yield self.tool.enable_udf_autosubscribe(False)
        enabled = yield self.tool.is_udf_autosubscribe_enabled()
        self.assertFalse(enabled)

        yield self.tool.enable_udf_autosubscribe(True)
        enabled = yield self.tool.is_udf_autosubscribe_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_enable_udf_autosubscribe(self):
        """Test for enable_udf_autosubscribe."""
        yield self.tool.enable_udf_autosubscribe(True)
        enabled = yield self.tool.is_udf_autosubscribe_enabled()
        self.assertTrue(enabled)

    @defer.inlineCallbacks
    def test_disable_udf_autosubscribe(self):
        """Test for disable_udf_autosubscribe."""
        yield self.tool.enable_udf_autosubscribe(False)
        enabled = yield self.tool.is_udf_autosubscribe_enabled()
        self.assertFalse(enabled)

    @defer.inlineCallbacks
    def test_refresh_volumes(self):
        """Test for refresh_volumes method."""
        d = defer.Deferred()
        def list_volumes():
            """Stub list_volumes."""
            d.callback(True)
        self.main.action_q.list_volumes = list_volumes
        yield self.tool.refresh_volumes()
        yield d

    @defer.inlineCallbacks
    def test_rescan_from_scratch(self):
        """Test for rescan_from_scratch method."""
        suggested_path = u'~/ñoño'
        udf = self._create_udf(uuid.uuid4(), 'node_id', suggested_path,
                               subscribed=True)
        yield self.main.vm.add_udf(udf)
        d = defer.Deferred()
        def rescan_from_scratch(volume_id):
            """Stub method."""
            d.callback(volume_id)
        self.main.action_q.rescan_from_scratch = rescan_from_scratch
        yield self.tool.rescan_from_scratch(udf.volume_id)
        vol_id = yield d
        self.assertEqual(vol_id, udf.volume_id)

    @defer.inlineCallbacks
    def test_rescan_from_scratch_missing_volume(self):
        """Test for rescan_from_scratch method with a non-existing volume.."""
        try:
            yield self.tool.rescan_from_scratch('volume_id')
        except DBusException, e:
            self.assertEqual('org.freedesktop.DBus.Python.ubuntuone'
                             '.syncdaemon.volume_manager.VolumeDoesNotExist',
                             e.get_dbus_name())
        else:
            self.fail('Should fail with a non-existing volume.')

    @defer.inlineCallbacks
    def test_get_dirty_nodes(self):
        """Test for get_dirty_nodes method."""
        # create some nodes
        path1 = os.path.join(self.root_dir, u'ñoño-1'.encode('utf-8'))
        mdid1 = self.main.fs.create(path1, "")
        path2 = os.path.join(self.root_dir, u'ñoño-2'.encode('utf-8'))
        mdid2 = self.main.fs.create(path2, "")
        path3 = os.path.join(self.root_dir, "path3")
        mdid3 = self.main.fs.create(path3, "")
        path4 = os.path.join(self.root_dir, "path4")
        mdid4 = self.main.fs.create(path4, "")

        # dirty some
        self.main.fs.set_by_mdid(mdid2, dirty=True)
        self.main.fs.set_by_mdid(mdid4, dirty=True)

        # check
        all_dirty = yield self.tool.get_dirty_nodes()
        dirty_mdids = dict((n['mdid'], n) for n in all_dirty)
        self.assertEqual(len(all_dirty), 2)
        self.assertIn(mdid2, dirty_mdids)
        self.assertIn(mdid4, dirty_mdids)
        self.assertNotIn(mdid1, dirty_mdids)
        self.assertNotIn(mdid3, dirty_mdids)
        # check that path de/encoding is done correctly
        self.assertEqual(repr(self.main.fs.get_by_mdid(mdid2).path),
                         repr(dirty_mdids[mdid2]['path'].encode('utf-8')))

    @defer.inlineCallbacks
    def test_get_root_dir(self):
        """Test the get_root_dir method"""
        result = yield self.tool.get_root_dir()
        self.assertEqual(self.main.root_dir, result)

    @defer.inlineCallbacks
    def test_get_shares_dir(self):
        """Test the get_shares_dir method"""
        result = yield self.tool.get_shares_dir()
        self.assertEqual(self.main.shares_dir, result)

    @defer.inlineCallbacks
    def test_get_shares_dir_link(self):
        """Test the get_shares_dir_link method"""
        result = yield self.tool.get_shares_dir_link()
        self.assertEqual(self.main.shares_dir_link, result)
