# tests.platform.windows.test_tools - test tools for SyncDaemon
#
# Authors: Manuel de la Pena <manuel@canonical.com>
#          Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

from mocker import Mocker, ANY

from twisted.trial.unittest import TestCase
from twisted.internet import defer
from twisted.spread.pb import (
    PBServerFactory,
    Root)

from ubuntuone.platform.windows import ipc
from ubuntuone.platform.windows.ipc import (
    Config,
    Events,
    Folders,
    FileSystem,
    PublicFiles,
    Shares,
    Status,
    SyncDaemon,
    ipc_server_listen,
)
from ubuntuone.platform.windows.tools import (
    U1_REG_PATH,
    SD_INSTALL_PATH,
    SyncDaemonTool,
)

TEST_TOOLS_PORT = 40405


class SaveProtocolServerFactory(PBServerFactory):
    """A PBServerFactory that saves the latest connected client."""

    protocolInstance = None

    def clientConnectionMade(self, protocol):
        """Keep track of the given protocol."""
        self.protocolInstance = protocol


class FakeRoot(Root):
    """Root object that exposes the diff referenceable objects."""

    def __init__(self, status, events, sync_daemon, file_system, shares,
                 config, folders, public_files):
        """Create a new instance that will expose the objects."""
        self._status = status
        self._events = events
        self._sync_daemon = sync_daemon
        self._file_system = file_system
        self._shares = shares
        self._config = config
        self._folders = folders
        self._public_files = public_files

    def remote_get_status(self):
        """Return the status remote object."""
        return self._status

    def remote_get_events(self):
        """Return the events remote object."""
        return self._events

    def remote_get_sync_daemon(self):
        """Return the sync daemon remote object."""
        return self._sync_daemon

    def remote_get_file_system(self):
        """Return the file system remote object."""
        return self._file_system

    def remote_get_shares(self):
        """Return the shares remote object."""
        return self._shares

    def remote_get_config(self):
        """Return the config remote object."""
        return self._config

    def remote_get_folders(self):
        """Return the folders remote object."""
        return self._folders

    def remote_get_public_files(self):
        """Return the public files remote object."""
        return self._public_files


class TestSyncDaemonTool(TestCase):
    """Various utility methods to test/play with the SyncDaemon."""

    def setUp(self):
        """Set up the different tests."""
        super(TestSyncDaemonTool, self).setUp()
        self.mocker = Mocker()
        self.root = self.mocker.mock()
        self.status = Status(None, None, None)
        self.status.syncdaemon_status = self.root
        self.events = Events(None)
        self.events.events = self.root
        self.daemon = SyncDaemon(None, None, None, None)
        self.daemon.service = self.root
        self.fs = FileSystem(None, None)
        self.fs.syncdaemon_filesystem  = self.root
        self.shares = Shares(None, None)
        self.shares.syncdaemon_shares = self.root
        self.config = Config(None, None)
        self.config.syncdaemon_config = self.root
        self.folders = Folders(None, None)
        self.folders.syncdaemon_folders = self.root
        self.files = PublicFiles(None, None)
        self.files.syncdaemon_public_files = self.root
        self.is_running = self.mocker.replace(
                                'ubuntuone.platform.windows.tools.is_running')
        # start pb
        self.syncdaemon_root = FakeRoot(self.status, self.events, self.daemon,
                                        self.fs, self.shares, self.config,
                                        self.folders, self.files)
        self.server_factory = SaveProtocolServerFactory(self.syncdaemon_root)
        # pylint: disable=E1101
        self.patch(ipc, "get_sd_pb_port", lambda: TEST_TOOLS_PORT)
        self.patch(ipc.ActivationClient, "get_active_port",
                             lambda _: defer.succeed(TEST_TOOLS_PORT))
        self.listener = ipc_server_listen(self.server_factory)
        self.sdtool = SyncDaemonTool()

    def tearDown(self):
        """Clean reactor."""
        if self.server_factory.protocolInstance is not None:
            self.server_factory.protocolInstance.transport.loseConnection()
        return defer.gatherResults([self._tearDownServer(),
                                    self._tearDownClient()])

    def _tearDownServer(self):
        """Teardown the server."""
        return defer.maybeDeferred(self.listener.stopListening)

    def _tearDownClient(self):
        """Tear down the client."""
        self.sdtool.client.disconnect()
        return defer.succeed(None)

    @defer.inlineCallbacks
    def _connect(self):
        """Connect to the running service."""
        yield self.sdtool.connected
        defer.returnValue(self.sdtool)

    def test_wait_connected(self):
        """Test that we wait until we connect to the server."""

    def test_get_current_downloads(self):
        """Test gettting the current downloads."""
        downloads = [dict(name='name', id='id'),
                     dict(name='name', id='id')]

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.current_downloads()
            self.mocker.result(downloads)
            self.mocker.replay()
            result = yield sdtool.get_current_downloads()
            self.mocker.restore()
            self.mocker.verify()
            self.assertEqual(downloads, result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_current_uploads(self):
        """Test that we get all the current uploads."""
        uploads = [dict(name='name', id='id'),
                   dict(name='name', id='id')]

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.current_uploads()
            self.mocker.replay()
            self.mocker.result(uploads)
            result = yield sdtool.get_current_uploads()
            self.mocker.restore()
            self.mocker.verify()
            self.assertEqual(uploads, result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_wait_for_nirvana(self, last_event_interval=5, verbose=False):
        """Test that we can reach nirvana."""
        last_event_interval = 5

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.wait_for_nirvana(last_event_interval, ANY, ANY)
            self.mocker.replay()
            yield sdtool.wait_for_nirvana()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_accept_share(self):
        """Test that we do accept the share."""
        share_id = 'share_id'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.accept_share(share_id, ANY, ANY)
            self.mocker.replay()
            yield sdtool.accept_share(share_id)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_reject_share(self):
        """Test that we do reject the share."""
        share_id = 'share_id'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.reject_share(share_id, ANY, ANY)
            self.mocker.replay()
            yield sdtool.reject_share(share_id)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_subscribe_share(self):
        """Test that we subscribe to the share."""
        share_id = 'share_id'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.subscribe(share_id)
            self.mocker.replay()
            yield sdtool.subscribe_share(share_id)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_unsubscribe_share(self):
        """Unsubscribe from a share given its id."""
        share_id = 'share_id'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.unsubscribe(share_id)
            self.mocker.replay()
            yield sdtool.unsubscribe_share(share_id)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_shares(self):
        """Test that we get the list of shares."""
        shares = [dict(name='name', id='id'),
                  dict(name='name', id='id')]

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_shares()
            self.mocker.result(shares)
            self.mocker.replay()
            yield sdtool.get_shares()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_refresh_shares(self):
        """Test that we can refresh the shares."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.refresh_shares()
            self.mocker.replay()
            yield sdtool.refresh_shares()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_offer_share(self):
        """Test that we can indeed offer a share."""
        path = 'path'
        username = 'username'
        name = 'name'
        access_level = 'level'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.create_share(path, username, name, access_level)
            self.mocker.replay()
            yield sdtool.offer_share(path, username, name, access_level)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_list_shared(self):
        """Test that we can list the shared."""
        shares = [dict(name='name', id='id'),
                  dict(name='name', id='id')]

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_shared()
            self.mocker.result(shares)
            self.mocker.replay()
            yield sdtool.list_shared()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_create_folder(self):
        """Test that we can create a folder."""
        path = 'path'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.create(path)
            self.mocker.replay()
            yield sdtool.create_folder(path)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_delete_folder(self):
        """Test that we can delete a folder."""
        folder_id = 'folder_id'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.delete(folder_id)
            self.mocker.replay()
            yield sdtool.delete_folder(folder_id)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_subscribe_folder(self):
        """Test that we can subscribe to a folder."""
        folder_id = 'folder_id' 

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.subscribe(folder_id )
            self.mocker.replay()
            yield sdtool.subscribe_folder(folder_id )
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_unsubscribe_folder(self):
        """Test that we can unsubscribe to a folder."""
        folder_id = 'folder_id'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.unsubscribe(folder_id)
            self.mocker.replay()
            yield sdtool.unsubscribe_folder(folder_id)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_folders(self):
        """Test that we can get the folders."""
        folders = [dict(name='name', id='id'),
                   dict(name='name', id='id')]

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_folders()
            self.mocker.result(folders)
            self.mocker.replay()
            yield sdtool.get_folders()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_folder_info(self):
        """Test that we can get the info of a folder."""
        path = 'path'

        @defer.inlineCallbacks
        def test_execution(client):
            """Actual test."""
            self.root.get_info(path)
            self.mocker.replay()
            yield client.get_folder_info(path)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_metadata(self):
        """Test that we can get the metadata."""
        path = 'path'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_metadata(path)
            self.mocker.replay()
            yield sdtool.get_metadata(path)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_change_public_access(self):
        """Test that we can change the public access."""
        path = 'path'
        share_id = 'share_id'
        node_id = 'node_id'
        is_public = False
        metadata = dict(share_id=share_id, node_id=node_id)

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_metadata(path)
            self.mocker.result(metadata)
            self.root.change_public_access(share_id, node_id, is_public)
            self.mocker.replay()
            yield sdtool.change_public_access(path, is_public)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_quit(self):
        """Test that we can quit."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.is_running()
            self.mocker.result(True)
            self.root.quit(None, None)
            self.mocker.replay()
            yield sdtool.quit()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_connect(self):
        """Test we can connect."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.connect()
            self.mocker.replay()
            yield sdtool.connect()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_disconnect(self):
        """Test we can disconnect."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.disconnect()
            self.mocker.replay()
            yield sdtool.disconnect()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_status(self):
        """Test we can get the status."""
        status = dict(is_error=False, is_online=True, is_connected=True)

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.current_status()
            self.mocker.result(status)
            self.mocker.replay()
            yield sdtool.get_status()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_waiting(self):
        """Test that we get the info of what is waitig."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.waiting()
            self.mocker.result({})
            self.mocker.replay()
            yield sdtool.waiting()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_waiting_metadata(self):
        """Test that we can get the waiting metadata."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.waiting_metadata()
            self.mocker.replay()
            yield sdtool.waiting_metadata()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_waiting_content(self):
        """Test that we get the waiting content."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.waiting_content()
            self.mocker.replay()
            yield sdtool.waiting_content()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_throttling_limits(self):
        """Test that we get the throttling limits."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_throttling_limits(ANY, ANY)
            self.mocker.replay()
            yield sdtool.get_throttling_limits()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_set_throttling_limits(self):
        """Test that we can set the limits."""
        download = 'download'
        upload = 'upload'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.set_throttling_limits(download, upload, ANY, ANY)
            self.mocker.replay()
            yield sdtool.set_throttling_limits(download, upload)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_is_throttling_enabled(self):
        """Test that we get the bool stating if they are active."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.bandwidth_throttling_enabled(ANY, ANY)
            self.mocker.result(True)
            self.mocker.replay()
            result = yield sdtool.is_throttling_enabled()
            self.mocker.restore()
            self.mocker.verify()
            self.assertTrue(result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_throttling_true(self):
        """Test that we can set them."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.enable_bandwidth_throttling(ANY, ANY)
            self.mocker.replay()
            yield sdtool.enable_throttling(True)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_throttling_false(self):
        """Test that we can set them."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.disable_bandwidth_throttling(ANY, ANY)
            self.mocker.replay()
            yield sdtool.enable_throttling(False)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_is_autoconnect_enabled(self):
        """Test that we can read the config."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.autoconnect_enabled()
            self.mocker.replay()
            self.mocker.result(True)
            result = yield sdtool.is_autoconnect_enabled()
            self.mocker.restore()
            self.mocker.verify()
            self.assertTrue(result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_autoconnect(self):
        """Test that we can enable the autoconnect."""
        enabled = False

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.set_autoconnect_enabled(enabled)
            self.mocker.replay()
            yield sdtool.enable_autoconnect(enabled)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_is_show_all_notifications_enabled(self):
        """Test if we can read the info."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.show_all_notifications_enabled()
            self.mocker.result(True)
            self.mocker.replay()
            result = yield sdtool.is_show_all_notifications_enabled()
            self.mocker.restore()
            self.mocker.verify()
            self.assertTrue(result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_show_all_notifications_true(self):
        """Enable/disable show_all_notifications."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.enable_show_all_notifications()
            self.mocker.replay()
            yield sdtool.enable_show_all_notifications(True)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_show_all_notifications_false(self):
        """Enable/disable show_all_notifications."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.disable_show_all_notifications()
            self.mocker.replay()
            yield sdtool.enable_show_all_notifications(False)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_is_share_autosubscribe_enabled(self):
        """Test that we can get the config."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.share_autosubscribe_enabled()
            self.mocker.result(True)
            self.mocker.replay()
            result = yield sdtool.is_share_autosubscribe_enabled()
            self.mocker.restore()
            self.mocker.verify()
            self.assertTrue(result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_share_autosubscribe_true(self):
        """Enable/disable share_autosubscribe."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.enable_share_autosubscribe()
            self.mocker.replay()
            yield sdtool.enable_share_autosubscribe(True)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_share_autosubscribe_false(self):
        """Enable/disable share_autosubscribe."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.disable_share_autosubscribe()
            self.mocker.replay()
            yield sdtool.enable_share_autosubscribe(False)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_is_udf_autosubscribe_enabled(self):
        """Test that we can get the config value."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.udf_autosubscribe_enabled()
            self.mocker.result(True)
            self.mocker.replay()
            result = yield sdtool.is_udf_autosubscribe_enabled()
            self.mocker.restore()
            self.mocker.verify()
            self.assertTrue(result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_udf_autosubscribe_true(self):
        """Test that we can get the config."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.enable_udf_autosubscribe()
            self.mocker.replay()
            yield sdtool.enable_udf_autosubscribe(True)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_enable_udf_autosubscribe_false(self):
        """Test that we can set the config."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.disable_udf_autosubscribe()
            self.mocker.replay()
            yield sdtool.enable_udf_autosubscribe(False)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_refresh_volumes(self):
        """Test that we can refresh the volumes."""

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.refresh_volumes()
            self.mocker.replay()
            yield sdtool.refresh_volumes()
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_rescan_from_scratch(self):
        """Test that we can rescan from scratch."""
        volume_id = 'volume_id'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.rescan_from_scratch(volume_id)
            self.mocker.replay()
            yield sdtool.rescan_from_scratch(volume_id)
            self.mocker.restore()
            self.mocker.verify()

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_dirty_nodes(self):
        """Test that we get the dirty nodes."""
        dirty_nodes = ['node_1', 'node_2', 'node_3']

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_dirty_nodes()
            self.mocker.result(dirty_nodes)
            self.mocker.replay()
            result = yield sdtool.get_dirty_nodes()
            self.mocker.restore()
            self.mocker.verify()
            self.assertEqual(dirty_nodes, result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_root_dir(self):
        """Test that we get the root dir."""
        root = '/path/to/root'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_rootdir()
            self.mocker.result(root)
            self.mocker.replay()
            result = yield sdtool.get_root_dir()
            self.mocker.restore()
            self.mocker.verify()
            self.assertEqual(root, result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_shares_dir(self):
        """Test that we get the shares dir."""
        shares_dir = '/path/to/shares'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_sharesdir()
            self.mocker.result(shares_dir)
            self.mocker.replay()
            result = yield sdtool.get_shares_dir()
            self.mocker.restore()
            self.mocker.verify()
            self.assertEqual(shares_dir, result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_get_shares_dir_link(self):
        """Test that we get the link dir."""
        link = 'path/to/link'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            self.root.get_sharesdir_link()
            self.mocker.result(link)
            self.mocker.replay()
            result = yield sdtool.get_shares_dir_link()
            self.mocker.restore()
            self.mocker.verify()
            self.assertEqual(link, result)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    @defer.inlineCallbacks
    def test_set_status_changed_handler(self):
        """The status handler is changed."""
        test_handler = object()
        sdtool = yield self._connect()
        yield sdtool.set_status_changed_handler(test_handler)
        self.assertEqual(sdtool.client.status.on_status_changed_cb,
                         test_handler)

    def test_start(self):
        """Test that we can indeed start the process."""
        cmd = 'notepad'

        @defer.inlineCallbacks
        def test_execution(sdtool):
            """Actual test."""
            open_key = self.mocker.replace('_winreg.OpenKey')
            query_value = self.mocker.replace('_winreg.QueryValueEx')
            subprocess = self.mocker.replace('subprocess.Popen')
            # set the expectations
            self.is_running()
            self.mocker.result(False)
            open_key(ANY, U1_REG_PATH)
            query_value(None, SD_INSTALL_PATH)
            self.mocker.result([cmd,])
            subprocess([cmd,])
            self.mocker.result(cmd)
            self.mocker.replay()
            process = yield sdtool.start()
            self.mocker.restore()
            self.mocker.verify()
            self.assertEqual(cmd, process)

        d = self._connect()
        # pylint: disable=E1101
        d.addCallback(test_execution)
        # pylint: enable=E1101
        return d

    def test_call_after_connection(self):
        """Test the _call_after_connection method."""
        self.patch(self.sdtool, "connected", defer.Deferred())
        collected_calls = []
        test_method = lambda *args: collected_calls.append(args)
        test_method = self.sdtool._call_after_connection(test_method)
        test_method(123)
        self.assertEqual(len(collected_calls), 0)
        self.sdtool.connected.callback("got connected!")
        self.assertEqual(len(collected_calls), 1)

    def test_should_wrap(self):
        """Only some attributes should be wrapped."""
        test_function = "sample_function"
        assert test_function not in SyncDaemonTool._DONT_VERIFY_CONNECTED
        self.assertTrue(self.sdtool._should_wrap(test_function))

    def test_should_not_wrap_listed_attributes(self):
        """Attributes listed in DONT_VERIFY_CONNECTED should not be wrapped."""
        for attribute_name in SyncDaemonTool._DONT_VERIFY_CONNECTED:
            self.assertFalse(self.sdtool._should_wrap(attribute_name))

    def test_should_not_wrap_underscore_attributes(self):
        """Attributes starting with underscore should not be wrapped."""
        dunder_function = "__sample_attribute__"
        assert dunder_function not in SyncDaemonTool._DONT_VERIFY_CONNECTED
        self.assertFalse(self.sdtool._should_wrap(dunder_function))

        under_function = "_sampleattribute"
        assert under_function not in SyncDaemonTool._DONT_VERIFY_CONNECTED
        self.assertFalse(self.sdtool._should_wrap(under_function))

    def test_getattributes_wraps_methods(self):
        """The common methods are wrapped with connect."""
        for attr_name in dir(self.sdtool):
            if self.sdtool._should_wrap(attr_name):
                attr = getattr(self.sdtool, attr_name)
                func_name = getattr(attr, "__name__", None)
                self.assertEqual(func_name, "call_after_connection_inner")

    def test_getattributes_does_not_wrap_special(self):
        """The special methods and attributes are not wrapped."""
        for attr_name in dir(self.sdtool):
            if not self.sdtool._should_wrap(attr_name):
                attr = getattr(self.sdtool, attr_name)
                func_name = getattr(attr, "__name__", None)
                self.assertNotEqual(func_name, "call_after_connection_inner")
