/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010-2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 *
 */

#include "config.h"
#include "syncdaemon-publicfiles-interface.h"
#include "utils.h"

G_DEFINE_TYPE(SyncdaemonPublicfilesInterface, syncdaemon_publicfiles_interface, SYNCDAEMON_TYPE_INTERFACE)

struct _SyncdaemonPublicfilesInterfacePrivate {
	GObject *proxy;
	GHashTable *public_files;
};

static void
syncdaemon_publicfiles_interface_finalize (GObject *object)
{
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (object);

	if (interface->priv != NULL) {
		if (interface->priv->public_files != NULL)
			g_hash_table_destroy (interface->priv->public_files);

		g_free (interface->priv);
	}

	G_OBJECT_CLASS (syncdaemon_publicfiles_interface_parent_class)->finalize (object);
}

static void
syncdaemon_publicfiles_interface_class_init (SyncdaemonPublicfilesInterfaceClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_publicfiles_interface_finalize;
}

static void
file_published_cb (DBusGProxy *proxy, GHashTable *hash, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (user_data);

    g_return_if_fail (hash != NULL);

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL) {
		SyncdaemonFileInfo *finfo;

		finfo = syncdaemon_file_info_new_from_hash_table (hash);
		if (syncdaemon_file_info_get_is_public (finfo))
			g_signal_emit_by_name (daemon, "file_published", TRUE, finfo);
		else
			g_signal_emit_by_name (daemon, "file_unpublished", TRUE, finfo);

		g_object_unref (G_OBJECT (finfo));
	}
}

static void
got_public_files_cb (DBusGProxy *proxy, GPtrArray *files, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (user_data);

    g_return_if_fail (files != NULL);

	if (interface->priv->public_files == NULL) {
		interface->priv->public_files = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
	}

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL) {
		guint i;

		for (i = 0; i < files->len; i++) {
			SyncdaemonFileInfo *finfo = syncdaemon_file_info_new_from_hash_table (g_ptr_array_index (files, i));

			if (SYNCDAEMON_IS_FILE_INFO (finfo)) {
				g_hash_table_insert (interface->priv->public_files,
						     g_strdup (syncdaemon_file_info_get_path (finfo)),
						     finfo);
				g_signal_emit_by_name (daemon, "file_published", TRUE, finfo);
			}
		}
	}
}

static void
public_access_change_error_cb (DBusGProxy *proxy, GHashTable *hash, gchar *error, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (user_data);

    g_return_if_fail (hash != NULL);

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL) {
		SyncdaemonFileInfo *finfo;

		finfo = syncdaemon_file_info_new_from_hash_table (hash);
		g_signal_emit_by_name (daemon, "file_published", FALSE, finfo);

		g_object_unref (G_OBJECT (finfo));
	}
}

static void
public_files_list_error_cb (DBusGProxy *proxy, gchar *error, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (user_data);

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL)
		g_signal_emit_by_name (daemon, "got_published_files", FALSE, NULL);
}

static void
syncdaemon_publicfiles_interface_init (SyncdaemonPublicfilesInterface *interface)
{
	interface->priv = g_new0 (SyncdaemonPublicfilesInterfacePrivate, 1);

	/* Setup DBus proxy */
	interface->priv->proxy = syncdaemon_interface_setup_proxy (SYNCDAEMON_INTERFACE (interface),
								   "com.ubuntuone.SyncDaemon",
								   "/publicfiles", "com.ubuntuone.SyncDaemon.PublicFiles");
	if (interface->priv->proxy != NULL) {
		/* Connect to DBus signals */
		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicAccessChanged",
					 dbus_g_type_get_map ("GHashTable",
							      G_TYPE_STRING,
							      G_TYPE_STRING),
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicAccessChanged",
					     G_CALLBACK (file_published_cb), interface, NULL);

		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicAccessChangeError",
					 dbus_g_type_get_map ("GHashTable",
							      G_TYPE_STRING,
							      G_TYPE_STRING),
					 G_TYPE_STRING,
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicAccessChangeError",
					     G_CALLBACK (public_access_change_error_cb), interface, NULL);

		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicFilesList",
					 dbus_g_type_get_collection ("GPtrArray",
								     dbus_g_type_get_map
								     ("GHashTable",
								      G_TYPE_STRING,
								      G_TYPE_STRING)),
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicFilesList",
					     G_CALLBACK (got_public_files_cb), interface, NULL);

		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicFilesListError",
					 G_TYPE_STRING,
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicFilesListError",
					     G_CALLBACK (public_files_list_error_cb), interface, NULL);
	}
}

/**
 * syncdaemon_publicfiles_interface_new:
 */
SyncdaemonPublicfilesInterface *
syncdaemon_publicfiles_interface_new (SyncdaemonDaemon *daemon)
{
	g_return_val_if_fail (SYNCDAEMON_IS_DAEMON (daemon), NULL);

	return g_object_new (SYNCDAEMON_TYPE_PUBLICFILES_INTERFACE, "daemon", daemon, NULL);
}

/**
 * syncdaemon_publicfiles_interface_change_public_access:
 */
void
syncdaemon_publicfiles_interface_change_public_access (SyncdaemonPublicfilesInterface *interface,
						       const gchar *share_id,
						       const gchar *node_id,
						       gboolean is_public)
{
	g_return_if_fail (SYNCDAEMON_IS_PUBLICFILES_INTERFACE (interface));
	g_return_if_fail (share_id != NULL);
	g_return_if_fail (node_id != NULL);

	dbus_g_proxy_begin_call (DBUS_G_PROXY (interface->priv->proxy), "change_public_access",
				 no_output_dbus_call_ended_cb, interface, NULL,
				 G_TYPE_STRING, share_id,
				 G_TYPE_STRING, node_id,
				 G_TYPE_BOOLEAN, is_public,
				 G_TYPE_INVALID);
}

/**
 * syncdaemon_publicfiles_interface_get_public_files:
 *
 * Return value: A #GSList of #SyncdaemonFileInfo objects, each of which describes
 * a published file for the user. When no longer needed, this list should be
 * freed by calling g_slist_free, but the data within the list should never
 * be freed, as it belongs to the library.
 */
GSList *
syncdaemon_publicfiles_interface_get_public_files (SyncdaemonPublicfilesInterface *interface)
{
	GSList *files_list = NULL;

	g_return_val_if_fail (SYNCDAEMON_IS_PUBLICFILES_INTERFACE (interface), NULL);

	if (interface->priv->public_files == NULL) {
		/* Only retrieve list if we haven't yet */
		dbus_g_proxy_begin_call (DBUS_G_PROXY (interface->priv->proxy), "get_public_files",
					 no_output_dbus_call_ended_cb, interface, NULL,
					 G_TYPE_INVALID);
	} else {
		GHashTableIter iter;
		gchar *path;
		SyncdaemonFileInfo *finfo;

		g_hash_table_iter_init (&iter, interface->priv->public_files);
		while (g_hash_table_iter_next (&iter, (gpointer *) &path, (gpointer *) &finfo))
			files_list = g_slist_append (files_list, finfo);
	}

	return files_list;
}
