# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

# Unused import QtGui
# pylint: disable=W0611
from PyQt4 import QtCore
# pylint: enable=W0611
from twisted.internet import defer

from ubuntuone.controlpanel import backend
from ubuntuone.controlpanel.logger import setup_logging, log_call
from ubuntuone.controlpanel.gui import (
    KILOBYTES,
    SETTINGS_ALLOW_NOTIFICATIONS,
    SETTINGS_AUTO_CONNECT,
    SETTINGS_BANDWIDTH,
    SETTINGS_BUTTON_APPLY,
    SETTINGS_BUTTON_DEFAULT,
    SETTINGS_FILE_SYNC,
    SETTINGS_KILOBITS_PER_SECOND,
    SETTINGS_LIMIT_DOWNLOAD,
    SETTINGS_LIMIT_UPLOAD,
    SETTINGS_SYNC_ALL_FOLDERS,
    SETTINGS_SYNC_ALL_SHARES,
)
from ubuntuone.controlpanel.gui.qt import force_wordwrap
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin
from ubuntuone.controlpanel.gui.qt.ui import preferences_ui


logger = setup_logging('qt.preferences')

CHECKED = QtCore.Qt.Checked
UNCHECKED = QtCore.Qt.Unchecked


def check_state(checked):
    """Return the QCheckBox check state according the value of 'checked'."""
    return CHECKED if checked else UNCHECKED


class PreferencesPanel(UbuntuOneBin):
    """The Preferences Tab Panel widget"""

    ui_class = preferences_ui
    logger = logger

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(PreferencesPanel, self)._setup()
        self.ui.apply_changes_button.setText(SETTINGS_BUTTON_APPLY)
        self.ui.bandwidth_settings.setTitle(SETTINGS_BANDWIDTH)
        self.ui.file_sync_settings.setTitle(SETTINGS_FILE_SYNC)
        self.ui.upload_speed_spinbox.setSuffix("  " +
            SETTINGS_KILOBITS_PER_SECOND)
        self.ui.download_speed_spinbox.setSuffix("  " +
            SETTINGS_KILOBITS_PER_SECOND)
        self.ui.restore_defaults_button.setText(SETTINGS_BUTTON_DEFAULT)

    # pylint: disable=E0202
    @defer.inlineCallbacks
    def load(self):
        """Load info."""
        self.is_processing = True
        info = yield self.backend.file_sync_settings_info()
        self.process_info(info)

    @log_call(logger.debug)
    def process_info(self, info):
        """Load settings info into the view."""
        autoconnect = check_state(info[backend.AUTOCONNECT_KEY])
        show_all_notifs = \
            check_state(info[backend.SHOW_ALL_NOTIFICATIONS_KEY])
        share_autosubscribe = \
            check_state(info[backend.SHARE_AUTOSUBSCRIBE_KEY])
        udf_autosubscribe = \
            check_state(info[backend.UDF_AUTOSUBSCRIBE_KEY])
        download_speed = info[backend.DOWNLOAD_KEY] // KILOBYTES
        upload_speed = info[backend.UPLOAD_KEY] // KILOBYTES

        self.ui.autoconnect_checkbox.setCheckState(autoconnect)
        self.ui.show_all_notifications_checkbox.setCheckState(show_all_notifs)
        self.ui.share_autosubscribe_checkbox.setCheckState(share_autosubscribe)
        self.ui.udf_autosubscribe_checkbox.setCheckState(udf_autosubscribe)
        # proper checkState will be set with the spinbox's valueChanged
        # signal handlers
        self.ui.download_speed_spinbox.setValue(download_speed)
        self.ui.upload_speed_spinbox.setValue(upload_speed)
        if upload_speed > 0:
            self.ui.limit_uploads_checkbox.setCheckState(CHECKED)
        else:
            self.ui.limit_uploads_checkbox.setCheckState(UNCHECKED)
        if download_speed > 0:
            self.ui.limit_downloads_checkbox.setCheckState(CHECKED)
        else:
            self.ui.limit_downloads_checkbox.setCheckState(UNCHECKED)
        self.is_processing = False

    # pylint: disable=C0103

    @QtCore.pyqtSlot()
    @defer.inlineCallbacks
    def on_apply_changes_button_clicked(self):
        """The apply changes button was clicked."""
        autoconnect = self.ui.autoconnect_checkbox.checkState()
        notifications = self.ui.show_all_notifications_checkbox.checkState()
        share_autosubscribe = self.ui.share_autosubscribe_checkbox.checkState()
        udf_autosubscribe = self.ui.udf_autosubscribe_checkbox.checkState()
        if self.ui.limit_uploads_checkbox.checkState() == UNCHECKED:
            upload_speed = -1
        else:
            upload_speed = self.ui.upload_speed_spinbox.value() * KILOBYTES
        if self.ui.limit_downloads_checkbox.checkState() == UNCHECKED:
            download_speed = -1
        else:
            download_speed = self.ui.download_speed_spinbox.value() * KILOBYTES

        settings = {
            backend.AUTOCONNECT_KEY: autoconnect == CHECKED,
            backend.SHOW_ALL_NOTIFICATIONS_KEY: notifications == CHECKED,
            backend.SHARE_AUTOSUBSCRIBE_KEY: share_autosubscribe == CHECKED,
            backend.UDF_AUTOSUBSCRIBE_KEY: udf_autosubscribe == CHECKED,
            backend.DOWNLOAD_KEY: download_speed,
            backend.UPLOAD_KEY: upload_speed,
        }

        self.is_processing = True
        yield self.backend.change_file_sync_settings(settings)
        self.is_processing = False

    @QtCore.pyqtSlot()
    @defer.inlineCallbacks
    def on_restore_defaults_button_clicked(self):
        """The restore defaults button was clicked."""
        self.is_processing = True
        yield self.backend.restore_file_sync_settings()
        settings = yield self.backend.file_sync_settings_info()

        self.process_info(settings)

    def resizeEvent(self, event):
        super(PreferencesPanel, self).resizeEvent(event)
        size_bandwidth = (self.ui.bandwidth_settings.width() -
            self.ui.upload_speed_spinbox.width())
        padding = 160  # Padding to allow shrinking
        size_sync = (self.ui.file_sync_settings.width() - padding)
        force_wordwrap(self.ui.limit_uploads_checkbox, size_bandwidth,
            SETTINGS_LIMIT_UPLOAD)
        force_wordwrap(self.ui.limit_downloads_checkbox, size_bandwidth,
            SETTINGS_LIMIT_DOWNLOAD)
        force_wordwrap(self.ui.autoconnect_checkbox, size_sync,
            SETTINGS_AUTO_CONNECT)
        force_wordwrap(self.ui.udf_autosubscribe_checkbox, size_sync,
            SETTINGS_SYNC_ALL_SHARES)
        force_wordwrap(self.ui.share_autosubscribe_checkbox, size_sync,
            SETTINGS_SYNC_ALL_FOLDERS)
        force_wordwrap(self.ui.show_all_notifications_checkbox, size_sync,
            SETTINGS_ALLOW_NOTIFICATIONS)
