# -*- coding: utf-8 -*-

# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Share Links tab."""

import os

from PyQt4 import QtGui, QtCore
from twisted.internet import defer

from ubuntuone.controlpanel.gui import (
    SEARCH_FILES,
    SHARED_FILES,
)
from ubuntuone.controlpanel.gui.tests import USER_HOME
from ubuntuone.controlpanel.gui.qt import share_links as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    BaseTestCase,
    FakeDesktopService,
)


# pylint: disable=W0212

class ShareLinksTestCase(BaseTestCase):
    """Test the qt control panel."""

    innerclass_ui = gui.share_links_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.ShareLinksPanel

    @defer.inlineCallbacks
    def setUp(self):
        yield super(ShareLinksTestCase, self).setUp()

    def test_setup(self):
        """Check that the strings are properly setted."""
        self.assertEqual(self.ui.ui.search_files_lbl.text(), SEARCH_FILES)
        self.assertEqual(self.ui.ui.shared_group.title(), SHARED_FILES)
        self.assertIsInstance(self.ui._enhanced_line, gui.EnhancedLineEdit)
        self.assertEqual(self.ui._enhanced_line.btn_operation.text(), '')
        self.assertFalse(self.ui._enhanced_line.btn_operation.isVisible())
        self.assertEqual(self.ui.ui.open_in_browser.cursor().shape(),
            QtCore.Qt.PointingHandCursor)
        self.assertEqual(self.ui.ui.back_to_file_list.cursor().shape(),
            QtCore.Qt.PointingHandCursor)
        self.assertFalse(self.ui._enhanced_line.btn_operation.isVisible())

    def test_share_file(self):
        """Check that the state of the widgets on share_file."""
        data = []
        self.patch(self.ui.backend, "set_public_access_changed_handler",
            data.append)
        self.patch(self.ui.backend, "set_public_access_change_error_handler",
            data.append)
        path = '/home/user/Ubuntu One/file1.txt'
        self.ui.share_file(path)
        self.assertTrue(self.ui.is_processing)
        widget = self.ui.ui.hbox_share_file.takeAt(0).widget()
        self.assertIsInstance(widget, gui.ShareFileWidget)
        self.assertEqual(widget.ui.lbl_filename.text(),
            os.path.basename(path))
        self.assertEqual(widget.ui.lbl_path.text(), path)
        self.assertEqual(len(data), 2)
        self.assertEqual(data[0], self.ui._file_shared)

    def test_open_file_properties(self):
        """Check that we clean the Details Page before adding a new item."""
        publicfiles = [
            {'path': '/home/file1', 'public_url': 'http:ubuntuone.com/asd123'},
            {'path': '/home/file2', 'public_url': 'http:ubuntuone.com/qwe456'},
        ]
        self.ui._load_public_files(publicfiles)
        item = self.ui.ui.tree_shared_files.topLevelItem(0)
        self.ui.ui.tree_shared_files.itemDoubleClicked.emit(item, 0)
        self.assertEqual(self.ui.ui.hbox_share_file.count(), 1)
        self.ui.ui.tree_shared_files.itemDoubleClicked.emit(item, 0)
        self.assertEqual(self.ui.ui.hbox_share_file.count(), 1)

    def test_share_file_actions(self):
        """Check the behaviour of share_file buttons."""
        path = '/home/user/Ubuntu One/file1.txt'
        self.ui.share_file(path)
        widget = self.ui.ui.hbox_share_file.takeAt(0).widget()
        self.ui.ui.line_copy_link.setText('link')
        self.assertEqual(self.ui.ui.line_copy_link.text(), 'link')
        widget.linkDisabled.emit()
        self.assertNotEqual(self.ui.ui.line_copy_link.text(), 'link')

    def test_file_shared(self):
        """Check the behavior of the widgets after the file is shared."""
        info = {'public_url': 'http://ubuntuone.com/asd123'}
        self.ui._file_shared(info)
        self.assertEqual(self.ui.ui.line_copy_link.text(), info['public_url'])
        self.assertEqual(self.ui.ui.stacked_widget.currentIndex(),
            gui.FILE_DETAILS_INDEX)
        self.assertFalse(self.ui.is_processing)

    def test_file_already_shared(self):
        """Check the behavior of the widgets when there is a shared file."""
        data = []

        def fake_method(self, *args):
            """Fake callback."""
            data.append((args))

        self.patch(self.ui.backend, "change_public_access", fake_method)
        path = '/home/user/Ubuntu One/file1.txt'
        shared = {
            '/home/user/Ubuntu One/file1.txt': 'http://ubuntuone.com/asd123'}
        self.ui._shared_files = shared
        self.ui.share_file(path)
        self.assertEqual(self.ui.ui.line_copy_link.text(), shared[path])
        self.assertEqual(self.ui.ui.stacked_widget.currentIndex(),
            gui.FILE_DETAILS_INDEX)
        self.assertEqual(data, [])

    def test_open_in_browser(self):
        """Test the execution of open_in_browser."""
        fake_desktop_service = FakeDesktopService()
        self.patch(QtGui, "QDesktopServices", fake_desktop_service)
        url = 'http://ubuntuone.com/asd123'
        self.ui.ui.line_copy_link.setText(url)
        self.ui._open_in_browser()
        expected = QtCore.QUrl(url)
        self.assertEqual(expected, fake_desktop_service.opened_url)

    def test_copy_link_from_line(self):
        """Test the execution of copy_link_from_line."""
        url = 'http://ubuntuone.com/asd123'
        self.ui.ui.line_copy_link.setText(url)
        self.ui._copy_link_from_line()
        clip = QtGui.QApplication.instance().clipboard()
        self.assertEqual(url, clip.text())

    def test_move_to_main_list(self):
        """Test that the stacked widget shows the proper index."""
        self.ui._move_to_main_list()
        self.assertEqual(self.ui.ui.stacked_widget.currentIndex(),
            gui.SHARED_FILES_INDEX)

    def test_get_public_files(self):
        """Test that the proper actions are executed on files requested.."""
        data = []
        self.patch(self.ui.backend, "set_public_files_list_handler",
            data.append)
        self.ui.get_public_files()
        self.assertTrue(self.ui.is_processing)
        self.assertEqual(self.ui.ui.stacked_widget.currentIndex(),
            gui.SHARED_FILES_INDEX)
        self.assertEqual(self.ui.home_dir, USER_HOME)
        self.assertEqual(len(data), 1)
        self.assertEqual(data[0], self.ui._load_public_files)

    def test_line_close_btn(self):
        """Check that the popup is hidden."""
        self.ui.ui.line_search.popup.show()
        self.addCleanup(self.ui.ui.line_search.popup.hide)
        self.ui._line_close_btn()
        self.assertFalse(self.ui.ui.line_search.popup.isVisible())
        self.assertFalse(self.ui._enhanced_line.btn_operation.isVisible())

    def test_hide_line_btn_close_show(self):
        """Check the state of the inline button."""
        self.ui.ui.line_search.popup.popupShown.emit()
        self.assertTrue(self.ui._enhanced_line.btn_operation.isVisible())

    def test_load_public_files(self):
        """Test if the list of public files is loaded properly."""
        publicfiles = [
            {'path': '/home/file1', 'public_url': 'http:ubuntuone.com/asd123'},
            {'path': '/home/file2', 'public_url': 'http:ubuntuone.com/qwe456'},
        ]
        self.ui._load_public_files(publicfiles)
        item = self.ui.ui.tree_shared_files.topLevelItem(0)
        self.assertEqual(item.text(0), os.path.basename('/home/file1'))
        self.assertEqual(item.toolTip(0), '/home/file1')
        label = self.ui.ui.tree_shared_files.itemWidget(item, 1)
        link = ('<a href="%s"><span style="font-size: 12px;'
                'color: #dd4814";>%s</span></a>'
                % ('http:ubuntuone.com/asd123', 'http:ubuntuone.com/asd123'))
        self.assertEqual(link, label.text())
        actions = self.ui.ui.tree_shared_files.itemWidget(item, 2)
        self.assertIsInstance(actions, gui.ActionsButtons)

        item = self.ui.ui.tree_shared_files.topLevelItem(1)
        self.assertEqual(item.text(0), os.path.basename('/home/file2'))
        self.assertEqual(item.toolTip(0), '/home/file2')
        label = self.ui.ui.tree_shared_files.itemWidget(item, 1)
        link = ('<a href="%s"><span style="font-size: 12px;'
                'color: #dd4814";>%s</span></a>'
                % ('http:ubuntuone.com/qwe456', 'http:ubuntuone.com/qwe456'))
        self.assertEqual(link, label.text())
        actions = self.ui.ui.tree_shared_files.itemWidget(item, 2)
        self.assertIsInstance(actions, gui.ActionsButtons)

    def test_load_search_results(self):
        """Test if the list of public files is loaded properly."""
        results = ['/home/file1', '/home/file2']
        self.ui._load_search_results(results, 'file')
        item = self.ui.ui.tree_search_results.topLevelItem(0)
        self.assertEqual(item.text(0), os.path.basename('/home/file1'))
        self.assertEqual(item.toolTip(0), '/home/file1')
        actions = self.ui.ui.tree_search_results.itemWidget(item, 1)
        self.assertIsInstance(actions, gui.ActionsResultsButtons)

        item = self.ui.ui.tree_search_results.topLevelItem(1)
        self.assertEqual(item.text(0), os.path.basename('/home/file2'))
        self.assertEqual(item.toolTip(0), '/home/file2')
        actions = self.ui.ui.tree_search_results.itemWidget(item, 1)
        self.assertIsInstance(actions, gui.ActionsResultsButtons)

        self.assertEqual(self.ui.ui.label_search_results.text(),
            gui.SEARCH_RESULTS % 'file')
        self.assertTrue(self.ui.ui.tree_search_results.isVisible())
        self.assertFalse(self.ui.ui.label_no_results.isVisible())

    def test_load_search_results_empty_list(self):
        """Test if the list of public files is loaded properly."""
        self.ui._load_search_results([], 'file')
        self.assertEqual(self.ui.ui.label_search_results.text(),
            gui.SEARCH_RESULTS % 'file')
        self.assertFalse(self.ui.ui.tree_search_results.isVisible())
        self.assertTrue(self.ui.ui.label_no_results.isVisible())

    def test_tree_item_clicked(self):
        """Check if the proper info is displayed when the item is clicked."""
        publicfiles = [
            {'path': '/home/file1', 'public_url': 'http:ubuntuone.com/asd123'},
            {'path': '/home/file2', 'public_url': 'http:ubuntuone.com/qwe456'},
        ]
        self.ui._load_public_files(publicfiles)
        item = self.ui.ui.tree_shared_files.topLevelItem(1)
        self.ui.ui.tree_shared_files.itemDoubleClicked.emit(item, 0)
        widget = self.ui.ui.hbox_share_file.itemAt(0).widget()

        file_path = unicode(item.toolTip(gui.FILE_NAME_COL))
        full_path = gui.expand_user(file_path.encode('utf-8')).decode('utf-8')
        self.assertEqual(widget.file_path, full_path)
        self.assertEqual(widget.ui.lbl_filename.text(),
            os.path.basename(file_path))
        self.assertEqual(widget.ui.lbl_path.text(), file_path)

    def test_hide_popup_on_tab_changed(self):
        """Test that the popup is hidden by the tab changed signal."""

        self.ui.ui.line_search.popup.show()
        self.ui.handle_current_tab_changed(0)
        self.assertFalse(self.ui.ui.line_search.popup.isVisible())
        self.assertFalse(self.ui.ui.line_search.hasFocus())


class ActionsButtonsTestCase(BaseTestCase):
    """Test the Actions Buttons."""

    def test_open(self):
        """Test the open method."""
        path = '/home/file1'
        link = 'http://ubuntuone.com/asd123'
        actions = gui.ActionsButtons(path, link)
        fake_desktop_service = FakeDesktopService()
        self.patch(QtGui, "QDesktopServices", fake_desktop_service)
        actions.open()
        file_path = QtCore.QUrl(u'file://%s' % path)
        self.assertEqual(file_path, fake_desktop_service.opened_url)

    def test_copy(self):
        """Test that the link is copied into the clipboard.."""
        path = '/home/file1'
        link = 'http://ubuntuone.com/asd123'
        actions = gui.ActionsButtons(path, link)
        fake_desktop_service = FakeDesktopService()
        self.patch(QtGui, "QDesktopServices", fake_desktop_service)
        actions.copy()
        clip = QtGui.QApplication.instance().clipboard()
        self.assertEqual(link, clip.text())


class EnhancedLineEditTestCase(BaseTestCase):
    """Test the EnhancedLineEdit."""

    def test_initialize(self):
        """Test initialization."""
        line = QtGui.QLineEdit()
        enhanced = gui.EnhancedLineEdit(line, self._set_called, 'text')
        self.assertEqual(line.layout().count(), 2)
        self.assertFalse(self._called)
        enhanced.btn_operation.click()
        self.assertEqual(self._called, ((False, ), {}))
