/* update-notifier.c
 * Copyright (C) 2004 Lukas Lipka <lukas@pmad.net>
 *           (C) 2004 Michael Vogt <mvo@debian.org>
 *           (C) Canonical
 *           (C) 2004 Michiel Sikkes <michiel@eyesopened.nl>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <signal.h>
#include <gnome.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "update-notifier.h"
#include "eggtrayicon.h"
#include "hal.h"
#include "update.h"
#include "hooks.h"

/* some prototypes */
extern gboolean up_get_clipboard (void);
gboolean update_timer_finished(gpointer data);
static void tray_destroyed_cb(GtkWidget *widget, void *data);

#define UPDATE_ICON_FILE PACKAGE_DATA_DIR"/pixmaps/update-icon.png"
#define HOOK_ICON_FILE PACKAGE_DATA_DIR"/pixmaps/hook-notifier.png"

// the time when we check for fam events
#define TIMEOUT_FAM 1000*5 /* 5 sec */

// the timeout (in msec) for apt-get update (changes in 
// /var/lib/apt/lists{/partial})
#define TIMEOUT_APT_GET_UPDATE 1000*30 /* 30 sec */

// the timeout (in sec) for a notification after apt/dpkg have finished
#define TIMEOUT_APT_RUN 60 /* 60 sec */


void
invoke_with_gksu(gchar *cmd, gchar *descr)
{
        gchar *full_descr = NULL;
	if(descr != NULL)
	   full_descr = g_strdup_printf(_("Please enter your password to run:\n %s"), descr);

        //g_print("invoke_update_manager ()\n");
	gchar *argv[5];
	argv[0] = "/usr/bin/gksudo";
	argv[1] = "--message";
	argv[2] = full_descr;
	argv[3] = cmd;
	argv[4] = NULL;

	g_spawn_async (NULL, argv, NULL, 0, NULL, NULL, NULL, NULL);
	g_free(full_descr);
}



gboolean
trayapplet_create (TrayApplet *un, char *name, char *iconfile)
{
        //g_print("trayicon_create()\n");
	GdkPixbuf *pixbuf;

	/* setup widgets */
        un->tooltip = gtk_tooltips_new ();
	un->tray_icon = egg_tray_icon_new (name);
	gtk_widget_show(un->tray_icon);
	un->eventbox = gtk_event_box_new ();
	gtk_widget_show(un->eventbox);
	un->icon = gtk_image_new ();
	gtk_widget_show(un->icon);

	/* load icon */
	pixbuf = gtk_image_get_pixbuf (GTK_IMAGE(un->icon));
	if (pixbuf)
		g_object_unref (G_OBJECT (pixbuf));
	pixbuf = gdk_pixbuf_new_from_file (iconfile, NULL);
	gtk_image_set_from_pixbuf (GTK_IMAGE (un->icon), pixbuf);

	/* build container */
	gtk_container_add (GTK_CONTAINER (un->eventbox), un->icon);
	gtk_container_add (GTK_CONTAINER (un->tray_icon), un->eventbox);
	gtk_widget_hide(un->tray_icon);

	return TRUE;
}



/* 
 the following files change:
 on "install":
  - /var/lib/dpkg/lock
  - /var/lib/dpkg/ *
  - /var/lib/update-notifier/dpkg-run-stamp
 on "update":
  - /var/lib/apt/lists/lock
  - /var/lib/apt/lists/ *
  - /var/lib/dpkg/lock
*/
gboolean 
fam_check(UpgradeNotifier *un)
{
   FAMEvent fe;
   gboolean dpkg_was_run = FALSE;
   gboolean apt_get_update_runing  = FALSE;
   gboolean new_update_hook = FALSE;
   static gboolean hook_pending = FALSE;
   static time_t last_apt_action = 0;
   int i;

   while(FAMPending(&un->fc)) {
      if(FAMNextEvent(&un->fc, &fe) < 0) {
	 g_critical("FAMNextEvent() failed");
	 break;
      }
      //       g_print("FAMPending() %i %s (reqnum %i)\n", 
      // 	      fe.code,fe.filename, fe.fr.reqnum); 

      if(!(fe.code == FAMChanged || fe.code == FAMDeleted 
	   || fe.code == FAMCreated))
	 continue;

      // we ignore lock file events because we can only get 
      // when a lock was taken, but not when it was removed
      if(g_str_has_suffix(fe.filename, "lock"))
	 continue;

      /* apt-get update */
      for(i=0;un->frq_apt[i].reqnum != -1; i++) {
	 if(fe.fr.reqnum == un->frq_apt[i].reqnum) {
	    // a timer is already runing, stop it and start a new one
	    if(un->update_finished_timer > 0) {
	       //g_print("removing old timer: %i\n", un->update_finished_timer);
	       g_source_remove(un->update_finished_timer);
	       un->update_finished_timer = 0;
	    }
	    apt_get_update_runing=TRUE;
	    break;
	 }
      }

      /* apt-get install/remove/upgrade run */
      if(strstr(fe.filename, "/var/lib/update-notifier/dpkg-run-stamp")) 
	 dpkg_was_run = TRUE;

      /* a update hook was registered */
      if(fe.fr.reqnum == un->frq_hooks.reqnum) 
	 new_update_hook = TRUE;
      
   }

   // FIXME: this is a stupid workaround because stupid gamin/inotify
   //        is oh so broken
   if(!dpkg_was_run) {
      static struct stat last_buf,buf;
      stat("/var/lib/update-notifier/dpkg-run-stamp", &buf);
      if(memcmp(&buf,&last_buf,sizeof(struct stat)) != 0) {
	 dpkg_was_run=TRUE;
	 last_buf = buf;
      }
   }

   if(dpkg_was_run) {
      update_check(un->update);
      // the update must be finished, otherwise apt-get install 
      // wouldn't run 
      if(un->update_finished_timer > 0) {
	 update_apt_is_running(un->update, FALSE);
	 g_source_remove(un->update_finished_timer);
      }
   } 

   if(apt_get_update_runing) {
      update_apt_is_running(un->update, TRUE);
      un->update_finished_timer = g_timeout_add(TIMEOUT_APT_GET_UPDATE,
						update_timer_finished, un);
   }

   // show the hooks only after apt/dpkg are finished
   time_t now = time(NULL);

   if(new_update_hook) {
      hook_pending = TRUE;
      last_apt_action = now;
   }

   // if there is a hook pending, check that no apt/dpkg/etc is
   // runing within the last N-sec
   if(hook_pending) {

      //g_print("hook_pending: now=%i last_action=%i (%i) \n", now, last_apt_action,now-last_apt_action);

      if(apt_get_update_runing || dpkg_was_run) {
	 last_apt_action = now;
      }

      if( (now - last_apt_action) >  TIMEOUT_APT_RUN) {
	 //g_print("checking hooks now\n");
	 check_update_hooks(un->hook);
	 hook_pending = FALSE;
	 last_apt_action = 0;
      }
   }

   return TRUE;
}



/* u_abort seems like an internal error notification.
 * End user might not understand the message at all */
void u_abort(gchar *msg)
{
   msg = g_strdup_printf("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s\n", _("Internal error"), msg);
   gtk_dialog_run(gtk_message_dialog_new_with_markup(NULL,0,
						     GTK_MESSAGE_ERROR,
						     GTK_BUTTONS_CLOSE,
						     msg));
   g_free(msg);
   exit(1);
}

// FIXME: get the apt directories with apt-config or something
gboolean fam_init(UpgradeNotifier *un)
{
   FAMRequest frq;

   // FIXME: if the number of directories is increased that are monitored
   // for apt, make sure that they alaway fir in UpgradeNotifier.frq_apt
   int aptindx=0;

   /* setup FAM/GAMIN, monitor the importend directories */
   if(FAMOpen(&un->fc) >= 0) {

      // monitor for apt getting updates
      if(FAMMonitorDirectory(&un->fc, "/var/lib/apt/lists/", 
			     &un->frq_apt[aptindx++], NULL)) 
	 u_abort("FAMMonitorFile() failed, make sure that fam/gamin "
		 "is installed");

      if(FAMMonitorDirectory(&un->fc, "/var/lib/apt/lists/partial/", 
			     &un->frq_apt[aptindx++], NULL))
	 u_abort("FAMMonitorFile() failed, , make sure that fam/gamin "
		 "is installed");

      // monitor for apt geting packages
      if(FAMMonitorDirectory(&un->fc, "/var/cache/apt/archives/", 
			     &un->frq_apt[aptindx++], NULL))
	 u_abort("FAMMonitorFile() failed, , make sure that fam/gamin "
		 "is installed");

      if(FAMMonitorDirectory(&un->fc, "/var/cache/apt/archives/partial/", 
			     &un->frq_apt[aptindx++], NULL))
	 u_abort("FAMMonitorFile() failed, , make sure that fam/gamin "
		 "is installed");

      // dpkg database
      if(FAMMonitorFile(&un->fc, 
			"/var/lib/dpkg/status", 
			&un->frq_apt[aptindx++], NULL))
	 u_abort("FAMMonitorFile() failed, make sure that fam/gamin "
		 "is installed");

      // monitor for hooks
      if(FAMMonitorDirectory(&un->fc, HOOKS_DIR, 
			     &un->frq_hooks, NULL))
	 u_abort("FAMMonitorFile() failed, make sure that fam/gamin "
		 "is installed");

      // monitor for installs
      if(FAMMonitorFile(&un->fc, 
			"/var/lib/update-notifier/dpkg-run-stamp", 
			&frq, NULL))
	 u_abort("FAMMonitorFile() failed, make sure that fam/gamin "
		 "is installed");
   } else {
      u_abort("FAMOpen() failed, make sure that fam/gamin "
	      "is installed");
   }

   un->frq_apt[aptindx].reqnum = -1;

   return TRUE;
}




static gboolean
tray_icons_init(UpgradeNotifier *un)
{
	/* new upates tray icon */
	un->update = g_new0 (TrayApplet, 1);
	trayapplet_create(un->update, "update-notifier", UPDATE_ICON_FILE);
	update_tray_icon_init(un->update);

	/* a destroy signal handler (e.g. when the gnome-panel is restarted */
	g_signal_connect(G_OBJECT(un->update->tray_icon), "destroy", 
			 G_CALLBACK(tray_destroyed_cb), un);


	/* update hook icon*/
	un->hook = g_new0 (TrayApplet, 1);
	trayapplet_create(un->hook, "hook-notifier", HOOK_ICON_FILE);
	hook_tray_icon_init(un->hook);
	
	return FALSE; // for the tray_destroyed_cb
}

static void
tray_destroyed_cb(GtkWidget *widget, void *data)
{
        g_critical("tray icon destroyed\n");
	// FIXME: handle this situation
	g_timeout_add(1000, (GSourceFunc*)(tray_icons_init), data);
}


int 
main (int argc, char *argv[])
{
	GnomeClient *client;
	UpgradeNotifier *un;

	gnome_program_init (PACKAGE, PACKAGE_VERSION, 
			    LIBGNOMEUI_MODULE,
			    argc, argv, 
			    GNOME_PARAM_NONE);

	notify_init("update-notifier");
	notify_glib_init("update-notifier", NULL);

        bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
        bind_textdomain_codeset(PACKAGE, "UTF-8");
        textdomain(PACKAGE);

	//g_print("starting update-notifier\n");
	
	client = gnome_master_client ();
	if (up_get_clipboard ())
	  gnome_client_set_restart_style (client, GNOME_RESTART_ANYWAY);
	else {
	   gnome_client_set_restart_style (client, GNOME_RESTART_NEVER);
	   g_warning ("already running?\n");
	   return 1;
	}

	/* Make sure we die when the session dies */
	g_signal_connect (G_OBJECT (client), "die",
			  G_CALLBACK (gtk_main_quit), NULL);
	
	/* Create the UpgradeNotifier object */
	un = g_new0 (UpgradeNotifier, 1);

	// check for .update-notifier dir and create if needed
	gchar *dirname = g_strdup_printf("%s/.update-notifier",
					 g_get_home_dir());
	if(!g_file_test(dirname, G_FILE_TEST_IS_DIR))
	   g_mkdir(dirname, 0700);
	g_free(dirname);

	// create the icons 
	tray_icons_init(un);

	// init hal (needs to be done _after_ the icons are created)
	/* setup hal so that inserted cdroms can be checked */
	LibHalContext *hal_ctx = up_do_hal_init();
	if (!hal_ctx) {
	   u_abort("failed to initialize HAL!\n");
	   return 1;
	}
	libhal_ctx_set_user_data(hal_ctx, un);
	
	// init fam
	fam_init(un);
	
	/* Set a time-out for the fam check */
	g_timeout_add (TIMEOUT_FAM, (GSourceFunc)fam_check, un);

	/* Start the main gtk loop */
	gtk_main ();

	FAMClose(&un->fc);
	return 0;
}
