#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>
#include <sys/types.h>
#include <sys/wait.h>


#include "update-notifier.h"
#include "hal.h"

#define CDROM_CHECKER PACKAGE_LIB_DIR"/update-notifier/apt-cdrom-check"

/* reposonses for the dialog */
enum {
   RES_START_PM=1,
   RES_DIST_UPGRADER=2
};

/* Returnvalues from apt-cdrom-check:
    # 0 - no ubuntu CD
    # 1 - CD with packages 
    # 2 - dist-upgrader CD
*/
enum {
   NO_CD, 
   CD_WITH_PACKAGES, 
   CD_WITH_DISTUPGRADER
};

static LibHalContext *hal_ctx;
static DBusConnection *dbus_connection;

void distro_cd_detected(UpgradeNotifier *un, int cdtype, char *mount_point)
{
   GtkWidget *dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL,
					      GTK_MESSAGE_QUESTION, 
					      GTK_BUTTONS_NONE,
					      NULL );
   gchar *title, *markup;
   switch(cdtype) {
   case CD_WITH_PACKAGES:
      title = _("Software packages volume detected");
      markup = _("<span weight=\"bold\" size=\"larger\">"
	    "A volume with software packages has "
	    "been detected.</span>\n\n"
	    "Would you like to open it with the "
	    "package manager?");
      gtk_dialog_add_buttons(GTK_DIALOG(dialog), 
			     GTK_STOCK_CANCEL,
			     GTK_RESPONSE_REJECT,
			     _("Start package manager"), 
			     RES_START_PM,
			     NULL);
      break;
   case CD_WITH_DISTUPGRADER:
      title = _("Upgrade volume detected");
      markup = _("<span weight=\"bold\" size=\"larger\">"
	    "A distribution volume with software packages has "
	    "been detected.</span>\n\n"
	    "Would you like to try to upgrade from it automatically? ");
      gtk_dialog_add_buttons(GTK_DIALOG(dialog), 
			     GTK_STOCK_CANCEL,
			     GTK_RESPONSE_REJECT,
			     _("Start package manager"), 
			     RES_START_PM,
			     _("Run upgrade"), 
			     RES_DIST_UPGRADER,
			     NULL);

      break;
   default:
      g_assert_not_reached();
   }

   gtk_window_set_title(GTK_WINDOW(dialog), title);
   gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);
   gtk_message_dialog_set_markup(GTK_MESSAGE_DIALOG(dialog), markup);

   int res = gtk_dialog_run (GTK_DIALOG (dialog));
   char *cmd;
   switch(res) {
   gchar *argv[3];
   case RES_START_PM:
      cmd = g_strdup_printf("synaptic --add-cdrom %s",mount_point);
      invoke_with_gksu(cmd, "/usr/share/applications/synaptic.desktop");
      break;
   case RES_DIST_UPGRADER:
      argv[0] = "/usr/lib/update-notifier/cddistupgrader";
      argv[1] = mount_point;
      argv[2] = NULL;
      g_spawn_async (NULL, argv, NULL, 0, NULL, NULL, NULL, NULL);
      break;
   default:
      /* do nothing */
      break;
   }
   gtk_widget_destroy (dialog);
}


/** Internal UP initialization function
 *
 * @functions			The LibHalFunctions to register as callbacks.
 * @return			The LibHalContext of the HAL connection or
 *				NULL on error.
 */
LibHalContext *
up_do_hal_init ()
{
        DBusError error;
	char **devices;
	int nr;

	hal_ctx = libhal_ctx_new ();
	if (!hal_ctx) {
	        g_warning ("failed to initialize HAL!\n");
		return NULL;
	}

	dbus_error_init (&error);
	if (!hal_mainloop_integration (hal_ctx, &error)) {
		g_warning ("hal_initialize failed: %s\n", error.message);
		dbus_error_free (&error);
		return NULL;
	}

	libhal_ctx_set_device_property_modified (hal_ctx,
		  				 hal_property_modified);

	if (!libhal_device_property_watch_all (hal_ctx, &error)) {
	        g_warning ("failed to watch all HAL properties!: %s\n", error.message);
		libhal_ctx_shutdown (hal_ctx, &error);
		return NULL;
	}

	if (!libhal_ctx_init (hal_ctx, &error)) {
		warn ("hal_initialize failed: %s\n", error.message);
		dbus_error_free (&error);
		libhal_ctx_free (hal_ctx);
		return NULL;
	}

	/*
	 * Do something to ping the HAL daemon - the above functions will
	 * succeed even if hald is not running, so long as DBUS is.  But we
	 * want to exit silently if hald is not running, to behave on
	 * pre-2.6 systems.
	 */
	devices = libhal_get_all_devices (hal_ctx, &nr, &error);
	if (!devices) {
		g_warning ("seems that HAL is not running\n");
		libhal_ctx_shutdown (hal_ctx, &error);
		if (dbus_error_is_set (&error))
			dbus_error_free (&error);
		libhal_ctx_free (hal_ctx);
		return NULL;
	}
	libhal_free_string_array (devices);

	return hal_ctx;
}

/** Invoked by GLib in response to a D-BUS disconnect event.
 *
 * @param  data                 Context pointer
 */
static gboolean
up_reconnect_to_hal (gpointer data __attribute__((__unused__)))
{
	static unsigned int retries = 0;

	g_message ("Trying a reconnect ...");
	hal_ctx = up_do_hal_init ();
	if (hal_ctx != NULL) {
		g_message ("Reconnected OK.");
		retries = 0;
		return FALSE;
	} else if (dbus_connection) { 
	   dbus_connection_unref (dbus_connection);
	   dbus_connection = NULL;
	}

	/* Retry later if it failed. */
	if (retries++ < 6000)
		return TRUE;

	/* Too many retries; clean up and bail. */
	warn("gvm_reconnect_to_hal: no reconnection after 6000 retries, "
	     "exiting\n");

	/* Too many retries; clean up and bail. */
	gtk_main_quit ();
	return FALSE;
}

static DBusHandlerResult
up_do_filter_dbus_msg (DBusConnection *connection __attribute__((__unused__)),
                        DBusMessage *message,
                        void *user_data __attribute__((__unused__)))
{
        DBusError error;

	if (dbus_message_is_signal (message,
	                            DBUS_INTERFACE_LOCAL,
	                            "Disconnected")) {
	        g_timeout_add(1000, up_reconnect_to_hal, NULL);
		libhal_ctx_shutdown (hal_ctx, &error);
		libhal_ctx_free (hal_ctx);
		hal_ctx = NULL;
		dbus_connection_unref (dbus_connection);
		dbus_connection = NULL;
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	else
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static dbus_bool_t
hal_mainloop_integration (LibHalContext *ctx, 
			   DBusError *error)
			  
{

	dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, error);

	if (dbus_error_is_set (error))
		return FALSE;

	dbus_connection_set_exit_on_disconnect (dbus_connection, FALSE);

        dbus_connection_setup_with_g_main (dbus_connection, NULL);
	dbus_connection_add_filter (dbus_connection, up_do_filter_dbus_msg, 
				    NULL, NULL);
	
	libhal_ctx_set_dbus_connection (ctx, dbus_connection);
	
	return TRUE;
}

/** Type for callback when a property of a device changes. 
 *
 *  @param  udi                 Unique Device Id
 *  @param  key                 Name of the property that has changed
 *  @param  is_removed          Property removed
 *  @param  is_added            Property added
 */
void 
hal_property_modified(LibHalContext *ctx, const char *udi, const char *key,
		      dbus_bool_t is_removed, dbus_bool_t is_added)
{
   char *mount_point;
   //g_print("hal_device_changed(): %s\n",key);

   /* we are only interessted in mount events from the cdrom*/
   if (g_strcasecmp (key, "volume.is_mounted") != 0)
      return;
   if(!libhal_device_get_property_bool(ctx, udi, key, NULL))
      return;

   char* storage_device = libhal_device_get_property_string (ctx, udi,
						    "block.storage_device",
							     NULL);
   if (!storage_device) {
      g_warning("cannot get storage_device\n");
      return;
   }
   char* media_type = libhal_device_get_property_string (ctx, storage_device, 
						      "storage.drive_type",
						      NULL);
   if(!media_type) {
      g_warning("cannot get storage.drive_type\n");
      return;
   }
   if (g_ascii_strcasecmp(media_type, "cdrom") != 0) {
      g_warning("no cdrom: %s\n",media_type);
      return;
   }
      
   mount_point = libhal_device_get_property_string (ctx, udi, 
						    "volume.mount_point",
						    NULL);
   //g_print("mounted: %s on %s\n", udi, mount_point);

   char *ubuntu_dir = g_strdup_printf("%s/ubuntu",mount_point);
   if(!g_file_test (ubuntu_dir, G_FILE_TEST_IS_SYMLINK)) {
      g_free(ubuntu_dir);
      return;
   }
   g_free(ubuntu_dir);

   /* this looks like a ubuntu CD, run the checker script to verify
    * this. We expect the following return codes:
    # 0 - no ubuntu CD
    # 1 - CD with packages 
    # 2 - dist-upgrader CD
    * (see data/apt-cdrom-check)
    */
   //g_print("this looks like a ubuntu-cdrom\n");
   char *cmd = g_strdup_printf(CDROM_CHECKER" %s",mount_point);
   int retval=-1;
   g_spawn_command_line_sync(cmd, NULL, NULL,  &retval, NULL);
   
   //g_print("retval: %i \n", WEXITSTATUS(retval));
   int cdtype = WEXITSTATUS(retval);
   if(cdtype > 0) {
     distro_cd_detected(libhal_ctx_get_user_data(ctx), cdtype, mount_point);
   }

   g_free(cmd);
}

