#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glade/glade.h>   
#include <libintl.h>
#include <sys/wait.h>

#include "update-notifier.h"
#include "update.h"

#define UPGRADE_CHECKER PACKAGE_LIB_DIR"/update-notifier/apt-check"

// command, description, desktopfile, needs_gksu
char* actions[][4] = {
   { "/usr/bin/update-manager", N_("Show updates"), 
     "/usr/share/applications/update-manager.desktop", FALSE },

   { "/usr/sbin/synaptic --dist-upgrade-mode --non-interactive --hide-main-window -o Synaptic::AskRelated=true",
     N_("Install all updates"), "/usr/share/applications/synaptic.desktop", TRUE
   },
   { "/usr/sbin/synaptic --update-at-startup --non-interactive --hide-main-window", 
     N_("Check for updates"), "/usr/share/applications/synaptic.desktop", TRUE },
   { "/usr/sbin/synaptic", N_("Start package manager"), 
     "/usr/share/applications/synaptic.desktop", TRUE},
   { NULL, NULL, NULL }
};


enum { NOTIFICATION_DEFAULT, NOTIFICATION_IGNORE, NOTIFICATION_SHOW_UPDATES };

static gboolean
activate_cb (GtkWidget *widget, 
	     TrayApplet *ta)
{
   int index = 0;
   invoke (actions[index][0], actions[index][2], (long)actions[index][3]);
   return TRUE;
}

static gboolean
popup_cb (GtkStatusIcon *status_icon,
	  guint          button,
	  guint          activate_time,
	  TrayApplet     *un)
{
   gtk_menu_set_screen (GTK_MENU (un->menu),
			gtk_status_icon_get_screen(un->tray_icon));
   gtk_menu_popup (GTK_MENU (un->menu), NULL, NULL, 
		   gtk_status_icon_position_menu, un->tray_icon,
		   button, activate_time);
   return TRUE;
}

void
update_trayicon_update_tooltip (TrayApplet *ta, int num_upgrades)
{
   //g_print("update_tooltip: %p %p %p\n", ta, ta->tooltip, ta->eventbox);
   gchar *updates;

   updates = g_strdup_printf(ngettext("There is %i update available",
				      "There are %i updates available",
				      num_upgrades),
			     num_upgrades);
   gtk_status_icon_set_tooltip(ta->tray_icon, updates);
   g_free(updates);
}


void 
cb_action(GObject *self, void *user_data)
{
   TrayApplet *ta = user_data;
	
   int i = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(self), "action"));
   invoke (actions[i][0], _(actions[i][2]), (long)actions[i][3]);

   UpdateTrayAppletPrivate *priv = (UpdateTrayAppletPrivate*)ta->user_data;
   gconf_client_set_int(priv->gconf, GCONF_KEY_DEFAULT_ACTION, i, NULL);
}

void 
cb_preferences(GObject *self, void *user_data)
{
   invoke_with_gksu("/usr/bin/software-properties-gtk",
		    "/usr/share/applications/software-properties.desktop",
		    FALSE);    
}

void 
cb_toggled_show_notifications(GObject *self, void *data)
{
      TrayApplet *ta = (TrayApplet*)data;
      UpdateTrayAppletPrivate *priv = (UpdateTrayAppletPrivate*)ta->user_data;

      gboolean b = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(self));
      gconf_client_set_bool(priv->gconf, GCONF_KEY_NO_UPDATE_NOTIFICATIONS, 
			    !b,NULL);
      

      NotifyNotification *n = priv->active_notification;
      if(n != NULL) {
	 notify_notification_close(n, NULL);
	 priv->active_notification = NULL;
      }
}


void
update_trayicon_create_menu(TrayApplet *ta)
{
	GtkWidget *menuitem;
	GtkAccelGroup *accelgroup;
	int i;

	ta->menu = gtk_menu_new ();

	for(i=0;actions[i][0]!=NULL;i++) {
	   menuitem = gtk_menu_item_new_with_label (_(actions[i][1]));
	   gtk_menu_shell_append (GTK_MENU_SHELL (ta->menu), menuitem);
	   g_object_set_data(G_OBJECT(menuitem), "action", GINT_TO_POINTER(i));
	   g_signal_connect(G_OBJECT(menuitem), "activate", 
			    G_CALLBACK(cb_action), ta);
	}

	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append (GTK_MENU_SHELL (ta->menu), menuitem);

	menuitem = gtk_check_menu_item_new_with_label(_("Show notifications"));
	UpdateTrayAppletPrivate *priv = (UpdateTrayAppletPrivate*)ta->user_data;
	gboolean b = gconf_client_get_bool(priv->gconf,
					   GCONF_KEY_NO_UPDATE_NOTIFICATIONS, 
					   NULL);
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem), !b);
	gtk_menu_shell_append (GTK_MENU_SHELL (ta->menu), menuitem);
	g_signal_connect(G_OBJECT(menuitem), "toggled", 
			 G_CALLBACK(cb_toggled_show_notifications), ta);
	

	menuitem = gtk_image_menu_item_new_from_stock (GTK_STOCK_PREFERENCES, accelgroup);
	gtk_menu_shell_append (GTK_MENU_SHELL (ta->menu), menuitem);
	g_signal_connect(G_OBJECT(menuitem), "activate", 
			 G_CALLBACK(cb_preferences), (void*)ta);

	gtk_widget_show_all (ta->menu);
}

/* this tells the trayicon that apt is downloading something */
void 
update_apt_is_running(TrayApplet *ta, gboolean is_running)
{
   UpdateTrayAppletPrivate *priv = (UpdateTrayAppletPrivate*)ta->user_data;
   //g_print("update_apt_is_running: %i\n",is_running);

   //  we load a different icon to show that apt is running
   // FIXME: we need block menu events
   gchar *inactive_name = g_strdup_printf("%s-inactive", ta->name);
   if(is_running)
      gtk_status_icon_set_from_icon_name(ta->tray_icon, inactive_name);
   else
      gtk_status_icon_set_from_icon_name(ta->tray_icon, ta->name);

   g_free(inactive_name);
   priv->apt_is_running = is_running;

   if(is_running) {
      gtk_status_icon_set_tooltip(ta->tray_icon,
				  _("A package manager is working"));
      if(priv->active_notification != NULL) {
	 notify_notification_close(priv->active_notification, NULL);
	 priv->active_notification = NULL;
      }
   }
}

// actually show the notification 
static gint 
show_notification(gpointer user_data)
{
   TrayApplet *ta = (TrayApplet *)user_data;
   UpdateTrayAppletPrivate *priv = (UpdateTrayAppletPrivate*)ta->user_data;

   // apt is runing, no point in showing a notification
   if(priv->apt_is_running) 
      return TRUE;

   GdkRectangle area;
   gtk_status_icon_get_geometry(ta->tray_icon, NULL, &area, NULL);

   // no usefull coordiante yet, do another timeout
   if(area.x <= 0 || area.y <= 0 || area.width <= 0 || area.height <= 0)
      return TRUE;

   // check if the update-icon is still visible (in the delay time a 
   // update may already have been performed)
   if(!gtk_status_icon_get_visible(ta->tray_icon))
      return FALSE;

   // now show a notification handle 
   NotifyNotification *n = notify_notification_new_with_status_icon(
			      _("Software updates available"),
			      _("Click on the notification"
				" icon to show the "
				"available updates.\n"),
			      NULL, 
			      ta->tray_icon);
 
   GdkPixbuf* pix= gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), 
					    GTK_STOCK_DIALOG_INFO, 48,0,NULL);
   notify_notification_set_icon_from_pixbuf (n, pix);
   g_object_unref(pix);
   notify_notification_set_timeout (n, 60*1000);
   notify_notification_show(n, NULL);
   // save the notification handle
   priv->active_notification = n;

   // remove this from the timeout now
   return FALSE;
}

gboolean
update_check (TrayApplet *ta)
{
   //g_print("update_check()\n");
   UpdateTrayAppletPrivate *priv = (UpdateTrayAppletPrivate*)ta->user_data;

   int num_upgrades = 0;
   int exit_status = 0;
   GError *error;

   char *cmd[] = { UPGRADE_CHECKER, NULL };
   char *ret;

   if(g_spawn_sync(NULL, cmd, NULL,  
		   G_SPAWN_STDOUT_TO_DEV_NULL, 
		   NULL, NULL, NULL, &ret,
		   &exit_status, &error)) {
      // read the value from childs stderr, E: indicates a problem
      if(ret[0] == 'E') {
	 gchar *error;
	 if(strlen(ret) > 3) 
	    error = g_strdup_printf(_("An error occurred, please run "
				      "Package Manager from the "
				      "right-click menu or apt-get in "
				      "a terminal to see what is wrong.\n"
				      "The error message was: '%s'"),
				    &ret[2]);
	 else
	    error = g_strdup_printf(_("An error occurred, please run "
				      "Package Manager from the "
				      "right-click menu or apt-get in "
				      "a terminal to see what is wrong."));
	 gtk_status_icon_set_tooltip(ta->tray_icon, 
				   _("This usually means that your installed "
				     "packages have unmet dependencies"));
	 gtk_status_icon_set_visible(ta->tray_icon, TRUE);
	 g_free(ret);
	 return TRUE;
	 
      } 
      num_upgrades = atoi(ret);
      g_free(ret);
   } else
      g_warning("Error launching %s", UPGRADE_CHECKER);
   
   if (num_upgrades == 0) {
      gtk_status_icon_set_visible (ta->tray_icon, FALSE);
      
      if(priv->active_notification != NULL) {
	 notify_notification_close(priv->active_notification, NULL);
	 priv->active_notification = NULL; 
      }
      return TRUE;
   } 

   // update the tooltip
   update_trayicon_update_tooltip(ta, num_upgrades);

   // if we are already visible, skip the rest
   if(gtk_status_icon_get_visible (ta->tray_icon))
      return TRUE;

   // show the icon
   gtk_status_icon_set_visible(ta->tray_icon, TRUE);

   // the user does not no notification messages
   if(gconf_client_get_bool(priv->gconf,
			    GCONF_KEY_NO_UPDATE_NOTIFICATIONS, NULL))
      return TRUE;

   // show the notification with some delay. otherwise on a login
   // the origin of the window is 0,0 and that looks ugly
   g_timeout_add(5000, show_notification, ta);
   
   return TRUE;
}


void 
update_tray_icon_init(TrayApplet *ta)
{
   // create the private data struct
   UpdateTrayAppletPrivate *priv = g_new0(UpdateTrayAppletPrivate, 1);
   priv->gconf = gconf_client_get_default();
   priv->apt_is_running = FALSE;
   priv->active_notification = NULL;
   ta->user_data = priv;

   // connect the signals we need
   g_signal_connect (G_OBJECT(ta->tray_icon),
		     "activate",
		     G_CALLBACK (activate_cb),
		     ta);
   g_signal_connect (G_OBJECT(ta->tray_icon),
		     "popup-menu",
		     G_CALLBACK (popup_cb),
		     ta);

   /* Menu initialization */
   update_trayicon_create_menu (ta);
   
   /* Check for updates for the first time */
   update_check (ta);
}
