#!/usr/bin/env python
# -*- coding: utf-8 -*-
##
## This script takes a JSON file with the following format as input:
## [
##      {
##          "test": "TestSite.js",
##          "login": "login",
##          "password": "password",
##          "url": [
##              "http://www.site-to-test.com/",
##              "http://www.site-to-test.de/"
##              ]
##      },
##      {
##          "test": "TestAnotherSite.js",
##          "login": "login",
##          "password": "password",
##          "url": [ "http://www.anothersite-to-test.com/" ]
##      },
##      {
##          "test": "TestYetAnotherSite.js",
##          "login" : "",
##          "password": "",
##          "url" : [ ]
##      },
##      ...
##  ]
##
## It executes test.js for each test script and URL.
##
import os, sys, time
import subprocess
import json
import junitxml
import unittest
import testscenarios

from optparse import OptionParser

CREDENTIALS_FILENAME = 'site-credentials'
TIMEOUT = 150 # Test limit: 150 seconds

class SiteTest(unittest.TestCase):

    scenarios = []

    def testSite(self):
        """ Test integration script """
        # Compose call with command line arguments
        call_params = [
            "./test.js",
            ]
        if (self.show_window):
            # Show window
            call_params.append("--show-window")
        if (self.login):
            # Login required
            call_params.append("--login")
            call_params.append(self.login)
        if (self.password):
            # Password required
            call_params.append("--password")
            call_params.append(self.password)
        if (self.url <> ''):
            # Non-default URL
            call_params.append("--url")
            call_params.append(self.url)
        call_params.append("../" + self.uri)
        # Executes test.js and captures stdout
        output = ''
        testjs = None
        time_begin = time.time()
        time_lapse = 0
        try:
            # Create process
            testjs = subprocess.Popen(call_params, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            # Check timeout
            seconds = 0
            while True:
                time_lapse = time.time() - time_begin
                if (testjs.poll() is not None):
                    # Process successfully finished
                    break
                elif (time_lapse > TIMEOUT):
                    # Timeout reached. Kill process
                    testjs.terminate()
                time.sleep(1)
        except subprocess.CalledProcessError:
            # Error executing the test
            pass
        # Processes the output
        if (testjs):
            stdout_log, stderr_log = testjs.communicate()
            # Check output
            if (testjs.returncode != 0):
                raise AssertionError(stderr_log + '\n' + stdout_log)
        return True

def test_suite_from_array(credential_array, script='', show_window=False):
    """ Returns a suite of tests using the credential array """
    # Loops over the site-credentials. Executes one test for earch site.
    suite = unittest.TestSuite()
    test = SiteTest('testSite')
    for site in credential_array:
        if (script <> '' and site['test'] <> script):
            # Site filter enabled. Script doesn't match it.
            continue
        if (site.has_key('url') and (len(site['url']) > 0)):
            # Loop over URLs
            for url in site['url']:
                test_name = ' '.join((site['test'], url, site['login']))
                # Adds test scenario
                test.scenarios.append((test_name, {
                    'show_window' : show_window,
                    'uri' : site['test'],
                    'login' : site['login'],
                    'password' : site['password'],
                    'url' : url }))
        else:
            # Test with default URL
            test_name = ' '.join((site['test'], site['login']))
            # Adds test scenario
            test.scenarios.append((test_name, {
                'show_window' : show_window,
                'uri' : site['test'],
                'login' : site['login'],
                'password' : site['password'],
                'url': ''}))
    # Run the test suite
    suite.addTests(testscenarios.generate_scenarios(test))
    return suite

def main(credential_array, script='', format='text', output=None, show_window=False):
    # Generate test suite
    suite = test_suite_from_array(credential_array, script, show_window)
    # Chooses output (file or stdout)
    stream = sys.stdout
    if (output):
        try:
            stream = open(output, 'w')
        except:
            sys.stderr.write('Cannot open output file\n')
            exit(1)
    # Chooses output format (text or xml)
    if (format == 'xml'):
        # jUnit XML output
        result = junitxml.JUnitXmlResult(stream)
        # Runs the test suite
        result.startTestRun()
        suite.run(result)
        result.stopTestRun()
        # Exits on failure
        if not result.wasSuccessful:
            exit(1)
    else:
        # Text output
        runner = unittest.TextTestRunner(stream=stream, verbosity=2)
        # Runs the test suite
        success = runner.run(suite).wasSuccessful()
        # Exits on failure
        if (not success):
            exit(1)

if __name__ == '__main__':
    # Checks arguments
    usage='Usage: test-runner.py [options] [credential_filename]'
    parser = OptionParser(usage=usage)
    parser.add_option('-f',
        '--format',
        dest='format',
        choices=['xml', 'text'],
        default='text',
        help='Format of the test results: xml or text')
    parser.add_option('-o',
        '--output',
        dest='output',
        help='Output results in jUnit XML')
    parser.add_option('-s',
        '--show-window',
        dest='show_window',
        action='store_true',
        default=False,
        help='Show window')
    parser.add_option('-t',
        '--test',
        dest='script',
        default="",
        help='Script to test. Example: --test Gmail.js')
    (options, args) = parser.parse_args()
    credentials_filename = CREDENTIALS_FILENAME # Default site credential filename
    if (len(args) > 0):
        credentials_filename = args[0]
    # Opens credentials file
    try:
        site_file = open(credentials_filename, 'r')
    except:
        parser.print_help()
        exit(1)
    # Reads file
    lines = ''.join(site_file.readlines())
    # Parses text into JSON
    sites = json.loads(lines)
    main(sites, script=options.script, format=options.format, output=options.output, show_window=options.show_window)
