/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef PATH_MAX
#define PATH_MAX 4096
#endif

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>

#include "keybindings.h"

#define KEYBIND_FILE "xfmedia/keybindings.rc"

#define KEYBIND_HASH_KEY(keyval, modmask) \
        GUINT_TO_POINTER((keyval << 16) | (modmask & 0xffff))
#define KEYBIND_UNHASH(hashval, keyval, modmask) \
    do {\
        keyval = (hashval & 0xffff0000) >> 16;\
        modmask = hashval & 0xffff;\
    } while(0)

/* this is lame */
static GHashTable *keys_hash = NULL;
static GHashTable *functions_hash = NULL;

static gboolean
parse_keybinding(const gchar *str, guint *keyval, GdkModifierType *modmask)
{
    gchar **parts;
    gint i;
    gboolean ret = TRUE;
    
    g_return_val_if_fail(str && keyval && modmask, FALSE);
    
    *keyval = 0;
    *modmask = 0;
    
    parts = g_strsplit(str, "+", -1);
    for(i = 0; parts[i]; i++) {
        if(!g_ascii_strcasecmp(parts[i], "none")) {
            *keyval = 0;
            *modmask = 0;
            break;
        } else if(!g_ascii_strcasecmp(parts[i], "shift"))
            *modmask |= GDK_SHIFT_MASK;
        else if(!g_ascii_strcasecmp(parts[i], "control") || !g_ascii_strcasecmp(parts[i], "ctrl"))
            *modmask |= GDK_CONTROL_MASK;
        else if(!g_ascii_strcasecmp(parts[i], "alt") || !g_ascii_strcasecmp(parts[i], "mod1"))
            *modmask |= GDK_MOD1_MASK;
        else if(!g_ascii_strcasecmp(parts[i], "super") || !g_ascii_strcasecmp(parts[i], "mod4"))
            *modmask |= GDK_MOD4_MASK;
        else {
            *keyval = gdk_keyval_from_name(parts[i]);
            if(*keyval == GDK_VoidSymbol) {
                ret = FALSE;
                break;
            }
            *keyval = gdk_keyval_to_lower(*keyval);
        }
    }
    
    if(parts)
        g_strfreev(parts);
    
    return ret;
}

static gboolean
xfmedia_keybindings_parse_file(const gchar *file)
{
    gboolean ret = FALSE;
    FILE *fp;
    gchar buf[2048], *p;
    gint line = 0;
    
    fp = fopen(file, "r");
    if(fp) {
        *buf = 0;
        while(fgets(buf, 2048, fp)) {
            guint keyval = -1;
            GdkModifierType modmask = 0;
            
            line++;
            
            if(*buf == '#' || !(p = strstr(buf, "=")))
                continue;
            
            while(buf[strlen(buf)-1] == '\n' || buf[strlen(buf)-1] == '\r')
                buf[strlen(buf)-1] = 0;
            
            if(!parse_keybinding(p+1, &keyval, &modmask)) {
                g_warning("Keybinding is invalid:%d: %s", line, buf);
                continue;
            }
            
            *p = 0;
            xfmedia_keybindings_modify(buf, keyval, modmask);
            
            *buf = 0;
        }
        fclose(fp);
        ret = TRUE;
    }
    
    return ret;
}

void
xfmedia_keybindings_init()
{    
    g_return_if_fail(!keys_hash || !functions_hash);
    
    keys_hash = g_hash_table_new(g_direct_hash, g_direct_equal);
    functions_hash = g_hash_table_new_full(g_str_hash, g_str_equal,
            (GDestroyNotify)g_free, NULL);
}

static gint
keybinding_compare(gconstpointer a, gconstpointer b)
{
    const XfmediaKeybinding *a_k = a, *b_k = b;
    return g_ascii_strcasecmp(a_k->keybind_id, b_k->keybind_id);
}

static void
keys_hash_to_list(gpointer key, gpointer value, gpointer user_data)
{
    guint binding_mask = GPOINTER_TO_UINT(value), keyval, modmask;
    gchar *keybind_id = key;
    GList **list = user_data;
    XfmediaKeybinding *kb = g_new0(XfmediaKeybinding, 1);
    
    KEYBIND_UNHASH(binding_mask, keyval, modmask);
    kb->keybind_id = keybind_id;
    kb->keyval = keyval;
    kb->modmask = modmask;
    
    *list = g_list_insert_sorted(*list, kb, keybinding_compare);
}

GList *
xfmedia_keybindings_get_list()
{
    GList *list = NULL;
    
    g_hash_table_foreach(functions_hash, keys_hash_to_list, &list);
    
    return list;
}

void
xfmedia_keybindings_free_list(GList *list)
{
    GList *l;
    
    for(l = list; l; l = l->next)
        g_free(l->data);
    if(list)
        g_list_free(list);
}

G_CONST_RETURN gchar *
xfmedia_keybindings_lookup(guint keyval, GdkModifierType modmask)
{
    g_return_val_if_fail(keys_hash, NULL);
    
    /* process only the modifiers we care about */
    modmask &= (GDK_SHIFT_MASK|GDK_CONTROL_MASK|GDK_MOD1_MASK|GDK_MOD4_MASK);
    
    if(modmask & GDK_SHIFT_MASK)
        keyval = gdk_keyval_to_lower(keyval);
    
    return g_hash_table_lookup(keys_hash, KEYBIND_HASH_KEY(keyval, modmask));
}

void
xfmedia_keybindings_cleanup()
{
    g_return_if_fail(keys_hash && functions_hash);
    
    g_hash_table_destroy(keys_hash);
    keys_hash = NULL;
    g_hash_table_destroy(functions_hash);
    functions_hash = NULL;
}

gboolean
xfmedia_keybindings_load(const gchar *file)
{
    gchar *default_file = NULL;
    gboolean ret = FALSE;
    
    if(!file) {
        default_file = xfce_resource_lookup(XFCE_RESOURCE_CONFIG, KEYBIND_FILE);
        file = default_file;
    }
    
    if(!file || !g_file_test(file, G_FILE_TEST_EXISTS)) {
        g_warning("No valid keybindings file found.");
        return FALSE;
    }
    
    if(xfmedia_keybindings_parse_file(file))
        ret = TRUE;
    else
        g_warning("Failed to parse keybindings file (%s)", file);
    
    if(default_file)
        g_free(default_file);
    
    return ret;
}

gboolean
xfmedia_keybindings_save(const gchar *file)
{
    FILE *fp;
    gchar *default_file = NULL, newfile[PATH_MAX];
    gboolean ret = FALSE;
    
    if(!file) {
        default_file = xfce_resource_save_location(XFCE_RESOURCE_CONFIG,
                KEYBIND_FILE, TRUE);
        file = default_file;
    }
    if(!file)
        return FALSE;
    
    g_snprintf(newfile, PATH_MAX, "%s.new", file);
    
    fp = fopen(newfile, "w");
    if(fp) {
        GList *kbs = xfmedia_keybindings_get_list(), *l;\
        
        for(l = kbs; l; l = l->next) {
            XfmediaKeybinding *kb = l->data;
            if(kb->keyval == 0 && kb->modmask == 0)
                fprintf(fp, "%s=none\n", kb->keybind_id);
            else {
                fprintf(fp, "%s=%s%s%s%s%s\n", kb->keybind_id,
                        (kb->modmask & GDK_CONTROL_MASK) ? "ctrl+" : "",
                        (kb->modmask & GDK_MOD1_MASK) ? "alt+" : "",
                        (kb->modmask & GDK_MOD4_MASK) ? "super+" : "",
                        (kb->modmask & GDK_SHIFT_MASK) ? "shift+" : "",
                        gdk_keyval_name(kb->keyval));
            }
        }
        xfmedia_keybindings_free_list(kbs);
        fclose(fp);
        
        if(!rename(newfile, file))
            ret = TRUE;
    } else
        g_warning("Unable to write to keybindings file '%s'", newfile);
    
    if(default_file)
        g_free(default_file);
    
    return ret;
}

void
xfmedia_keybindings_modify(const gchar *function, guint keyval,
        GdkModifierType modmask)
{
    gpointer old_keyhash, new_keyhash;
    gchar *function_dup;
    
    g_return_if_fail(function);
    
    old_keyhash = g_hash_table_lookup(functions_hash, function);
    if(old_keyhash)
        g_hash_table_remove(keys_hash, old_keyhash);
    
    function_dup = g_strdup(function);
    new_keyhash = KEYBIND_HASH_KEY(keyval, modmask);
    g_hash_table_insert(keys_hash, new_keyhash, function_dup);
    g_hash_table_replace(functions_hash, function_dup, new_keyhash);
}
