/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 *               2004 Jean-François Wauthy (pollux@xfce.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* !HAVE_CONFIG_H */

#include <sys/time.h>
#include <sys/types.h>
#include <string.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <libxfce4mcs/mcs-client.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>
#include <printing-system.h>

#include "print_dialog.h"

#define CHANNEL         "xfprint"

extern void print_dialog_run (const gchar * ifile);

int
main (int argc, char **argv)
{
  gchar *ifile = NULL;
  gchar *decoded_ifile = NULL;
  McsClient *client;
  Display *dpy;
  int screen;
  PrintingSystem *ps = NULL;
  
#if 0
#ifdef ENABLE_NLS
  /* This is required for UTF-8 at least - Please don't remove it */
  bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
#ifdef HAVE_BIND_TEXTDOMAIN_CODESET
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
  textdomain (GETTEXT_PACKAGE);
#endif
#else
  /* This is required for UTF-8 at least - Please don't remove it */
  xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");
#endif

  gtk_set_locale ();
  gtk_init (&argc, &argv);

  /* Load the printing system module */
  dpy = GDK_DISPLAY ();
  screen = XDefaultScreen (dpy);

  if (!mcs_client_check_manager (dpy, screen, "xfce-mcs-manager"))
    g_warning ("%s: mcs manager not running\n", PACKAGE);
  client = mcs_client_new (dpy, screen, NULL, NULL, NULL);
  if (client) {
    McsSetting *setting;

    mcs_client_add_channel (client, CHANNEL);

    if (mcs_client_get_setting (client, "XfPrint/system", CHANNEL, &setting) == MCS_SUCCESS) {
      if (g_ascii_strcasecmp (setting->data.v_string, "none") != 0) {
        ps = printing_system_new (setting->data.v_string);
        if (ps == NULL)
          g_warning ("Unable to load printing system module %s", setting->data.v_string);
      }
      mcs_setting_free (setting);
    }

    mcs_client_destroy (client);
  }


  if (argc == 1) {
    struct timeval tv;
    fd_set fds;
    gchar *file;

    FD_ZERO (&fds);
    FD_SET (STDIN_FILENO, &fds);

    tv.tv_sec = 0;
    tv.tv_usec = 20 * 1000;

    /* If no input is available on stdin, ask the users to
     * choose a file to print.
     */
    if (select (1, &fds, NULL, NULL, &tv) < 1) {
      GtkWidget *selection;
      gint response;

      selection = gtk_file_chooser_dialog_new (_("Select file to print"),
                                         NULL,
                                         GTK_FILE_CHOOSER_ACTION_OPEN,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);
      response = gtk_dialog_run (GTK_DIALOG (selection));

      if (response != GTK_RESPONSE_OK)
        exit (EXIT_SUCCESS);

      file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (selection));

      if (!g_file_test (file, G_FILE_TEST_IS_REGULAR)) {
        xfce_err (_("The specified file does not exist"));
        g_free (file);
        exit (EXIT_FAILURE);
      }

      ifile = file;

      gtk_widget_destroy (selection);
    }
    else {
      /* Ok, data ready on stdin */
      ifile = g_strdup ("-");
    }
  }
  else if (!strcmp (argv[1], "--version") || !strcmp (argv[1], "-V")) {
    g_print ("\tThis is %s version %s for Xfce %s\n", PACKAGE, VERSION, xfce_version_string ());
    g_print ("\tbuilt with GTK+-%d.%d.%d, ", GTK_MAJOR_VERSION, GTK_MINOR_VERSION, GTK_MICRO_VERSION);
    g_print ("linked with GTK+-%d.%d.%d.\n", gtk_major_version, gtk_minor_version, gtk_micro_version);
    if (ps)
      g_print ("\tCurrently it uses the %s printing system.\n", ps->name);
    else
      g_print ("\tNo printing system is currently used.\n");
    
    exit (EXIT_SUCCESS);
  }
  else {
    ifile = g_strdup (argv[1]);
  }

  if (g_str_has_prefix (ifile, "file://"))
    decoded_ifile = g_filename_from_uri (ifile, NULL, NULL);
  else if (g_path_is_absolute (ifile) || strcmp (ifile, "-") == 0)
    decoded_ifile = g_strdup (ifile);
  else {
    gchar *current_dir = NULL;

    current_dir = g_get_current_dir ();
    decoded_ifile = g_build_filename (current_dir, ifile, NULL);
    
    g_free (current_dir);
  }
  
  DBG ("Printing file: %s", decoded_ifile);
  
  gtk_dialog_run (GTK_DIALOG (print_dialog_new (ps, decoded_ifile)));
  
  g_free (decoded_ifile);
  g_free (ifile);

  return (EXIT_SUCCESS);
}
