#------------------------------------------------------------------------------
# File:         ExifTool.pm
#
# Description:  Utility to read EXIF information from image files
#
# URL:          http://owl.phy.queensu.ca/~phil/exiftool/
#
# Revisions:    Nov. 12/03 - P. Harvey Created
#               (See html/history.html for revision history)
#
# Legal:        Copyright (c) 2003-2004 Phil Harvey (phil@owl.phy.queensu.ca)
#               This library is free software; you can redistribute it and/or
#               modify it under the same terms as Perl itself.
#------------------------------------------------------------------------------

package Image::ExifTool;

use strict;
require 5.002;
require Exporter;
use File::RandomAccessFile;

use vars qw($VERSION @ISA @EXPORT_OK);
$VERSION = '3.45';
@ISA = qw(Exporter);
@EXPORT_OK = qw(ImageInfo Options ClearOptions ExtractInfo GetInfo CombineInfo
                GetTagList GetFoundTags GetRequestedTags GetValue GetDescription
                GetGroup GetGroups BuildCompositeTags GetTagName GetShortcuts
                GetAllTags GetAllGroups GetByteOrder SetByteOrder ToggleByteOrder
                Get16u Get16s Get32u Get32s GetFloat GetDouble);

# group hash for ExifTool-generated tags
my %allGroupsExifTool = ( 0 => 'ExifTool', 1 => 'ExifTool', 2 => 'ExifTool' );

# extra tags that aren't truly EXIF tags, but are generated by the script
%Image::ExifTool::extraTags = (
    GROUPS => { 0 => 'File', 1 => 'File', 2 => 'Image' },
    Comment     => { Name => 'Comment' },
    FileName    => { Name => 'FileName' },
    FileSize    => { Name => 'FileSize',  PrintConv => 'sprintf("%.0fKB",$val/1024)' },
    FileType    => { Name => 'FileType' },
    ImageWidth  => { Name => 'ImageWidth' },
    ImageHeight => { Name => 'ImageHeight' },
    ExifData    => { Name => 'ExifData',  PrintConv => '\$val' },
    ExifToolVersion => {
        Name        => 'ExifToolVersion',
        Description => 'ExifTool Version Number',
        Groups      => \%allGroupsExifTool
    },
    Error       => { Name => 'Error',   Groups => \%allGroupsExifTool },
    Warning     => { Name => 'Warning', Groups => \%allGroupsExifTool },
);

# static private ExifTool variables

my %allTables;      # list of all tables loaded (except composite tags)

# composite tags (accumulation of all Composite tag tables)
%Image::ExifTool::compositeTags = ( GROUPS => { 0 => 'Composite', 1 => 'Composite' } );

# special tag names (not used for tag info)
my %specialTags = ( PROCESS_PROC=>1, FORMAT=>1, GROUPS=>1,
                    TAG_PREFIX=>1, FIRST_ENTRY=>1, PRINT_CONV=>1 );


#------------------------------------------------------------------------------
# New - create new ExifTool object
# Inputs: 0) reference to exiftool object or ExifTool class name
sub new
{
    my $that = shift;
    my $class = ref($that) || $that || 'Image::ExifTool';
    my $self = bless {}, $class;

    $self->ClearOptions();      # create default options hash

    return $self;
}

#------------------------------------------------------------------------------
# ImageInfo - return specified information from image file
# Inputs: 0) [optional] ExifTool object reference
#         1) filename, file reference, or scalar data reference
#         2-N) list of tag names to find (or tag list reference or options reference)
# Returns: reference to hash of tag/value pairs (with "Error" entry on error)
# Notes:
#   - if no tags names are specified, the values of all tags are returned
#   - tags may be specified with leading '-' to exclude
#   - can pass a reference to list of tags to find, in which case the list will
#     be updated with the tags found in the proper case and in the specified order.
#   - can pass reference to hash specifying options
#   - returned tag values may be scalar references indicating binary data
#   - see ClearOptions() below for a list of options and their default values
# Examples:
#   use Image::ExifTool 'ImageInfo';
#   my $info = ImageInfo($file, 'DateTimeOriginal', 'ImageSize');
#    - or -
#   my $exifTool = new Image::ExifTool;
#   my $info = $exifTool->ImageInfo($file, \@tagList, {Sort=>'Group0'} );
sub ImageInfo($;@)
{
    # get our ExifTool object ($self) or create one if necessary
    my $self;
    if (ref $_[0] eq 'Image::ExifTool') {
        $self = shift;
    } else {
        $self = new Image::ExifTool;
    }
    my $saveOptions = $self->{OPTIONS};     # save original options

    # initialize file information
    $self->{FILENAME} = $self->{RAF} = undef;

    $self->ParseArguments(@_);              # parse our function arguments
    $self->ExtractInfo(undef);              # extract meta information from image
    my $info = $self->GetInfo(undef);       # get requested information

    $self->{OPTIONS} = $saveOptions;        # restore original options

    return $info;   # return requested information
}

#------------------------------------------------------------------------------
# Get/set ExifTool options
# Inputs: 0) ExifTool object reference,
#         1) Parameter name, 2) Value to set the option
#         3-N) More parameter/value pairs
# Returns: original value of last option specified
sub Options($$;@)
{
    my $self = shift;
    my $oldVal;

    while (@_) {
        my $param = shift;
        my $value = shift;
        $oldVal = $self->{OPTIONS}->{$param};
        $self->{OPTIONS}->{$param} = $value if defined $value;
    }
    return $oldVal;
}

#------------------------------------------------------------------------------
# ClearOptions - set options to default values
# Inputs: 0) ExifTool object reference
sub ClearOptions($)
{
    my $self = shift;

    # create options hash with default values
    # (commented out options don't need initializing)
    $self->{OPTIONS} = {
        Binary      => 0,       # flag to extract binary values even if tag not specified
        Composite   => 1,       # flag to calculate Composite tags
    #   DateFormat  => undef,   # format for date/time
        Duplicates  => 0,       # flag to save duplicate tag values
    #   Exclude     => undef,   # tags to exclude
    #   Group#      => undef,   # return tags for specified groups in family #
        PrintConv   => 1,       # flag to enable print conversion
        Sort        => 'Input', # order to sort found tags (Input, File, Alpha, Group#)
        Unknown     => 0,       # flag to get values of unknown tags (0-2)
        Verbose     => 0,       # print verbose messages (0-3, higher # = more verbose)
    };
}

#------------------------------------------------------------------------------
# Extract meta information from image
# Inputs: 0) ExifTool object reference
#         1-N) Same as ImageInfo()
# Returns: 1 if this was a valid image, 0 otherwise
# Notes: pass an undefined value to avoid parsing arguments
sub ExtractInfo($;@)
{
    my $self = shift;
    my $options = $self->{OPTIONS};     # pointer to current options
    my %saveOptions;

    if (defined $_[0]) {
        %saveOptions = %{$self->{OPTIONS}}; # save original options

        # only initialize filename if called with arguments
        $self->{FILENAME} = undef;      # name of file (or '' if we didn't open it)
        $self->{RAF} = undef;           # RandomAccessFile object reference

        $self->ParseArguments(@_);      # initialize from our arguments
    }
    # initialize ExifTool object members
    $self->{FOUND_TAGS} = [ ];          # list of found tags
    $self->{NUM_TAGS_FOUND} = 0;        # total number of tags found (incl. duplicates)
    $self->{FILE_ORDER} = { };          # hash of tag order in file
    $self->{VALUE_CONV} = { };          # hash of converted tag values
    $self->{PRINT_CONV} = { };          # hash of print-converted values
    $self->{TAG_INFO}   = { };          # hash of tag information
    $self->{EXIF_DATA} = undef;         # the EXIF data block

    # load our main Exif tag table
    GetTagTable("Image::ExifTool::Exif::Main");

    my $filename = $self->{FILENAME};   # image file name ('' if already open)
    my $raf = $self->{RAF};             # RandomAccessFile object

    # return our version number
    $self->FoundTag('ExifToolVersion', $VERSION);

    local *EXIFTOOL_FILE;   # avoid clashes with global namespace

    unless ($raf) {
        # save file name
        if ($filename) {
            unless ($filename eq '-') {
                my $name = $filename;
                $name =~ s/.*\///;  # remove path
                $self->FoundTag('FileName', $name);
                my $fileSize = -s $filename;
                $self->FoundTag('FileSize', $fileSize) if defined $fileSize;
            }
            # open the file
            if (open(EXIFTOOL_FILE,$filename)) {
                my $filePt = \*EXIFTOOL_FILE;
                # create random access file object
                # (note: disable buffering for a normal file -- $filename ne '-')
                $raf = new File::RandomAccessFile($filePt, $filename ne '-');
                $self->{RAF} = $raf;
            } else {
                $self->Error('Error opening file');
            }
        } else {
            $self->Error('No file specified');
        }
    }

    if ($raf) {
        # process the image
        $raf->BinMode();    # set binary mode before we start reading

        # read tags from the file
        my (@fileTypes, $pos);
        if ($filename and $filename ne '-') {
            # use extension of file name to determine file type
            @fileTypes = ( $filename );
        } else {
            # scan through all major supported types in this order
            @fileTypes = ( '.jpg', '.crw', '.tif', '.gif' );
            # must test our input file for the ability to seek
            # since it isn't a regular file
            $raf->SeekTest();
            $pos = $raf->Tell();   # get file position so we can rewind
        }
        # loop through possible file types
        for (;;) {
            my $type = shift @fileTypes;
            if ($type =~ /\.(jpg|jpeg|thm)$/i) {
                $self->JpgInfo() and last;
            } elsif ($type =~ /\.gif$/i) {
                $self->GifInfo() and last;
            } elsif ($type =~ /\.tiff{0,1}$/i) {
                # A TIFF file starts with an EXIF block -- Easy huh?
                $self->ExifInfo('TIFF', $raf) and last;
            } elsif ($type =~ /\.nef$/i) {
                # An NEF file is standard TIFF format
                $self->ExifInfo('NEF', $raf) and last;
            } elsif ($type =~ /\.cr2$/i) {
                # A CR2 file is standard TIFF format too
                $self->ExifInfo('CR2', $raf) and last;
            } elsif ($type =~ /\.crw$/i) {
                GetTagTable('Image::ExifTool::CanonRaw::Main'); # load the raw tables
                Image::ExifTool::CanonRaw::RawInfo($self) and last;
            } else {
                $self->Error('Unknown image file type');
            }
            last unless @fileTypes;     # all done if no more types
            # seek back to try again from the same position in the file
            unless ($raf->Seek($pos, 0)) {
                $self->Error('Error seeking in file');
                last;
            }
        }
        # extract binary EXIF data block only if requested
        if (defined $self->{EXIF_DATA} and
            grep /^ExifData$/i, @{$self->{REQUESTED_TAGS}})
        {
            $self->FoundTag('ExifData', $self->{EXIF_DATA});
        }
        # calculate composite tags
        $self->BuildCompositeTags() if $options->{Composite};

        $raf->Close() if $filename;     # close the file if we opened it
    }

    # restore original options
    %saveOptions and $self->{OPTIONS} = \%saveOptions;

    return exists $self->{PRINT_CONV}->{Error} ? 0 : 1;
}

#------------------------------------------------------------------------------
# Get hash of extracted meta information
# Inputs: 0) ExifTool object reference
#         1-N) options hash reference, tag list reference or tag names
# Returns: Reference to information hash
# Notes: - pass an undefined value to avoid parsing arguments
#        - If groups are specified, first groups take precidence if duplicate
#          tags found but Duplicates option not set.
sub GetInfo($;@)
{
    my $self = shift;
    my %saveOptions;
    my $rtnTags;    # reference to list of tags for which we will return info
    my $tag;

    unless (@_ and not defined $_[0]) {
        %saveOptions = %{$self->{OPTIONS}}; # save original options
        $self->ParseArguments(@_);
    }
    my $options = $self->{OPTIONS};
    my $reqTags = $self->{REQUESTED_TAGS} || [ ];
    my $duplicates = $options->{Duplicates};

    # get list of all existing tags
    my @allTags = keys %{$self->{PRINT_CONV}};

    # get information for requested groups
    my @groupOptions = sort grep /^Group/, keys %$options;
    if (@groupOptions) {
        my %wantGroup;
        my $family;
        my $allGroups = 1;
        # build hash of requested/excluded group names for each group family
        my $wantOrder = 0;
        my $groupOpt;
        foreach $groupOpt (@groupOptions) {
            $groupOpt =~ /^Group(\d*)/ or next;
            $family = $1 || 0;
            $wantGroup{$family} or $wantGroup{$family} = { };
            my $groupList;
            if (ref $options->{$groupOpt} eq 'ARRAY') {
                $groupList = $options->{$groupOpt};
            } else {
                $groupList = [ $options->{$groupOpt} ];
            }
            foreach (@$groupList) {
                # groups have priority in order they were specified
                ++$wantOrder;
                my ($groupName, $want);
                if (/^-(.*)/) {
                    # excluded group begins with '-'
                    $groupName = $1;
                    $want = 0;          # we don't want tags in this group
                } else {
                    $groupName = $_;
                    $want = $wantOrder; # we want tags in this group
                    $allGroups = 0;     # don't want all groups if we requested one
                }
                $wantGroup{$family}->{$groupName} = $want;
            }
        }
        # loop through all tags and decide which ones we want
        my (%groupTags, %bestTag);
G_TAG:  foreach $tag (@allTags) {
            my $wantTag = $allGroups;   # want tag by default if want all groups
            foreach $family (keys %wantGroup) {
                my $group = $self->GetGroup($tag, $family);
                my $wanted = $wantGroup{$family}->{$group};
                next unless defined $wanted;
                next G_TAG unless $wanted;      # skip tag if group excluded
                # take lowest non-zero want flag
                next if $wantTag and $wantTag < $wanted;
                $wantTag = $wanted;
            }
            next unless $wantTag;
            if ($duplicates) {
                $groupTags{$tag} = $wantTag;
            } else {
                my $tagName = GetTagName($tag);
                my $bestTag = $bestTag{$tagName};
                if (defined $bestTag) {
                    next if $wantTag > $groupTags{$bestTag};
                    # this tag is better, so delete old best tag
                    delete $groupTags{$bestTag};
                }
                $groupTags{$tag} = $wantTag;    # keep this tag (for now...)
                $bestTag{$tagName} = $tag;      # this is our current best tag
            }
        }
        my @tags = keys %groupTags;
        $rtnTags = \@tags;
    } else {
        $rtnTags = \@allTags unless @$reqTags;
    }

    # exclude specified tags if requested
    if ($rtnTags and $options->{Exclude}) {
        my $exclude = $options->{Exclude};
        my @filteredTags;
        foreach $tag (@$rtnTags) {
            my $tagName = GetTagName($tag);
            next if grep /^$tagName$/i, @$exclude;
            push @filteredTags, $tag;
        }
        $rtnTags = \@filteredTags;      # use new filtered tag list
    }

    # get information for requested tags
    if (@$reqTags) {
        $rtnTags or $rtnTags = [ ];
        # scan through the requested tags and generate a list of tags we found
        foreach $tag (@$reqTags) {
            my @matches;
            my $fileOrder = $self->{FILE_ORDER};
            if (defined $fileOrder->{$tag} and not $duplicates) {
                $matches[0] = $tag;
            } else {
                # do case-insensitive check
                if ($duplicates) {
                    # must also look for tags like "Tag (1)"
                    @matches = grep /^$tag(\s|$)/i, keys %$fileOrder;
                } else {
                    # find first matching value
                    # (use in list context to return value instead of count)
                    ($matches[0]) = grep /^$tag$/i, keys %$fileOrder;
                    defined $matches[0] or undef @matches;
                }
                unless (@matches) {
                    # put entry in return list even without value (value is undef)
                    $matches[0] = $tag;
                    # bogus file order entry to avoid warning if sorting in file order
                    $fileOrder->{$tag} = 999;
                }
            }
            push @$rtnTags, @matches;
        }
    }
    # save found tags
    $self->{FOUND_TAGS} = $rtnTags;

    # build hash of tag information
    my %info;
    foreach (@$rtnTags) {
        my $printConv = $self->{PRINT_CONV}->{$_};
        next unless defined $printConv;
        $info{$_} = $printConv;
    }

    # return sorted tag list if provided with a list reference
    if ($self->{IO_TAG_LIST}) {
        # use file order by default if no tags specified
        # (no such thing as 'Input' order in this case)
        my $sortOrder = $self->{OPTIONS}->{Sort};
        unless (@$reqTags or ($sortOrder and $sortOrder ne 'Input')) {
            $sortOrder = 'File';
        }
        # return tags in specified sort order
        @{$self->{IO_TAG_LIST}} = $self->GetTagList($rtnTags, $sortOrder);
    }

    # restore original options
    %saveOptions and $self->{OPTIONS} = \%saveOptions;

    return \%info;
}

#------------------------------------------------------------------------------
# Combine information from a list of info hashes
# Unless Duplicates is enabled, first entry found takes priority
# Inputs: 0) ExifTool object reference, 1-N) list of info hash references
# Returns: Combined information hash reference
sub CombineInfo($;@)
{
    my $self = shift;
    my (%combinedInfo, $info);

    if ($self->{OPTIONS}->{Duplicates}) {
        while ($info = shift) {
            my $key;
            foreach $key (keys %$info) {
                $combinedInfo{$key} = $$info{$key};
            }
        }
    } else {
        my (%haveInfo, $tag);
        while ($info = shift) {
            foreach $tag (keys %$info) {
                my $tagName = GetTagName($tag);
                next if $haveInfo{$tagName};
                $haveInfo{$tagName} = 1;
                $combinedInfo{$tag} = $$info{$tag};
            }
        }
    }
    return \%combinedInfo;
}

#------------------------------------------------------------------------------
# Inputs: 0) ExifTool object reference
#         1) [optional] reference to info hash or tag list ref (default is found tags)
#         2) [optional] sort order ('File', 'Input', ...)
# Returns: List of tags in specified order
sub GetTagList($;$$)
{
    my ($self, $info, $sortOrder) = @_;

    my $foundTags;
    if (ref $info eq 'HASH') {
        my @tags = keys %$info;
        $foundTags = \@tags;
    } elsif (ref $info eq 'ARRAY') {
        $foundTags = $info;
    } else {
        $foundTags = $self->{FOUND_TAGS} or return undef;
        $sortOrder = $info if $info and not $sortOrder;
    }
    $sortOrder or $sortOrder = $self->{OPTIONS}->{Sort};

    # return original list if no sort order specified
    return @$foundTags unless $sortOrder and $sortOrder ne 'Input';

    my %groupOrder;
    my $fileOrder = $self->{FILE_ORDER};

    if ($sortOrder eq 'Alpha') {
        return sort @$foundTags;
    } else {
        if ($sortOrder =~ /^Group(\d*)/) {
            my $family = $1 || 0;
            # want to maintain a basic file order with the groups
            # ordered in the way they appear in the file
            my %groupCount;
            my $numGroups = 0;
            my $tag;
            foreach $tag (sort { $$fileOrder{$a} <=> $$fileOrder{$b} } @$foundTags) {
                my $group = $self->GetGroup($tag);
                my $num = $groupCount{$group};
                $num or $num = $groupCount{$group} = ++$numGroups;
                $groupOrder{$tag} = $num;
            }
            return sort { $groupOrder{$a} <=> $groupOrder{$b} or
                           $$fileOrder{$a} <=> $$fileOrder{$b} } @$foundTags;
        }
        return sort { $$fileOrder{$a} <=> $$fileOrder{$b} } @$foundTags;
    }
}

#------------------------------------------------------------------------------
# Get list of found tags in specified sort order
# Inputs: 0) ExifTool object reference, 1) sort order ('File', 'Input', ...)
# Returns: List of tags in specified order
# Notes: If not specified, sort order is taken from OPTIONS
sub GetFoundTags($;$)
{
    my $self = shift;
    my $foundTags = $self->{FOUND_TAGS} or return undef;
    return $self->GetTagList($foundTags, shift);
}

#------------------------------------------------------------------------------
# Get list of requested tags
# Inputs: 0) ExifTool object reference
sub GetRequestedTags($)
{
    return @{$_[0]->{REQUESTED_TAGS}};
}

#------------------------------------------------------------------------------
# Get tag value
# Inputs: 0) ExifTool object reference, 1) tag name
#         2) Value type (PrintConv or ValueConv, defaults to PrintConv)
sub GetValue($$;$)
{
    my ($self, $tag, $type) = @_;
    my $value;
    if ($type and $type eq 'ValueConv') {
        $value = $self->{VALUE_CONV}->{$tag};
    } else {
        $value = $self->{PRINT_CONV}->{$tag};
    }
    return $value;
}

#------------------------------------------------------------------------------
# Get description for specified tag
# Inputs: 0) ExifTool object reference, 1) tag name
# Notes: Will always return a defined value, even if description isn't available
sub GetDescription($$)
{
    my ($self, $tag) = @_;
    my $tagInfo = $self->{TAG_INFO}->{$tag};
    # ($tagInfo should be defined for any extracted tag,
    # but we might as well handle the case where it isn't)
    my $desc = $$tagInfo{Description} if $tagInfo;
    # just make the tag more readable if description doesn't exist
    unless ($desc) {
        # start with the tag name and force first letter to be upper case
        $desc = ucfirst(GetTagName($tag));
        $desc =~ tr/_/ /;       # translate underlines to spaces
        # put a space between lower-UPPER case combinations
        if ($desc =~ s/([a-z])([A-Z\d])/$1 $2/g) {
            # repair hex numbers of unknown tags
            while ($desc =~ s/ 0x([a-f\d]{0,3}) / 0x$1/) { }
        }
        # put a space between acronyms and words
        $desc =~ s/([A-Z])([A-Z][a-z])/$1 $2/g;
        # put spaces after numbers
        $desc =~ s/(\d)([A-Z])/$1 $2/g;
        # save description in tag information
        $$tagInfo{Description} = $desc if $tagInfo;
    }
    return $desc;
}

#------------------------------------------------------------------------------
# Get group name for specified tag
# Inputs: 0) ExifTool object reference
#         1) tag name
#         2) [optional] group family number
# Returns: Scalar context: Group name (for family 0 if not otherwise specified)
#          Array context: Group name if family specified, otherwise list of
#          group names for each family.
sub GetGroup($$;$)
{
    my ($self, $tag, $family) = @_;
    my $tagInfo = $self->{TAG_INFO}->{$tag} or return 'ErrNoInfo';  # this shouldn't happen
    # fill in default groups unless already done
    unless ($$tagInfo{GotGroups}) {
        my $tagTablePtr = $$tagInfo{Table};
        if ($tagTablePtr) {
            # construct our group list
            my $groups = $$tagInfo{Groups};
            $groups or $groups = $$tagInfo{Groups} = { };
            # fill in default groups
            my $defaultGroups = $$tagTablePtr{GROUPS};
            foreach (keys %$defaultGroups) {
                $$groups{$_} or $$groups{$_} = $$defaultGroups{$_};
            }
        }
        # set flag indicating group list was built
        $$tagInfo{GotGroups} = 1;
    }
    unless (defined $family) {
        if (wantarray) {
            # return all groups in array context
            my @groups;
            my $tagGroups = $$tagInfo{Groups};
            foreach (keys %$tagGroups) {
                $groups[$_] = $tagGroups->{$_};
            }
            return @groups;
        } else {
            $family = 0;
        }
    }
    return $tagInfo->{Groups}->{$family} || 'Other';
}

#------------------------------------------------------------------------------
# Get group names for specified tags
# Inputs: 0) ExifTool object reference
#         1) [optional] information hash reference (default all extracted info)
#         2) [optional] group family number (default 0)
# Returns: List of group names in alphabetical order
sub GetGroups($;$$)
{
    my $self = shift;
    my $info = shift;
    my $family;

    # figure out our arguments
    if (ref $info ne 'HASH') {
        $family = $info;
        $info = $self->{PRINT_CONV};
    } else {
        $family = shift;
    }
    $family = 0 unless defined $family;

    # get a list of all groups in specified information
    my ($tag, %groups);
    foreach $tag (keys %$info) {
        $groups{ $self->GetGroup($tag, $family) } = 1;
    }
    return sort keys %groups;
}

#------------------------------------------------------------------------------
# Build composite tags from required tags
# Inputs: 0) ExifTool object reference
# Note: Tag values are calculated in alphabetical order unless a tag Require's
#       or Desire's another composite tag, in which case the calculation is
#       deferred until after the other tag is calculated.
sub BuildCompositeTags($)
{
    my $self = shift;
    my @tagList = sort keys %Image::ExifTool::compositeTags;

    my $dataPt = \$self->{EXIF_DATA};   # set data ptr to be used in eval
    for (;;) {
        my %notBuilt;
        foreach (@tagList) {
            $notBuilt{$_} = 1;
        }
        my @deferredTags;
        my $tag;
COMPOSITE_TAG:
        foreach $tag (@tagList) {
            next if $specialTags{$tag};
            my $tagInfo = $self->GetTagInfo(\%Image::ExifTool::compositeTags, $tag);
            next unless $tagInfo;
            # put required tags into array and make sure they all exist
            my (@val, @valPrint, $type);
            foreach $type ('Require','Desire') {
                my $req = $$tagInfo{$type};
                $req or next;
                # save Require'd and Desire'd tag values in list
                my $index;
                foreach $index (keys %$req) {
                    my $reqTag = $$req{$index};
                    # calculate this tag later if it relies on another
                    # Composite tag which hasn't been calculated yet
                    if ($notBuilt{$reqTag}) {
                        push @deferredTags, $tag;
                        next COMPOSITE_TAG;
                    }
                    unless (defined $self->{VALUE_CONV}->{$reqTag}) {
                        # don't continue if we require this tag
                        $type eq 'Require' and next COMPOSITE_TAG;
                    }
                    $val[$index] = $self->{VALUE_CONV}->{$reqTag};
                    $valPrint[$index] = $self->{PRINT_CONV}->{$reqTag};
                }
            }
            delete $notBuilt{$tag}; # this tag is OK to build now
            unless ($$tagInfo{ValueConv}) {
                warn "Can't build composite tag $tag (no ValueConv)\n";
                next;
            }
            $self->FoundTag($tagInfo, undef, \@val, \@valPrint);
        }
        last unless @deferredTags;
        if (@deferredTags == @tagList) {
            # everything was deferred in the last pass,
            # must be a circular dependency
            warn "Circular dependency in Composite tags\n";
            last;
        }
        @tagList = @deferredTags; # calculate deferred tags now
    }
}

#------------------------------------------------------------------------------
# Get tag name (removes copy index)
# Inputs: Tag key
sub GetTagName($)
{
    $_[0] =~ /^(\S+)/;
    return $1;
}

#------------------------------------------------------------------------------
# Get list of shortcuts
# Returns: Shortcut list (sorted alphabetically)
sub GetShortcuts()
{
    require Image::ExifTool::Shortcuts;
    return sort keys %Image::ExifTool::Shortcuts::Main;
}

#------------------------------------------------------------------------------
# Get list of all available tags
# Returns: tag list (sorted alphabetically)
sub GetAllTags()
{
    my %allTags;

    LoadAllTables();    # first load all our tables

    # list of tables to load
    # Note: Do Composite last (tables are popped off the end of the list)
    my @tableNames = ('Image::ExifTool::compositeTags', keys %allTables);

    # recursively load all tables referenced by the current tables
    while (@tableNames) {
        my $table = GetTagTable(pop @tableNames);
        # save all tag names and look for any SubDirectory tables
        my $key;
        foreach $key (TagTableKeys($table)) {
            my @infoArray = GetTagInfoArray($table,$key);
            my $tagInfo;
            foreach $tagInfo (@infoArray) {
                my $tag = $$tagInfo{Name} || $key;
                $allTags{$tag} = 1;
                my $subdir = $$tagInfo{SubDirectory} or next;
                my $tableName = $$subdir{TagTable} or next;
                next if $allTables{$tableName}; # next if table already loaded
                push @tableNames, $tableName;   # must scan this one too
            }
        }
    }
    return sort keys %allTags;
}

#------------------------------------------------------------------------------
# Get list of all group names
# Inputs: 1) Group family number
# Returns: List of group names (sorted alphabetically)
sub GetAllGroups($)
{
    my $family = shift || 0;

    LoadAllTables();    # first load all our tables

    # list of tables to load
    # Note: Do Composite last (tables are popped off the end of the list)
    my @tableNames = ( 'Image::ExifTool::compositeTags', keys %allTables );

    # recursively load all tables referenced by the current tables
    my %allGroups;
    while (@tableNames) {
        my $table = GetTagTable(pop @tableNames);
        my $defaultGroup;
        $defaultGroup = $table->{GROUPS}->{$family} if $table->{GROUPS};
        $defaultGroup = 'Other' unless defined $defaultGroup;
        $allGroups{$defaultGroup} = 1;
        # look for any SubDirectory tables
        foreach (TagTableKeys($table)) {
            my @infoArray = GetTagInfoArray($table,$_);
            my ($tagInfo, $groups, $group);
            foreach $tagInfo (@infoArray) {
                unless ($groups = $$tagInfo{Groups} and $group = $$groups{$family}) {
                    $group = $defaultGroup;
                }
                $allGroups{$group} = 1;
                # recursively scan subdirectories if they exist
                my $subdir = $$tagInfo{SubDirectory} or next;
                my $tableName = $$subdir{TagTable} or next;
                next if $allTables{$tableName}; # next if table already loaded
                push @tableNames, $tableName;   # must scan this one too
            }
        }
    }
    return sort keys %allGroups;
}

#==============================================================================
# Functions below this are not part of the public API

#------------------------------------------------------------------------------
# Init - initialize member variables from arguments
# Inputs: Same as ImageInfo()
# - sets REQUESTED_TAGS, IO_TAG_LIST, FILENAME, RAF, OPTIONS
sub ParseArguments($;@)
{
    my $self = shift;
    my $options = $self->{OPTIONS};
    my @exclude;
    my @oldGroupOpts = grep /^Group/, keys %{$self->{OPTIONS}};
    my $wasExcludeOpt;

    $self->{REQUESTED_TAGS} = [ ];
    $self->{IO_TAG_LIST} = undef;

    # handle our input arguments
    while (@_) {
        my $arg = shift;
        if (ref $arg) {
            if (ref $arg eq 'ARRAY') {
                $self->{IO_TAG_LIST} = $arg;
                foreach (@$arg) {
                    if (/^-(.*)/) {
                        push @exclude, $1;
                    } else {
                        push @{$self->{REQUESTED_TAGS}}, $_;
                    }
                }
            } elsif (ref $arg eq 'HASH') {
                my $opt;
                foreach $opt (keys %$arg) {
                    # a single new group option overrides all old group options
                    if (@oldGroupOpts and $opt =~ /^Group/) {
                        foreach (@oldGroupOpts) {
                            delete $options->{$_};
                        }
                        undef @oldGroupOpts;
                    }
                    $options->{$opt} = $$arg{$opt};
                    $opt eq 'Exclude' and $wasExcludeOpt = 1;
                }
            } elsif (ref $arg eq 'GLOB' or ref $arg eq 'SCALAR') {
                next if defined $self->{RAF};
                $self->{RAF} = new File::RandomAccessFile($arg);
                # set filename to empty string to indicate that
                # we have a file but we didn't open it
                $self->{FILENAME} = '';
            } else {
                warn "Don't understand ImageInfo argument $arg\n";
            }
        } elsif (defined $self->{FILENAME}) {
            if ($arg =~ /^-(.*)/) {
                push @exclude, $1;
            } else {
                push @{$self->{REQUESTED_TAGS}}, $arg;
            }
        } else {
            $self->{FILENAME} = $arg;
        }
    }
    # expand shortcuts in tag arguments if provided
    ExpandShortcuts($self->{REQUESTED_TAGS}) if @{$self->{REQUESTED_TAGS}};

    if (@exclude or $wasExcludeOpt) {
        # must make a copy of exclude list so we can modify it
        if ($options->{Exclude}) {
            if (ref $options->{Exclude} eq 'ARRAY') {
                # make copy of list so we can modify it below
                $options->{Exclude} = [ @{$options->{Exclude}} ];
            } else {
                # turn single exclude into list reference
                $options->{Exclude} = [ $options->{Exclude} ];
            }
            # add our new exclusions
            push @{$options->{Exclude}}, @exclude;
        } else {
            $options->{Exclude} = \@exclude;
        }
        # expand shortcuts in new exclude list
        ExpandShortcuts($options->{Exclude});
    }
}

#------------------------------------------------------------------------------
# Add warning tag
# Inputs: 0) reference to ExifTool object
#         1) warning message
sub Warn($$)
{
    $_[0]->FoundTag('Warning', $_[1]);
}

#------------------------------------------------------------------------------
# Add error tag
# Inputs: 0) reference to ExifTool object
#         1) error message
sub Error($$)
{
    $_[0]->FoundTag('Error', $_[1]);
}

#------------------------------------------------------------------------------
# Load all tag tables
sub LoadAllTables()
{
    # load all of our non-referenced tables (Exif table first)
    GetTagTable('Image::ExifTool::Exif::Main');
    GetTagTable('Image::ExifTool::CanonRaw::Main');
    GetTagTable('Image::ExifTool::Photoshop::Main');
    GetTagTable('Image::ExifTool::GeoTiff::Main');
    GetTagTable('Image::ExifTool::extraTags');
}

#------------------------------------------------------------------------------
# Expand shortcuts
# Inputs: 0) reference to list of tags
# Notes: Handles leading '-' indicating excluded tag
sub ExpandShortcuts($)
{
    my $tagList = shift || return;

    require Image::ExifTool::Shortcuts;

    # expand shortcuts
    my @expandedTags;
    my ($entry, $tag);
EXPAND_TAG:
    foreach $entry (@$tagList) {
        ($tag = $entry) =~ s/^-//;  # remove leading '-'
        foreach (keys %Image::ExifTool::Shortcuts::Main) {
            /^$tag$/i or next;
            if ($tag eq $entry) {
                push @expandedTags, @{$Image::ExifTool::Shortcuts::Main{$_}};
            } else {
                # entry starts with '-', so exclude all tags in this shortcut
                foreach (@{$Image::ExifTool::Shortcuts::Main{$_}}) {
                    /^-/ and next;  # ignore excluded exclude tags
                    push @expandedTags, "-$_";
                }
            }
            next EXPAND_TAG;
        }
        push @expandedTags, $entry;
    }
    @$tagList = @expandedTags;
}

#------------------------------------------------------------------------------
# Add hash of composite tags to our composites
# Inputs: 0) reference to hash of composite tags
sub AddCompositeTags($)
{
    my $add = shift;
    my $groups = $$add{GROUPS};

    # make sure default groups are defined in families 0 and 1
    if ($groups) {
        $groups->{0} or $groups->{0} = 'Composite';
        $groups->{1} or $groups->{1} = 'Composite';
    } else {
        $groups = $$add{GROUPS} = { 0 => 'Composite', 1 => 'Composite' };
    }
    SetupTagTable($add);

    my $tag;
    foreach $tag (keys %$add) {
        next if $specialTags{$tag}; # must skip special tags
        # ignore duplicate composite tags
        next if $Image::ExifTool::compositeTags{$tag};
        # add this composite tag to our main composite table
        $Image::ExifTool::compositeTags{$tag} = $$add{$tag};
    }
}

#------------------------------------------------------------------------------
# Set up tag table (must be done once for each tag table used)
# Inputs: 0) Reference to tag table
# Notes: - generates 'Name' field from key if it doesn't exist
#        - stores 'Table' pointer
#        - expands 'Flags' for quick lookup
sub SetupTagTable($)
{
    my $tagTablePtr = shift;
    my $tagVal;
    foreach $tagVal (TagTableKeys($tagTablePtr)) {
        my @infoArray = GetTagInfoArray($tagTablePtr,$tagVal);
        # process conditional tagInfo arrays
        my $tagInfo;
        foreach $tagInfo (@infoArray) {
            $$tagInfo{Table} = $tagTablePtr;
            my $tag = $$tagInfo{Name};
            unless (defined $tag) {
                # generate name equal to tag value if 'Name' doesn't exist
                $tag = $tagVal;
                $$tagInfo{Name} = $tag;
            }
            # add Flags to tagInfo hash for quick lookup
            if ($$tagInfo{Flags}) {
                my $flags = $$tagInfo{Flags};
                if (ref $flags eq 'ARRAY') {
                    foreach (@$flags) {
                        $$tagInfo{$_} = 1;
                    }
                } elsif (ref $flags eq 'HASH') {
                    my $key;
                    foreach $key (keys %$flags) {
                        $$tagInfo{$key} = $$flags{$key};
                    }
                } else {
                    $$tagInfo{$flags} = 1;
                }
            }
        }
    }
}

#------------------------------------------------------------------------------
# Utility routines to for reading binary data values from file

my $swapBytes;               # set if EXIF header is not native byte ordering
my $currentByteOrder = 'MM'; # current byte ordering ('II' or 'MM')
my %unpackMotorola = ( S => 'n', L => 'N' );
my %unpackIntel    = ( S => 'v', L => 'V' );
my %unpackStd = %unpackMotorola;

# Unpack value, letting unpack() handle byte swapping
# Inputs: 0) unpack template, 1) data reference, 2) offset (default 0)
# Returns: unpacked number
# - uses value of %unpackStd to determine the unpack template
# - can only be called for 'S' or 'L' templates since these are the only
#   templates for which you can specify the byte ordering.
sub DoUnpackStd(@)
{
    $_[2] and return unpack("x$_[2] $unpackStd{$_[0]}", ${$_[1]});
    return unpack($unpackStd{$_[0]}, ${$_[1]});
}

# Unpack value, handling the byte swapping manually
# Inputs: 0) # bytes, 1) unpack template, 2) data reference, 3) offset (default 0)
# Returns: unpacked number
# - uses value of $swapBytes to determine byte ordering
sub DoUnpack(@)
{
    my ($bytes, $template, $dataPt, $pos) = @_;
    $pos = 0 unless defined $pos;
    my $val;
    if ($swapBytes) {
        $val = '';
        $val .= substr($$dataPt,$pos+$bytes,1) while $bytes--;
    } else {
        $val = substr($$dataPt,$pos,$bytes);
    }
    defined($val) or return undef;
    return unpack($template,$val);
}

# Inputs: 0) data reference, 1) offset into data (zero if not defined)
sub Get16u($;$)    { return DoUnpackStd('S', @_); }
sub Get16s($;$)    { return DoUnpack(2, 's', @_); }
sub Get32u($;$)    { return DoUnpackStd('L', @_); }
sub Get32s($;$)    { return DoUnpack(4, 'l', @_); }
sub GetFloat($;$)  { return DoUnpack(4, 'f', @_); }
sub GetDouble($;$) { return DoUnpack(8, 'd', @_); }

#------------------------------------------------------------------------------
# Get current byte order ('II' or 'MM')
sub GetByteOrder() { return $currentByteOrder; }

#------------------------------------------------------------------------------
# set byte ordering
# Inputs: 0) 'II'=intel, 'MM'=motorola
# Returns: 1 on success
sub SetByteOrder($)
{
    my $order = shift;

    if ($order eq 'MM') {       # big endian (Motorola)
        %unpackStd = %unpackMotorola;
    } elsif ($order eq 'II') {  # little endian (Intel)
        %unpackStd = %unpackIntel;
    } else {
        return 0;
    }
    my $val = unpack('S',"A ");
    my $nativeOrder;
    if ($val == 0x4120) {       # big endian
        $nativeOrder = 'MM';
    } elsif ($val == 0x2041) {  # little endian
        $nativeOrder = 'II';
    } else {
        warn sprintf("Unknown native byte order! (pattern %x)\n",$val);
        return 0;
    }
    # swap bytes if our native CPU byte ordering is not the same as the EXIF
    $swapBytes = ($order ne $nativeOrder);
    $currentByteOrder = $order;  # save current byte order

    return 1;
}

#------------------------------------------------------------------------------
# change byte order
sub ToggleByteOrder()
{
    SetByteOrder(GetByteOrder() eq 'II' ? 'MM' : 'II');
}

#------------------------------------------------------------------------------
# Dump data in hex and ASCII to console
# Inputs: 0) data reference, 1) length, 2-N) Options:
# Options: Start => offset to start of data (default=0)
#          Addr => address to print for data start (default=Start)
#          Width => width of printout (bytes, default=16)
#          Prefix => prefix to print at start of line (default='')
sub HexDump($;$%)
{
    my $dataPt = shift;
    my $len    = shift;
    my %opts   = @_;
    my $start  = $opts{Start}  || 0;
    my $addr   = $opts{Addr}   || $start;
    my $wid    = $opts{Width}  || 16;
    my $prefix = $opts{Prefix} || '';

    defined $len or $len = length($$dataPt) - $start;

    my $i;
    for ($i=0; $i<$len; $i+=$wid) {
        $wid > $len-$i and $wid = $len-$i;
        printf "$prefix%8.4x: ", $addr+$i;
        my $dat = substr($$dataPt, $i+$start, $wid);
        printf "%-48s", join(' ',unpack("H*",$dat) =~ /../g);
        $dat =~ tr /\x00-\x1f\x7f-\xff/./;
        print "[$dat]\n";
    }
}

#------------------------------------------------------------------------------
# Dump tag data in hex and ASCII to console
# Inputs: 0) Tag number, 1) data reference, 2) length, 3-N) Options (See HexDump())
sub HexDumpTag($$;$%)
{
    my $tag    = shift;
    my $dataPt = shift;
    my $len    = shift;
    printf("  Tag 0x%.4x Hex Dump (%d bytes):\n",$tag, $len);
    HexDump($dataPt, $len, @_);
}

#------------------------------------------------------------------------------
# return printable value
# Inputs: 0) value to print
sub Printable($)
{
    my $outStr = shift;
    return '(undef)' unless defined $outStr;
    $outStr =~ tr/\x01-\x1f\x80-\xff/\./;
    $outStr =~ s/\x00//g;
    return $outStr;
}

#------------------------------------------------------------------------------
# Convert date/time from Exif format
# Inputs: 0) ExifTool object reference, 1) Date/time in EXIF format
# Returns: Formatted date/time string
sub ConvertDateTime($$)
{
    my ($self, $date) = @_;
    my $dateFormat = $self->{OPTIONS}->{DateFormat};
    # only convert date if a format was specified and the date is recognizable
    if ($dateFormat and $date =~ /^(\d+):(\d+):(\d+)\s+(\d+):(\d+):(\d+)/) {
        if (eval 'require POSIX') {
            $date = POSIX::strftime($dateFormat, $6, $5, $4, $3, $2-1, $1-1900);
        }
    }
    return $date;
}

#------------------------------------------------------------------------------
# JpgInfo : extract EXIF information from a jpg image
# Inputs: 0) ExifTool object reference, 1) file handle
# Returns: 1 if this was a valid JPG file
sub JpgInfo($)
{
    my $self = shift;
    my ($ch,$s,$length,$buff) = (0,0,0,0);
    my ($a,$b,$c,$d);
    my $verbose = $self->{OPTIONS}->{Verbose};
    my $raf = $self->{RAF};

    # check to be sure this is a valid JPG file
    return 0 unless $raf->Read($s,2) == 2 and $s eq "\xff\xd8";

    $self->FoundTag('FileType', 'JPG');    # set file type

    # set input record separator to 0xff (the JPEG marker) to make reading quicker
    my $oldsep = $/;
    $/ = "\xff";

    # read file until we reach an end of image (EOI) or start of scan (SOS)
    Marker: for (;;) {
        # Find next marker (JPEG markers begin with 0xff)
        $raf->ReadLine($buff) or last;
        # JPEG markers can be padded with unlimited 0xff's
        do { $raf->Read($ch, 1) or last Marker } while ord($ch) == 0xff;
        if (ord($ch) >= 0xc0 and ord($ch) <= 0xc3) {
            last unless $raf->Read($buff, 3) == 3;
            last unless $raf->Read($s, 4) == 4;
            ($a,$b,$c,$d) = unpack("C"x4,$s);
            # calculate the image size;
            my $w = ($c << 8) | $d;
            my $h = ($a << 8) | $b;
            $self->FoundTag('ImageWidth', $w);
            $self->FoundTag('ImageHeight', $h);
            # ignore stand-alone markers 0x01 and 0xd0-0xd7
        } elsif (ord($ch)!=0x01 and (ord($ch)<0xd0 or ord($ch)>0xd7)) {
            # We must skip data blocks, since 0xff's within data
            # are not valid JPEG markers
            last unless $raf->Read($s, 2) == 2;
            ($a, $b) = unpack("C"x2,$s);
            $length = ($a << 8) | $b;   # get length
            last if (!defined($length) || $length < 2);
            $length -= 2;   # get length of data alone
            last unless $raf->Read($buff, $length) == $length;
            $verbose > 2 and printf("JPG marker 0x%x, len $length\n",ord($ch));
            if (ord($ch) == 0xe1) {             # EXIF data marker
                if ($length<6 or substr($buff,0,6) ne "Exif\0\0") {
                    # Hmmm.  Could be XMP, let's see
                    if ($buff =~ /<exif:/ || $buff =~ /^http/) {
                        my $tagTablePtr = GetTagTable('Image::ExifTool::XMP::Main');
                        my %dirInfo = (
                            DataPt   => \$buff,
                            DataLen  => $length,
                            DirStart => 0,
                            DirLen   => $length,
                            DirBase  => 0,
                            Nesting  => 0,
                        );
                        next if $self->ProcessTagTable($tagTablePtr, \%dirInfo);
                    }
                    $verbose and $self->Warn("Ignored EXIF block length $length (bad header)");
                    last;
                }
                # get the data block (into a common variable)
                $self->{EXIF_DATA} = substr($buff, 6);
                # extract the EXIF information
                $self->ExifInfo($length-6);
            } elsif (ord($ch) == 0xed) {        # APP13 data marker
                if ($buff =~ /^Photoshop 3.0\0/) {
                    # process Photoshop APP13 record
                    my $tagTablePtr = GetTagTable('Image::ExifTool::Photoshop::Main');
                    my %dirInfo = (
                        DataPt   => \$buff,
                        DataLen  => $length,
                        DirStart => 14,     # directory starts after identifier
                        DirLen   => $length-14,
                        DirBase  => 0,
                        Nesting  => 0,
                    );
                    $self->ProcessTagTable($tagTablePtr, \%dirInfo);
                } elsif ($buff =~ /^\x1c\x02/) {
                    # this is written in IPTC format by photoshop, but is
                    # all messed up, so we ignore it
                } else {
                    $self->Warn("Unknown APP13 data in $self->{FILENAME}");
                    if ($verbose > 1) {
                        print "  APP13 Data:\n";
                        HexDump(\$buff);
                    }
                }
            } elsif (ord($ch) == 0xfe) {        # JPEG comment
                $self->FoundTag('Comment',$buff);
            } elsif ($verbose > 2 and ord($ch)) {
                HexDump(\$buff);
            }
        }
    }
    $/ = $oldsep;     # restore separator to original value
    return 1;
}

#------------------------------------------------------------------------------
# get GIF size and comments (no EXIF blocks in GIF files though)
# Inputs: 0) ExifTool object reference
# Returns: 1 if this was a valid GIF file
sub GifInfo($)
{
    my $self = shift;
    my ($type, $s, $ch, $length, $buff);
    my $raf = $self->{RAF};

    # verify this is a valid GIF file
    # (must do a RAF read until we know the file is ours)
    return 0 unless $self->{RAF}->Read($type, 6) == 6
        and $type =~ /^GIF8[79]a$/
        and $self->{RAF}->Read($s, 4) == 4;

    $self->FoundTag('FileType','GIF');     # set file type

    my ($a,$b,$c,$d) = unpack("C"x4, $s);
    my $w = ($b << 8) | $a;
    my $h = ($d << 8) | $c;
    $self->FoundTag('ImageWidth', $w);
    $self->FoundTag('ImageHeight', $h);
    if ($raf->Read($s, 3) == 3) {
        if (ord($s) & 0x80) { # does this image contain a color table?
            # calculate color table size
            $length = 3 * (2 << (ord($s) & 0x07));
            $raf->Read($buff, $length);  # skip color table
            my $comment;
            for (;;) {
                last unless $raf->Read($ch, 1);
                if (ord($ch) == 0x2c) {
                    # image descriptor
                    last unless $raf->Read($buff, 8) == 8;
                    last unless $raf->Read($ch, 1);
                    if (ord($ch) & 0x80) { # does color table exist?
                        $length = 3 * (2 << (ord($ch) & 0x07));
                        # skip the color table
                        last unless $raf->Read($buff, $length) == $length;
                    }
                    # skip "LZW Minimum Code Size" byte
                    last unless $raf->Read($buff, 1);
                    # skip image blocks
                    for (;;) {
                        last unless $raf->Read($ch, 1);
                        last unless ord($ch);
                        last unless $raf->Read($buff, ord($ch));
                    }
                    next;  # continue with next field
                }
#               last if ord($ch) == 0x3b;  # normal end of GIF marker
                # check for a valid marker
                last unless ord($ch) == 0x21;
                last unless $raf->Read($s, 2) == 2;
                # get marker and block size
                ($a,$length) = unpack("C"x2, $s);
                if ($a == 0xfe) {  # is this a comment?
                    while ($length) {
                        last unless $raf->Read($buff, $length) == $length;
                        if (defined $comment) {
                            $comment .= $buff;  # add to comment string
                        } else {
                            $comment = $buff;
                        }
                        last unless $raf->Read($ch, 1);  # read next block header
                        $length = ord($ch);  # get next block size
                    }
                    last;  # all done once we have found the comment
                } else {
                    # skip the block
                    while ($length) {
                        last unless $raf->Read($buff, $length) == $length;
                        last unless $raf->Read($ch, 1);  # read next block header
                        $length = ord($ch);  # get next block size
                    }
                }
            }
            $self->FoundTag('Comment', $comment) if $comment;
        }
    }
    return 1;
}

#------------------------------------------------------------------------------
# Process EXIF block
# Inputs: 0) ExifTool object reference,
#         1) data length (or FileType if RAF specified)
#         2) RAF pointer if we must read the data ourself
# Returns: 1 if this looked like a valid EXIF block, 0 otherwise
sub ExifInfo($$;$)
{
    my ($self, $length, $raf) = @_;
    my $dataPt = \$self->{EXIF_DATA};

    # read start of EXIF block from file if not done already
    $raf->Read($$dataPt,8)==8 or return 0 if $raf;

    # set byte ordering
    SetByteOrder(substr($$dataPt,0,2)) or return 0;

    # make sure our swapping works
    Get16u($dataPt, 2) == 0x2a or return 0;

    my $offset = Get32u($dataPt, 4);
    $offset >= 8 or return 0;

    # read the EXIF data and main directory if required
    if ($raf) {
        $length and $self->FoundTag('FileType', $length);
        # read up to and including the IFD directory count
        my $buff;
        $raf->Read($buff,$offset-6)==$offset-6 or return 0;
        $$dataPt .= $buff;
        $length = 12 * Get16u($dataPt,$offset);
        # read the directory
        $raf->Read($buff,$length)==$length or return;
        $$dataPt .= $buff;
        # set length to the total number of bytes read so far
        $length += $offset + 2;
    }

    # get reference to the main EXIF table
    my $tagTablePtr = GetTagTable('Image::ExifTool::Exif::Main');

    # build directory information hash
    my %dirInfo = (
        DataPt   => $dataPt,
        DataLen  => $length,
        DirStart => $offset,
        DirLen   => $length,
        DirBase  => 0,
        Nesting  => 0,
        RAF      => $raf,
    );
    # process the directory
    $self->ProcessTagTable($tagTablePtr, \%dirInfo);

    return 1;
}

#------------------------------------------------------------------------------
# Return list of tag table keys (ignoring special keys)
# Inputs: 0) reference to tag table
# Returns: List of table keys
sub TagTableKeys($)
{
    my $tagTablePtr = shift;
    my @keyList;
    foreach (keys %$tagTablePtr) {
        push(@keyList, $_) unless $specialTags{$_};
    }
    return @keyList;
}

#------------------------------------------------------------------------------
# GetTagTable
# Inputs: 0) table name
# Returns: tag table reference, or undefined if not found
# Notes: Always use this function instead of requiring module and using table
# directly since this function also does the following the first time the table
# is loaded:
# - requires new module if necessary
# - generates default GROUPS hash and Group 0 name from module name
# - registers Composite tags if Composite table found
# - saves descriptions for tags in specified table
# - generates default TAG_PREFIX to be used for unknown tags
sub GetTagTable($)
{
    my $tableName = shift or return undef;

    my $table = $allTables{$tableName};

    unless ($table) {
        no strict 'refs';
        unless (defined %$tableName) {
            # try to load module for this table
            if ($tableName =~ /(.*)::/) {
                my $module = $1;
                if (eval "require $module") {
                    # look for 'Composite' table and add it to our composites
                    if (defined %{"${module}::Composite"}) {
                        no strict 'refs';
                        AddCompositeTags(\%{"${module}::Composite"});
                    }
                } else {
                    $@ and warn $@;
                }
            }
            unless (defined %$tableName) {
                warn "Can't find table $tableName\n";
                return undef;
            }
        }
        no strict 'refs';
        $table = \%$tableName;
        use strict 'refs';
        # set default group 0 and 1 from module name unless already specified
        my $defaultGroups = $$table{GROUPS};
        $defaultGroups or $defaultGroups = $$table{GROUPS} = { };
        unless ($$defaultGroups{0} and $$defaultGroups{1}) {
            if ($tableName =~ /Image::.*?::([^:]*)/) {
                $$defaultGroups{0} = $1 unless $$defaultGroups{0};
                $$defaultGroups{1} = $1 unless $$defaultGroups{1};
            } else {
                $$defaultGroups{0} = $tableName unless $$defaultGroups{0};
                $$defaultGroups{1} = $tableName unless $$defaultGroups{1};
            }
        }
        # generate a tag prefix for unknown tags if necessary
        unless ($$table{TAG_PREFIX}) {
            my $tagPrefix;
            if ($tableName =~ /Image::.*?::(.*)::Main/ || $tableName =~ /Image::.*?::(.*)/) {
                ($tagPrefix = $1) =~ s/::/_/g;
            } else {
                $tagPrefix = $tableName;
            }
            $$table{TAG_PREFIX} = $tagPrefix;
        }
        # save all descriptions in the new table
        SetupTagTable($table);
        # insert newly loaded table into list
        $allTables{$tableName} = $table;
    }
    return $table;
}

#------------------------------------------------------------------------------
# Process specified tag table
# Inputs: 0) ExifTool object reference, 1) Tag table reference
#         2) Directory information reference
# Returns: Result from processing (1=success)
sub ProcessTagTable($$$)
{
    my ($self, $tagTablePtr, $dirInfo) = @_;
    $tagTablePtr or return 0;
    my $processProc = $$tagTablePtr{PROCESS_PROC};
    # process as EXIF directory unless otherwise specified
    $processProc or $processProc = \&Image::ExifTool::Exif::ProcessExif;
    return &$processProc($self, $tagTablePtr, $dirInfo);
}

#------------------------------------------------------------------------------
# Return special tag
# Inputs: 0) Tag name
sub GetSpecialTag($)
{
    return $specialTags{$_[0]};
}

#------------------------------------------------------------------------------
# Get array of tag information
# Inputs: 0) Tag table reference, 1) tag value
# Returns: Array of tag information references
# Notes: Generates tagInfo hash if necessary
sub GetTagInfoArray($$)
{
    my $tagTablePtr = shift;
    my $tagVal = shift;
    my $tagInfo = $$tagTablePtr{$tagVal};

    my @infoArray;
    if (ref $tagInfo eq 'ARRAY') {
        @infoArray = @$tagInfo;
    } elsif ($tagInfo) {
        if (ref $tagInfo ne 'HASH') {
            # create hash with name
            $tagInfo = $$tagTablePtr{$tagVal} = { Name => $tagInfo };
        }
        push @infoArray, $tagInfo;
    }
    return @infoArray;
}

#------------------------------------------------------------------------------
# Find tag information, processing conditional tags
# Inputs: 0) ExifTool object reference, 1) tagTable pointer, 2) tag key
# Returns: pointer to tagInfo hash, or undefined if none found
# Notes: You should always call this routine to find a tag in a table because
# this routine will evaluate conditional tags.
sub GetTagInfo($$$)
{
    my ($self, $tagTablePtr, $tag) = @_;

    my @infoArray = GetTagInfoArray($tagTablePtr, $tag);
    # evaluate condition
    my $tagInfo;
    foreach $tagInfo (@infoArray) {
        my $condition = $$tagInfo{Condition};
        if ($condition) {
            # set old value for use in condition if needed
            my $oldVal = $self->{VALUE_CONV}->{$$tagInfo{Name}};
            unless (eval $condition) {
                $@ and warn $@;
                next;
            }
        }
        # return the tag information we found
        return $tagInfo;
    }
    # generate information for unknown tags if required
    # (only numerical tags, since XMP tags won't be known)
    if ($self->{OPTIONS}->{Unknown} and $tag =~ /^\d+$/) {
        my $printConv;
        if (defined $$tagTablePtr{PRINT_CONV}) {
            $printConv = $$tagTablePtr{PRINT_CONV};
        } else {
            # limit length of printout (can be very long)
            $printConv = 'length($val) > 60 ? substr($val,0,55) . "[...]" : $val';
        }
        $tagInfo = {
            Name => sprintf("%s_0x%.4x",$$tagTablePtr{TAG_PREFIX} || 'Unknown', $tag),
            #Name => sprintf("%s_0x%.4x",$$tagTablePtr{GROUPS}->{0},$tag),
            Groups => $$tagTablePtr{GROUPS},
            PrintConv => $printConv,
        };
    } else {
        undef $tagInfo;
    }
    return $tagInfo;
}

#------------------------------------------------------------------------------
# found specified tag
# Inputs: 0) reference to ExifTool object
#         1) reference to tagInfo hash or tag name
#         2) data value (may be undefined if building composite tag)
#         3) optional reference to list of values used to build composite tags
#         4) optional reference to list of print values for composite tags
sub FoundTag($$$;$$)
{
    my ($self, $tagInfo, $val, $valListPt, $valPrintPt) = @_;
    my ($tag, @val, @valPrint);
    my $verbose = $self->{OPTIONS}->{Verbose};

    if (ref($tagInfo) eq 'HASH') {
        $tag = $$tagInfo{Name} or warn("No tag name\n"), return;
    } else {
        $tag = $tagInfo;
        # look for tag in extraTags
        $tagInfo = $self->GetTagInfo(GetTagTable('Image::ExifTool::extraTags'), $tag);
        # make temporary hash if tag doesn't exist in extraTags
        # (not advised to do this since the tag won't show in list)
        $tagInfo or $tagInfo = { Name => $tag, Groups => \%allGroupsExifTool };
    }
    # initialize arrays used to build composite tags if necessary
    if ($valListPt) {
        @val = @$valListPt;
        @valPrint = @$valPrintPt;
    }

    # convert the value into a usable form
    # (always do this conversion even if we don't want to return
    #  the value because the conversion may have side-effects)
    my $valueConv;
    if ($$tagInfo{ValueConv}) {
        my $valueConversion = $$tagInfo{ValueConv};
        if (ref($valueConversion) eq 'HASH') {
            $valueConv = $$valueConversion{$val};
            defined $valueConv or $valueConv = "Unknown ($val)";
        } else {
            my $dataPt = \$self->{EXIF_DATA};   # set $dataPt for use in eval
            $valueConv = eval $valueConversion;
            $@ and warn $@;
            # treat it as if the tag doesn't exist if ValueConv returns undef
            return unless defined $valueConv;
        }
    } elsif (defined $val) {
        $valueConv = $val;
    } else {
        # this is a composite tag that could not be calculated
        $verbose and warn "Can't get value for $tag\n";
        return;
    }
    $val = $valueConv;
    $verbose and print "  $tag = ",Printable($val),"\n";

    # do the print conversion if required
    my $printConv;
    my $doPrintConversion = $self->{OPTIONS}->{PrintConv};
    if (not $doPrintConversion) {
        $printConv = $val;
    } elsif ($$tagInfo{PrintConv}) {
        my $printConversion = $$tagInfo{PrintConv};
        if (ref($printConversion) eq 'HASH') {
            $printConv = $$printConversion{$val};
            defined $printConv or $printConv = "Unknown ($val)";
        } else {
            my $dataPt = \$self->{EXIF_DATA};   # set $dataPt for use in eval
            $printConv = eval $printConversion;
            $@ and warn $@;
            if (defined $printConv) {
                # terminate string at null unless this is a data reference
                $printConv =~ s/\0.*// unless ref $printConv;
            } else {
                $printConv = 'Undefined';
            }
            # WARNING: do not change $val after this (because $printConv
            # could be a reference to $val for binary data types)
        }
    } else {
        # terminate string at null since printConv is enabled
        ($printConv = $val) =~ s/\0.*//;
    }

    # handle duplicate tag names
    if (defined $self->{PRINT_CONV}->{$tag}) {
        if ($$tagInfo{List}) {
            # form a comma-separated list of print converted values
            $printConv = "$self->{PRINT_CONV}->{$tag}, $printConv";
        } else {
            # rename existing tag to make room for duplicate values
            my $nextTag;  # next available copy number for this tag
            my $i;
            for ($i=1; ; ++$i) {
                $nextTag = "$tag ($i)";
                last unless exists $self->{PRINT_CONV}->{$nextTag};
            }
            # change the name of existing tags in all hashes
            $self->{VALUE_CONV}->{$nextTag} = $self->{VALUE_CONV}->{$tag};
            $self->{PRINT_CONV}->{$nextTag} = $self->{PRINT_CONV}->{$tag};
            $self->{FILE_ORDER}->{$nextTag} = $self->{FILE_ORDER}->{$tag};
            $self->{TAG_INFO}->{$nextTag} = $self->{TAG_INFO}->{$tag};
        }
    }

    # save the converted values, file order, and tag groups
    $self->{VALUE_CONV}->{$tag} = $valueConv;
    $self->{PRINT_CONV}->{$tag} = $printConv;
    $self->{FILE_ORDER}->{$tag} = ++$self->{NUM_TAGS_FOUND};
    $self->{TAG_INFO}->{$tag} = $tagInfo;
}

#------------------------------------------------------------------------------
# delete specified tag
# Inputs: 0) reference to ExifTool object
#         1) tag name
sub DeleteTag($$)
{
    my ($self, $tag) = @_;
    delete $self->{VALUE_CONV}->{$tag};
    delete $self->{PRINT_CONV}->{$tag};
    delete $self->{FILE_ORDER}->{$tag};
    delete $self->{TAG_INFO}->{$tag};
}

#------------------------------------------------------------------------------
# extract binary data from file
# 0) ExifTool object reference, 1) offset, 2) length, 3) tag name if conditional
# Returns: binary data, or undef on error
# Notes: Returns "Binary data #### bytes" instead of data unless tag is
#        specifically requested or the Binary option is set
sub ExtractBinary($$$;$)
{
    my ($self, $offset, $length, $tag) = @_;

    if ($tag and not $self->{OPTIONS}->{Binary}
        and not grep /^$tag$/i, @{$self->{REQUESTED_TAGS}})
    {
        return "Binary data $length bytes";
    }
    my $buff;
    unless ($self->{RAF}->Seek($offset,0)
        and $self->{RAF}->Read($buff,$length) == $length)
    {
        $tag or $tag = 'binary data';
        $self->Warn("Error reading $tag from file");
        return undef;
    }
    return $buff;
}

#------------------------------------------------------------------------------
# process binary data
# Inputs: 0) ExifTool object reference
#         1) pointer to tag table, 2) data reference, 3) dirInfo reference
# Returns: 1 on success
sub ProcessBinaryData($$$)
{
    my ($self, $tagTablePtr, $dirInfo) = @_;
    my $dataPt = $dirInfo->{DataPt};
    my $offset = $dirInfo->{DirStart};
    my $size = $dirInfo->{DirLen};
    my %formatSize = ( short => 2,
                       ushort => 2,
                       long => 4,
                       ulong => 4,
                       string => 1,
                       char => 1,
                       uchar => 1,
                       shortrational => 4,
                       longrational => 8,
                     );

    # get default format ('Short' unless specified)
    my $defaultFormat = $$tagTablePtr{FORMAT} || 'Short';
    my $increment = $formatSize{lc($defaultFormat)};
    unless ($increment) {
        warn "Unknown format $defaultFormat\n";
        $defaultFormat = 'Short';
        $increment = $formatSize{lc($defaultFormat)};
    }
    # prepare list of tag number to extract
    my @tags;
    if ($self->{OPTIONS}->{Unknown} > 1 and defined $$tagTablePtr{FIRST_ENTRY}) {
        # scan through entire binary table
        @tags = ($$tagTablePtr{FIRST_ENTRY}..(int($size/$increment) - 1));
    } else {
        # extract known tags in numerical order
        @tags = sort { $a <=> $b } TagTableKeys($tagTablePtr);
    }
    my $index;
    foreach $index (@tags) {
        my $tagInfo = $self->GetTagInfo($tagTablePtr, $index);
        next unless $tagInfo;
        my $entry = $index * $increment;
        # don't go beyond end of data
        last if $entry >= $size;
        my $format = $$tagInfo{Format} || $defaultFormat;
        my $val;
        if ($format =~ /^Short$/i) {
            $val = Get16s($dataPt, $entry + $offset);
        } elsif ($format =~ /^UShort$/i) {
            $val = Get16u($dataPt, $entry + $offset);
        } elsif ($format =~ /^Long$/i) {
            $val = Get32s($dataPt, $entry + $offset);
        } elsif ($format =~ /^ULong$/i) {
            $val = Get32u($dataPt, $entry + $offset);
        } elsif ($format =~ /^String\[(\d+)\]$/i) {
            $val = substr($$dataPt, $entry + $offset, $1);
        } elsif ($format =~ /^Char$/i) {
            $val = unpack('c', $val);
        } elsif ($format =~ /^UChar$/i) {
            $val = unpack('C', $val);
        } elsif ($format =~ /^ShortRational$/i) {
            my $denom = Get16s($dataPt, $entry + $offset + 2) || 1;
            $val = Get16s($dataPt, $entry + $offset) / $denom;
        } elsif ($format =~ /^LongRational$/i) {
            my $denom = Get32s($dataPt, $entry + $offset + 4) || 1;
            $val = Get32s($dataPt, $entry + $offset) / $denom;
        } else {
            warn "Unknown format $format\n";
        }
        $self->FoundTag($tagInfo,$val);
    }
    return 1;
}

#------------------------------------------------------------------------------
1;  # end

__END__

=head1 NAME

Image::ExifTool - Extract meta information from image files

=head1 SYNOPSIS

use Image::ExifTool 'ImageInfo';

$info = B<ImageInfo>('a.jpg');    # simple!

$exifTool = new Image::ExifTool;

$info = $exifTool-E<gt>B<ImageInfo>($file, \%options, \@tagList, $tag, ...);

$oldValue = $exifTool-E<gt>B<Options>(Option =E<gt> 'Value', ...);

$exifTool-E<gt>B<ClearOptions>();

$success = $exifTool-E<gt>B<ExtractInfo>($file, \%options, \@tagList, ...);

$info = $exifTool-E<gt>B<GetInfo>(\%options, \@tagList, $tag, ...);

$combined = $exifTool-E<gt>B<CombineInfo>($info1, $info2, ...);

@tagList = $exifTool-E<gt>B<GetTagList>($info, $sortOrder);

@foundTags = $exifTool-E<gt>B<GetFoundTags>($sortOrder);

@requestedTags = $exifTool-E<gt>B<GetRequestedTags>();

$value = $exifTool-E<gt>B<GetValue>($tag, $type);

$description = $exifTool-E<gt>B<GetDescription>($tag);

$group = $exifTool-E<gt>B<GetGroup>($tag, $family);

@groups = $exifTool-E<gt>B<GetGroups>($info, $family);

$exifTool-E<gt>B<BuildCompositeTags>();

$tagName = Image::ExifTool::B<GetTagName>($tag);

@shortcuts = Image::ExifTool::B<GetShortcuts>();

@allTags = Image::ExifTool::B<GetAllTags>();

@allGroups = Image::ExifTool::B<GetAllGroups>($family);

=head1 DESCRIPTION

ExifTool provides an extensible set of perl modules to extract and parse
EXIF, IPTC, XMP and GeoTIFF meta information from JPEG, GIF, TIFF, THM, CRW
(Canon RAW), CR2 (Canon 1D Mk II RAW) and NEF (Nikon Electronic image
Format) images.  ExifTool currently reads the maker notes of images from
Canon, Casio, FujiFilm, Minolta, Nikon, Olympus, Pentax, Sanyo and Sigma
digital cameras.

=head1 METHODS

=head2 new

Creates a new ExifTool object.

Example:
    $exifTool = new Image::ExifTool;

=head2 ImageInfo

Obtain meta information from image.  This is the one step function for
obtaining meta information from an image.  Internally, B<ImageInfo> calls
B<ExtractInfo> to extract the information, B<GetInfo> to generate the
information hash, and B<GetTagList> for the returned tag list.

=over 4

=item Examples:

$info = ImageInfo($filename, $tag1, $tag2)

$info = $exifTool-E<gt>ImageInfo(\*FILE)

$info = ImageInfo(\$imageData, \@tagList, \%options)

=item Inputs:

B<ImageInfo> is very flexible about the input arguments, and interprets
them based on their type.  It may be called with one or more arguments.
The one required argument is either a SCALAR (the image file name), a GLOB
reference (a reference to the image file) or a SCALAR reference (a
reference to the image in memory).  Other arguments are optional.  The
order of the arguments is not significant, except that the first SCALAR is
taken to be the file name unless a file reference or scalar reference came
earlier in the argument list.

Below is an explanation of how the B<ImageInfo> function arguments are
interpreted:

=over 4

=item ExifTool ref

B<ImageInfo> may be called with an ExifTool object if desired.  The
advantage of using the object-oriented form is that the options may be set
before calling B<ImageInfo>, and the object may be used afterward to access
member functions.

=item SCALAR

The first scalar argument is taken to be the file name unless an earlier
argument specified the image data via a file reference (GLOB ref) or data
reference (SCALAR ref). The remaining scalar arguments are names of tags for
requested information.  If no tags are specified, all possible information
is extracted.  Tag names may begin with '-' indicating tags to exclude.  The
tag names are case-insensitive, so note that the returned tags may not be
exactly the same as the requested tags. For this reason it is best to use
either the keys of the returned hash or the elements of the tag array when
accessing the return values

=item GLOB ref

A reference to an open image file.

=item SCALAR ref

A reference to image data in memory.

=item ARRAY ref

Reference to a list of tag names.  On entry, any elements in the list are
added to the list of requested tags.  Tags with names beginning with '-' are
excluded.  On return, this list is updated to contain a sorted list of tag
names in the proper case.

=item HASH ref

Reference to a hash containing the options settings.  See B<Options>
documentation below for a list of available options.  Options specified
as arguments to B<ImageInfo> take precidence over B<Options> settings.

=back

=item Return Values:

B<ImageInfo> returns a reference to a hash of tag/value pairs. The keys of
the hash are the tag identifiers, which are similar to the tag names but my
have an embedded copy number if more than one tag with that name was found
in the image.  Use B<GetTagName> to remove the copy number from the tag.
Note that the case of the tags may not be the same as requested. Here is a
simple example to print out the information returned by B<ImageInfo>:

    foreach (sort keys %$info) {
        print "$_ =E<gt> $$info{$_}\n";
    }

The values of the returned hash are usually simple scalars, but a scalar
reference is used to indicate binary data.  Note that binary values are not
necessarily extracted unless specifically requested or the Binary option is
set.  If not extracted the value is a reference to a string of the form
"Binary data ##### bytes".  The code below gives an example of how to handle
these return values, as well as illustrating the use of other ExifTool
functions:

    use Image::ExifTool;
    my $exifTool = new Image::ExifTool;
    $exifTool->Options(Unknown => 1);
    my $info = $exifTool->ImageInfo('a.jpg');
    my $group = '';
    my $tag;
    foreach $tag ($exifTool->GetFoundTags('Group0')) {
        if ($group ne $exifTool->GetGroup($tag)) {
            $group = $exifTool->GetGroup($tag);
            print "---- $group ----\n";
        }
        my $val = $info->{$tag};
        if (ref $val eq 'SCALAR') {
            if ($$val =~ /^Binary data/) {
                $val = "($$val)";
            } else {
                my $len = length($$val);
                $val = "(Binary data $len bytes)";
            }
        }
        printf("%-32s : %s\n", $exifTool->GetDescription($tag), $val);
    }

As well as tags representing information extracted from the image,
the following tags generated by ExifTool may be returned:

    ExifToolVersion - The ExifTool version number.

    Error - An error message if the image could not be read.

    Warning - A warning message if problems were encountered
              while extracting information from the image.

=back

=head2 Options

Get/set ExifTool options.  This function can be called to set the default
options for an ExifTool object.  Options set this way are in effect for
all function calls but may be overridden by options passed as arguments
to a specific function.

Examples:
    $exifTool-E<gt>Options( Exclude =E<gt> 'OwnerName' );

    $exifTool-E<gt>Options( Group1 =E<gt> [ 'EXIF', 'MakerNotes' ] );

    $exifTool-E<gt>Options( Group0 =E<gt> '-XMP' );  # ignore XMP tags

    $exifTool-E<gt>Options( Sort =E<gt> 'Group2', Unknown =E<gt> 1 );

    $oldSetting = $exifTool-E<gt>Options( Duplicates =E<gt> 1 );

Inputs:
    0) ExifTool object reference.
    1) Option parameter name.
    2) [optional] Option parameter value.
    3-N) [optional] Additional parameter/value pairs.

=over 4

=item Option Parameters:

=over 4

=item Binary

Flag to get the value of all binary tags.  Unless set, large
binary values may only be extracted for specifically requested tags.
Default is 0.

=item Composite

Flag to calculate Composite tags automatically.  Default
is 1.

=item DateFormat

Format for printing EXIF date/time.  See L<strftime> for
details about the format string.

=item Duplicates

Flat to preserve values of duplicate tags (instead of
overwriting existing value).  Default is 0.

=item Group#

Extract tags only for specified groups in family # (Group0 assumed if #
not given).  The option value may be a single group name or a reference
to a list of groups.  Case is significant in group names.  Specify a
group to be excluded by preceeding group name with a '-'.  See
B<GetAllGroups> for a list of available groups.

=item Exclude

Exclude specified tags from tags extracted from an image. The
option value is either a tag name or reference to a list of tag names
to exclude.  The case of tag names is not significant. This option is
ignored for specifically requested tags.  Tags may also be excluded
by preceeding their name with a '-' in the arguments to B<ImageInfo>.

=item PrintConv

Flag to enable automatic print conversion.  Default is 1.

=item Sort

Specifies order to sort tags in returned list:

  Alpha  - Sort alphabetically
  File   - Sort in order that tags were found in the file
  Group# - Sort by tag group, where # is the group family
           number.  If # is not specified, Group0 is assumed.
           See GetGroup for a list of groups.
  Input  - Sort in same order as input tag arguments (default)

=item Unknown

Flag to get the values of unknown tags.  If set to 1, unknown tags are
extracted from EXIF directories.  If set to 2, unknown tags are also
extracted from binary data blocks.  Default is 0.

=item Verbose

Flag to print verbose messages.  May be set to a value from 0 to 3 to be
increasingly verbose.  Default is 0.

=back

=item Return Values:

The original value of the last specified parameter.

=back

=head2 ClearOptions

Reset all options to their default values.

Example:
    $exifTool-E<gt>ClearOptions();

Inputs:
    0) ExifTool object reference

=over 4

=item Return Values:

(none)

=back

=head2 ExtractInfo

Extract meta information from an image.  Call B<GetInfo> to access the
information after extracting it from the image.

Example:
    $success = $exifTool-E<gt>ExtractInfo('image.jpg', \%options);

=over 4

=item Inputs:

B<ExtractInfo> takes exactly the same arguments as B<ImageInfo>.  The only
difference is that a list of tags is not returned if an ARRAY reference is
given.  The following options are effective in the call to B<ExtractInfo>:

Binary, Composite, DateFormat, PrintConv, Unknown and Verbose.

=item Return Value:

1 if image was valid, 0 otherwise (and 'Error' tag set).

=back

=head2 GetInfo

B<GetInfo> is called to return meta information after it has been extracted
from the image by a previous call to B<ExtractInfo> or B<ImageInfo>. This
function may be called repeatedly after a single call to B<ExtractInfo> or
B<ImageInfo>.

Examples:
    $info = $exifTool-E<gt>GetInfo('ImageWidth', 'ImageHeight');

    $info = $exifTool-E<gt>GetInfo(\@ioTagList);

    $info = $exifTool-E<gt>GetInfo({Group2 =E<gt> ['Author', 'Location']});

=over 4

=item Inputs:

Inputs are the same as B<ExtractInfo> and B<ImageInfo> except that an image
can not be specified.  Options in effect are:

Duplicates, Exclude, Group#, (and Sort if tag list reference is given).

=item Return Value:

Reference to information hash, the same as with B<ImageInfo>.

=back

=head2 CombineInfo

Combine information from more than one information hash into a single hash.

Example:
    $info = $exifTool-E<gt>CombineInfo($info1, $info2, $info3);

Inputs:
    0) ExifTool object reference
    1-N) Information hash references

If the Duplicates option is disabled and duplicate tags exist, the order of
the hashes is significant.  In this case, the value used is the first value
found as the hashes are scanned in order of input.  The Duplicates option
is the only option that is in effect for this function.

=head2 GetTagList

Get a sorted list of tags from the specified information hash or tag list.

Example:
    @tags = $exifTool-E<gt>GetTagList($info, 'Group0');

Inputs:
    0) ExifTool object reference
    1) [optional] Information hash reference or tag list reference
    2) [optional] Sort order ('File', 'Input', 'Alpha' or 'Group#')

If the information hash or tag list reference is not provided, then the list
of found tags from the last call to B<ImageInfo> or B<GetInfo> is used
instead, and the result is the same as if B<GetFoundTags> was called.  If
sort order is not specified, the sort order from the current options is used.

=over 4

=item Return Values:

A list of tags in the specified order.

=back

=head2 GetFoundTags

Get list of found tags in specified sort order.  The found tags are the
tags for the information returned from the most recent call to B<ImageInfo>
or B<GetInfo> for this object.

Example:
    @tags = $exifTool-E<gt>GetFoundTags('File');

Inputs:
    0) ExifTool object reference
    1) [optional] Sort order ('File', 'Input', 'Alpha' or 'Group#')

If sort order is not specified, the sort order from the ExifTool options is
used.

=over 4

=item Return Values:

A list of tags in the specified order.

=back

=head2 GetRequestedTags

Get list of requested tags.  These are the tags that were specified in the
arguments of the most recent call to B<ImageInfo>, B<ExtractInfo> or
B<GetInfo>, including tags specified via a tag list reference. Shortcut
tags are expanded in the list.

Example:
    @tags = $exifTool-E<gt>GetRequestedTags();

Inputs:
    (none)

=over 4

=item Return Values:

List of requested tags in the same order that they were specified.
Note that this list will be empty if tags were not specifically requested
(ie. If extracting all tags).

=back

=head2 GetValue

Get the value of specified tag.  By default this routine returns the
human-readable (PrintConv) value, but optionally returns the
machine-readable (ValueConv) value. Note that the PrintConv value will
only differ from the ValueConv value if the PrintConv option is enabled
(which it is by default).  The PrintConv values are the values returned
by B<ImageInfo> and B<GetInfo> in the tag/value hash.

Example:
    $val = $exifTool-E<gt>GetValue('ISO', 'ValueConv');

Inputs:
    0) ExifTool object reference
    1) Tag key
    2) [optional] Value type, 'PrintConv' (default) or 'ValueConv'

=over 4

=item Return Values:

The value of the specified tag.

=back

=head2 GetDescription

Get description for specified tag.  This function will always return a
defined value.  In the case where the description doesn't exist, the tag
name is returned.

Inputs:
    0) ExifTool object reference
    1) Tag key

=over 4

=item Return Values:

A description for the specified tag.

=back

=head2 GetGroup

Get group name for specified tag.

Example:
    $group = $exifTool-E<gt>GetGroup($tag, 0);

Inputs:
    0) ExifTool object reference
    1) Tag key
    2) [optional] Group family number

=over 4

=item Return Values:

Group name (or 'Other' if tag has no group).  If no group family is
specified, B<GetGroup> returns the name of the group in family 0 when called
in scalar context, or the names of groups for all families in list context.
See B<GetAllGroups> for a list of groups in each famly.

=back

=head2 GetGroups

Get list of group names for specified information.

Example:
    @groups = $exifTool-E<gt>GetGroups($info, 2);

Inputs:
    0) ExifTool object reference
    1) [optional] Info hash ref (default is all extracted info)
    2) [optional] Group family number (default 0)

=over 4

=item Return Values:

List of group names in alphabetical order. If information hash is not
specified, the group names are returned for all extracted information.

=back

=head2 BuildCompositeTags

Builds composite tags from required tags.  The composite tags are
convenience tags which are derived from the values of other tags.  This
routine is called automatically by B<ImageInfo> and B<ExtractInfo> if the
Composite option is set.

Inputs:
    0) ExifTool object reference

=over 4

=item Return Values:

(none)

=item Notes:

Tag values are calculated in alphabetical order unless a tag Require's or
Desire's another composite tag, in which case the calculation is deferred
until after the other tag is calculated. Composite tags may need to read
data from the image for their value to be determined, so for these
B<BuildCompositeTags> must be called while the image is available.  This is
only a problem if B<ImageInfo> is called with a filename (as opposed to a
file reference or scalar reference) since in this case the file is closed
before B<ImageInfo> returns.  However if you enable the Composite
option, B<BuildCompositeTags>  is called from within B<ImageInfo> before the
file is closed.  (Note: As of ExifTool version 3.10, only the PreviewImage
required access to the image data.)

=back

=head2 GetTagName [static]

Get name of tag from tag key.  This is a convenience function that
strips the embedded copy number, if it exists, from the tag key.

Note: "static" in the heading above indicates that the function does not
require an ExifTool object reference as the first argument.  All functions
documented below are also static.

Example:
    $tagName = Image::ExifTool::GetTagName($tag);

Inputs:
    0) Tag key

=over 4

=item Return Value:

Tag name.  This is the same as the tag key but has the copy number removed.

=back

=head2 GetShortcuts [static]

Get a list of shortcut tags.

Inputs:
    (none)

=over 4

=item Return Values:

List of shortcut tags (as defined in Image::ExifTool::Shortcuts).

=back

=head2 GetAllTags [static]

Get list of all available tag names.

Example:
    @tagList = Image::ExifTool::GetAllTags();

Inputs:
    (none)

=over 4

=item Return Values:

A list of all available tags in alphabetical order.

=back

=head2 GetAllGroups [static]

Get list of all group names in specified family.

Example:
    @groupList = Image::ExifTool::GetAllGroups($family);

Inputs:
    0) Group family number (0-2)

=over 4

=item Return Values:

A list of all groups in the specified family in alphabetical order.

=back

Three families of groups are currently defined: 0, 1 and 2.  Families 0 and
1 are based on the file structure and are equivalent except that the
individual manufacturer's groups of family 0 are combined into a single
MakerNotes group in family 1.  Family 2 classifies information in logical
groups based on what the information refers to.  Here is a complete list of
groups for each family:

=over 4

=item Family 0:

Canon, CanonCustom, CanonRaw, Casio, Composite, EXIF, ExifTool, File,
FujiFilm, GPS, GeoTiff, IPTC, MakerUnknown, Minolta, Nikon, Olympus, Pentax,
Photoshop, PrintIM, Sanyo, Sigma, Sony, XMP

=item Family 1:

Composite, EXIF, ExifTool, File, GPS, GeoTiff, IPTC, MakerNotes, Photoshop,
PrintIM, XMP

=item Family 2:

Author, Camera, ExifTool, Image, Location, Other, Printing, Time, Unknown

=back

=head1 AUTHOR

Copyright 2003-2004, Phil Harvey (phil@owl.phy.queensu.ca)

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 CREDITS

Thanks to the following people for their help:

=over 4

=item Malcolm Wotton for his help with the D30 Custom Functions

=item David Anson for his help sorting out binary file problems on Windows

=item Leon Booyens for his suggestions

=item Jeremy Brown for the 35efl tags

=item Dan Heller for his bug reports, detailed suggestions and guidance

=item Wayne Smith for his help figuring out the Pentax maker notes

=item Michael Rommel for his bug fixes and additions to the Canon maker notes

=item Joseph Heled for his help figuring out some of the Nikon D70 maker notes

=back

=head1 SEE ALSO

L<Image::Info|Image::Info>

=cut
