// $Id: UMLRadioButtonPanel.java,v 1.5 2003/06/29 23:50:03 linus Exp $
// Copyright (c) 1996-2002 The Regents of the University of California. All
// Rights Reserved. Permission to use, copy, modify, and distribute this
// software and its documentation without fee, and without a written
// agreement is hereby granted, provided that the above copyright notice
// and this paragraph appear in all copies.  This software program and
// documentation are copyrighted by The Regents of the University of
// California. The software program and documentation are supplied "AS
// IS", without any accompanying services from The Regents. The Regents
// does not warrant that the operation of the program will be
// uninterrupted or error-free. The end-user understands that the program
// was developed for research purposes and is advised not to rely
// exclusively on the program for any reason.  IN NO EVENT SHALL THE
// UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
// SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
// ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
// THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY
// WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
// PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
// CALIFORNIA HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
// UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

// $Id: UMLRadioButtonPanel.java,v 1.5 2003/06/29 23:50:03 linus Exp $
package org.argouml.uml.ui;

import java.awt.GridLayout;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.ButtonModel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.border.TitledBorder;

import org.argouml.model.uml.UmlModelEventPump;
import org.argouml.ui.targetmanager.TargetEvent;
import org.argouml.ui.targetmanager.TargetListener;
import org.tigris.gef.presentation.Fig;

import ru.novosoft.uml.MBase;
import ru.novosoft.uml.MElementEvent;
import ru.novosoft.uml.MElementListener;

/**
 * A panel that shows a group of radiobuttons. An action can be added to the
 * panel which will be executed when one of the radiobuttons is pressed. Via the 
 * name of the button (settext), the action can find out which button is
 * pressed.
 * @author jaap.branderhorst@xs4all.nl	
 * @since Jan 4, 2003
 */
public abstract class UMLRadioButtonPanel
    extends JPanel
    implements TargetListener, MElementListener {

    /**
     * The target object of which some attribute is shown via this panel.
     */
    private Object _target;

    /**
     * The name of the MEvent that is fired when the target object has changed
     * the attribute that is shown here.
     */
    private String _propertySetName;

    /**
     * The group of buttons
     */
    private ButtonGroup _buttonGroup = new ButtonGroup();

    /**
     * Constructs a new UMLRadioButtonPanel. 
     * @param isDoubleBuffered @see JPanel
     * @param title The title of the titledborder around the buttons. If the
     * title is null, there is no border shown.
     * @param labeltextsActioncommands A map of keys containing the texts for
     * the buttons and values containing the actioncommand that permits the
     * setAction to logically recognize the button.
     * @param propertySetName the name of the MEvent that is fired when the
     * property that is showns changes value.
     * @param setAction the action that should be registred with the buttons and
     * that's executed when one of the buttons is pressed.
     * @param horizontal when true the buttons should be layed out horizontaly.
     */
    public UMLRadioButtonPanel(
			       boolean isDoubleBuffered,
			       String title,
			       Map labeltextsActioncommands,
			       String propertySetName,
			       Action setAction,
			       boolean horizontal) {
        super(isDoubleBuffered);
        setLayout(horizontal ? new GridLayout() : new GridLayout(0, 1));
        setDoubleBuffered(true);
        if (title != null)
            setBorder(new TitledBorder(title));
        setButtons(labeltextsActioncommands, setAction);
        setPropertySetName(propertySetName);
    }

    /**
     * Constructs a new UMLRadioButtonPanel. 
     * @param title The title of the titledborder around the buttons.
     * @param labeltextsActioncommands A map of keys containing the texts for
     * the buttons and values containing the actioncommand that permits the
     * setAction to logically recognize the button.
     * @param propertySetName the name of the MEvent that is fired when the
     * property that is showns changes value.
     * @param setAction the action that should be registred with the buttons and
     * that's executed when one of the buttons is pressed
     * @param horizontal when true the buttons should be layed out horizontaly.
     */
    public UMLRadioButtonPanel(String title,
			       Map labeltextsActioncommands,
			       String propertySetName,
			       Action setAction,
			       boolean horizontal) {
        this(true, title, labeltextsActioncommands, 
	     propertySetName, setAction, horizontal);
    }

    /**
     * Initially constructs the buttons.
     * @param labeltextsActioncommands A map of keys containing the texts for
     * the buttons and values containing the actioncommand that permits the
     * setAction to logically recognize the button.
     * @param setAction the action that should be registred with the buttons and
     * that's executed when one of the buttons is pressed
     * @param horizontal when true the buttons should be layed out horizontaly.
     */
    private void setButtons(Map labeltextsActioncommands, Action setAction) {
        Enumeration en = _buttonGroup.getElements();
        while (en.hasMoreElements()) {
            AbstractButton button = (AbstractButton) en.nextElement();
            _buttonGroup.remove(button);
        }
        removeAll();
        Iterator it = labeltextsActioncommands.keySet().iterator();
        while (it.hasNext()) {
            String keyAndLabel = (String) it.next();
            JRadioButton button = new JRadioButton(keyAndLabel);
            button.addActionListener(setAction);
            button.setActionCommand(
				    (String) labeltextsActioncommands.get(keyAndLabel));
            _buttonGroup.add(button);
            add(button);
        }
    }

    /**
     * @see ru.novosoft.uml.MElementListener#listRoleItemSet(ru.novosoft.uml.MElementEvent)
     */
    public void listRoleItemSet(MElementEvent e) {
    }

    /**
     * @see ru.novosoft.uml.MElementListener#propertySet(ru.novosoft.uml.MElementEvent)
     */
    public void propertySet(MElementEvent e) {
        if (e.getName().equals(_propertySetName))
            buildModel();
    }

    /**
     * @see ru.novosoft.uml.MElementListener#recovered(ru.novosoft.uml.MElementEvent)
     */
    public void recovered(MElementEvent e) {
    }

    /**
     * @see ru.novosoft.uml.MElementListener#removed(ru.novosoft.uml.MElementEvent)
     */
    public void removed(MElementEvent e) {
    }

    /**
     * @see ru.novosoft.uml.MElementListener#roleAdded(ru.novosoft.uml.MElementEvent)
     */
    public void roleAdded(MElementEvent e) {
    }

    /**
     * @see ru.novosoft.uml.MElementListener#roleRemoved(ru.novosoft.uml.MElementEvent)
     */
    public void roleRemoved(MElementEvent e) {
    }

    /**
     * Returns the target.
     * @return Object
     */
    public Object getTarget() {
        return _target;
    }

    /**
     * Sets the target and removes/adds this as a listener to the target.
     * @param target The target to set
     */
    public void setTarget(Object target) {
        target = target instanceof Fig ? ((Fig) target).getOwner() : target;
        if (_target instanceof MBase) {
            UmlModelEventPump.getPump().removeModelEventListener(
								 this,
								 (MBase) _target,
								 _propertySetName);
        }
        _target = target;
        if (_target instanceof MBase) {
            // UmlModelEventPump.getPump().removeModelEventListener(this, (MBase)_target, _propertySetName);
            UmlModelEventPump.getPump().addModelEventListener(
							      this,
							      (MBase) _target,
							      _propertySetName);
        }
        if (_target != null)
            buildModel();
    }

    /**
     * Returns the propertySetName.
     * @return String
     */
    public String getPropertySetName() {
        return _propertySetName;
    }

    /**
     * Sets the propertySetName.
     * @param propertySetName The propertySetName to set
     */
    public void setPropertySetName(String propertySetName) {
        _propertySetName = propertySetName;
    }

    /**
     * Builds the model. That is: it selects the radiobutton showing the value
     * of the attribute shown. The name of this method is choosen to be
     * compliant with for example UMLModelElementListModel2
     */
    public abstract void buildModel();

    /**
     * Selects the radiobutton with the given actionCommand
     * @param actionCommand The actionCommand of the button that should be
     * selected.
     */
    public void setSelected(String actionCommand) {
        Enumeration en = _buttonGroup.getElements();
        ButtonModel model = null;
        while (en.hasMoreElements()) {
            model = ((AbstractButton) en.nextElement()).getModel();
            if (actionCommand.equals(model.getActionCommand())) {
                model.setSelected(true);
                break;
            }
        }
    }

    /**
     * @see org.argouml.ui.targetmanager.TargetListener#targetAdded(org.argouml.ui.targetmanager.TargetEvent)
     */
    public void targetAdded(TargetEvent e) {
    }

    /**
     * @see org.argouml.ui.targetmanager.TargetListener#targetRemoved(org.argouml.ui.targetmanager.TargetEvent)
     */
    public void targetRemoved(TargetEvent e) {
        setTarget(e.getNewTargets()[0]);
    }

    /**
     * @see org.argouml.ui.targetmanager.TargetListener#targetSet(org.argouml.ui.targetmanager.TargetEvent)
     */
    public void targetSet(TargetEvent e) {
        setTarget(e.getNewTargets()[0]);
    }

}
