//
// File:        CommandLineDriver.java
// Package:     gov.llnl.babel
// Revision:    $Revision: 4434 $
// Modified:    $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: main driver for the babel IDL compiler
//
// Copyright (c) 2000-2004, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel;

import gov.llnl.babel.backend.BuildGenerator;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeGenerationFactory;
import gov.llnl.babel.backend.CodeGenerator;
import gov.llnl.babel.backend.FileListener;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.parsers.sidl.Parser;
import gov.llnl.babel.parsers.sidl.SIDLException;
import gov.llnl.babel.repository.Repository;
import gov.llnl.babel.repository.RepositoryException;
import gov.llnl.babel.repository.RepositoryFactory;
import gov.llnl.babel.symbols.AssertionException;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolNotFoundException;
import gov.llnl.babel.symbols.SymbolRedefinitionException;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.RegexMatch;
import gov.llnl.babel.url.URLUtilities;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.StringTokenizer;


/**
 * This class is the main driver for running Babel from the command line.
 */
public class CommandLineDriver
{
  /**
   * The command line parser.
   */
  private static UserOptions d_option_parser = UserOptions.getInstance();

  /**
   * The user options and basic babel configuration information.
   */
  private static BabelConfiguration s_babel_config = 
                   BabelConfiguration.getInstance();

  /**
   * The file generation manager.
   */
  private FileManager d_file_manager = null;

  /**
   * The symbol table
   */
  private SymbolTable d_symbol_table = null;

  /**
   * The handle to the singleton instance of this class.
   */
  private static CommandLineDriver s_my_instance = null;


  /**
   * The protected singleton constructor instantiates the options parser,
   * <code>UserOptions</code>.
   *
   * @see UserOptions
   */
  protected CommandLineDriver() {
  } 

  /**
   * Return the singleton instance of this class.
   */
  private static CommandLineDriver getInstance() {
    if ( s_my_instance == null ) {
      s_my_instance = new CommandLineDriver();
    }
    return s_my_instance;
  } 

  /**
   * Extract the options from the command line arguments and perform
   * associated set up as appropriate.  
   *
   * @see BabelConfiguration
   */
  private void processCommandline( String args[] )
  { 
    /*
     * First extract the options from the arguments and determine if
     * we should proceed with the compilation.
     */
    int firstUnusedArg = d_option_parser.parseCommandlineOptions(s_babel_config,
                                                                 args);

    if ( !d_option_parser.canProceed() ) {
      System.exit(1);
    }

    /*
     * Now continue processing the remainder of the arguments and
     * create associated processing elements as needed.
     */
    resolveRepositoryPaths();

    setupFileManager();

    try { 
      resolveRemainingArgs( args, firstUnusedArg );
    } catch ( CollectionException ex ) { 
      System.err.println("Babel: FATAL: Unable to resolve remaining args.");
      ex.printStackTrace(System.err);
      System.exit(1);
    }

    /*
     * Stop now if only checking parsing.
     */
    if ( s_babel_config.parseCheckOnly() ) {
      System.exit(0);
    }
  } 

  /**
   * Create <code>Repository</code> instances based on the user-specified
   * repository path and add them to the <code>SymbolTable</code> singleton.
   * 
   * @see SymbolTable
   */
  private void resolveRepositoryPaths() 
  { 
    String repoPath               = s_babel_config.getRepositoryPath();
    RepositoryFactory the_factory = RepositoryFactory.getInstance();

    d_symbol_table                = SymbolTable.getInstance();

    for( StringTokenizer tokenizer = new StringTokenizer( repoPath, ";" );
         tokenizer.hasMoreTokens(); ) {
      String path = tokenizer.nextToken();
      if ( (path == null) || (path.equals("")) || (path.equals("null")) ) { 
        continue;
      } 
      try { 
        d_symbol_table.addSymbolResolver(the_factory.createRepository(path));
      } catch ( RepositoryException ex ) { 
        System.err.println("Babel: Warning: Unable to create repository using "
              + "path \"" + path + "\".  Will use default.");
        ex.printStackTrace( System.err );
      }
    }
  }

  /**
   * Initialize the file generation manager with configuration-specific
   * information as needed.  
   *
   * @see BabelConfiguration
   * @see FileManager
   */
  private void setupFileManager()
  { 
    d_file_manager   = FileManager.getInstance();
    if ( s_babel_config == null ) {
      System.err.println("Babel: Warning: Configuration was not populated prior"
            + " to setting up the file manager. Will set it now.");
      s_babel_config = BabelConfiguration.getInstance();
    }

    try { 
      d_file_manager.setFileGenerationRootDirectory(
                                           s_babel_config.getOutputDirectory());
    } catch ( CodeGenerationException ex ) { 
      System.err.println("Babel: Warning: Unable to set output directory. Will "
            + "use default.");
      ex.printStackTrace( System.err );
    }
    
    d_file_manager.setJavaStylePackageGeneration(
                                           s_babel_config.makePackageSubdirs());
    d_file_manager.setGlueSubdirGeneration(s_babel_config.makeGlueSubdirs());

    try { 
      d_file_manager.setVPathDirectory( s_babel_config.getVPathDirectory() );
    } catch ( CodeGenerationException ex ) { 
      System.err.println("Babel: Warning: Unable to set VPATH directory.");
      System.err.println( ex.getMessage() );
      ex.printStackTrace( System.err );
    }
  }

  private CollectionException addToCollection(CollectionException ce,
                                              String uri, Exception ex)
  {
    if (ce == null) {
      ce = new CollectionException(uri, ex);
    } else {
      ce.addException(uri, ex);
    }
    return ce;
  }

  /**
   * Process the remainder of the command line, assuming the contents
   * reflect one or more SIDL files.  
   */
  private void resolveRemainingArgs( String args[], int firstUnusedArg ) 
    throws CollectionException
  { 
    /*
     * First parse the remaining arguments.
     */
    CollectionException ce = null;
    for( int i=firstUnusedArg; i<args.length; i++ ) {
      try { 
        File f = new File( args[i] );
        if ( f.exists() && f.isFile() && f.canRead() ) { 
           /* 
            * Parse the URL for the readable file. 
            */
          String filename = f.toURL().toString();
          if (!filename.endsWith(".sidl") && !filename.endsWith(".idl")) {
            System.out.println("Babel: Warning: Will probably not be able to "
                  + "parse \"" + filename + "\" since only support parsing "
                  + "SIDL files.");
          }
          parseURL(filename);
          s_babel_config.setSIDLFileInput(true);
        } else if ( f.exists() && f.isDirectory() ) {
          /*
           * If it's a directory, assume user meant a symbol, so resolve it.
           */
          resolveSymbolByString( args[i] );
        } else if ( args[i].indexOf(':') != -1 ) { 
          /*
           * Try it as a URL if there's a colon.
           */
          parseURL( URLUtilities.expandURL(args[i]) );
	  s_babel_config.setSIDLFileInput(true);
        } else if ( resolveSymbolByString( args[i] ) == null ) { 
          /*
           * Otherwise, it must be tried as a symbol.
           */
          throw new CodeGenerationException("Babel: Error: \"" + args[i] 
                      + "\" fails to resolve as a symbol or file.");
        }
      } catch ( Exception ex ) { 
        ce = addToCollection(ce, args[i], ex);
      }
    } // end for loop
    
    /*
     * Now resolve all the symbols by first handling those by SIDL base class.
     */
    if ( d_symbol_table == null ) {
      System.err.println("Babel: Warning: Symbol table was not populated prior "
            + "to symbol resolution. Attempting to obtain instance now.");
      d_symbol_table = SymbolTable.getInstance();
    }
    
    try {
      d_symbol_table.resolveSymbol("sidl");
      d_symbol_table.resolveAllReferences();
    } catch ( Exception ex ) {
      ce = addToCollection(ce, null, ex);
    } 
    if (ce != null) {
      throw ce;
    }
  }

  /**
   * Open and parse contents of the specified URL.
   */
  private static void parseURL( String url ) 
    throws SIDLException, MalformedURLException, IOException
  { 
    URL         u = new URL( url );
    InputStream s = null;
    try {
      s = u.openStream();
      Parser parser = new Parser(new BufferedInputStream(s));
      System.out.println("Babel: Parsing URL \"" + url + "\".");
      parser.setSourceURL( url );
      String[] warnings = parser.beginParse();

      if ( warnings != null ) { 
        for ( int w=0; w<warnings.length; w++ ) { 
          System.err.println("Babel: Warning: " + warnings[w] );
        }
      }
    } finally {
      if (s != null) s.close();
    }
  }

  /**
   * Try to parse string as a symbol.  Return null if it fails.
   */
  private Symbol resolveSymbolByString( String symbol_string ) 
    throws NumberFormatException, SymbolRedefinitionException 
  {
    int    split        = symbol_string.indexOf("-v");
    String symbol_name  = ( split>0 && split<symbol_string.length() ) ?
                        symbol_string.substring(0,split) : symbol_string;
    String version_name = ( split>0 && (split+2)<symbol_string.length() ) ?
                        symbol_string.substring(split+2) : null;
    Symbol symbol       = null;

    if ( (version_name == null) || (version_name == "") ) { 
      symbol = d_symbol_table.resolveSymbol( symbol_name );
    } else { 
      SymbolID id = 
        new SymbolID( symbol_name, 
                      new gov.llnl.babel.symbols.Version( version_name ));
      symbol = d_symbol_table.resolveSymbol( id );
    }

    if ( symbol != null ) { 
      System.out.println("Babel: Resolved symbol \"" + symbol_string + "\".");
      symbol.setUserSpecified(true);

      if (symbol.isPackage()) {
        /*
         * Add patterns for all symbols specifed on the command line, so that
         * code will be generated for relevant subpackages and the package
         * contents (if the symbol is a package).
         */ 
        s_babel_config.addIncluded(symbol_string + ".");
      } else {
        System.out.println("Babel: Warning: Will not be able to generate a file"
              + " for \"" + symbol_string + "\" since it is not a package.");
      }
    }

    return symbol;
  }

  /**
   * Generate the appropriate source code based on user configuration 
   * parameters.
   */
  private void generateSourceCode()
  {
    if ( d_symbol_table == null ) {
      System.out.println("Babel: Warning: Symbol table was not populated prior "
            + "to generating source code. Obtaining instance now.");
      d_symbol_table = SymbolTable.getInstance();
    }
    Set symbols = d_symbol_table.getSymbolNames();

    if ( s_babel_config == null ) {
      System.out.println("Babel: Warning: Configuration was not populated prior"
            + " to generating the source code. Obtaining instance now.");
      s_babel_config = BabelConfiguration.getInstance();
    }

    if (s_babel_config.excludeExternal()) {
      symbols = excludeExternalSymbols(symbols);
    }
    symbols = excludeSymbols(symbols);

    /*
     * Now that the symbols are loaded, it's time to make sure they are valid
     * _before_ any code can be generated.
     */
    try {
      d_symbol_table.resolveAllParents();
      d_symbol_table.validateAssertions();
    } catch (AssertionException aex) {
      System.err.println("Babel: FATAL: Unable to validate the assertions of "
            + "the symbols in the symbol table.");
      aex.printStackTrace( System.err );
      System.exit(1);
    } catch (SymbolNotFoundException snfe) {
      System.err.println("Babel: Warning: Cannot find all parent packages.");
      System.err.println(snfe.getMessage());
    }

    /*
     * NOW it should be safe to generate code.
     */
    String targetLanguage = s_babel_config.getTargetLanguage();
    if ( s_babel_config.generateText() ) { 
      if ( targetLanguage.equals("xml") ) {
        generateXMLCode(symbols);
      } else {
        try {
          d_symbol_table.resolveAllParents();
          generateCode(symbols, "text", targetLanguage);
        }
        catch (SymbolNotFoundException snfe) {
          System.err.println("Babel: Error: SIDL backend cannot find all parent"
                + " packages.");
          System.err.println(snfe.getMessage());
          System.exit(1);
        }
      }
    } else {
      BuildGenerator mg = getBuildGen(targetLanguage);
      if (mg instanceof FileListener) {
        d_file_manager.addListener((FileListener)mg);
      }
      if ( s_babel_config.generateClient() ) {
        generateClientSource(symbols);
        generateMakefiles(mg);
      } else if ( s_babel_config.generateServer() ) {
        generateServerSource(symbols);
        generateMakefiles(mg);
      }
    }
  }

  /**
   * Generate the XML repository code.
   */
  private void generateXMLCode( Set symbols )
  {
    Repository repository;
    if ( s_babel_config == null ) {
      System.err.println("Babel: Warning: Configuration was not populated prior"
            + " to generating the XML code. Otaining instance now.");
      s_babel_config = BabelConfiguration.getInstance();
    }

    try { 
      repository = RepositoryFactory.getInstance().createRepository(
                                s_babel_config.getOutputDirectory());
      if ( repository != null ) { 
        repository.writeSymbols(symbols);
      }
    } catch ( RepositoryException ex ) { 
      System.err.println("Babel: FATAL: Unable to create output repository.");
      System.err.println( ex.getMessage() );
      ex.printStackTrace( System.err );
      repository = null;
      System.exit(1);
    }
  }

  private void generateCode(Set symbols, String type, String language)
  {
    //TLD-To check contents of symbols...printSymbolNames(symbols);
    CodeGenerator gen = getCodeGen(language, type);
    if (gen != null) {
      if (gen.getUserSymbolsOnly()) {
        symbols = filterSymbols(symbols);
      }
      try {
        gen.generateCode(symbols);
      } catch (CodeGenerationException ex) {
        System.err.println("Babel: FATAL: Unable to generate " + type 
              + " code.");
        System.err.println( ex.getMessage() );
        ex.printStackTrace( System.err );
        System.exit(1);
      }
    } else {
      System.err.println("Babel: Warning: Unable to generate the " + type 
            + " sources in the " + language + " language.");
    }
  }

  /**
   * Generate the server source code.
   */
  private void generateServerSource( Set symbols )
  {
    /*
     * First generate the client code.
     */
    generateClientSource(symbols);

    /*
     * Now generate the server-specific code.
     */
    String targetLanguage = s_babel_config.getTargetLanguage();
    generateCode(symbols, "skel", "ior");
    generateCode(symbols, "skel", targetLanguage);
  }

  /**
   * Generate the client source code.
   */
  private void generateClientSource( Set symbols )
  {
    String targetLanguage = s_babel_config.getTargetLanguage();
    generateCode(symbols, "stub", "ior");
    generateCode(symbols, "stub", targetLanguage);
  }

  private BuildGenerator getBuildGen(String myLang)
  {
    BuildGenerator bg = 
      CodeGenerationFactory.getInstance().getBuildGenerator(myLang);
    if (bg == null){
      System.err.println("Babel: Warning: No build generator returned for the "
            + myLang + " language.");
    }
    return bg;
  }

  /**
   * Instantiate the appropriate code generator associated with the desired
   * language and mode.
   *
   * @see CodeGenerator
   */
  private CodeGenerator getCodeGen( String myLang, String mode )
  {
    CodeGenerator cg;

    cg = CodeGenerationFactory.getInstance().getCodeGenerator( myLang, mode );
    if ( cg == null ) { 
      System.err.println("Babel: Warning: No code generator returned for the " 
            + myLang + "language and " + mode + " mode.");
    }
    return cg;
  }

  public static boolean notExcluded(SymbolID id)
  {
    Iterator i = s_babel_config.getExcludedList().iterator();
    while (i.hasNext()) {
      if (((RegexMatch)i.next()).match(id)) return false;
    }
    return true;
  }

  private static Set excludeSymbols(Collection symbols)
  {
    Set result = new HashSet(symbols.size());
    Iterator i = symbols.iterator();
    while (i.hasNext()) {
      SymbolID id = (SymbolID)i.next();
      if (notExcluded(id)) {
        result.add(id);
      }
    }
    return result;
  }

  public static boolean isIncluded(SymbolID id) 
  {
    final String fullname = id.getFullName();
    Iterator incl = s_babel_config.getIncludedList().iterator();
    while(incl.hasNext()) {
      if (fullname.startsWith((String)incl.next())) return true;
    }
    return false;
  }

  private Set excludeExternalSymbols(Collection symbols) {
    Set result = new HashSet(symbols.size());
    if (s_babel_config.excludeExternal()) {
      Iterator i = symbols.iterator();
      while (i.hasNext()) {
        SymbolID id = (SymbolID)i.next();
        if (d_symbol_table.lookupSymbol(id).getUserSpecified()) {
          result.add(id);
        } else {
          /* 
           * Check if current symbol is contained within the scope of 
           * a user-specified (on the command line) symbol.
           */
          if (isIncluded(id)) {
            d_symbol_table.lookupSymbol(id).setUserSpecified(true);
            result.add(id);
          }
        }
      }
    }
    else {
      result.addAll(symbols);
    }
    return result;
  }

  private static boolean filterSymbol(SymbolID id) {
    return !BabelConfiguration.isSIDLBaseClass(id);
  }

  /**
   * Depending on the configuration settings, return either all symbols or
   * just the user symbols, those symbols outside the SIDL namespace.
   */
  private static Set filterSymbols(Set symbols) 
  {
    Set result;
    if (s_babel_config.generateStdlib()) {
      result = symbols;
    } else {
      result = new HashSet(symbols.size());
      Iterator i = symbols.iterator();
      while (i.hasNext()) {
        SymbolID id = (SymbolID)i.next();
        if (filterSymbol(id)) result.add(id);
      }
    }
    return result;
  }

  /**
   * Generate the appropriate makefile fragments.
   */
  private void generateMakefiles(BuildGenerator mg) {
    /*
     * Now generate the fragments.
     */
    try { 
      mg.createAll();
    } catch ( IOException ex ) { 
      System.err.println("Babel: Warning: Unable to generate makefiles.");
      System.err.println(ex.getMessage());
      ex.printStackTrace( System.err );
      
    }
  }

  /**
   * Print all symbols to output.  This is obviously intended only to facilitate
   * debugging.
   */
  public void printSymbolNames(Set symbols) {
    System.out.println("Symbol set:");
    Iterator i = symbols.iterator();
    while (i.hasNext()) {
      SymbolID id = (SymbolID)i.next();
      System.out.println("  " + id.getFullName());
    }
  }

  /****************************************************************
   * Main babel entry point.
   */
  public static void main( String args[] ) {
    try {
      /*
       * Set up the configuration parameters based on command line arguments.
       */
      CommandLineDriver myDriver = getInstance();
      myDriver.processCommandline(args);
      
      /*
       * Generate the source code.
       */
      myDriver.generateSourceCode();
    } catch (Throwable t) {
      System.err.println("Babel.CommandLineDriver: Unable to generate code.");
      String msg = t.getMessage();
      if (msg != null) {
        System.err.println(t.getMessage());
      }
      t.printStackTrace();
      Runtime.getRuntime().exit(1);
    }
    finally {
      /* workaround for JNI bug in Linux 1.3.1 JVM */
      Runtime.getRuntime().exit(0);
    }
  }
} 
