//
// File:        CodeGenerationFactory.java
// Package:
// Revision:     @(#) $Id: CodeGenerationFactory.java 4434 2005-03-17 17:05:29Z epperly $
// Description:
//
// Copyright (c) 2000-2002, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend;

import java.util.HashMap;
import java.util.Iterator;
import java.io.PrintStream;
import gov.llnl.babel.backend.CodeGenerator;


/**
 * This class stores and serves up instances CodeGenerator based
 * on the string name of the code that the user wishes to generate.
 * Classes derived from CodeGenerator will register themselves
 * with this class (usually as part of a static initialization block).
 * If a match based on the string name is not available, this class
 * will try to dynamically load an instance based on some simple
 * naming rules.
 *
 * @see CodeGenerator
 */
public class CodeGenerationFactory {
  private static String[] s_classList = {
  	"gov.llnl.babel.backend.ior.GenerateIORClient",
  	"gov.llnl.babel.backend.ior.GenerateIORServer",
    "gov.llnl.babel.backend.c.GenerateClientC",
    "gov.llnl.babel.backend.c.GenerateServerC",
    "gov.llnl.babel.backend.python.GenPythonClient",
    "gov.llnl.babel.backend.python.GenPythonServer",
    "gov.llnl.babel.backend.cxx.GenerateCxxClient",
    "gov.llnl.babel.backend.cxx.GenerateCxxServer",
    "gov.llnl.babel.backend.ucxx.GenerateCxxClient",
    "gov.llnl.babel.backend.ucxx.GenerateCxxServer",
    "gov.llnl.babel.backend.fortran.GenerateClient",
    "gov.llnl.babel.backend.fortran.GenerateServer",
    "gov.llnl.babel.backend.jdk.GenerateClientJava",
    "gov.llnl.babel.backend.jdk.GenerateServerJava",
    "gov.llnl.babel.backend.MakefileGenerator",
    "gov.llnl.babel.backend.python.DualGenerator",
    "gov.llnl.babel.backend.sidl.GenerateSidl"
  };

  /**
   * static instance of the class
   */
  private static CodeGenerationFactory s_instance = null;

  /**
   * map from string name to CodeGenerator
   */
  private HashMap m_hashmap;

  /**
   * map from string name to BuildGenerator
   */
  private HashMap m_buildmap;

  /**
   * Create a new instance of <code>CodeGenerationFactory</code> and
   * initialize with reasonable defaults.
   * This is kept from the public interface because this class
   * is a singleton.  As such, the singleton instance
   * can only be created and accessed through the
   * <code>getInstance</code> method.
   */
  private CodeGenerationFactory() {
    m_hashmap = new HashMap();
    m_buildmap = new HashMap();
  }

  /**
   * Return the singleton instance of the class.
   * If the <code>CodeGenerationFactory</code> has not yet been created,
   * then it will be created by this call.
   */
  public static CodeGenerationFactory getInstance() {
    if (s_instance == null) {
      s_instance = new CodeGenerationFactory();
      for(int i = 0; i < s_classList.length; ++i ){
        try {
          Class cls = Class.forName(s_classList[i]);
          Object obj = cls.newInstance();
          boolean matches = false;
          if (obj instanceof CodeGenerator) {
            matches = true;
            s_instance.registerCodeGenerator((CodeGenerator)obj);
          }
          if (obj instanceof BuildGenerator) {
            matches = true;
            s_instance.registerBuildGenerator((BuildGenerator)obj);
          }
          if (!matches) {
            System.err.println("Babel: Warning: backend extension '" +
                               s_classList[i] + 
                               "' is neither a code or build generator. Ignoring it.");
          }
        }
        catch (ClassNotFoundException cnfe) {
          System.err.println("Babel: Error: Unable to find backend '" +
                             s_classList[i] + "'. Check your Java classpath.");
          cnfe.printStackTrace(System.err);
        }
        catch (Exception e) {
          System.err.println("Babel: Error: Unable to instantiate backend '" +
                             s_classList[i] + "'.");
          e.printStackTrace(System.err);
        }
        catch (LinkageError e) {
          System.err.println("Babel: Error: Unable to load backend '" +
                             s_classList[i] + "'. Linkage problem.");
          e.printStackTrace(System.err);
        }
      }
    }
    return s_instance;
  }

  /**
   * Register codeGenerators with the factory.
   *
   * @param gen instance of a CodeGenerator;
   */
  private void registerCodeGenerator( CodeGenerator gen)
  {
    Iterator i = gen.getLanguages().iterator();
    final String mode = gen.getType();
    while (i.hasNext()) {
      String lang = (String)i.next();
      String key = hashLanguageAndType(lang, mode);
      m_hashmap.put(key, gen);
    }
  }

  private void registerBuildGenerator( BuildGenerator gen)
  {
    Iterator i = gen.getLanguages().iterator();
    while (i.hasNext()) {
      m_buildmap.put((String)i.next(), gen);
    }
  }

  /**
   * Get the build generator appropriate for this language.
   */
  public BuildGenerator getBuildGenerator(String language)
  {
    return (BuildGenerator)m_buildmap.get(language);
  }

  /**
   * Get a registered codeGenerator from the factory.
   * @param language String name of the language
   * @param mode Usually one of "stub", "skel", "ior", or "text"
   * @return Live CodeGenerator or null if no match found.
   */
  public CodeGenerator getCodeGenerator( String language, String mode )
  {
    String key = hashLanguageAndType( language, mode );
    return (CodeGenerator)m_hashmap.get(key);
  }

  /**
   * Hash the language and mode into a single (reproducable) string.
   */
  private String hashLanguageAndType( String language, String mode ) {
    String hash = language + "-" + mode;
    return hash.toLowerCase();
  }

  public void printRegisteredGenerators( PrintStream out ) {
    out.println("Registered CodeGenerators = " + m_hashmap.size() );
    for( Iterator it = m_hashmap.keySet().iterator(); it.hasNext(); ) {
      String key = (String)it.next();
      CodeGenerator gen = (CodeGenerator)m_hashmap.get(key);
      out.println("   key = \"" + key);
    }
  }
}
