//
// File:        StubHeader.java
// Package:     gov.llnl.babel.backend.c
// Revision:    @(#) $Id: StubHeader.java 4449 2005-03-21 18:00:39Z leek2 $
// Description: write C client (i.e., stub) header to a pretty writer stream
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.ior.IORSource;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Package;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.BabelConfiguration;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Class <code>StubHeader</code> writes a C client header to a language
 * writer output stream.  The constructor takes a language writer stream
 * and method <code>generateCode</code> writes the C client header code
 * for the specified symbol to the output stream.  The language writer
 * output stream is not closed by this object.
 */
public class StubHeader {
  private final static String SIDL_EXCEPTION =
    BabelConfiguration.getBaseExceptionClass();
  private final static String SIDL_EXCEPTION_INTERFACE =
    BabelConfiguration.getBaseExceptionInterface();

  public final static String s_epv       = IOR.getEPVVar(IOR.PUBLIC_EPV);
  public final static String s_self      = Utilities.s_self;
  public final static String s_sepv_func = "_getSEPV()";

  /**
   * Indices associated with the special, stub-only built-in methods.
   */
  public final static int DUMP_STATS       = 0;
  public final static int SET_CHECKING     = 1;
  public final static int SET_INTERCEPTORS = 2;

  private LanguageWriterForC d_writer;

  private boolean d_comment_all;

  /**
   * This is a convenience utility function that writes the C client
   * header information into the provided language writer output stream.
   * The output stream is not closed on exit.  A code generation
   * exception is thrown if an error is detected.
   *
   * @param symbol the symbol for which a C client header will
   *               be written.
   *
   * @param writer the output writer to which the header will
   *               be written. This will not be closed.
   *
   * @exception gov.llnl.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateCode(Symbol symbol, LanguageWriterForC writer)
    throws CodeGenerationException
  {
    StubHeader header = new StubHeader(writer);
    header.generateCode(symbol);
  }

  /**
   * Create a <code>StubHeader</code> object that will write symbol
   * information to the provided output language writer stream.
   *
   * @param writer the output writer to which the header will
   *               be written. This will not be closed.
   */
  public StubHeader(LanguageWriterForC writer) {
    d_writer = writer;
    d_comment_all = !BabelConfiguration.getInstance().getCommentLocalOnly();
  }

  /**
   * Write C client header information for the provided symbol to the
   * language writer output stream provided in the constructor.  This
   * method does not close the writer output stream and may be called
   * for more than one symbol (although the written header may not be
   * valid input for the C compiler).  A code generation exception is
   * written if an error is detected.
   *
   * @param symbol the <code>Symbol</code> whose header will be
   *               written.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble
   *    or violations of the data type invariants.
   */
  public void generateCode(Symbol symbol) throws CodeGenerationException {
    if (symbol != null) {
      if ( symbol.getSymbolType() == Symbol.PACKAGE ) { 
	  d_writer.skipIncludeGuard(); 
      }
      generatePrologue(symbol);

      switch (symbol.getSymbolType()) {
        case Symbol.CLASS:
          generateExtendable((Extendable) symbol);
          break;
        case Symbol.ENUM:
          generateEnumeration((Enumeration) symbol);
          break;
        case Symbol.INTERFACE:
          generateExtendable((Extendable) symbol);
          break;
        case Symbol.PACKAGE:
          generatePackage((Package) symbol);
          break;
        default:
          throw new CodeGenerationException("Unsupported symbol type.");
      }
    } else {
      throw new CodeGenerationException("Unexpected null Symbol.");
    }
  }

  /**
   * Output a banner and the main header guard at the top of the file.
   *
   * @param sym the <code>Symbol</code> whose header will be written.
   */
  private void generatePrologue(Symbol sym) {
    final SymbolID id     = sym.getSymbolID();
    String         header = C.getHeaderFile(id);

    d_writer.writeBanner(sym, header, false, 
      CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
    d_writer.openHeaderGuard(header);
  }

  /**
   * Output the closing block methods as needed at the end of the file.
   *
   * @param closeExtern the boolean used to indicate if the C++ extern block
   *     needs to be closed.
   */
  private void generateEpilogue(boolean closeExtern) {
    /*
     * Conclude the header file by closing the C++ extern block, if
     * indicated, and the header include guard.
     */
    if (closeExtern) {
      d_writer.closeCxxExtern();
    }
    d_writer.closeHeaderGuard();
  }

  /**
   * Generate the C header for a SIDL enumerated type.  This method simply
   * includes the IOR enumerated type header file and typedefs that type
   * to a C type.
   *
   * @param enm the <code>Enumeration</code> whose header is being written.
   */
  private void generateEnumeration(Enumeration enm) {
    SymbolID id = enm.getSymbolID();
    ArrayMethods ar = new ArrayMethods(id, true);

    d_writer.generateInclude(IOR.getHeaderFile(id), true);
    d_writer.println();

    d_writer.openCxxExtern();
    ar.generateHeader(d_writer);
    generateEpilogue(true);
  }

  /**
   * Generate a C client header for a SIDL class or interface description.
   * The header file consists of the typedef the defines the symbol type.
   * Note that the typedef comes before any external includes to solve
   * the problem with forward references.  After the typedef comes the
   * external includes, followed by special methods such as cast and new,
   * followed by the regular methods.  The header concludes with close
   * statements for the header guards.
   *
   * @param ext the <code>Extendable</code> whose header is being written.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble
   *    or violations of the data type invariants.
   */
  private void generateExtendable(Extendable ext) throws CodeGenerationException
  {
    SymbolID id = ext.getSymbolID();

    /*
     * Output a documentation comment, forward declare the object, and
     * define the symbol typedef.
     */
    d_writer.writeComment(ext, true);
    d_writer.println(IOR.getObjectName(id) + ";");
    d_writer.println(IOR.getArrayName(id) + ";");
    d_writer.print("typedef " + IOR.getObjectName(id) + "* ");
    d_writer.println(C.getObjectName(id) + ";");
    d_writer.println();

    /*
     * Generate the includes and open the C++ extern block.
     */
    generateIncludes(ext);
    d_writer.openCxxExtern();

    /*
     * Output standard function signatures for all methods.
     */
    generateMethodPrototypes(ext);

    /*
     * Output array method signatures
     */
    ArrayMethods ar = new ArrayMethods(id, false);
    ar.generateHeader(d_writer);

    /*
     * Conclude the header file by closing block(s) as needed.
     */
    generateEpilogue(true);
  }

  /**
   * Generate the C client header for a SIDL package description.  The
   * package header file consists of the standard header information along
   * with include statements for all package symbols.
   *
   * @param p the <code>Package</code> whose header is being written.
   */
  private void generatePackage(Package p) {
    SymbolID id   = p.getSymbolID();

    /*
     * Write out the C include files for each of the symbols within
     * the package.
     */
    List entries = Utilities.sort(p.getSymbols().keySet());
    for (Iterator i = entries.iterator(); i.hasNext(); ) {
      String include = C.getHeaderFile((SymbolID) i.next());
      d_writer.generateInclude(include, true);
    }
    d_writer.println();

    generateEpilogue(false);
  }

  /**
   * Generate the list of include files required to satisfy data
   * dependencies within this header file and output the appropriate
   * include lines.  If any of the symbols do not exist in the symbol
   * table, then throw a code generation exception.
   */
  private void generateIncludes(Extendable ext) throws CodeGenerationException {
    /*
     * Create the set of include symbol identifiers.
     */
    Set includes = new HashSet();

    for (Iterator i = ext.getMethods(true).iterator(); i.hasNext(); ) {
      Method method = (Method) i.next();
      includes.addAll(method.getSymbolReferences());

      if (!method.getThrows().isEmpty()) {
        Symbol symbol = Utilities.lookupSymbol(SIDL_EXCEPTION);
        includes.add(symbol.getSymbolID());
        symbol = Utilities.lookupSymbol(SIDL_EXCEPTION_INTERFACE);
        includes.add(symbol.getSymbolID());
      }
    }

    /*
     * We should always include the base header file.
     */
    d_writer.writeComment("Includes for all header dependencies.", false);
    d_writer.generateInclude("sidl_header.h", true);

    /*
     * Remove this symbol from the dependency set and iterate over symbols if
     * there are any remaining symbols in the set.
     */
    includes.remove(ext.getSymbolID());
    if (!includes.isEmpty()) {
      List entries = Utilities.sort(includes);
      for (Iterator i = entries.iterator(); i.hasNext(); ) {
        String header = C.getHeaderFile((SymbolID) i.next());
        d_writer.generateInclude(header, true);
      }
      d_writer.println();
    }
    d_writer.generateInclude(C.getHeaderFile(
              Utilities.lookupSymbol("sidl.io.Serializer").getSymbolID()), 
              true);
    d_writer.generateInclude(C.getHeaderFile(
              Utilities.lookupSymbol("sidl.io.Deserializer").getSymbolID()), 
              true);
  }

  /**
   * Generate prototypes for the specified built-in method.
   */
  private void generateBuiltinPrototypes(SymbolID id, int stubType, 
                                         boolean doStatic) 
    throws CodeGenerationException 
  {
    d_writer.writeComment(getBuiltinComment(stubType, doStatic), true);
    generateBuiltinSignature(d_writer, stubType, id, doStatic, ";");
    d_writer.println();
  }

  /**
   * Generate prototypes for the specified assertion method(s).
   */
  private void generateAssertionPrototypes(SymbolID id, boolean doStatic) 
    throws CodeGenerationException 
  {
    generateBuiltinPrototypes(id, SET_CHECKING, doStatic);
    generateBuiltinPrototypes(id, DUMP_STATS, doStatic);
  }
 
   /**
    * Generate method prototypes for the methods in the interface or class.
    */
  private void generateMethodPrototypes(Extendable ext)
    throws CodeGenerationException 
  {
    /*
     * Generate the name for this entry point vector as well as a pointer
     * to "self" for this structure.  For classes, self will be the object
     * structure whereas for interfaces it is void*.
     */
    SymbolID id   = ext.getSymbolID();

    /*
     * Output the normal and remote constructors if this extendable is a class.
     */
    if (!ext.isAbstract()) {
      d_writer.writeComment("Constructor function for the class.", true);
      d_writer.println(C.getSymbolObjectPtr(id));
      d_writer.println(C.getFullMethodName(id, "_create") + "(void);");
      d_writer.println();

      d_writer.writeComment("RMI constructor function for the class.", true);
      d_writer.println(C.getSymbolName(id));

      d_writer.println(C.getFullMethodName(id, "_createRemote") 
        + "(const char *, sidl_BaseInterface *_ex);");

      d_writer.println();

    }

    /*
     * Output the assertion and/or interceptor method(s) if appropriate.
     */
    boolean hasStatic = ext.hasStaticMethod(true);
    if (!ext.isAbstract() && !ext.isInterface()) {
      if (IOR.supportAssertions(ext)) {
        if (hasStatic) {
          generateAssertionPrototypes(id, true);
        }
        generateAssertionPrototypes(id, false);
      }
      if (IOR.supportInterceptors(ext)) {
        if (hasStatic) {
          generateBuiltinPrototypes(id, SET_INTERCEPTORS, true);
        }
        generateBuiltinPrototypes(id, SET_INTERCEPTORS, false);
      }
    }

    /*
     * Output each of the method prototypes from the interface or class.
     */
    boolean isIfc = ext.isInterface();
    Collection methods = ext.getMethods(true);
    for (Iterator m = methods.iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      generateMethodSignature(id, isIfc, method, 
                              d_comment_all || ext.isLocal(method));
      d_writer.println();
    }

    /*
     * Output the cast methods for the class or interface.
     */
    d_writer.writeComment("Cast method for interface and class type "
      + "conversions.", true);
    d_writer.println(C.getSymbolObjectPtr(id));
    d_writer.println(C.getFullMethodName(id, "_cast") + "(");
    d_writer.tab();
    d_writer.println("void* obj);");
    d_writer.backTab();
    d_writer.println();

    d_writer.writeComment("String cast method for interface and class type "
      + "conversions.", true);
    d_writer.println("void*");
    d_writer.println(C.getFullMethodName(id, "_cast2") + "(");
    d_writer.tab();
    d_writer.println("void* obj,");
    d_writer.println("const char* type);");
    d_writer.backTab();
    d_writer.println();
    
    /*
     * Output the exec methods for the class or interface.
     */
    Method m_exec = C.getExecMethod();
    generateMethodSignature(id, isIfc, m_exec, true);

    m_exec = C.getSExecMethod();
    generateMethodSignature(id, isIfc, m_exec, true);
  }

  /**
   * Generate the method signature.
   *
   * @param id the <code>SymbolID</code> of the <code>Extendable</code> whose
   *   header is being written.
   *
   * @param isInterface TRUE if the <code>Extendable</code> is an interface,
   *                    otherwise, use FALSE.
   * 
   * @param method the <code>Method</code> whose signature is being output.
   * 
   * @param docComment iff <code>true</code> this will print a document
   *                   comment for the method if one is available. If
   *                   <code>false</code>, not document comment will
   *                   be printed.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateMethodSignature(SymbolID id, boolean isInterface,
                                       Method method, boolean docComment)
    throws CodeGenerationException
  {
    if (docComment) {
      d_writer.writeComment(method, true);
    }
    d_writer.println(getReturnString(method.getReturnType()));
    d_writer.print(C.getFullMethodName(id, method));
    d_writer.println("(");
    d_writer.tab();
    C.generateArgumentList(d_writer, s_self, isInterface, id, method, true, 
                           true, false, true, false, true, true);
    d_writer.println(");");
    d_writer.backTab();
  }

   /**
    * Generate a return string for the specified SIDL type.  Most
    * of the SIDL return strings are listed in the static structures defined
    * at the start of this class.  Symbol types and array types require
    * special processing.
    */
   private static String getReturnString(Type type)
     throws CodeGenerationException
   {
     return IOR.getReturnString(type, false, true);
   }


  /**
   * Return the comment description associated with the specified
   * built-in stub.
   */
  public static String getBuiltinComment(int type, boolean doStatic) {
    String cmt  = null;
    String desc = doStatic ? "static " : "";
    switch (type) {
      case SET_CHECKING:
        cmt = "Method to set the " + desc + "assertion checking level.";
        break;
      case DUMP_STATS:
        cmt = "Method to dump " + desc +  "assertion checking statistics.";
        break;
      case SET_INTERCEPTORS:
        cmt = "Method to enable/disable " + desc +  "interceptor execution.";
        break;
    }
    return cmt;
  }

  /**
   * Generate the specified stub-only built-in method signature.
   */
  public static void generateBuiltinSignature(LanguageWriterForC lw, int type, 
                                              SymbolID id, boolean doStatic,
                                              String terminator)
  {
    String suffix = doStatic ? IOR.s_static_suffix : "";
    String basename = null;
    switch (type) {
      case SET_CHECKING:
        basename = "_set_checking";
        break;
      case DUMP_STATS:
        basename = "_dump_stats";
        break;
      case SET_INTERCEPTORS:
        basename = "_set_interceptors";
        break;
    }
    lw.println("void");
    lw.println(C.getFullMethodName(id, basename + suffix) + "(");
    lw.tab();
    if (!doStatic) {
      lw.println(C.getFullSelfDecl(id) + ",");
    }
    switch (type) {
      case SET_CHECKING:
        lw.println("int32_t level,");
        lw.println("double  rate,");
        lw.println("int32_t resetCounters)" + terminator);
        break;
      case DUMP_STATS:
        lw.println("const char* filename)"+ terminator);
        break;
      case SET_INTERCEPTORS:
        lw.println("int32_t on)" + terminator);
        break;
    }
    lw.backTab();
  }

  /**
   * Return the comma-separated list of arguments associated with the 
   * specified built-in method.  The entries must match the names and
   * ordering found (above) in generateStubBuiltinSignature().
   */
  public static String getBuiltinArgList(int type) {
    String args = null;
    switch (type) {
      case SET_CHECKING:
        args = "level, rate, resetCounters";
        break;
      case DUMP_STATS:
        args = "filename";
        break;
      case SET_INTERCEPTORS:
        args = "on";
        break;
    }
    return args;
  }

  /**
   * Return a string that dereferences the specified IOR function pointer.
   */
  public static String getDerefFunctionPtr(String baseName, boolean doStatic) {
    String base = doStatic ? s_sepv_func : "*" + s_self + "->" + s_epv;
    return "(" + base + "->" + IOR.getVectorEntry(baseName) + ")";
  }
}
