//
// File:        GenerateClientJava.java
// Package:     gov.llnl.babel.backend.jdk
// Revision:    @(#) $Id: GenerateClientJava.java 4434 2005-03-17 17:05:29Z epperly $
// Description: generate Java client code based on a set of symbol identifiers
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.jdk;

import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeGenerator;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.jdk.Java;
import gov.llnl.babel.backend.jdk.ClientJava;
import gov.llnl.babel.backend.jdk.ClientJNI;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.writers.LanguageWriterForJava;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import java.io.PrintWriter;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

/**
 * Class <code>GenerateClientJava</code> is the main entry point for the Java
 * client-side code generation routines.  This class generates the Java files
 * and the C JNI files.  It is assumed that all symbols necessary to generate
 * Java code are available in the symbol table.
 */
public class GenerateClientJava implements CodeGenerator {

  /**
   * The constructor does nothing interesting.  The entry point for
   * the <code>GenerateClientJava</code> class is <code>generateCode</code>.
   */
  public GenerateClientJava() {
  }

  /**
   * Generate Java client-side code for each symbol identifier in the set
   * argument.  This routine assumes that all necessary symbol references
   * are available in the symbol table.
   */
  public void generateCode(Set symbols) throws CodeGenerationException {
    for (Iterator s = symbols.iterator(); s.hasNext(); ) {
      SymbolID id = (SymbolID) s.next();
      Symbol symbol = SymbolTable.getInstance().lookupSymbol(id);
      if (symbol != null) {
        generateJava(symbol);
        generateJNI(symbol);
      }
    }
  }

  /**
   * Generate Java code for the specified symbol.  Since Java code must follow
   * Java-style directory conventions, we must force the file manager to output
   * the Java code in a directory structure.  Reset the flags so that the other
   * C JNI code follows the default conventions.  We do not need to generate any
   * code for packages.
   */
  private void generateJava(Symbol symbol) throws CodeGenerationException {
    SymbolID id = symbol.getSymbolID();
    int type = symbol.getSymbolType();
    if ((type != Symbol.PACKAGE)) {
      PrintWriter pw = null;
      try {
        String file = Java.getClientJavaFile(id);
        boolean f = FileManager.getInstance().getJavaStylePackageGeneration();
        FileManager.getInstance().setJavaStylePackageGeneration(true);
        pw = FileManager.getInstance().createFile(id, type, "STUBJAVA", file);
        FileManager.getInstance().setJavaStylePackageGeneration(f);
        LanguageWriterForJava writer = new LanguageWriterForJava(pw);
        ClientJava.generateCode(symbol, writer);
      } finally {
        if (pw != null) {
          pw.close();
        }
      }
    }
  }

  /**
   * Generate C JNI stub code for the specified class or interface.  Stub
   * files are not generated for package symbols or enumerated types.
   */
  private void generateJNI(Symbol symbol) throws CodeGenerationException {
    final int type = symbol.getSymbolType();
    final SymbolID id = symbol.getSymbolID();
    if (symbol instanceof Extendable) {
      PrintWriter pw = null;
      try {
        String file = Java.getClientJNIFile(id);
        pw = FileManager.getInstance().
          createFile(id, type, "STUBSRCS", file);
        LanguageWriterForC writer = new LanguageWriterForC(pw);
        ClientJNI.generateCode((Extendable) symbol, writer);
      } finally {
        if (pw != null) {
          pw.close();
        }
      }
    }
  }

  public String getType()
  {
    return "stub";
  }

  public boolean getUserSymbolsOnly()
  {
    return false;
  }

  public Set getLanguages()
  {
    Set result = new TreeSet();
    result.add("java");
    return result;
  }
}
