//
// File:        GrammarException.java
// Package:     gov.llnl.babel.symbols
// Revision:    @(#) $Id: GrammarException.java 4434 2005-03-17 17:05:29Z epperly $
// Description: sidl parser exception for parsing errors in the grammar
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.parsers.sidl;

import gov.llnl.babel.parsers.sidl.Grammar;
import gov.llnl.babel.parsers.sidl.ParseException;
import gov.llnl.babel.parsers.sidl.SIDLException;

/**
 * A <code>GrammarException</code> is thrown if an error is detected during
 * the parse of the sidl grammar.  This exception inherits from the base
 * sidl exception class <code>SIDLException</code>.
 */
public class GrammarException extends SIDLException {
   private static final String EOL = "\n";

   private String   d_error_token;
   private String[] d_expected_tokens;

   /**
    * Create a new exception object using the specified parser exception
    * and production identifier.  Initialize the base class with the location
    * of the error and the associated production number.  Extract error token
    * and expected token information from the parse exception object.
    */
   public GrammarException(ParseException ex, int production) {
      super(ex.currentToken.next.beginLine,
            ex.currentToken.next.beginColumn,
            production);

      /*
       * Extract the expected token and the expected tokens.  Note that this
       * only works if the token look-ahead is one token (which is the case
       * for the sidl grammar).
       */

      if (ex.currentToken.next.kind == ParserConstants.EOF) {
         d_error_token = "END-OF-FILE";
      } else {
         d_error_token = ex.currentToken.next.image;
      }

      int [][] expected = ex.expectedTokenSequences;
      int n = expected.length;
      d_expected_tokens = new String[n];
      for (int i = 0; i < n; i++) {
         d_expected_tokens[i] = Grammar.getTokenString(expected[i][0]);
      }
   }

   /**
    * Return the token string causing the grammar error.
    */
   public String getErrorToken() {
      return d_error_token;
   }

   /**
    * Return the array of tokens expected by the grammar parser.
    */
   public String[] getExpectedTokens() {
      return d_expected_tokens;
   }

   /**
    * Generate an error message corresponding to this exception.  The
    * error message will report the line number, column number, production,
    * and the error and expected tokens.
    */
   public String getMessage() {
      StringBuffer buffer = new StringBuffer();

      buffer.append("Grammar exception at line=");
      buffer.append(String.valueOf(getLineNumber()));
      buffer.append(" column=");
      buffer.append(String.valueOf(getColumnNumber()));
      buffer.append(EOL);

      buffer.append("Grammar production: ");
      buffer.append(Grammar.getProduction(getProductionNumber()));
      buffer.append(EOL);

      buffer.append("Encountered \"");
      buffer.append(d_error_token);
      buffer.append("\""+EOL);

      int n = d_expected_tokens.length;
      if (n == 1) {
         buffer.append("Expected ");
      } else {
         buffer.append("Expected one of ");
      }
      for (int i = 0; i < n; i++) {
         buffer.append(d_expected_tokens[i]);
         if (i != n-1) {
            buffer.append(", ");
         }
      }

      return buffer.toString();
   }
}
