//
// File:        Extendable.java
// Package:     gov.llnl.babel.symbols
// Revision:    @(#) $Id: Extendable.java 4434 2005-03-17 17:05:29Z epperly $
// Description: sidl base symbol for both classes and interfaces
//
// Copyright (c) 2000-2003, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.symbols;

import gov.llnl.babel.symbols.Assertion;
import gov.llnl.babel.symbols.AssertionException;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Metadata;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Set;

/**
 * Class <code>Extendable</code> is a base class for SIDL symbols of type
 * class and interface.  It brings together common methods and implementation
 * for both final classes.  Class and interfaces have a set of methods and
 * interface inheritance in common.  Classes have the additional property
 * that they can be extended by another class; that functionality is not
 * defined here.  Many of the member functions take a boolean argument that
 * selects whether the method refers to this particular extendable only or
 * to this extendable and all of its parents in the SIDL inheritance system.
 * Constraints on the validity of methods added to this object must be checked
 * by the parser.
 *
 * Key design goals include providing mechanisms to:
 * - ensure method lookups in O(1) by both long and short names;
 * - ensure fast return of abstract, static, non-static, local, and both
 *   local and parent (or all) methods;
 * - ensure original ordering of methods preserved when returning lists;
 */
public abstract class Extendable extends Symbol {
  private ArrayList d_all_abstract_methods;
  private HashSet   d_all_basicarrays;
  private HashMap   d_all_interfaces;
  private ArrayList d_all_methods;
  private HashMap   d_all_methods_long;
  private HashMap   d_all_methods_short;
  private HashSet   d_all_references;
  private ArrayList d_all_nonstatic_methods;
  private ArrayList d_all_static_methods;

  private HashMap   d_local_interfaces;
  private ArrayList d_local_methods;
  private HashMap   d_local_methods_long;
  private HashMap   d_local_methods_short;
  private ArrayList d_local_nonstatic_methods;
  private ArrayList d_local_static_methods;

  private ArrayList d_invariants;		// Local invariants ONLY


  /**
   * Create an empty <code>Extendable</code> object that will be constructed
   * by calls to other member functions.
   *
   * @param  id       The symbol id of the new object
   * @param  type     The value of the type of the new object
   * @param  comment  The comment associated with the object
   */
  public Extendable(SymbolID id, int type, Comment comment) {
    super(id, type, comment);
    d_all_abstract_methods    = new ArrayList();
    d_all_basicarrays         = new HashSet();
    d_all_interfaces          = new HashMap();
    d_all_methods             = new ArrayList();
    d_all_methods_long        = new HashMap();
    d_all_methods_short       = new HashMap();
    d_all_references          = new HashSet();
    d_all_nonstatic_methods   = new ArrayList();
    d_all_static_methods      = new ArrayList();

    d_local_interfaces        = new HashMap();
    d_local_methods           = new ArrayList();
    d_local_methods_long      = new HashMap();
    d_local_methods_short     = new HashMap();
    d_local_nonstatic_methods = new ArrayList();
    d_local_static_methods    = new ArrayList();

    d_invariants              = new ArrayList();

    d_all_references.add(id);
  }

  /**
   * Create an empty <code>Extendable</code> object that will be constructed
   * by calls to other member functions.
   *
   * @param  id       The symbol id of the new object
   * @param  type     The value of the type of the new object
   * @param  comment  The comment associated with the object
   * @param  m        Object metadata
   */
  public Extendable(SymbolID id, int type, Comment comment, Metadata m) {
    super(id, type, comment, m);
    d_all_abstract_methods    = new ArrayList();
    d_all_basicarrays         = new HashSet();
    d_all_interfaces          = new HashMap();
    d_all_methods             = new ArrayList();
    d_all_methods_long        = new HashMap();
    d_all_methods_short       = new HashMap();
    d_all_references          = new HashSet();
    d_all_nonstatic_methods   = new ArrayList();
    d_all_static_methods      = new ArrayList();

    d_local_interfaces        = new HashMap();
    d_local_methods           = new ArrayList();
    d_local_methods_long      = new HashMap();
    d_local_methods_short     = new HashMap();
    d_local_nonstatic_methods = new ArrayList();
    d_local_static_methods    = new ArrayList();

    d_invariants              = new ArrayList();

    d_all_references.add(id);
  }

  /**
   * Return TRUE if this object contains any abstract methods; otherwise, 
   * return FALSE.  A class is abstract if and only if it has any abstract 
   * methods.  An interface must always be abstract, even if it contains no 
   * methods.
   */
  abstract public boolean isAbstract();

  /**
   * Return TRUE if this object represents an interface, FALSE if it is a 
   * class.
   */
  abstract public boolean isInterface();

  /**
   * Add the specified new method to this object.  No checking is done whether 
   * this method is valid for this particular extendable.  The new method will 
   * over-write any existing method unless the new method is abstract.  The
   * references and arrays will also be cached in this object.
   *
   * @param  method  The method to be added to this object
   */
  public void addMethod(Method method) {
    addToMethodLists(method, false);
    d_all_references.addAll(method.getSymbolReferences());
    d_all_basicarrays.addAll(method.getBasicArrays());
  }

  /**
   * Return the number of methods associated with this extendable.
   *
   * @param  all  If TRUE, then return the number of local and parent methods; 
   *              otherwise, return the number of local methods only.
   */
  public int getNumberOfMethods(boolean all) {
    return all ? d_all_methods.size() : d_local_methods.size();
  }

  /**
   * Return the methods in this interface as a <code>Collection</code>.
   * Each element in the collection is of type <code>Method</code>.
   *
   * @param  all  If TRUE, then return local and parent methods; otherwise,
   *              return only local methods.
   */
  public Collection getMethods(boolean all) {
    return all ? d_all_methods : d_local_methods;
  }

  /**
   * Return the <code>Collection</code> non-static methods in this interface.
   * Each element in the collection is of type <code>Method</code>.
   *
   * @param  all  If TRUE, then return local and parent non-static methods; 
   *              otherwise, return only local non-static methods.
   */
  public Collection getNonstaticMethods(boolean all) {
    return all ? d_all_nonstatic_methods : d_local_nonstatic_methods;
  }

  /**
   * Return the static methods in this interface as a <code>Collection</code>.
   * Each element in the collection is of type <code>Method</code>.
   *
   * @param  all  If TRUE, then return local and parent static methods; 
   *              otherwise, return only local static methods.
   */
  public Collection getStaticMethods(boolean all) {
    return all ? d_all_static_methods : d_local_static_methods;
  }

  /**
   * Return the abstract methods for this class or interface, which
   * includes all parent classes and interfaces.  Each element in the
   * collection is of type <code>Method</code>.
   */
  public Collection getAbstractMethods() {
    return d_all_abstract_methods;
  }

  /**
   *  Return all abstract methods and all local methods (both abstract and concreate).
   *  This is a special case for Java abstract classes, although maybe it
   *  will be useful elsewhere.
   */
  public Collection getAbstractAndLocalMethods() {
    ArrayList ret = (ArrayList) d_local_methods.clone();
    //Add all Abstracts that don't collide with locals
    for (Iterator abs = d_all_abstract_methods.iterator(); abs.hasNext(); ) {
      Method absM = (Method) abs.next();
      for (Iterator local = d_local_methods.iterator(); local.hasNext(); ) {
        Method locM = (Method) local.next();
        if(absM == locM)
          break;
        if(!local.hasNext())
          ret.add(absM);
      }
    }
    return ret;
  }
    

  /**
   * Return <code>true</code> if and only if the method given is locally
   * defined in this extendable.
   *
   * @param  m  The method of interest.
   * @return <code>true</code> if <code>m</code> is locally defined
   *         in this Extendable.
   */
  public boolean isLocal(Method m)
  {
    return (m != null) &&
      m.equals(d_local_methods_long.get(m.getLongMethodName()));
  }

  /**
   * Return the <code>Method</code> with the specified long method name.  
   * If there is none, return null.
   *
   * @param  name  The short method name for the method to be located.
   * @param  all   If TRUE then all local and parent methods are to 
   *               be searched; otherwise, only local methods are to
   *               be searched.
   */
  public Method lookupMethodByLongName(String name, boolean all) {
    return (Method) (all ? d_all_methods_long.get(name) 
                     : d_local_methods_long.get(name));
  }

  
  /**
   * Return the <code>Collection</code> of methods, each in <code>Method</code>
   * form, that are found to have the specified short method name, if any.  
   * If none are found, then return null.
   *
   * @param  name  The short method name for the method to be located.
   * @param  all   If TRUE then all local and parent methods are to 
   *               be searched; otherwise, only local methods are to
   *               be searched.
   */
  public Collection lookupMethodByShortName(String name, boolean all) {
    return (Collection) (all ? d_all_methods_short.get(name)
                         : d_local_methods_short.get(name));
  }

  /**
   * Return TRUE if the specified method exists by long name; otherwise,
   * return FALSE.
   *
   * @param  name  The long method name for the method to be located.
   * @param  all   If TRUE then all local and parent methods are to 
   *               be searched; otherwise, only local methods are to
   *               be searched.
   */
  public boolean hasMethodByLongName(String name, boolean all) {
    return all ? d_all_methods_long.containsKey(name) 
      : d_local_methods_long.containsKey(name);
  }

  /**
   * Return TRUE if the specified method exists by short name; otherwise,
   * return FALSE.  Recall there may be multiple methods with the same short 
   * name but here we only care if there is at least one.
   *
   * @param  name  The short method name to be located
   * @param  all   If TRUE, then all local and parent methods are to be 
   *               searched; otherwise, only local methods are to be 
   *               searched
   */
  public boolean hasMethodByShortName(String name, boolean all) {
    return all ? d_all_methods_short.containsKey(name) 
      : d_local_methods_short.containsKey(name);
  }

  /**
   * Return TRUE if any of the methods are static methods; otherwise,
   * return FALSE.
   *
   * @param  all  If TRUE, then local and parent methods are to be searched;
   *              otherwise, only local methods are to be searched.
   */
  abstract public boolean hasStaticMethod(boolean all);

  /**
   * Return TRUE if any of the methods throws an exception; otherwise, return
   * FALSE.  
   *
   * @param  all  If TRUE, then local and parent methods are to be searched;
   *              otherwise, only local methods are to be searched.
   */
  public boolean hasExceptionThrowingMethod(boolean all) {
    boolean has_exception_throwing_method = false;
     
    for (Iterator i = getMethods(all).iterator(); i.hasNext(); ) {
      Method method = (Method) i.next();
      if (method.getThrows().size()>0) { 
        has_exception_throwing_method = true;
        break;
      }
    }
    return has_exception_throwing_method;
  }

  /**
   * Add a new parent interface to this object.  This method will be
   * implemented by the <code>Class</code> and <code>Interface</code>
   * subclasses.
   *
   * @param  parent  The parent interface to be added to this object.
   */
  abstract public void addParentInterface(Interface parent);

  /**
   * Return the parent interfaces in a <code>Collection</code>.  Each
   * member of the collection is an <code>Interface</code>.  
   *
   * @param  all  If TRUE, then all parents are to be returned; otherwise,
   *              only direct parents are to be returned.
   */
  public Collection getParentInterfaces(boolean all) {
    return all ? d_all_interfaces.values() : d_local_interfaces.values();
  }

  /**
   * Return TRUE if the specified extendable can be found in the given 
   * list; otherwise, return FALSE.
   *
   * @param  list  The list being searched.
   * @param  ext   The extendable that is to be located.
   */
  private boolean findExistingEntry(ArrayList list, Extendable ext)
  {
    boolean    found = false;
    Extendable item  = null;
    for (ListIterator iter = list.listIterator(); iter.hasNext() && !found; )
    {
      item = (Extendable) iter.next();
      if (item == ext) {
        found = true;
      }
    }
    return found;
  }

  /**
   * Return the parent interfaces and/or classes in hierarchical order
   * from top-most down to direct parents in an <code>ArrayList</code>.  
   * Each member of the list is an <code>Extendable</code> and there
   * are NO duplicates.
   */
  public ArrayList getAllParentsInOrder() {
    ArrayList  ancestors = new ArrayList();
    Collection parents   = getParents(false);
    Iterator   i         = parents.iterator();
    while (i.hasNext()) {
      Extendable parent  = (Extendable)i.next();
      if (!findExistingEntry(ancestors, parent)) {
        ancestors.addAll(parent.getAllParentsInOrder());
        ancestors.add(parent);
      }
    }
    return ancestors;
  }

  /**
   * Return the parent interfaces and/or class in a <code>Collection</code>.
   * Each member of the collection is an <code>Extendable</code>.  No 
   * duplicate entries are returned in the collection.
   *
   * @param  all  If TRUE, all ancestors are to be returned; otherwise, only
   *              direct parents.  
   */
  public Collection getParents(boolean all) {
    return getParentInterfaces(all);
  }

  /**
   * Return TRUE if this object implements or extends the specified 
   * <code>Extendable</code> directly or indirectly.  Otherwise, return 
   * FALSE.
   *
   * @param  ext  The potential ancestor
   */
  public boolean hasAncestor(Extendable ext)
  {
    if (ext == null) return false;
    Collection parents = getParents(false);
    Iterator i = parents.iterator();
    while (i.hasNext()) {
      Extendable parent = (Extendable)i.next();
      if (ext == parent) return true;
    }
    i = parents.iterator();
    while (i.hasNext()) {
      Extendable parent = (Extendable)i.next();
      if (parent.hasAncestor(ext)) return true;
    }
    return false;
  }

  /**
   * Return TRUE if this class has the specified parent interface; otherwise,
   * return FALSE.  
   *
   * @param  id   The parent interface to be located
   * @param  all  Indicator of whether all parents (TRUE) or only direct
   *              parents (FALSE) are to be searched
   */
  public boolean hasParentInterface(SymbolID id, boolean all) {
    return all ? d_all_interfaces.containsKey(id)
      : d_local_interfaces.containsKey(id);
  }

  /**
   * Return the <code>Set</code> of symbol references for this object, each
   * element of which is in <code>SymbolID</code> form.  These are
   * defined as all references for this object as well as its parents.
   * The set of references includes this symbol name.
   */
  public Set getSymbolReferences() {
    return d_all_references;
  }

  /**
   * Return the <code>Set</code> of all symbol references for this object, each
   * element of which is in <code>SymbolID</code> form.  Note the references
   * include those from this object's parents.
   */
  public Set getAllSymbolReferences() {
    return getSymbolReferences();
  }

  /**
   * Return the <code>Set</code> of basic array types for this object, each
   * element of the set being in <code>SymbolID</code> form.
   */
  public Set getBasicArrayRefs() {
    return d_all_basicarrays;
  }

  /**
   * Protected method called by parents to add their relevant information,
   * such as methods, interfaces, references, and arrays to this object.
   *
   * @param  ext  The parent whose methods and related information are to be 
   *              added
   */
  protected void addParentData(Extendable ext) {
    /*
     * Add all parent methods to the list of all methods in this
     * class.  Do not override an existing method in the methods
     * list if the new method is abstract, since abstract methods
     * cannot override other methods.  If the method is abstract,
     * then add it to the list of abstract methods; otherwise, remove
     * it from the list.
     */
    for (Iterator m = ext.getMethods(true).iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      addToMethodLists(method, true);
    }

    /*
     * Add the parent and its parents to the list of parents.  That is
     * a lot of parents, let me tell you.
     */
    if (ext.isInterface()) {
      d_all_interfaces.put(ext.getSymbolID(), ext);
      d_local_interfaces.put(ext.getSymbolID(), ext);
    }

    Collection parents = ext.getParentInterfaces(true);
    for (Iterator p = parents.iterator(); p.hasNext(); ) {
      Extendable e = (Extendable) p.next();
      d_all_interfaces.put(e.getSymbolID(), e);
    }

    d_all_references.addAll(ext.getSymbolReferences());
    d_all_basicarrays.addAll(ext.getBasicArrayRefs());
  }

  /**
   *  Get all methods with the same short name as the passed in method name.
   *  (This list includes the passed in method, so the collection has a
   *  minimum size of 1)
   *  This is necessary for F90 method overloading
   */
  public Collection getOverloadedMethodsByName(String shortName) {
    ArrayList ret = new ArrayList();
    for(Iterator i = d_all_methods.iterator(); i.hasNext();) {
      Method m = (Method) i.next();
      if(m.getShortMethodName().compareTo(shortName) == 0) {
        ret.add(m);
      }
    }
    return ret;
  }

  /**
   * Locate the corresponding <code>Method</code> in the given list of 
   * <code>Method</code> elements, removing and optionally replacing
   * it.
   *
   * Assumptions:  
   * 1) There is only one entry in the list for each long method name.
   *    (This is appropriate since the long method name is supposed to be 
   (    unique.)
   *
   * @param  list       List of methods to be searched.
   * @param  method     The method whose existing entry is to be located
   * @param  replaceIt  Indicator of whether or not the existing method is
   *                    to be replaced
   */
  private void findExistingEntry(ArrayList list, Method method, 
                                 boolean replaceIt)
  {
    String  name  = method.getLongMethodName();
    boolean found = false;
    Method  item  = null;

    for (ListIterator iter = list.listIterator(); iter.hasNext() && !found; )
      {
        item = (Method) iter.next();
        if (item.getLongMethodName().equals(name)) {
          found = true;
          iter.remove();
        }
      }
    if (replaceIt) {
      list.add(method);
    }
  }

  /**
   * Add the given method to the specified short-name method list.
   *
   * @param  list    The short-name method list
   * @param  method  The method to be added to the list
   */
  private void addToShortMethodList(HashMap list, Method method) {
    String name = method.getShortMethodName();

    if (list.containsKey(name)) {
      findExistingEntry((ArrayList) list.get(name), method, true);
    } else {
      ArrayList sublist = new ArrayList();
      sublist.add(method);
      list.put(name, sublist);
    }
  }

  /**
   * Add the specified method to all and, if not isParent, local method lists
   * provided the method does not exist or is not abstract.  Note the latter 
   * case is used to overwrite existing methods.
   *
   * @param  method    The method to be added
   * @param  isParent  TRUE if method belongs to the parent; otherwise, FALSE.
   */
  private void addToMethodLists(Method method, boolean isParent) {
    String long_name = method.getLongMethodName();

    if (!(hasMethodByLongName(long_name, true) && method.isAbstract())) {
      boolean is_static  = method.isStatic();

      /* 
       * First process the lists of local and parent methods (i.e., the
       * "all" lists).
       */
      findExistingEntry(d_all_methods, method, true);
      d_all_methods_long.put(long_name, method);
      addToShortMethodList(d_all_methods_short, method);
      if (is_static) {
        findExistingEntry(d_all_static_methods, method, true);
      } else {
        findExistingEntry(d_all_nonstatic_methods, method, true);
      }
      if (method.isAbstract()) {
        findExistingEntry(d_all_abstract_methods, method, true);
      } else {
        findExistingEntry(d_all_abstract_methods, method, false);
      }

      /* 
       * Now process the lists of local methods, provided the method does
       * not belong to a parent.
       */
      if (!isParent) {
        findExistingEntry(d_local_methods, method, true);
        d_local_methods_long.put(long_name, method);
        addToShortMethodList(d_local_methods_short, method);
        if (is_static) {
          findExistingEntry(d_local_static_methods, method, true);
        } else {
          findExistingEntry(d_local_nonstatic_methods, method, true);
        }
      }
    }
  }

  /**
   * Add the invariant to this object.  
   *
   * Assumptions:
   * 1) The constraints on the assertion are being enforced elsewhere.
   * 2) Only invariants local to this extendable are going to be added!
   *
   * @param   assertion  The invariant assertion to be added
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                     The exception raised if the assertion cannot be added.
   */
  public void addInvariant(Assertion assertion) throws AssertionException {
    if (!assertion.isInvariant()) {
      throw new AssertionException("Extendable: " + getFullName(), "Cannot add "
                + "a(n) \"" + assertion.getTypeName() + "\" assertion to the "
                + "invariants list.");
    } else {
      d_invariants.add(assertion);
    } 
  }

  /**
   * Return TRUE if the extendable has any invariants; otherwise, return FALSE.
   *
   * @param   all   If TRUE then check inherited and local; otherwise, check
   *                only local invariants.
   */
  public boolean hasInvariants(boolean all) {
    boolean has = false;
    if (getNumberInvariants(false) > 0) {
      has = true;
    } else {
      has = getNumberInvariants(all) > 0 ? true : false;
    }
    return has;
  }

  /**
   * Return the number of invariants in this interface.
   *
   * @param   all   If TRUE then return the number of inherited and local
   *                invariants; otherwise, return only the number of local ones.
   */
  public int getNumberInvariants(boolean all) {
    return all ? d_invariants.size() : getAllInvariants().size();
  }

  /**
   * Return the invariants in this interface as an <code>ArrayList</code>,
   * each element of which is in <code>Assertion</code> form.
   */
  public ArrayList getInvariants() {
    return d_invariants;
  }

  /**
   * Return the list of all invariants associated with this object as an 
   * <code>ArrayList</code>, each element of which is in <code>Assertion</code>
   * form.
   */
  public ArrayList getAllInvariants() {
    ArrayList  list    = new ArrayList();
    ArrayList  parents = getAllParentsInOrder();
    Iterator i = parents.iterator();
    while (i.hasNext()) {
      Extendable parent = (Extendable)i.next();
      list.addAll(parent.getAllInvariants());
    }
    list.addAll(d_invariants);
    return list;
  }

  /**
   * Validate the invariants associated with this object.
   *
   * @param   all   If TRUE then all invariants are to be validated; otherwise,
   *                will only validate the local invariants.
   * @param   skip  If TRUE then will skip the validation process if the
   *                expression has already been marked as valid.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                The exception raised if errors are detected.
   */
  public void validateInvariants(boolean all, boolean skip) 
       throws AssertionException 
  {
    ArrayList list = (all ? getAllInvariants() : d_invariants);
    Iterator i = list.iterator();
    while (i.hasNext()) {
      Assertion inv = (Assertion)i.next();
      inv.validateExpression(this, null, skip);
    }
  }

  /**
   * Return TRUE if the extendable has any assertions; otherwise, return FALSE.
   */
  public boolean hasAssertions() {
    boolean has = false;
    if (hasInvariants(true)) {
      has = true;
    } else {
      for (Iterator i = getMethods(true).iterator(); i.hasNext() && (!has); ) {
        Method method = (Method) i.next();
        String lname  = method.getLongMethodName();
        if (  (getNumberPreconditions(lname)>0 ) 
           || (getNumberPostconditions(lname)>0) ) { 
          has = true;
        }
      }
    }
    return has;
  }

  /**
   * Return the number of assertions associated with the method in this
   * object, if it is present.  Will return an 0 if there is no such method
   * in this object's hierarchy or if there are no assertions.  Since the
   * invariants are invoked along with the preconditions and again with the
   * postconditions, their number counts twice.
   *
   * @param  longname  The long name of the method the number of whose 
   *                   assertions is to be returned.
   */
  public int getNumberAssertions(String longname) {
    return getNumberPreconditions(longname) 
         + 2 * getNumberInvariants(true)
         + getNumberPostconditions(longname);
  }

  /**
   * Return the number of preconditions associated with the method in this
   * object, if it is present.  Will return an 0 if there is no such method
   * in this object's hierarchy or if there are no preconditions.
   *
   * @param  longname  The long name of the method the number of whose 
   *                   preconditions is to be returned.
   */
  public int getNumberPreconditions(String longname) {
    return getAllPreconditions(longname).size();
  }

  /**
   * Return all preconditions associated with the method in this object,
   * if there is one, in an <code>ArrayList</code> with each entry being
   * in <code>Assertion</code> form.  Will return an empty list if no such 
   * method is in this object's hierarchy.
   *
   * WARNING:  
   *   There is no attempt at this point to optimize the preconditions.
   *   That is, duplicate preconditions are NOT removed.
   *
   * @param  longname  The long name of the method whose preconditions are to 
   *                   be returnede
   */
  public ArrayList getAllPreconditions(String longname) {
    ArrayList list    = new ArrayList();
    ArrayList parents = getAllParentsInOrder();
    Iterator  i       = parents.iterator();
    while (i.hasNext()) {
      Extendable parent = (Extendable)i.next();
      list.addAll(parent.getAllPreconditions(longname));
    }
    Method meth = lookupMethodByLongName(longname, false);
    if (meth != null) {
      list.addAll(meth.getPreconditions());
    }
    return list;
  }

  /**
   * Validate the preconditions associated with this object and the specified 
   * method.  By virtue of using getAllPreconditions(), it starts with the 
   * top-most inherited preconditions and proceeds "down" the inheritance 
   * hierarchy to the level of this class.
   *
   * @param   meth  The method whose preconditions are to be validated.
   * @param   all   If TRUE then will validate all preconditions; otherwise,
   *                will validate only the locally defined preconditions.
   * @param   skip  If TRUE then will skip the validation process if the
   *                expression has already been marked as valid.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                The exception raised if errors are detected.
   */
  public void validatePreconditions(Method meth, boolean all, boolean skip) 
       throws AssertionException 
  {
    if (meth != null) {
      ArrayList list;
      if (all) {
        String longname = meth.getLongMethodName();
        list = getAllPreconditions(longname);
      } else {
        list = meth.getPreconditions();
      }
      Iterator i = list.iterator();
      while (i.hasNext()) {
        Assertion pre = (Assertion)i.next();
        pre.validateExpression(this, meth, skip);
      }
    } else {
      throw new AssertionException("Extendable: " + getFullName(), 
                  "Cannot validate preconditions for specified method "
                + "when no method given.");
    }
  }

  /**
   * Validate the preconditions associated with this object and the specified 
   * method.  
   *
   * @param   longname  The long name of the method whose preconditions are to 
   *                    be validated.
   * @param   all       If TRUE then will validate all preconditions; otherwise,
   *                    will validate only the locally defined preconditions.
   * @param   skip      If TRUE then will skip the validation process if the
   *                    expression has already been marked as valid.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                    The exception raised if errors are detected.
   */
  public void validatePreconditions(String longname, boolean all, boolean skip) 
       throws AssertionException 
  {
    validatePreconditions(lookupMethodByLongName(longname, true), all, skip);
  }

  /**
   * Validate the preconditions associated with this object.
   *
   * @param   skip  If TRUE then will skip the validation process if the
   *                expression has already been marked as valid.
   * @param   all   If TRUE then will validate all preconditions; otherwise,
   *                will validate only the locally defined preconditions.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                The exception raised if errors are detected.
   */
  public void validatePreconditions(boolean all, boolean skip) 
       throws AssertionException 
  {
    Iterator im;
    if (all) {
      im = d_all_methods.iterator();
    } else {
      im = d_local_methods.iterator();
    }
    while (im.hasNext()) {
      Method meth = (Method)im.next();
      validatePreconditions(meth, all, skip);
    }
  }

  /**
   * Return the number of postconditions associated with the method in this
   * object, if it is present.  Will return an 0 if there is no such method
   * in this object's hierarchy or if there are no postconditions.
   *
   * @param  longname  The long name of the method the number of whose 
   *                   postconditions is to be returned.
   */
  public int getNumberPostconditions(String longname) {
    return getAllPostconditions(longname).size();
  }

  /**
   * Return all postconditions associated with the method in this object,
   * if there is one, in an <code>ArrayList</code> with each element in
   * <code>Assertion</code> form.  Will return an empty list if no such 
   * method in this object's hierarchy.
   *
   * WARNING:  
   *   There is no attempt at this point to optimize the postconditions.
   *   That is, duplicate postconditions are NOT removed.
   *
   * @param  longname  The long name of the method whose postconditions are to 
   *                   be returned.
   */
  public ArrayList getAllPostconditions(String longname) {
    ArrayList  list    = new ArrayList();
    ArrayList  parents = getAllParentsInOrder();
    Iterator   i       = parents.iterator();
    while (i.hasNext()) {
      Extendable parent = (Extendable)i.next();
      list.addAll(parent.getAllPostconditions(longname));
    }
    Method meth = lookupMethodByLongName(longname, false);
    if (meth != null) {
      list.addAll(meth.getPostconditions());
    }
    return list;
  }

  /**
   * Validate the postconditions associated with this object and the specified 
   * method.  By virtue of using getAllPostconditions(), it starts with the 
   * top-most inherited postconditions and proceeds "down" the inheritance 
   * hierarchy to the level of this class.
   *
   * @param   meth  The method whose postconditions are to be validated.
   * @param   all   If TRUE then all postconditions of the method will be
   *                validated; otherwise, only the locally defined ones will.
   * @param   skip  If TRUE then will skip the validation process if the
   *                expression has already been marked as valid.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                The exception raised if errors are detected.
   */
  public void validatePostconditions(Method meth, boolean all, boolean skip) 
       throws AssertionException 
  {
    if (meth != null) {
      ArrayList list;
      if (all) {
        String longname = meth.getLongMethodName();
        list = getAllPostconditions(longname);
      } else {
        list = meth.getPostconditions();
      }
      Iterator i = list.iterator();
      while (i.hasNext()) {
        Assertion post = (Assertion)i.next();
        post.validateExpression(this, meth, skip);
      }
    } else {
      throw new AssertionException("Extendable: " + getFullName(), 
                  "Cannot validate postconditions for specified method "
                + "when no method given.");
    }
  }

  /**
   * Validate the postconditions associated with this object and the 
   * specified method.  
   *
   * @param   longname  The long name of the method whose postconditions are to 
   *                    be validated.
   * @param   all       If TRUE then all postconditions of the method will be
   *                    validated; otherwise, only the locally defined ones 
   *                    will.
   * @param   skip      If TRUE then will skip the validation process if the
   *                    expression has already been marked as valid.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                    The exception raised if errors are detected.
   */
  public void validatePostconditions(String longname, boolean all, boolean skip)
       throws AssertionException 
  {
    validatePostconditions(lookupMethodByLongName(longname, true), all, skip);
  }

  /**
   * Validate postconditions associated with this object.
   *
   * @param   skip  If TRUE then will skip the validation process if the
   *                expression has already been marked as valid.
   * @param   all   If TRUE then all postconditions of the method will be
   *                validated; otherwise, only the locally defined ones will.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                The exception raised if errors are detected.
   */
  public void validatePostconditions(boolean all, boolean skip) 
       throws AssertionException 
  {
    Iterator im;
    if (all) {
      im = d_all_methods.iterator();
    } else {
      im = d_local_methods.iterator();
    }
    while (im.hasNext()) {
      Method meth = (Method)im.next();
      validatePostconditions(meth, all, skip);
    }
  }

  /**
   * Validate assertions associated with this object.  That is, validate all 
   * invariants as well as any pre- and post- conditions associated with any 
   * methods.
   *
   * @param   skip  If TRUE then will skip the validation process if the
   *                expression has already been marked as valid.
   * @param   all   If TRUE then all assertions including those for all 
   *                associated methods will be validated; otherwise, only 
   *                validate the locally defined ones.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                The exception raised if errors are detected.
   */
  public void validateAssertions(boolean all, boolean skip) 
       throws AssertionException 
  {
    validateInvariants(all, skip);
    Iterator im;
    if (all) {
      im = d_all_methods.iterator();
    } else {
      im = d_local_methods.iterator();
    }
    while (im.hasNext()) {
      Method meth = (Method)im.next();
      validatePreconditions(meth, all, skip);
      validatePostconditions(meth, all, skip);
    }
  }

  /**
   * Returns true if the specified reserved method call is present in any
   * of the assertions; otherwise, returns false.
   */
  public boolean hasReservedMethodAssertion(int type) {
    boolean   hasIt = false;
    ArrayList list  = getAllInvariants();
    Iterator i = list.iterator();
    while (i.hasNext() && !hasIt) {
      Assertion inv = (Assertion)i.next();
      if (inv.hasReservedMethod(type)) {
        hasIt = true;
      }
    }

    if (!hasIt) {
      Iterator im = d_all_methods.iterator();
      while (im.hasNext() && !hasIt) {
        Method meth = (Method)im.next();
        if (meth.hasReservedMethodAssertion(type)) {
          hasIt = true;
        }
      }
    }
    return hasIt;
  }
}
