//
// File:        GenerateServerC.java
// Package:     gov.llnl.babel.backend.c
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: GenerateServerC.java,v 1.18 2003/09/08 23:04:59 epperly Exp $
// Description: generate C server code based on a set of symbol identifiers
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeGenerator;
import gov.llnl.babel.backend.CodeGenerationFactory;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.c.ImplHeader;
import gov.llnl.babel.backend.c.ImplSource;
import gov.llnl.babel.backend.c.SkelSource;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.writers.LineCountingFilterWriter;
import gov.llnl.babel.backend.writers.LineRedirector;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import java.io.PrintWriter;
import java.io.Writer;
import java.io.IOException;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

/**
 * Class <code>GenerateServerC</code> is the main entry point for the C
 * server-side code generation routines.  This class generates the skeleton,
 * implementation header, and implementation source files.  It is assumed
 * that all symbols necessary to generate C code are available in the symbol
 * table (call <code>resolveAllReferences</code> before calling this routine).
 */
public class GenerateServerC implements CodeGenerator {

   /**
    * The constructor does nothing interesting.  The entry point for
    * the <code>GenerateServerC</code> class is <code>generateCode</code>.
    */
   public GenerateServerC() {
   }

   /**
    * Generate C server-side code for each symbol identifier in the set
    * argument.  This routine assumes that all necessary symbol references
    * are available in the symbol table.  Files are only generated for
    * class symbol types.
    *
    * @param     symbols  the set of all symbols to be processed. Server
    *                     code will be generated for members of this
    *                     set that are <code>Class<code>es.
    * @exception gov.llnl.babel.backend.CodeGenerationException
    *    this is a catch all exception. It can be caused by I/O trouble or
    *    violations of the data type invariants.
    */
   public void generateCode(Set symbols) 
     throws CodeGenerationException
  {
    SymbolTable table = SymbolTable.getInstance();
      for (Iterator s = symbols.iterator(); s.hasNext(); ) {
         SymbolID id     = (SymbolID) s.next();
         Symbol   symbol = table.lookupSymbol(id);
         if ((symbol != null) && (symbol.getSymbolType() == Symbol.CLASS)) {
            Class cls = (Class) symbol;
            generateSkeleton(cls);
            generateImplHeader(cls);
            generateImplSource(cls);
         }
      }
   }

   /**
    * Generate C skeleton source code for the specified class. The skeleton
    * code is the layer between the IOR and the developers implementation of
    * the class. It initializes the entry point vectors and provides methods
    * to access the object's private data pointer.
    *
    * @param     cls  the class for which a skeleton source 
    * @exception gov.llnl.babel.backend.CodeGenerationException
    *    this is a catch all exception. It can be caused by I/O trouble or
    *    violations of the data type invariants.
    */
   private void generateSkeleton(Class cls) throws CodeGenerationException {
      final SymbolID id = cls.getSymbolID();
      PrintWriter pw = null;
      try {
         String f = C.getSkelFile(id);
         pw = FileManager.getInstance().createFile(id, Type.CLASS, 
                                                   "SKELSRCS", f);
         LanguageWriterForC writer = new LanguageWriterForC(pw);
         SkelSource.generateCode(cls, writer);
      } finally {
         if (pw != null) {
            pw.close();
         }
      }
   }

   /**
    * Generate C implementation header code for the specified class.
    *
    * @param cls    the class for which an implementation header will
    *               be created.
    * @exception gov.llnl.babel.backend.CodeGenerationException
    *    this is a catch all exception. It can be caused by I/O trouble or
    *    violations of the data type invariants.
    */
   private void generateImplHeader(Class cls) 
     throws CodeGenerationException
   {
      final SymbolID id = cls.getSymbolID();
      PrintWriter pw = null;
      try {
         String f = C.getImplHeaderFile(id);
         CodeSplicer splicer = 
           FileManager.getInstance().getCodeSplicer(id, Type.CLASS, f);
         Writer fw = FileManager.getInstance().
           createWriter(id, Type.CLASS, "IMPLHDRS", f);
         LineCountingFilterWriter lcfw = new LineCountingFilterWriter(fw);
         pw = new PrintWriter( lcfw );
         LanguageWriterForC writer = new LanguageWriterForC( pw, lcfw );
         splicer.setLineRedirector( (LineRedirector) writer );
         ImplHeader.generateCode(cls, writer, splicer);
      } catch ( IOException ex ) { 
        throw new CodeGenerationException( ex.getMessage() );
      } finally {
         if (pw != null) {
            pw.close();
         }
      }
   }


   /**
    * Generate C implementation source code for the specified class.
    * Previous edits to the C implementation file are preserved in a
    * code splicer object and are spliced into the new file.
    *
    * @param cls    the class for which an implementation source file
    *               will be created.
    * @exception gov.llnl.babel.backend.CodeGenerationException
    *    this is a catch all exception. It can be caused by I/O trouble or
    *    violations of the data type invariants.
    */
   private void generateImplSource(Class cls)
     throws CodeGenerationException
   {
      final SymbolID id = cls.getSymbolID();
      PrintWriter pw = null;
      try {
         String f = C.getImplSourceFile(id);
         CodeSplicer splicer = 
           FileManager.getInstance().getCodeSplicer(id, Type.CLASS, f);
         Writer fw = FileManager.getInstance().
           createWriter(id, Type.CLASS, "IMPLSRCS", f);
         LineCountingFilterWriter lcfw = new LineCountingFilterWriter( fw );
         pw = new PrintWriter( lcfw );
         LanguageWriterForC writer = new LanguageWriterForC(pw, lcfw);
         splicer.setLineRedirector( (LineRedirector) writer );
         ImplSource.generateCode(cls, writer, splicer);
      } catch ( IOException ex ) { 
        throw new CodeGenerationException( ex.getMessage() );
      } finally {
         if (pw != null) {
            pw.close();
         }
      }
   }

  public String getType()
  {
    return "skel";
  }

  public boolean getUserSymbolsOnly()
  {
    return true;
  }

  public Set getLanguages()
  {
    Set result = new TreeSet();
    result.add("c");
    return result;
  }
}




