//
// File:        ImplHeader.java
// Package:     gov.llnl.babel.backend.c
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: ImplHeader.java,v 1.34 2003/02/04 23:49:08 epperly Exp $
// Description: generate C implementation header to a pretty writer stream
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Class <code>ImplHeader</code> generates a C implementation header to a
 * language writer output stream.  The constructor takes a language writer
 * stream and method <code>generateCode</code> generates the implementation
 * header file for the specified symbol to the output stream.  The language
 * writer stream is not closed by this object.
 */
public class ImplHeader {
  /**
   * The writer is the output stream for the new file. The content of the
   * new file is writtern to this writer.
   */
  private LanguageWriterForC d_writer;

  private SymbolTable        d_symtab;

  /**
   * The splicer is used when replacing an existing file. Sections from the
   * previous file are stored in the splicer and can be grafted back into
   * the file being generated.
   */
  private CodeSplicer        d_splicer;

  /**
   * Create a <code>ImplHeader</code> object that will write symbol
   * information to the provided output language writer stream.
   *
   * Assumption:  This code is completed generated so no code splicer
   * capabilities are needed.
   *
   * @param writer  the output writer for the new implementation header
   *                file. This writer will not be closed by this method.
   */
  public ImplHeader(LanguageWriterForC writer, CodeSplicer splicer) {
    d_splicer = splicer;
    d_writer  = writer;
    d_symtab  = SymbolTable.getInstance();
  }

  /**
   * This is a convenience utility function that writes the symbol
   * header information into the provided language writer output stream.
   * The output stream is not closed on exit.
   *
   * @param cls     the <code>Class</code> whose implementation header file
   *                is to be created.
   * @param writer  the output writer with which the new header file is
   *                created.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   *
   */
  public static void generateCode(Class cls,
                                  LanguageWriterForC writer,
                                  CodeSplicer splicer)
     throws CodeGenerationException
  {
    ImplHeader header = new ImplHeader(writer, splicer);
    header.generateCode(cls);
  }

  /**
   * Output a banner and the main header guard at the top of the file.
   *
   * @param cls the <code>Class</code> whose header will be written.
   */
  private void writePrologue(Class cls) {
    final SymbolID id = cls.getSymbolID();
    String header = C.getImplHeaderFile(id);
    d_writer.writeBanner(cls, header, true, 
      CodeConstants.C_DESC_IMPL_PREFIX + id.getFullName());
    d_writer.openHeaderGuard(header);
  }

  /**
   * Output the includes for the implementation header file.
   *
   * @param cls  the <code>Class</code> whose header will be written.
   */
  private void writeIncludeSection(Class cls,
                                   Collection localRefs) {
    final SymbolID id = cls.getSymbolID();
    String includes = id.getFullName() + "." + "_includes";
    d_writer.generateInclude("SIDL_header.h", true);

    Iterator i = localRefs.iterator();
    while (i.hasNext()) {
      SymbolID sid = (SymbolID)i.next();
      d_writer.generateInclude(C.getHeaderFile(sid), true);
    }
    i = null;

    d_writer.println();
    if (d_splicer.hasSymbol(includes)) {
      d_splicer.outputSymbolEdits(includes, d_writer.getPrintWriter());
    }
    else{
      d_writer.writeCommentLine(CodeSplicer.getBeginString(includes));
      d_writer.writeCommentLine("Put additional include files here...");
      d_writer.writeCommentLine(CodeSplicer.getEndString(includes));
    }
    d_writer.println();
  }

  /**
   * Write any necessary forward struct declarations.
   *
   * @param cls  the <code>Class</code> whose header will be written.
   */
  private void writeForwardStructDecls(Class cls, Collection localRefs) {
    // no forward struct declarations are needed
  }

  /**
   * Output the signatures for the get private data access methods.
   *
   * @param cls  the <code>Class</code> whose header will be written.
   */
  private void writeGetPrivateDataDecl(Class cls) {
    final SymbolID id = cls.getSymbolID();
    d_writer.println("extern " + C.getDataName(id) + "*");
    d_writer.println(C.getDataGetName(id) + "(");
    d_writer.increaseTabLevel();
    d_writer.println(C.getObjectName(id)+ ");");
    d_writer.decreaseTabLevel();
    d_writer.println();
  }

  /**
   * Output the signatures for the set private data access methods.
   *
   * @param cls  the <code>Class</code> whose header will be written.
   */
  private void writeSetPrivateDataDecl(Class cls) {
    final SymbolID id = cls.getSymbolID();
    d_writer.println("extern void");
    d_writer.println(C.getDataSetName(id) + "(");
    d_writer.increaseTabLevel();
    d_writer.println(C.getObjectName(id) + ",");
    d_writer.println(C.getDataName(id) + "*);");
    d_writer.decreaseTabLevel();
    d_writer.println();
  }

  /**
   * This will write an external declaration for a function that
   * wraps a method from the IOR. This routine implemented in the skeleton
   * provides access to the IOR method.
   *
   * @param cls       the <code>Class</code> whose header will be written.
   * @param funcName  the name of the wrapper function.
   * @param iorMethod a description of the IOR method to be wrapped.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void writeIORWrapperDecl(Class cls,
                                   String funcName,
                                   Method iorMethod)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    Iterator args = Utilities.extendArgs(id, iorMethod).iterator();
    d_writer.println("extern void");
    d_writer.print(funcName);
    d_writer.println("(");
    d_writer.increaseTabLevel();
    while (args.hasNext()) {
      Argument a = (Argument)args.next();
      d_writer.print(C.getArgumentString(a));
      if (args.hasNext()) {
        d_writer.print(", ");
      }
    }
    d_writer.println(");");
    d_writer.decreaseTabLevel();
    d_writer.println();
  }

  /**
   * Write a method declaration for a generic implementation method.
   *
   * @param lw       the language writer to write to
   * @param m        the method description
   * @param id       the name of the symbol owning the method
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private static void writeMethodDecl(LanguageWriter lw,
                                     Method m,
                                     SymbolID id)
    throws CodeGenerationException
  {
    Iterator arguments = Utilities.extendArgs(id, m).iterator();
    lw.print("extern ");
    lw.println(C.getReturnString(m.getReturnType()));
    lw.print(C.getMethodImplName(id, m.getLongMethodName()));
    lw.println("(");
    if (arguments.hasNext()){ 
      lw.increaseTabLevel();
      while (arguments.hasNext()) {
        Argument arg = (Argument)arguments.next();
        lw.print(C.getArgumentString(arg));
        if (arguments.hasNext()) {
          lw.println(",");
        }
      }
      lw.println(");");
      lw.decreaseTabLevel();
    }
    else {
      lw.print("void);");
    }
    lw.println();
  }

  /**
   * Write external declarations for all non-<code>abstract</code> methods
   * in the class.
   *
   * @param lw       the language writer to write to.
   * @param cls      the <code>Class</code> whose header will be written.
   * @param isStatic <code>true</code> means write only static methods;
   *                 <code>false</code> means write only non-static
   *                 methods.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void writeMethodDecls(LanguageWriter lw,
                                      Class cls,
                                      boolean isStatic)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    Iterator i = null;
    if (isStatic) {
      i = cls.getStaticMethods(false).iterator();
    } else {
      i = cls.getNonstaticMethods(false).iterator();
    }
    while (i.hasNext()) {
      Method m = (Method)i.next();
      if (!m.isAbstract()) {
        writeMethodDecl(lw, m, id);
      }
    }
  }

  /**
   * Write external declarations for the <code>Class</code> constructor and
   * destructor.
   *
   * @param lw   the language writer to write to.
   * @param cls  the <code>Class</code> whose header will be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void writeBuiltinDecls(LanguageWriter lw,
                                       Class cls)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    writeMethodDecl(lw,IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id), id);
    writeMethodDecl(lw,IOR.getBuiltinMethod(IOR.DESTRUCTOR, id), id);
  }

  public Collection getLocalReferences(Extendable ext) {
    boolean hasThrows = false;
    Set refs = new HashSet();
    Iterator i = ext.getMethods(false).iterator();
    refs.add(ext.getSymbolID());
    while (i.hasNext()) {
      Method m = (Method)i.next();
      refs.addAll(m.getSymbolReferences());
      if (!(hasThrows || m.getThrows().isEmpty())) {
        Symbol sym =
          d_symtab.lookupSymbol(BabelConfiguration.getBaseException());
        if (sym != null) {
          hasThrows = true;
          refs.add(sym.getSymbolID());
        }
      }
    }
    return refs;
  }

  /**
   * Define the data structure for the private class data. Each
   * implementation class has some private data associated with it. The
   * struct to hold that information is defined by this method.
   * 
   * @param cls  the <code>Class</code> whose data header will be written.
   */
  private void writePrivateDataType(Class cls) {
    final SymbolID id = cls.getSymbolID();
    String name = id.getFullName();
    d_writer.writeComment("Private data for class " + name, false);

    d_writer.println(C.getDataName(id) +" {");
    d_writer.increaseTabLevel();

    String data = cls.getFullName() + "." + "_data";
    if (d_splicer.hasSymbol(data)) {
      d_splicer.outputSymbolEdits(data, d_writer.getPrintWriter());
    } else {
      d_writer.writeCommentLine(CodeSplicer.getBeginString(data));
      d_writer.writeCommentLine("Put private data members here...");
      d_writer.println(
        "int ignore; " +
        "/* dummy to force non-empty struct; remove if you add data */");
      d_writer.writeCommentLine(CodeSplicer.getEndString(data));
    }

    d_writer.decreaseTabLevel();
    d_writer.println("};");
    d_writer.println();
  }

  /**
   * Write C implementation header information for the provided class
   * to the language writer output stream provided in the constructor.
   * This method does not close the writer output stream.
   *
   * @param cls  the <code>Class</code> whose header will be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public void generateCode(Class cls)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    Collection localRefs = getLocalReferences(cls);
    writePrologue(cls);
    writeIncludeSection(cls, localRefs);
    writeForwardStructDecls(cls, localRefs);
    writePrivateDataType(cls);
    d_writer.openCxxExtern();
    d_writer.writeComment(
      "Access functions for class private data and built-in methods",
                          false);
    writeGetPrivateDataDecl(cls);
    writeSetPrivateDataDecl(cls);
    if (cls.getParentClass() == null) {
      writeIORWrapperDecl(cls, C.getPrivateDestructor(id),
                          IOR.getBuiltinMethod(IOR.DELETE, id));
    }
    writeBuiltinDecls(d_writer, cls);
    d_writer.writeComment("User-defined object methods", false);
    writeMethodDecls(d_writer, cls, true);  // static methods
    writeMethodDecls(d_writer, cls, false); // object methods

    /*
     * Close the header file and include unused splicer symbols, if any.
     */
    if (d_splicer.hasUnusedSymbolEdits()) {
      d_writer.println();
      d_writer.beginBlockComment(true);
      d_writer.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT1);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT2);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT3);
      d_writer.endBlockComment(true);
      d_splicer.outputUnusedSymbolEdits(d_writer.getPrintWriter());
      d_writer.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
    }
    d_writer.closeCxxExtern();
    d_writer.closeHeaderGuard();
  }
}
