//
// File:        SkelSource.java
// Package:     gov.llnl.babel.backend.c
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.25 $
// Date:        $Date: 2003/02/04 23:49:09 $
// Description: generate C skeleton based for a particular class/interface
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * <p>
 * Class <code>SkelSource</code> generates a C skeleton source file to the
 * language writer output stream. The skeleton provides the glue between
 * the independent object representation (IOR) and the developer's C
 * implementation of the class.
 * </p>
 * <p>
 * The skeleton source contains free functions to fill the entry point
 * vector and optionally the static entry point vector. These functions
 * are what the IOR requires.
 * </p>
 * <p>
 * The skeleton source contains free functions to get/set the private
 * data pointer from the IOR. These functions are what the implementation
 * source requires. For the base class (i.e. the class without a parent
 * class), the skeleton also include a function to destroy the object.
 * This function is required for the implemention of
 * <code>deleteRef</code>.
 * </p>
 */
public class SkelSource {

  private static final String s_ensureOrderConstant[] = {
    "SIDL_general_order",
    "SIDL_column_major_order",
    "SIDL_row_major_order"
  };

  /**
   * Write the skeleton file for a particular class to the language writer
   * provided. The skeleton file is the glue between the IOR and the
   * developer's implementation of the class.
   *
   * @param cls    a skeleton file for this class will be written to
   *               <code>writer</code>
   * @param writer this is the output device to which the skeleton
   *               file will be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateCode(Class cls, LanguageWriterForC writer)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    writer.writeBanner(
      cls, C.getSkelFile(cls.getSymbolID()), false,
      CodeConstants.C_DESC_SKEL_PREFIX + id.getFullName());
    writer.generateInclude(IOR.getHeaderFile(cls.getSymbolID()), false);
    writer.generateInclude(C.getHeaderFile(cls.getSymbolID()), false);
    writer.printlnUnformatted("#include <stddef.h>");
    writer.println();

    if (cls.getParentClass() == null) {
      writeIORCall(cls, C.getPrivateDestructor(id),
                   IOR.getBuiltinMethod(IOR.DELETE, id), writer);
    }

    ImplHeader.writeBuiltinDecls(writer,cls);
    ImplHeader.writeMethodDecls(writer, cls, true);
    ImplHeader.writeMethodDecls(writer, cls, false);

    writeSkelMethods(cls, writer);
    
    writeInitializeEPV(cls, writer);
    if (cls.hasStaticMethod(true)) {
      writeInitializeSEPV(cls, writer);
    }

    writeGetDataPointer(cls, writer);
    writeSetDataPointer(cls, writer);
  }

  /**
   * Write skeleton functions for methods that need them.  For the C
   * skeleton, only functions with array ordering specifications require
   * a skeleton method.
   *
   * @param  cls   the class whose skeleton methods are to be written.
   * @param writer the output device where the skeleton methods are to be
   *               written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private static void writeSkelMethods(Class cls, LanguageWriter writer) 
    throws CodeGenerationException
  {
    Iterator i = cls.getMethods(false).iterator();
    while (i.hasNext()){
      Method m = (Method)i.next();
      if (m.hasArrayOrderSpec()) {
        writeSkelMethod(cls.getSymbolID(), m, writer);
      }
    }
  }

  private static String getArgValue(Argument arg)
  {
    return  (arg.getMode() == Argument.IN) ? arg.getFormalName()
      : "*" + arg.getFormalName();
  }

  /**
   * Write a skeleton function for a method with an array ordering
   * specification.
   *
   * @param id	   the name of the class who owns the method.
   * @param m      the method itself.
   * @param writer the output device where the skeleton function
   *               will be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private static void writeSkelMethod(SymbolID id, 
                                      Method m, 
                                      LanguageWriter writer)
    throws CodeGenerationException
  {
    List extArgs = Utilities.extendArgs(id, m);
    Iterator args = extArgs.iterator();
    Type methodReturn = m.getReturnType();
    final String method_name = C.getMethodSkelName(id, m);
    writer.println("static " + C.getReturnString(methodReturn));
    writer.println(method_name + "(");

    if (args.hasNext()) {
      writer.increaseTabLevel();
      while (args.hasNext()) {
        Argument arg = (Argument)args.next();
        writer.print(C.getArgumentWithFormal(arg));
        if (args.hasNext()) {
          writer.println(",");
        }
      }
      writer.println(")");
      writer.decreaseTabLevel();
    }
    else {
      writer.println("void)");
    }
    writer.println("{");
    writer.increaseTabLevel();
    if (methodReturn.getType() != Type.VOID) {
      writer.println(C.getReturnString(methodReturn) + " _return;");
      if (methodReturn.hasArrayOrderSpec()) {
        writer.println(C.getReturnString(methodReturn) + " _return_proxy;");
      }
    }
    args = extArgs.iterator();
    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      if (arg.hasArrayOrderSpec()) {
        final Type argType = arg.getType();
        writer.print(C.getReturnString(argType) +
                       " " + arg.getFormalName() + "_proxy");
        if (arg.getMode()!= Argument.OUT) {
          writer.println(" = " + C.getEnsureArray(argType.getArrayType()) +
                           "(" + getArgValue(arg) + ", " + 
                           argType.getArrayDimension() + ", " +
                           s_ensureOrderConstant[argType.getArrayOrder()] +
                           ");");
        }
        else{
          writer.println(" = NULL;");
        }
      }
    }
    args = extArgs.iterator();
    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      if (arg.hasArrayOrderSpec() && arg.getMode() == Argument.INOUT) {
        writer.println(C.getDelRefArray(arg.getType().getArrayType()) + 
                         "(" + getArgValue(arg) + ");");
      }
    }
    if (methodReturn.getType() != Type.VOID) {
      if (methodReturn.hasArrayOrderSpec()) {
        writer.println("_return_proxy =");
      }
      else {
        writer.println("_return =");
      }
      writer.increaseTabLevel();
    }
    writer.println(C.getMethodImplName(id, m.getLongMethodName()) + "(");
    writer.increaseTabLevel();
    args = extArgs.iterator();
    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      if (arg.hasArrayOrderSpec()) {
        if (Argument.IN != arg.getMode()) {
          writer.print("&");
        }
        writer.print(arg.getFormalName() + "_proxy");
      }
      else {
        writer.print(arg.getFormalName());
      }
      if (args.hasNext()) {
        writer.println(",");
      }
    }
    writer.println(");");
    writer.decreaseTabLevel();
    if (methodReturn.getType() != Type.VOID) {
       writer.decreaseTabLevel();
    }
    args = extArgs.iterator();
    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      if (arg.hasArrayOrderSpec()) {
        final Type argType = arg.getType();
        if (arg.getMode() != Argument.IN) {
          writer.println(getArgValue(arg) + " = " +
                           C.getEnsureArray(argType.getArrayType()) + "(" +
                           arg.getFormalName() + "_proxy, " +
                           arg.getType().getArrayDimension() + ", " +
                           s_ensureOrderConstant[argType.getArrayOrder()] +
                           ");");
        }
        writer.println(C.getDelRefArray(argType.getArrayType()) +
                         "(" + arg.getFormalName() + "_proxy);");
      }
    }
    if (methodReturn.getType() != Type.VOID) {
      if (methodReturn.hasArrayOrderSpec()) {
        writer.println("_return = " +
                         C.getEnsureArray(methodReturn.getArrayType()) +
                         "(_return_proxy, " +
                         methodReturn.getArrayDimension() + ", " +
                         s_ensureOrderConstant[methodReturn.getArrayOrder()]
                         + ");");
        writer.println(C.getDelRefArray(methodReturn.getArrayType()) +
                         "(_return_proxy);");
      }
      writer.println("return _return;");
    }
    writer.decreaseTabLevel();
    writer.println("}");
    writer.println();
  }

  /**
   * Write the support code to get the data pointer from the class.
   */
  private static void writeGetDataPointer(Class cls, LanguageWriter writer) {
    SymbolID id = cls.getSymbolID();
    writer.println(C.getDataName(id) + "*");
    writer.println(C.getDataGetName(id) + "(" + C.getObjectName(id) + " self)");
    writer.println("{");
    writer.increaseTabLevel();
    writer.println("return (" + C.getDataName(id) + "*)" +
                   "(self ? self->d_data : NULL);");
    writer.decreaseTabLevel();
    writer.println("}");
    writer.println();
  }

  /**
   * Write the support code to set the data pointer in the class.
   */
  private static void writeSetDataPointer(Class cls, LanguageWriter writer) {
    SymbolID id = cls.getSymbolID();
    writer.println("void " + C.getDataSetName(id) + "(");
    writer.increaseTabLevel();
    writer.println(C.getObjectName(id) + " self,");
    writer.println(C.getDataName(id) + "* data)");
    writer.decreaseTabLevel();
    writer.println("{");
    writer.increaseTabLevel();
    writer.println("if (self) {");
    writer.increaseTabLevel();
    writer.println("self->d_data = data;");
    writer.decreaseTabLevel();
    writer.println("}");
    writer.decreaseTabLevel();
    writer.println("}");
  }

  /**
   * Write a function to initialize entries in the entry point vector for
   * a particular class. This will generate an assignment statement for
   * each non-<code>static</code> method defined locally in the class.
   *
   * @param cls    the class for which an routine will be written.
   * @param writer the output writer to which the routine will be written.
   */
  private static void writeInitializeEPV(Class cls,
                                         LanguageWriter writer)
  {
    SymbolID id = cls.getSymbolID();
    writer.println("void");
    writer.println(C.getSetEPVName(id) + "(" + IOR.getEPVName(id) + " *epv)");
    writer.println("{");
    writer.increaseTabLevel();
    initializeMethodPointer
      (writer, IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id), id);
    initializeMethodPointer
      (writer, IOR.getBuiltinMethod(IOR.DESTRUCTOR, id), id);
    Iterator i = cls.getMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      initializeMethodPointer(writer, m, id);
    }
    writer.decreaseTabLevel();
    writer.println("}");
    writer.println();
  }

  /**
   * For non-<code>static</code> methods, write an assignment statement to
   * initialize a particular membor of the entry point
   * vector. Initialization of <code>static</code> methods appears
   * elsewhere.
   *
   * @param writer     the output writer to which the assignment statement
   *                   is written.
   * @param m          a description of the method to initialize
   * @param id         the name of class that owns the method
   */
  private static void initializeMethodPointer(LanguageWriter writer,
                                              Method m,
                                              SymbolID id)
  {
    final String methodName = m.getLongMethodName();
    switch (m.getDefinitionModifier()) {
    case Method.FINAL:
    case Method.NORMAL:
      writer.print("epv->");
      writer.print(IOR.getVectorEntry(methodName));
      writer.print(" = ");
      writer.print(C.getMethodSkelName(id, m));
      writer.println(";");
      break;
    case Method.ABSTRACT:
      writer.print("epv->");
      writer.print(IOR.getVectorEntry(methodName));
      writer.println(" = NULL;");
      break;
    default:
      /* do nothing */
      break;
    }
  }

  /**
   * Write a function to initialize entries in the static entry point vector
   * for a particular class. This will generate an assignment statement for
   * each <code>static</code> method defined locally in the class.
   *
   * @param cls    the class for which an routine will be written.
   * @param writer the output writer to which the routine will be written.
   */
  private static void writeInitializeSEPV(Class cls,
                                          LanguageWriter writer)
  {
    SymbolID id = cls.getSymbolID();
    writer.println("void");
    writer.println(C.getSetSEPVName(id)+"("+IOR.getSEPVName(id)+" *sepv)");
    writer.println("{");
    writer.increaseTabLevel();
    Iterator i = cls.getMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method) i.next();
      if (m.isStatic()) {
        writer.print("sepv->");
        writer.print(IOR.getVectorEntry(m.getLongMethodName()));
        writer.print(" = ");
        writer.print(C.getMethodSkelName(id, m));
        writer.println(";");
      }
    }
    writer.decreaseTabLevel();
    writer.println("}");
    writer.println();
  }

  /**
   * Create a wrapper function in the skeleton for a particular IOR method.
   *
   * @param cls       the class for which the routine is created.
   * @param funcName  the name of the wrapper function to be written.
   * @param iorMethod the description of the IOR method to be wrapped.
   * @param writer    the output writer to which the routine will be
   *                  written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private static void writeIORCall(Class cls,
                                   String funcName,
                                   Method iorMethod,
                                   LanguageWriter writer)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    List extArgs = Utilities.extendArgs(id, iorMethod);
    Iterator args = extArgs.iterator();
    writer.println("void");
    writer.print(funcName);
    writer.print("(");
    while (args.hasNext()) {
      Argument a = (Argument)args.next();
      writer.print(IOR.getArgumentWithFormal(a));
      if (args.hasNext()) {
        writer.print(", ");
      }
    }
    writer.println(") {");
    writer.increaseTabLevel();
    writer.println("if (self) {");
    writer.increaseTabLevel();
    writer.writeCommentLine("call the IOR method");
    writer.print("self->d_epv->");
    writer.print(IOR.getVectorEntry(iorMethod.getLongMethodName()));
    writer.print("(");
    args = extArgs.iterator();
    while (args.hasNext()){
      Argument a = (Argument)args.next();
      writer.print(a.getFormalName());
      if (args.hasNext()) {
        writer.print(", ");
      }
    }
    writer.println(");");
    writer.decreaseTabLevel();
    writer.println("}");
    writer.decreaseTabLevel();
    writer.println("}");
    writer.println();
  }
}
