//
// File:        GenerateClient.java
// Package:     gov.llnl.babel.backend.fortran
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.8 $
// Description: generate C code to allow FORTRAN to access BABEL
//
// Copyright (c) 2000-2002, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
//
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
//
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


package gov.llnl.babel.backend.fortran;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.CommandLineDriver;
import gov.llnl.babel.ResourceLoader;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeGenerationFactory;
import gov.llnl.babel.backend.CodeGenerator;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.fortran.ModuleSource;
import gov.llnl.babel.backend.fortran.StubDoc;
import gov.llnl.babel.backend.fortran.StubSource;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.writers.LanguageWriterForFortran;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import java.io.PrintWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;


/**
 * This class implements the {@link gov.llnl.babel.backend.CodeGenerator
 * CodeGenerator} interface for the FORTRAN client side code generator.
 * This generator creates C source code to provide the glue between
 * FORTRAN clients and the IOR.
 */
public class GenerateClient implements CodeGenerator 
{
  private final static int[] s_basic_array_types = {
    Type.BOOLEAN,
    Type.CHAR,
    Type.DCOMPLEX,
    Type.DOUBLE,
    Type.FCOMPLEX,
    Type.FLOAT,
    Type.INT,
    Type.LONG,
    Type.OPAQUE,
    Type.STRING
  };
  
  /**
   * Create a new instance.
   */
  public GenerateClient() {
    
  }

  private void copyArrayStub(String typeName,
                             Writer out)
    throws CodeGenerationException
  {
    final boolean isF90 = 
      "f90".equals(BabelConfiguration.getInstance().getTargetLanguage());
    final String context =
      "gov/llnl/babel/backend/fortran/" +
      (isF90 ? "f90" : "f77");
    ResourceLoader rl = new ResourceLoader();
    InputStream in = null;
    byte[] buffer = new byte[1024];
    int count;
    try {
      try {
        in = rl.getResourceStream(context + "/SIDL_" + typeName + "_fStub.c");
        while ((count = in.read(buffer)) >= 0) {
          String str = new String(buffer,0,count);
          out.write(str);
        }
      }
      finally {
        if (in != null) {
          in.close();
        }
      }
    }
    catch (IOException ioe) {
      throw new CodeGenerationException(ioe.getMessage());
    }
  }

  private void copyArrayAbbrev(String typeName,
                               Writer out)
    throws CodeGenerationException
  {
    final String context = "gov/llnl/babel/backend/fortran/f90";
    ResourceLoader rl = new ResourceLoader();
    InputStream in = null;
    byte[] buffer = new byte[1024];
    int count;
    try {
      try {
        in = rl.getResourceStream(context + 
                                  "/SIDL_" + typeName + "_fAbbrev.h");
        while ((count = in.read(buffer)) >= 0) {
          String str = new String(buffer,0,count);
          out.write(str);
        }
      }
      finally {
        if (in != null) {
          in.close();
        }
      }
    }
    catch (IOException ioe) {
      throw new CodeGenerationException(ioe.getMessage());
    }
  }

  /**
   * Write F90array stubs for all the basic types.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception.
   */
  private void generateBasicArrayStubs()
    throws CodeGenerationException
  {
    PrintWriter pw = null;
    Writer w = null;
    LanguageWriterForFortran forWriter = null;
    boolean isF90 = "f90".
      equals(BabelConfiguration.getInstance().getTargetLanguage());

    try {
      int i;
      for(i = 0; i < s_basic_array_types.length; ++i) {
        final int type = s_basic_array_types[i];
        final String typeName = (new Type(type)).getTypeString();
        final SymbolID id = new SymbolID("SIDL." + typeName,
                                         new Version());
        final String filename = Fortran.getArrayFile(id);
        if (CommandLineDriver.notExcluded(id)) {
          if (isF90) {
            pw = FileManager.getInstance().createFile(id, type, 
                                                      "ARRAYMODULESRCS",
                                                      filename);
            forWriter = new LanguageWriterForFortran(pw);
            ArrayModule am = new ArrayModule(id, type, forWriter);
            am.generateStub();
            forWriter.close();
          }
        }
        w = FileManager.getInstance().createFile(id, type,
                                                 "STUBSRCS",
                                                 Fortran.getStubFile(id));
        copyArrayStub(typeName, w);
        w.close();

        if (isF90) {
          w = FileManager.getInstance().
            createFile(id, type, "STUBHDRS",
                       Fortran.getStubNameFile(id));
          copyArrayAbbrev(typeName, w);
        }
      }
    }
    catch (IOException ioe) {
      throw new CodeGenerationException(ioe.getMessage());
    }
    finally {
      try {
        if (w != null) {
          w.close();
        }
      }
      catch (IOException ioe) {
        throw new CodeGenerationException(ioe.getMessage());
      }
      finally {
        if (pw != null) {
          pw.close();
        }
      }
    }
  }

  /**
   * If <code>symbol</code> is a class or interface, create a stub file for
   * it.
   *
   * @param symbol  a particlar symbol.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception.
   */
  private void generateStubSource(Symbol symbol)
    throws CodeGenerationException 
  {
    PrintWriter pw = null, dw = null, nw = null, mw = null;
    LanguageWriterForFortran forWriter;
    final SymbolID id = symbol.getSymbolID();
    final int type = symbol.getSymbolType();
    boolean isF90 = "f90".
      equals(BabelConfiguration.getInstance().getTargetLanguage());
    boolean writeBasicArrayStubs = false;
    switch (type){
    case Symbol.CLASS:
    case Symbol.INTERFACE:
      try {
        String f = Fortran.getStubFile(id);
        pw       = FileManager.getInstance().createFile(id, type, 
                                                        "STUBSRCS", f);

        LanguageWriterForC writer = new LanguageWriterForC(pw);
        StubSource.generateCode(symbol, writer);
        writeBasicArrayStubs = true;

        if (isF90) {
          String nf = Fortran.getStubNameFile(id);
          nw        = FileManager.getInstance().
            createFile(id, type, "STUBHDRS", nf);
          writer    = new LanguageWriterForC(nw);
          AbbrevHeader.generateCode(symbol, writer);

          String mf = Fortran.getModuleFile(id);
          mw        = FileManager.getInstance().
            createFile(id, type, "STUBMODULESRCS", mf);
	  forWriter = new LanguageWriterForFortran(mw);
          ModuleSource.generateCode(symbol, forWriter);
          forWriter.close();

          String amf = Fortran.getArrayFile(id);
          mw         = FileManager.getInstance().
            createFile(id, type, "ARRAYMODULESRCS", amf);
          forWriter = new LanguageWriterForFortran(mw);
          ArrayModule am = new ArrayModule(id, type, forWriter);
          forWriter.writeBanner(symbol, amf, false,
                                CodeConstants.C_FORTRAN_TYPE_MODULE_PREFIX +
                                symbol.getFullName());
          am.generateStub();
          forWriter.close();


          String tf = Fortran.getTypeFile(id);
          mw        = FileManager.getInstance().
            createFile(id, type, "TYPEMODULESRCS", tf);
          forWriter = new LanguageWriterForFortran(mw);
          TypeModule.generateCode(symbol, forWriter);
        } else {
          String df = Fortran.getStubDocFile(id);
          dw        = FileManager.getInstance().
            createFile(id, type, "STUBDOCS", df);
	  forWriter = new LanguageWriterForFortran(dw);
          StubDoc.generateCode(symbol, forWriter);
        }
      }
      finally {
        if (pw != null) {
          pw.close();
        }
        if (dw != null) {
          dw.close();
        }
        if (nw != null) {
          nw.close();
        }
        if (mw != null) {
          mw.close();
        }
      }
      break;
    case Symbol.ENUM:
      try {
        String nf = Fortran.getStubNameFile(id);
        String f = Fortran.getEnumStubImpl(id);
        pw = FileManager.getInstance().createFile(id, type, "STUBSRCS", f);
        writeBasicArrayStubs = true;
        LanguageWriterForC clw = new LanguageWriterForC(pw);
        StubSource.generateCode(symbol, clw);
        if (isF90) {
          nw  = FileManager.getInstance().
            createFile(id, type, "STUBHDRS", nf);
          clw = new LanguageWriterForC(nw);
          AbbrevHeader.generateCode(symbol, clw);
          String mf = Fortran.getModuleFile(id);
          mw        = FileManager.getInstance().
            createFile(id, type, "STUBMODULESRCS", mf);
	  forWriter = new LanguageWriterForFortran(mw);
          ModuleSource.generateCode(symbol, forWriter);
          forWriter.close();

          String amf = Fortran.getArrayFile(id);
          mw         = FileManager.getInstance().
            createFile(id, type, "ARRAYMODULESRCS", amf);
          forWriter = new LanguageWriterForFortran(mw);
          ArrayModule am = new ArrayModule(id, type,
                                           forWriter);
          forWriter.writeBanner(symbol, amf, false,
                                CodeConstants.C_FORTRAN_TYPE_MODULE_PREFIX +
                                symbol.getFullName());
          am.generateStub();
          forWriter.close();

          String tf = Fortran.getTypeFile(id);
          mw        = FileManager.getInstance().
            createFile(id, type, "TYPEMODULESRCS", tf);
          forWriter = new LanguageWriterForFortran(mw);
          TypeModule.generateCode(symbol, forWriter);
        }
        else {
          f = Fortran.getEnumStubFile(id);
          pw = FileManager.getInstance().
            createFile(id, type, "STUBFORTRANINC", f);
          LanguageWriterForFortran writer = new LanguageWriterForFortran(pw);
          StubSource.generateCode(symbol, writer);
          writer.close();
        }
      }
      finally {
        if (pw != null) {
          pw.close();
        }
        if (nw != null) {
          nw.close();
        }
        if (mw != null) {
          mw.close();
        }
      }
      break;
    }
    if (writeBasicArrayStubs) {
      generateBasicArrayStubs();
    }
  }
  
  /**
   * Given a set of symbol ids, this method will generate FORTRAN stubs
   * (written in C) for all the symbols in the set as needed.  Some symbol
   * types may not require a stub file, for example packages, and these are 
   * silently skipped.
   *
   * @param symbols    a set of symbol id (symbol names) for whom stubs
   *                   should be written as needed.  Each object in the
   *                   set should be a {@link
   *                   gov.llnl.babel.symbols.SymbolID SymbolID}.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   */
  public void generateCode(Set symbols)
    throws CodeGenerationException 
  {
    Iterator i = symbols.iterator();
    SymbolTable table = SymbolTable.getInstance();
    while (i.hasNext()) {
      SymbolID id     = (SymbolID)i.next();
      Symbol   symbol = table.lookupSymbol(id);
      if (symbol != null) {
        generateStubSource(symbol);
      }
    }
  }

  public String getType()
  {
    return "stub";
  }

  public boolean getUserSymbolsOnly()
  {
    return false;
  }

  public Set getLanguages()
  {
    Set result = new TreeSet();
    result.add("f77");
    result.add("f90");
    return result;
  }
}
