//
// File:        ModuleSource.java
// Package:     gov.llnl.babel.backend.fortran
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.10 $
// Description: Generate a module file for FORTRAN 90 clients
//
// Copyright (c) 2002-2003, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
//
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
//
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


package gov.llnl.babel.backend.fortran;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.fortran.Fortran;
import gov.llnl.babel.backend.fortran.StubSource;
import gov.llnl.babel.backend.writers.LanguageWriterForFortran;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * This class provides the ability to write a FORTRAN 90 module for
 * a SIDL class/interface. 
 */
public class ModuleSource {
  private static final int s_CASTS_PER_LINE = 7;

  /**
   * Indicator of version of FORTRAN being generated.
   */
  private final static boolean d_isF90 = "f90".
    equals(BabelConfiguration.getInstance().getTargetLanguage());

  /**
   * This is the output device.
   */
  private LanguageWriterForFortran d_lw;

  private static final String[] s_intent_spec = {
    ", intent(in)",
    ", intent(inout)",
    ", intent(out)"
  };

  /**
   * Generate an instance to write the module for a FORTRAN 90
   * client.
   * 
   * @param writer  the output device to which the FORTRAN 90 module
   *                  should be written.
   */
  public ModuleSource(LanguageWriterForFortran writer) {
    d_lw = writer;
  }


  /**
   * Generate the argument list for the subroutine, including the
   * opening and closing parens.
   *
   * @param List     List of arguments to be printed
   */
  private void printArgs(List args)
  {
    d_lw.print("(");
    Iterator i = args.iterator();
    while (i.hasNext()) {
      d_lw.print(((Argument)i.next()).getFormalName());
      if (i.hasNext()) {
        d_lw.print(", ");
      }
    }
    d_lw.println(")");
  }

  /**
   * Write the FORTRAN 90 module for a subroutine that corresponds to a 
   * SIDL class/interface method. This writes method signatures and declares 
   * the types of the arguments.
   * 
   * @param m     the method whose module method is to be written.
   * @param id    the name of the class that owns this method.
   * @param isInterface true iff the method is in an interface (as opposed
   *                    to a class.
   * @exception   gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   */
  private void extendAndGenerate(Method   m,
                                 SymbolID id)
    throws CodeGenerationException
  {
    List extendedArgs = StubSource.extendArgs(id, m);
    final String methodName = m.getLongMethodName() + "_s";
    final String stubName = Fortran.getMethodStubName(id, m);
    d_lw.print("recursive subroutine " + methodName);
    printArgs(extendedArgs);

    d_lw.increaseTabLevel();
    d_lw.println("implicit none");

    Iterator i = extendedArgs.iterator();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      d_lw.writeCommentLine(a.getArgumentString());
      d_lw.println(Fortran.getReturnString(a.getType()) + " " +
                   s_intent_spec[a.getMode()] + " :: " 
                   + a.getFormalName());
    }
    d_lw.println();
    d_lw.println("external " + stubName);
    d_lw.print("call " + stubName);
    printArgs(extendedArgs);

    d_lw.decreaseTabLevel();
    d_lw.println();
    d_lw.println("end subroutine " + methodName);
  }


  /**
   * Generate a create method object.
   */
  private Method create(SymbolID id)
  {
    Method m = new Method();
    m.setMethodName("new");
    m.setDefinitionModifier(Method.STATIC);
    String[] s = new String[1];
    s[0] = "Create an instance of class " + id.getFullName();
    m.setComment(new Comment(s));
    m.setReturnType(new Type(id));
    return m;
  }

  public static Set extendedReferences(Extendable ext)
    throws CodeGenerationException
  {
    Set s = StubSource.extendedReferences(ext);
    s.add(ext.getSymbolID());
    Iterator i = ext.getParents(true).iterator();
    while (i.hasNext()) {
      s.add(((Symbol)i.next()).getSymbolID());
    }
    return s;
  }


  /**
   * Add the implicit stub methods to the list of those that must
   * be included.
   *
   * @param ext the class whose module file is being written.
   */
  private Collection extendMethods(Extendable ext) {
    Collection allMethods = ext.getMethods(true);
    final SymbolID id = ext.getSymbolID();
    ArrayList  extendedMethods = new ArrayList(allMethods.size()+1);
    if (!ext.isAbstract()) {
      extendedMethods.add(create(id));
    }

    extendedMethods.addAll(allMethods);
    return extendedMethods;
  }


  /**
   * Write #include for all the abbreviation files for
   * referenced symbols.
   */
  private void writeIncludes(Extendable ext)
    throws CodeGenerationException
  {
    Set s = extendedReferences(ext);
    s.add(ext.getSymbolID());
    Iterator i = s.iterator();
    d_lw.disableLineBreak();
    while (i.hasNext()) {
      SymbolID id = (SymbolID)i.next();
      d_lw.printlnUnformatted("#include \"" +
                              Fortran.getStubNameFile(id) + "\"");
    }
    d_lw.enableLineBreak();
  }

  private void includeType(SymbolID id)
  {
    d_lw.println("use " + Fortran.getTypeModule(id));
  }

  private static void checkType(Type t, Set result)
  {
    if (t.getDetailedType() == Type.ARRAY) {
      final Type arrayType = t.getArrayType();
      final int detailedType = arrayType.getDetailedType();
      if ((detailedType > Type.VOID) && (detailedType <= Type.STRING)) {
        result.add(new SymbolID("SIDL." + arrayType.getTypeString(),
                                new Version()));
      }
    }
  }

  private static Set basicArrayReferences(Extendable ext)
  {
    HashSet result = new HashSet();
    Iterator i = ext.getMethods(true).iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      checkType(m.getReturnType(), result);
      Iterator j = m.getArgumentList().iterator();
      while (j.hasNext()) {
        checkType(((Argument)j.next()).getType(), result);
      }
    }
    return result;
  }

  private void includeTypes(Extendable ext) 
    throws CodeGenerationException
  {
    final SymbolID id = ext.getSymbolID();
    final SymbolTable table = SymbolTable.getInstance();
    Set s = extendedReferences(ext);
    s.add(id);
    Iterator i = s.iterator();
    while (i.hasNext()) {
      SymbolID current = (SymbolID)i.next();
      Symbol sym = table.lookupSymbol(current);
      if (sym instanceof Extendable) {
        includeType(current);
      }
    }
    s = basicArrayReferences(ext);
    i = s.iterator();
    while (i.hasNext()) {
      SymbolID basic_array = (SymbolID)i.next();
      d_lw.println("use " + Fortran.getArrayModule(basic_array));
    }
  }

  /**
   * Generate a CAST function in the module.
   */
  private void generateCast(SymbolID oldType,
                            SymbolID newType,
                            int      num)
  {
    String castMethod = Fortran.getMethodStubName
      (newType, StubDoc.createCast(newType));
    d_lw.beginBlockComment(false);
    d_lw.println("Static function to cast from " + oldType.getFullName());
    d_lw.println("to " + newType.getFullName() + ".");
    d_lw.endBlockComment(false);
    d_lw.println("subroutine cast_" + num + "(oldType, newType)");
    d_lw.increaseTabLevel();
    d_lw.println("implicit none");
    d_lw.println("type(" + Fortran.getTypeName(oldType) + 
                 "), intent(in) :: oldType");
    d_lw.println("type(" + Fortran.getTypeName(newType) + 
                 "), intent(out) :: newType");
    d_lw.println("external " + castMethod);
    d_lw.println();
    d_lw.println("call " + castMethod + "(oldType, newType)");
    d_lw.decreaseTabLevel();
    d_lw.println("end subroutine cast_" + num);
    d_lw.println();
  }

  private void generateCastMethods(Extendable ext,
                                   Collection parents)
  {
    /*
     * Experience has shown that we cannot count on parents to
     * have a consistent ordering from run to run. Therefore we
     * must sort parents.
     */
    final Object[] parentArray = parents.toArray();
    final SymbolID  id = ext.getSymbolID();
    int num = 0;
    Arrays.sort(parentArray);
    for(int i = 0; i < parentArray.length; ++i) {
      SymbolID parentID = ((Extendable)parentArray[i]).getSymbolID();
      generateCast(id, parentID, num++);
      generateCast(parentID, id, num++);
    }
  }

  private void writeCastList(int numCasts)
  {
    d_lw.print("cast_0");
    for(int i = 1; i < numCasts; ++i) {
      if (((numCasts % s_CASTS_PER_LINE) == 0) 
          && (i < (numCasts - 1))) {
        d_lw.println(", &");
        if (numCasts == 7) {
          d_lw.increaseTabLevel();
        }
      }
      else {
        d_lw.print(", ");
      }
      d_lw.print("cast_" + i);
    }
  }

  private void writeCastInterface(int numCasts)
  {
    if (numCasts > 0) {
      int i;
      d_lw.print("private :: ");
      writeCastList(numCasts);
      d_lw.println();
      if (numCasts > 7) {
        d_lw.decreaseTabLevel();
      }
      d_lw.println("interface cast");
      d_lw.increaseTabLevel();
      d_lw.print("module procedure ");
      writeCastList(numCasts);
      d_lw.println();
      if (numCasts > 7) {
        d_lw.decreaseTabLevel();
      }
      d_lw.decreaseTabLevel();
      d_lw.println("end interface");
    }
  }

  private void generateNullSubroutines(Extendable ext)
  {
    d_lw.println("logical function is_null_s(ext)");
    d_lw.increaseTabLevel();
    d_lw.println("type(" + Fortran.getTypeName(ext.getSymbolID()) +
                 "), intent(in) :: ext");
    d_lw.println("is_null_s = (ext%d_ior .eq. 0)");
    d_lw.decreaseTabLevel();
    d_lw.println("end function is_null_s");
    d_lw.println();
    d_lw.println("logical function not_null_s(ext)");
    d_lw.increaseTabLevel();
    d_lw.println("type(" + Fortran.getTypeName(ext.getSymbolID()) +
                 "), intent(in) :: ext");
    d_lw.println("not_null_s = (ext%d_ior .ne. 0)");
    d_lw.decreaseTabLevel();
    d_lw.println("end function not_null_s");
    d_lw.println();
    d_lw.println("subroutine set_null_s(ext)");
    d_lw.increaseTabLevel();
    d_lw.println("type(" + Fortran.getTypeName(ext.getSymbolID()) +
                 "), intent(out) :: ext");
    d_lw.println("ext%d_ior = 0");
    d_lw.decreaseTabLevel();
    d_lw.println("end subroutine set_null_s");
    d_lw.println();
  }
  
  private void writeMethodInterface(String methodName)
  {
    d_lw.println("private :: " + methodName + "_s");
    d_lw.println("interface " + methodName);
    d_lw.increaseTabLevel();
    d_lw.println("module procedure " + methodName + "_s");
    d_lw.decreaseTabLevel();
    d_lw.println("end interface");
    d_lw.println();
  }

  private void writeMethodInterfaces(Collection methods)
  {
    Iterator i = methods.iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      writeMethodInterface(m.getLongMethodName());
    }
  }


  /**
   * Generate the FORTRAN 90 module file for a SIDL class.  
   * 
   * @param ext    the SIDL class whose module is to be written.
   * @exception    gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   */
  public void generateCode(Extendable ext)
    throws CodeGenerationException
  {
    Collection methods = extendMethods(ext);
    Collection parents = ext.getParents(true);
    final SymbolID id = ext.getSymbolID();
    final String name = Fortran.getModule(id);

    d_lw.writeBanner(ext, Fortran.getModuleFile(id), false,
                     CodeConstants.C_FORTRAN_MODULE_PREFIX 
                     + id.getFullName());
    d_lw.println();
    d_lw.writeComment(ext, false);
    d_lw.println();
    writeIncludes(ext);
    d_lw.println();
    d_lw.println("module " + name);
    d_lw.println();
    d_lw.increaseTabLevel();

    includeTypes(ext);

    writeCastInterface(parents.size()*2);
    writeMethodInterfaces(methods);
    writeMethodInterface("not_null");
    writeMethodInterface("is_null");
    writeMethodInterface("set_null");

    d_lw.println();
    d_lw.decreaseTabLevel();
    d_lw.println("contains");
    d_lw.println();

    d_lw.increaseTabLevel();
    Iterator i = methods.iterator();
    while (i.hasNext()) {
      d_lw.println();
      d_lw.println();

      extendAndGenerate((Method)i.next(), id);
    }
    d_lw.println();
    generateCastMethods(ext, parents);
    generateNullSubroutines(ext);

    d_lw.decreaseTabLevel();
    d_lw.println();
    d_lw.println("end module " + name);
  }

  /**
   * Generate the FORTRAN 90 module file for a SIDL enumerated type.  
   * 
   * @param enm    the SIDL enumeration whose module is to be written.
   * @exception    gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   */
  public void generateCode(Enumeration enm)
    throws CodeGenerationException
  {
    final SymbolID id = enm.getSymbolID();
    final String name = Fortran.getModule(id);
    d_lw.writeBanner(enm, Fortran.getModuleFile(id), false,
                     CodeConstants.C_FORTRAN_MODULE_PREFIX 
                     + id.getFullName());
    d_lw.println();
    d_lw.println("module " + name);
    d_lw.writeComment(enm, false);
    d_lw.increaseTabLevel();
    d_lw.println();
    Iterator  i = enm.getEnumerators().iterator();
    while (i.hasNext()){
      String sym = (String)i.next();
      Comment cmt = enm.getEnumeratorComment(sym);
      d_lw.writeComment(cmt, true);
      d_lw.print(Fortran.getReturnString(new Type(id)));
      d_lw.print(", parameter :: ");
      d_lw.print(sym);
      d_lw.println(" = " + enm.getEnumeratorValue(sym));
      if (cmt != null) {
        d_lw.println();
      }
    }
    d_lw.decreaseTabLevel();
    d_lw.println("end module " + name);
  }


  /**
   * Generate the FORTRAN 90 module file for a SIDL class.  
   *
   * Note:  This is the assumed entry point; otherwise, the test for
   * the version of the language should be repeated.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
  */
  public static void generateCode(Symbol                   sym,
                                  LanguageWriterForFortran writer)
    throws CodeGenerationException
  {
    if (d_isF90) {
      if (sym instanceof Extendable) {
        ModuleSource modFile = new ModuleSource(writer);
        modFile.generateCode((Extendable)sym);
      }
      else if (sym instanceof Enumeration) {
        ModuleSource modFile = new ModuleSource(writer);
        modFile.generateCode((Enumeration)sym);
      }
    } else {  // Assuming this means F77
      throw new CodeGenerationException("Generation of module files only " +
                                        "supported for FORTRAN 90");
    }
  }
}
