//
// File:        ClientJNI.java
// Package:     gov.llnl.babel.backend.jdk
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: ClientJNI.java,v 1.7 2003/08/05 15:31:11 epperly Exp $
// Description: write Java client (stub) JNI code that links Java with the IOR
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.jdk;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.jdk.Java;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Class <code>ClientJNI</code> writes the JNI C code that links the Java
 * client to the IOR.  The constructor takes a C language writer stream and
 * method <code>generateCode</code> writes the C JNI code for the specified
 * symbol to the output stream.  The language writer output stream is not
 * closed by this object.
 */
public class ClientJNI {
  private LanguageWriterForC d_writer;

  /**
   * This is a convenience utility function that writes the JNI client
   * information into the provided language writer output stream.  The
   * output stream is not closed on exit.  A code generation exception
   * is thrown if an error is detected, such as I/O trouble or a violation
   * of data type invariants.
   */
  public static void generateCode(Extendable symbol, LanguageWriterForC writer)
      throws CodeGenerationException {
    ClientJNI jni = new ClientJNI(writer);
    jni.generateCode(symbol);
  }

 /**
   * Create a <code>ClientJNI</code> object that will write symbol
   * information to the provided output language writer stream.
   */
  public ClientJNI(LanguageWriterForC writer) {
    d_writer = writer;
  }

  /**
   * Return the number of methods that the client stub declares.
   */
  private int countMethods(Extendable ext)
  {
    final boolean isInterface = ext.isInterface();
    Collection methods = ext.getMethods(isInterface);
    int count;
    if (isInterface) {
      count = methods.size();
    }
    else {
      count = 0;
      Iterator i = methods.iterator();
      while (i.hasNext()) {
        Method m = (Method)i.next();
        if (!m.isAbstract()) {
          ++count;
        }
      }
    }
    return count;
  }

  /**
   * Write Java JNI information for the provided symbol to the language
   * writer output stream provided in the class constructor.  This method
   * does not close the writer output stream.  Code is currently generated
   * only for SIDL interfaces and classes, since enumerations and packages
   * do not require JNI support.
   */
  public void generateCode(Extendable ext) throws CodeGenerationException {
    if (ext == null) {
      throw new CodeGenerationException("Unexpected null extendable object");
    }

    /*
     * Output the file banner.  Include the IOR and SIDL Java include files.
     */
    SymbolID id = ext.getSymbolID();
    String file = Java.getClientJNIFile(id);
    d_writer.writeBanner(ext, file, CodeConstants.C_IS_NOT_IMPL,
                         CodeConstants.C_DESC_CJNI_PREFIX + id.getFullName());

    d_writer.generateInclude("SIDL_Java.h", false);
    d_writer.generateInclude("SIDL_Loader.h", false);
    d_writer.generateInclude("SIDL_String.h", false);
    d_writer.generateInclude(IOR.getHeaderFile(id), false);
    d_writer.generateInclude("babel_config.h", false);
    d_writer.println();

    /*
     * Output defines that convert between pointes and jlongs.
     */
    d_writer.writeComment("Convert between jlong and void* pointers.", false);
    d_writer.disableLineBreak();
    d_writer.println("#if (SIZEOF_VOID_P == 8)");
    d_writer.println("#define JLONG_TO_POINTER(x) ((void*)(x))");
    d_writer.println("#define POINTER_TO_JLONG(x) ((jlong)(x))");
    d_writer.println("#else");
    d_writer.println("#define JLONG_TO_POINTER(x) ((void*)(int32_t)(x))");
    d_writer.println("#define POINTER_TO_JLONG(x) ((jlong)(int32_t)(x))");
    d_writer.println("#endif");
    d_writer.println();
    d_writer.println("#ifndef NULL");
    d_writer.println("#define NULL 0");
    d_writer.println("#endif");
    d_writer.println();
    d_writer.enableLineBreak();

    /*
     * Output the method to extract the IOR reference from the object.
     * We only need the _get_ior function if the object is an interface
     * or if the object is a class with a non-static, non-abstract method.
     */
    boolean need_ior_function = ext.isInterface();

    if (!need_ior_function) {
      Iterator m = ext.getMethods(false).iterator();
      while (m.hasNext()) {
        Method method = (Method) m.next();
        if (!method.isAbstract() && !method.isStatic()) {
          need_ior_function = true;
          break;
        }
      }
    }

    if (need_ior_function) {
      d_writer.writeComment(
        "Function to extract IOR reference from the Java object.", false);
      d_writer.println("static " + IOR.getObjectName(id) + "* _get_ior(");
      d_writer.increaseTabLevel();
      d_writer.println("JNIEnv* env,");
      d_writer.println("jobject obj)");
      d_writer.decreaseTabLevel();
      d_writer.println("{");
      d_writer.increaseTabLevel();
      d_writer.println("void* ptr = NULL;");
      d_writer.println("static jmethodID mid = (jmethodID) NULL;");
      d_writer.println();
      d_writer.println("if (mid == (jmethodID) NULL) {");
      d_writer.increaseTabLevel();
      d_writer.println("jclass cls = (*env)->GetObjectClass(env, obj);");
      d_writer.println(
        "mid = (*env)->GetMethodID(env, cls, \"_get_ior\", \"()J\");");
      d_writer.println("(*env)->DeleteLocalRef(env, cls);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
      d_writer.println(
        "ptr = JLONG_TO_POINTER((*env)->CallLongMethod(env, obj, mid));");
      d_writer.println("return (" + IOR.getObjectName(id) + "*) ptr;");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }

    /*
     * If this is a non-abstract class, then output a static pointer to
     * the IOR external data structure (for the constructor pointer).
     */
    if (!ext.isAbstract()) {
      d_writer.writeComment("External reference to IOR methods.", false);
      d_writer.println("static const "
                     + IOR.getExternalName(id)
                     + "* s_external = NULL;");
      d_writer.println();
    }

    /*
     * If this is a class with static methods, then output a pointer to the
     * static EPV structure that will be initialized at class loading.
     */
    if (ext.hasStaticMethod(false)) {
      d_writer.writeComment("External reference to static EPV.", false);
      d_writer.println("static const "
                     + IOR.getSEPVName(id)
                     + "* s_sepv = NULL;");
      d_writer.println();
    }

    /*
     * If this is a non-abstract class, then output the function that will
     * create an instance of the object and return the pointer as a jlong
     * reference.
     */
    if (!ext.isAbstract()) {
      d_writer.writeComment(
        "Create object instance and return reference.", false);
      d_writer.println("static jlong jni__create_ior(");
      d_writer.increaseTabLevel();
      d_writer.println("JNIEnv* env,");
      d_writer.println("jclass  cls)");
      d_writer.decreaseTabLevel();
      d_writer.println("{");
      d_writer.increaseTabLevel();
      d_writer.println("(void) env;");
      d_writer.println("(void) cls;");
      d_writer.println(
        "return POINTER_TO_JLONG((*s_external->createObject)());");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }

    /*
     * Output the glue code for all methods in the class or interface.  For
     * an interface, we must define all methods.  For a class, we only define
     * new methods.
     */
    List methods = (List) ext.getMethods(ext.isInterface());
    for (Iterator m = methods.iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      if (ext.isInterface() || !method.isAbstract()) {
        generateMethod(id, method);
        d_writer.println();
      }
    }

    /*
     * Output the registration method.  This is the only external symbol.
     * First output the method signature and the arguments.
     */
    final int nmethods = countMethods(ext) + (ext.isAbstract() ? 0 : 1);
    d_writer.writeComment("Register JNI methods with the Java JVM.", false);
    d_writer.println("void " + Java.getRegisterFunction(id) + "(JNIEnv* env)");
    d_writer.println("{");
    d_writer.increaseTabLevel();
    if (nmethods > 0) {
      d_writer.println("JNINativeMethod methods[" +
                       Integer.toString(nmethods) + "];");
      d_writer.println("jclass cls;");
      d_writer.println();

      /*
       * If this is a non-abstract class, then output the code that dynamically
       * loads the external reference.  If there is an error, then throw an
       * unsatisfied link error.
       */
      if (!ext.isAbstract()) {
        if (BabelConfiguration.isSIDLBaseClass(id)) {
          d_writer.println("s_external = " + IOR.getExternalFunc(id) + "();");
        } else {
          d_writer.printlnUnformatted("#ifdef SIDL_STATIC_LIBRARY");
          d_writer.println("s_external = " + IOR.getExternalFunc(id) + "();");
          d_writer.printlnUnformatted("#else");
          d_writer.println("void* address = SIDL_Loader_lookupSymbol(\""
                           + IOR.getExternalFunc(id)
                           + "\");");
          d_writer.println("if (address != NULL) {");
          d_writer.increaseTabLevel();
          d_writer.println("s_external = (*(("
                           + IOR.getExternalName(id)
                           + "* (*)(void)) address))();");
          d_writer.decreaseTabLevel();
          d_writer.println("} else {");
          d_writer.increaseTabLevel();
          d_writer.println("jclass e = (*env)->FindClass(env, "
                           + "\"java/lang/UnsatisfiedLinkError\");");
          d_writer.println("if (e != NULL) {");
          d_writer.increaseTabLevel();
          d_writer.disableLineBreak();
          d_writer.println("(*env)->ThrowNew(env, e, \""
                           + "Could not find implementation for SIDL class "
                           + id.getFullName()
                           + "\");");
          d_writer.enableLineBreak();
          d_writer.println("(*env)->DeleteLocalRef(env, e);");
          d_writer.decreaseTabLevel();
          d_writer.println("}");
          d_writer.decreaseTabLevel();
          d_writer.println("}");
          d_writer.printlnUnformatted("#endif");
          d_writer.println();
        }
      }
      
      /*
       * If this class has a static method, then initialize the static pointer.
       */
      if (ext.hasStaticMethod(false)) {
        d_writer.println("s_sepv = (*(s_external->getStaticEPV))();");
        d_writer.println();
      }

      /*
       * Initialize the method array of names, signatures, and pointers in
       * preparation for registering the methods.
       */
      int idx = 0;
      if (!ext.isAbstract()) {
        d_writer.println("methods[0].name      = \"_create_ior\";");
        d_writer.println("methods[0].signature = \"()J\";");
        d_writer.println("methods[0].fnPtr     = (void *)jni__create_ior;");
        idx++;
      }
      
      for (Iterator m = methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        if (ext.isInterface() || !method.isAbstract()) {
          String prefix = "methods[" + Integer.toString(idx) + "].";
          d_writer.println(prefix
                           + "name      = \""
                           + method.getShortMethodName()
                           + "\";");
          d_writer.disableLineBreak();
          d_writer.println(prefix
                           + "signature = \""
                           + Java.getJavaSignature(method)
                           + "\";");
          d_writer.enableLineBreak();
          d_writer.println(prefix
                           + "fnPtr     = (void *)"
                           + Java.getJNIFunction(method)
                           + ";");
          ++idx;
        }
      }
      d_writer.println();

      /*
       * Register the methods with the Java virtual machine.
       */
      String lookup_name = Java.getFullJavaSymbolName(id).replace('.', '/');
      if (ext.isInterface()) {
        lookup_name = lookup_name + "$Wrapper";
      }
      d_writer.println("cls = (*env)->FindClass(env, \"" + lookup_name + "\");");
      d_writer.println("if (cls) {");
      d_writer.increaseTabLevel();
      d_writer.println("(*env)->RegisterNatives(env, cls, methods, "
                       + String.valueOf(nmethods)
                       + ");");
      d_writer.println("(*env)->DeleteLocalRef(env, cls);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
    }
    else {
      d_writer.writeComment("Intentionally empty: no methods to register",
                            false);
    }
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }

  /**
   * Generate the method glue code that hooks the Java to the C IOR.  Output
   * the signature, the glue code to morph Java arguments to IOR arguments,
   * make the call, and then glue code to morph out and return arguments from
   * IOR to Java.
   */
  private void generateMethod(SymbolID id, Method method)
      throws CodeGenerationException {
    boolean throws_exception = !method.getThrows().isEmpty();

    /*
     * Output the method signature and the agrument list.  Static methods do
     * not take an object reference but rather a class reference.  All user
     * arguments are prefixed with "_arg_" and the temporary variables are
     * prefixed with "_tmp_" to avoid name collisions with other arguments.
     */
    d_writer.writeComment(method, false);

    Type return_type = method.getReturnType();
    d_writer.println("static " + Java.getJNIReturnType(return_type));
    d_writer.println(Java.getJNIFunction(method) + "(");
    d_writer.increaseTabLevel();
    d_writer.println("JNIEnv* env,");
    if (!method.isStatic()) {
      d_writer.print("jobject obj");
    } else {
      d_writer.print("jclass  cls");
    }

    /*
     * Declare the remaining arguments in the argument list.
     */
    List args = method.getArgumentList();
    d_writer.println(args.isEmpty() ? ")" : ",");

    for (Iterator a = args.iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      d_writer.print(Java.getJNIFormalArgument(arg));
      d_writer.println(a.hasNext() ? "," : ")");
    }

    /*
     * Done with the argument list - decrease indentation, print the curly
     * brace, and then indent for the method implementation.
     */
    d_writer.decreaseTabLevel();
    d_writer.println("{");
    d_writer.increaseTabLevel();

    /*
     * Begin the method body with the declarations required for the return
     * type, object reference (if not a static method), and the temporary
     * variables used to translate Java constructs to C.
     */
    d_writer.writeComment("Declare return and temporary variables.", false);
    if (!method.isStatic()) {
      d_writer.println(IOR.getObjectName(id) + "* _ior = NULL;");
    }
    for (Iterator a = args.iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      Java.declareIORVariable(d_writer, arg, "_tmp_");
    }
    if (return_type.getType() != Type.VOID) {
      Java.declareIORVariable(d_writer, return_type, "_ior_res");
      Java.declareJavaVariable(d_writer, return_type, "_res");
    }
    
    if (throws_exception) {
      d_writer.println("struct SIDL_BaseException__object* _ex = NULL;");
    }
    d_writer.println();

    /*
     * Grab the pointer to the IOR object and pre-process all of the method
     * arguments to convert data from Java to C.
     */
    d_writer.writeComment("Preprocess Java types and convert into IOR.", false);
    if (!method.isStatic()) {
      d_writer.println("_ior = _get_ior(env, obj);");
    }
    for (Iterator a = args.iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      Java.preprocessJNIArgument(d_writer, arg, "_tmp_");
    }
    d_writer.println();

    /*
     * Call the IOR method.  If there is a return type, then store the return
     * value in <code>_ior_result</code>.  We need to lookup the appropriate
     * function pointer in the structure and then pass the appropriate this
     * pointer.  Pass each of the arguments to the output method.
     */
    d_writer.writeComment("Call the IOR method through the EPV.", false);
    if (return_type.getType() != Type.VOID) {
      d_writer.print("_ior_res = ");
    }
    d_writer.print("(*(");
    d_writer.print(method.isStatic() ? "s_sepv" : "_ior->d_epv");
    d_writer.print("->f_" + method.getLongMethodName() + "))(");
    if (method.isStatic() && args.isEmpty() && !throws_exception) {
      d_writer.println(");");
    } else {
      d_writer.println();
      d_writer.increaseTabLevel();
      if (!method.isStatic()) {
        d_writer.print(method.isAbstract() ? "_ior->d_object" : "_ior");
        d_writer.println(args.isEmpty() && !throws_exception ? ");" : ",");
      }
      for (Iterator a = args.iterator(); a.hasNext(); ) {
        Argument arg = (Argument) a.next();
        if (arg.getMode() != Argument.IN) {
          d_writer.print("&");
        }
        d_writer.print("_tmp_" + arg.getFormalName());
        d_writer.println((a.hasNext() || throws_exception) ? "," : ");");
      }
      if (throws_exception) {
        d_writer.println("&_ex);");
      }
      d_writer.decreaseTabLevel();
    }
    d_writer.println();

    /*
     * Post-process the method arguments if necessary.  This will involve
     * processing the INOUT and OUT arguments and the return argument.
     */
    d_writer.writeComment(
      "Postprocess OUT, INOUT, returns, and exceptions.", false);
    for (Iterator a = args.iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      Java.postprocessJNIArgument(d_writer, arg, "_tmp_");
    }
    if (return_type.getType() != Type.VOID) {
      Java.postprocessJNIReturn(d_writer, return_type, "_ior_res", "_res");
    }
    if (throws_exception) {
      d_writer.println("SIDL_Java_CheckException(");
      d_writer.increaseTabLevel();
      d_writer.println("env,");
      d_writer.println("_ex,");
      for (Iterator t = method.getThrows().iterator(); t.hasNext(); ) {
        d_writer.println("\"" + ((SymbolID) t.next()).getFullName() + "\",");
      }
      d_writer.println("NULL);");
      d_writer.decreaseTabLevel();
    }
    if (return_type.getType() != Type.VOID) {
      d_writer.println();
      d_writer.println("return _res;");
    }

    /*
     * Done with the method body.  Decrease tab level and close the curly brace.
     */
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }
}
