/* PLEASE READ:
 *****************************************************************************
 * NOTES/ IMPLEMENTATION ISSUES: (Sarah Knoop 8/14/03)
 * 1.) Compilation errors arise in generated code that throws exceptions. Will have 
 * to generate proper import statements or set proper classpath to handle the
 * current errors.Exception handling has not yet been tested.
 * 2.) I don't believe that interface generation has been enabled.Should be easy 
 * addition/switch.
 ******************************************************************************/

package gov.llnl.babel.backend.jdk;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeGenerator;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.jdk.Java;
import gov.llnl.babel.backend.FileManager;

import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForJava;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolUtilities;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import java.io.File;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Create and write a source file for a Java extension class to wrap a 
 * BABEL extendable in a Java object. 
 */
public class JavaImplSource {
    
    private Extendable d_ext = null;
    private LanguageWriterForJava d_writer = null;
    private CodeSplicer d_splicer = null;
    private String d_classname = null;  // So files will be named with _Impl suffix, for naming collisions

    /**
     * Create an object capable of generating the source file for Java implementation
     *
     * @param ext   an interface or class symbol that needs source
     *              file for a Java extension class.
     */
    public JavaImplSource(Extendable ext) throws CodeGenerationException {
	if (ext == null) {
	    throw new CodeGenerationException("Unexpected null extendable object");
	}
	
	d_ext= ext;
	
	SymbolID id = d_ext.getSymbolID();
	d_classname = id.getShortName() + "_Impl"; 

	int type = d_ext.getSymbolType();
	String filename = Java.getJavaImplSourceFile(id);
	System.out.println("Impl file: " + filename);
	
	System.out.println("******************* Set up - Java Impl **************************");
	// Set up the writer and the splicer
	try{
	    // To generate _Impl file in correct directory
	    FileManager.getInstance().setJavaStylePackageGeneration(true);

	    d_splicer = FileManager.getInstance().getCodeSplicer(id, type, filename);
	    System.out.println("Got Splicer");
	    d_writer = new LanguageWriterForJava((FileManager.getInstance()).createFile(id,
											type,
											"JAVAIMPL",
											filename));
	    System.out.println("Made Writer");
	    FileManager.getInstance().setJavaStylePackageGeneration(false);
	    d_writer.writeBanner(d_ext, filename,true, CodeConstants.C_DESC_IMPL_PREFIX + id.getFullName());

	} catch(java.io.IOException e){
	    if (d_writer != null) {
		d_writer.close();
		d_writer = null;
	    }
	    throw new CodeGenerationException("IOException : " + e.getMessage() );
	}
    }
  
    /**
     * Writes Java implementation based on the provided symbol
     *
     * @param ext   an interface or class symbol that needs source
     *              file for a Java extension class.
     */
    public static void generateCode(Extendable ext) 
	throws CodeGenerationException
    {
	JavaImplSource source = new JavaImplSource(ext);
	source.generateCode();
    }
    

    /**
     * Writes Java implimentation based on the symbol given to the constructor
     */
    public synchronized void generateCode() 
	throws CodeGenerationException 
    {
	System.out.println("******************* Generate code - Java Impl **************************");
	
	writePackageImports();
	writeClassBeginning();
        writeCtorDtor();
        writeSIDLDefinedMethods();
        d_splicer.splice( d_ext.getSymbolID().getFullName() + "._misc",
			  d_writer, "Put miscellaneous code here");
	d_writer.println();
        writeClassEnd();
        checkSplicer();
	
	d_writer.close();	
    }



    /********************************************************************************************
     *                           Private support methods
     ********************************************************************************************/

    /**
     * Writes the necessary package and import statements needed
     */
    private void writePackageImports() {
	writePackage();
        writeImports();
	spliceImports();
    }
    
    private void writePackage() {
	String pkg = SymbolUtilities.getParentPackage(d_ext.getSymbolID().getFullName());
	d_writer.println("package " + pkg + ";");
	d_writer.println();
    }
    
    private void writeImports() {
	Iterator i = d_ext.getSymbolReferences().iterator();
	while (i.hasNext()) {
	    SymbolID id = (SymbolID) i.next();
	    if (! id.getFullName().equals(d_ext.getSymbolID().getFullName())) {
		d_writer.println("import " + id.getFullName() + ";");
	    }
    }
	d_writer.println();
    }
    
    private void spliceImports() {
	d_splicer.splice(d_ext.getSymbolID().getFullName() + "._imports",
			 d_writer, "Put additional imports here...");
	d_writer.println();
    }

    /**
     * Writes class declaration and splicer block for private data
     */
    private void writeClassBeginning() {
	String name = d_ext.getSymbolID().getShortName();
	SymbolID id = d_ext.getSymbolID();
	String splicer_symbol_data = d_ext.getSymbolID().getFullName() + "._data";
	String splicer_symbol_inherit = d_ext.getSymbolID().getFullName() + "._inherits";
	
	d_writer.writeComment(d_ext, true);
	d_writer.println("public class " + d_classname);
	d_splicer.splice(splicer_symbol_inherit, d_writer, 
			 "Put additional inheritance here...");
	d_writer.println("{");
	d_writer.println();
	d_writer.increaseTabLevel();
	d_writer.writeCommentLine("Pointer back to IOR.");
	d_writer.writeCommentLine("Use this to dispatch back through IOR vtable");
	d_writer.println("private " + name + " IORself;");
	d_writer.println();
	
	d_splicer.splice(splicer_symbol_data, d_writer, 
			 "Put additional private data here...");
	d_writer.println();
    }
    
   
    
    /**
     * Writes Java constructor and a "destructor", method that can be utilized to 
     * "destruct" a Java object.
     */
    private void writeCtorDtor() {
	String name = d_ext.getSymbolID().getShortName();
	// Constructor
	d_writer.writeComment("User defined constructor", true);
	d_writer.println("public " +d_classname + "(){");
	d_writer.increaseTabLevel();
	d_splicer.splice(d_ext.getSymbolID().getFullName() + "." + name,
			 d_writer,
			 "add construction details here");
	d_writer.decreaseTabLevel();
	d_writer.println("}");
	d_writer.println();
	
	// "destructor"
	d_writer.writeComment("User defined destructing method", true);
	d_writer.println("public void dtor() {");
	d_writer.increaseTabLevel();
	d_splicer.splice(d_ext.getSymbolID().getFullName() + "._dtor",
			 d_writer,
			   "add destruction details here");
	d_writer.decreaseTabLevel();
	d_writer.println("}");
	d_writer.println();		     
    }
    
    /**
     * Writes both static and nonstatic methods defined in the SIDL file
     */
    private void writeSIDLDefinedMethods() {
	List static_methods = (List) d_ext.getStaticMethods(false);
	if (static_methods.size() > 0) {
	    d_writer.writeCommentLine("user defined static methods:");
	    for (Iterator m = static_methods.iterator(); m.hasNext(); ) {
		Method method = (Method) m.next();
		generateMethod(method);
	    }
	} 
	else {
	    d_writer.writeCommentLine("user defined static methods: (none)");
	}
	
	d_writer.println();
	List nonstatic_methods = (List) d_ext.getNonstaticMethods(false);
	if (nonstatic_methods.size() > 0) {
	    d_writer.writeCommentLine("user defined non-static methods:");
	    for (Iterator m = nonstatic_methods.iterator(); m.hasNext(); ) {
		Method method = (Method) m.next();
		if (!method.isAbstract()) {
		    System.out.println("Generating method: " + method.getShortMethodName());
		    generateMethod(method);
		}
	    }
	} 
	else {
	    d_writer.writeCommentLine("user defined non-static methods: (none)");
	}
	d_writer.println();
    }
    
    /**
     * Writes complete method header and comments for given parameter. 
     * Includes a default return if necessary.
     */
    private void generateMethod( Method method ) 
    {
	if ( method == null ) { return; }
	
	generateMethodBeginning( method );
	
	if (method.getArgumentList().size() > 0) {
	    generateArgumentList(method);  
	} 
	else {
	    d_writer.println(" () ");
	}

	if(method.getThrows().size() > 0) {
	    generateThrowsList(method); 
	}
	generateMethodBody(method); 
    }


    private void generateMethodBeginning ( Method method )
    {
	d_writer.writeComment (method, true);
	d_writer.print("public ");
	if(method.isStatic()){
	    d_writer.print("static ");
	}
	d_writer.print(Java.getJavaReturnType(method.getReturnType()) + " ");
	d_writer.print(method.getShortMethodName());
    }

    private void generateArgumentList( Method method )
    {
	if ( method == null ) { return; }
   
	List args = method.getArgumentList();

	d_writer.println(" (");
	d_writer.increaseTabLevel();
	for( Iterator a = args.iterator(); a.hasNext(); ) { 
	    Argument arg = (Argument) a.next();
	    d_writer.print( Java.getJavaArgument( arg )+ " " + arg.getFormalName());
	    if ( a.hasNext() ) { 
		d_writer.println(",");
	    }
	}
	d_writer.println(" ) ");
	d_writer.decreaseTabLevel();
    }
    
    private void generateThrowsList( Method method )  
    {
	if ( method == null ) { return; }	
 
	Set exceptions = method.getThrows();
	
	d_writer.print( "throws " );
	d_writer.increaseTabLevel();
	for( Iterator e = exceptions.iterator(); e.hasNext(); ) { 
	    SymbolID id = (SymbolID) e.next();
	    d_writer.print( Java.getFullJavaSymbolName(id) );
	    if ( e.hasNext() ) { 
		d_writer.println(", ");
	    }
	}
	d_writer.println();
	d_writer.decreaseTabLevel();
	
    }
    

    private void generateMethodBody ( Method method ) 
    {
	d_writer.println("{");
	d_writer.increaseTabLevel();
	d_splicer.splice(d_ext.getSymbolID().getFullName() + "." + method.getLongMethodName(),
			 d_writer, "insert implementation here", 
			 "return " + Java.getDefaultReturnValue(method) + ";");
	d_writer.decreaseTabLevel();
	d_writer.println("}");
	d_writer.println();

    }

  
    /**
     * Closes out class
     */
    private void writeClassEnd() 
    {
	String name = d_ext.getSymbolID().getShortName();
	d_writer.decreaseTabLevel();
	d_writer.print("} "); // end of class
	d_writer.writeCommentLine("end class " + name );
	d_writer.println();
    }

    /**
     * Print any remaining symbols in the splicer
     */
    private void checkSplicer() 
    {
	if (d_splicer.hasUnusedSymbolEdits()) {
	    d_writer.beginBlockComment(true);
	    d_writer.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
	    d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT1);
	    d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT2);
	    d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT3);
	    d_writer.endBlockComment(true);
	    d_splicer.outputUnusedSymbolEdits( d_writer.getPrintWriter() );
	    d_writer.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
	}
    }
}
