//
// File:        Python.java
// Package:     gov.llnl.babel.backend.python
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.34 $
// Date:        $Date: 2003/09/29 20:43:10 $
// Description: Python naming conventions shared by Python code generators
// 
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.python;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.writers.LanguageWriterForPython;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Iterator;

/**
 * <p>
 * This class defines some of the fundamental mapping rules for translating
 * a symbol in the SIDL file into Python. These fundamental mapping rules
 * are used repeatedly in the generation of client and server side bindings,
 * so they are seperated into a class that can be shared by all Python code
 * generators.
 * </p>
 * <p>
 * Some of the particular features include:
 * <ul>
 * <li>Provide the header file name</li>
 * </ul>
 * </p>
 */
public class Python {
  /**
   * The class is going to be generating a number of files, so we keep
   * a reference to a file manager.  This is primarily a keystroke
   * saving decision to avoid having FileManager.getInstance() everytime 
   * a file manager is needed.  It also keeps this class from
   * assuming that FileManager is a singleton.
   */
  private static FileManager s_fileMan = FileManager.getInstance();

  /**
   * Generate an include file for a symbol.
   */
  public static String getIncludeGuard(Symbol symbol,
                                       String modifier)
  {
    String fullname = symbol.getFullName().replace('.','_');
    StringBuffer buf = 
      new StringBuffer(2 + fullname.length() + modifier.length());
    buf.append(fullname)
      .append('_')
      .append(modifier.toUpperCase());
    return buf.toString();
  }

  public static String headerFilename(Symbol symbol,
                                      String modifier)
  {
    SymbolID id = symbol.getSymbolID();
    String name = id.getFullName().replace('.','_');
    StringBuffer buf = 
      new StringBuffer(name.length() + modifier.length() + 3);
    buf.append(name).append('_').append(modifier).append(".h");
    return buf.toString();
  }

  public static String sourceFilename(Symbol symbol,
                                      String modifier)
  {
    SymbolID id = symbol.getSymbolID();
    String longname = id.getFullName().replace('.','_');
    StringBuffer buf = 
      new StringBuffer(longname.length() + modifier.length() + 3);
    buf.append(longname).append('_').append(modifier).append(".c");
    return buf.toString();
  }

  public static String skelFilename(Symbol symbol,
                                    String modifier)
  {
    final String name = symbol.getFullName().replace('.','_');
    StringBuffer buf = 
      new StringBuffer(name.length() + modifier.length() + 3);
    buf.append(name).append('_').append(modifier).append(".c");
    return buf.toString();
  }

  public static String implFilename(Symbol symbol)
  {
    return symbol.getSymbolID().getShortName() + "_Impl.py";
  }

  public static String getAPIVarName(Symbol symbol)
  {
    String fullname = symbol.getFullName().replace('.','_');
    StringBuffer buf = new StringBuffer(fullname.length() + 5);
    buf.append(fullname).append("__API");
    return buf.toString();
  }

  public static String getInternalGuard(Symbol symbol)
  {
    String fullname = symbol.getFullName().replace('.','_');
    StringBuffer buf = 
      new StringBuffer(fullname.length() + 9);
    buf.append(fullname)
      .append("_INTERNAL");
    return buf.toString();
  }

  /**
   * Build a Python support object.
   */
  public Python()
  {
  }
   
  public static String getCHeaderPath(Symbol symbol,
                                      String modifier)
  {
    return headerFilename(symbol, modifier);
  }

  public static LanguageWriterForC createCHeader(Symbol symbol,
                                                 String modifier,
                                                 String description)
    throws CodeGenerationException
  {
    final SymbolID id = symbol.getSymbolID();
    final int type = symbol.getSymbolType();
    String filename = headerFilename(symbol, modifier);
    LanguageWriterForC lw = null;
    lw = new LanguageWriterForC
        (s_fileMan.createFile(id, type, "PYMOD_HDRS", filename));
    lw.writeBanner(symbol, filename, CodeConstants.C_IS_IMPL, 
                   description);
    return lw;
  }
   
  /**
   * Generate an IO stream to receive the C stub file for the Python clients.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public static LanguageWriterForC createStub(Symbol symbol,
                                              String description)
    throws CodeGenerationException
  {
    final boolean saveState = s_fileMan.getJavaStylePackageGeneration();
    final SymbolID id = symbol.getSymbolID();
    final int type = symbol.getSymbolType();
    String filename = sourceFilename(symbol, "Module");
    LanguageWriterForC lw = null;
    try {
      s_fileMan.setJavaStylePackageGeneration(true);
      lw = new LanguageWriterForC
        (s_fileMan.createFile(id, type, "PYMOD_SRCS", filename));
      lw.writeBanner(symbol, filename, CodeConstants.C_IS_NOT_IMPL, 
                     description);
    }
    finally {
      s_fileMan.setJavaStylePackageGeneration(saveState);
    }
    return lw;
  }

  /**
   * Generate an IO stream to receive the C skeleton file for the Python
   * implementations.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public static LanguageWriterForC createSkel(Symbol symbol,
                                              String description)
    throws CodeGenerationException
  {
    final SymbolID id = symbol.getSymbolID();
    final int type = symbol.getSymbolType();
    String filename = skelFilename(symbol, "pSkel");
    LanguageWriterForC lw = null;
    lw = new LanguageWriterForC
      (s_fileMan.createFile(id, type, "SKELSRCS", filename));
    lw.writeBanner(symbol, filename, CodeConstants.C_IS_NOT_IMPL,
                   description);
    return lw;
  }

  /**
   * Generate an IO stream to receive the C skeleton file for the Python
   * implementations.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public static LanguageWriterForC createLaunch(Symbol symbol,
                                                String description)
    throws CodeGenerationException
  {
    String filename = skelFilename(symbol, "pLaunch");
    final SymbolID id = symbol.getSymbolID();
    final int type = symbol.getSymbolType();
    LanguageWriterForC lw = null;
    lw = new LanguageWriterForC
      (s_fileMan.createFile(id, type, "LAUNCHSRCS", filename));
    lw.writeBanner(symbol, filename, CodeConstants.C_IS_NOT_IMPL,
                   description);
    return lw;
  }

  /**
   * Return the name of the function that should be used for the
   * skeleton.
   *
   * @param id  the symbol who owns the method.
   * @param m   the method
   */
  public static String getSkelMethod(SymbolID id, Method m)
  {
    return "pSkel_" + id.getShortName() + "_" + m.getLongMethodName();
  }

  /**
   * Return the name of the function that should be used for the
   * Python stub code.
   *
   * @param id  the symbol who owns the method.
   * @param m   the method
   */
  public static String getStubMethod(SymbolID id, Method m)
  {
    return "pStub_" + id.getShortName() + "_" + m.getLongMethodName();
  }

  /**
   * If <code>filename</code> already exists, extract the code splicer
   * blocks from it and store the contents in the returned code splicer;
   * otherwise, return an empty code splicer.
   * 
   * @param symbol   the symbol whose splicer is to be returned.
   * @param filename the name of the file
   * @return a valid (though possibly empty) <code>CodeSplicer</code>
   * @exception java.io.IOException
   *   where there is IO, there is the possibility for an
   *   <code>IOException</code>.
   */
  public static CodeSplicer getPySplicer(Symbol symbol, 
                                         String filename)
    throws java.io.IOException
  {
    final boolean saveState = s_fileMan.getJavaStylePackageGeneration();
    try {
      s_fileMan.setJavaStylePackageGeneration(true);
      return s_fileMan.getCodeSplicer(symbol.getSymbolID(),
                                      symbol.getSymbolType(),
                                      filename);
    }
    finally {
      s_fileMan.setJavaStylePackageGeneration(saveState);
    }
  }
   
  /**
   * Create a Python <code>LanguageWriterForPython</code> with a banner
   * comment a documentation string in the <code>FileManager</code> group
   * PYTHON.
   * 
   * @param symbol	 the symbol for which the
   *                    <code>LanguageWriter</code> is being created.
   * @param file	 the name of the file to be created. This contains
   *                    no directory references.
   * @param description a brief statement of the purpose of the file.
   *                    This string should have no newlines.
   * @exception gov.llnl.backend.CodeGenerationException
   *    something went wrong while trying to create the file.
   */
  public static LanguageWriterForPython createPyWriter(Symbol symbol,
                                                       String file,
                                                       String description)
    throws CodeGenerationException
  {
    final boolean saveState = s_fileMan.getJavaStylePackageGeneration();
    LanguageWriterForPython out = null;
    try {
      s_fileMan.setJavaStylePackageGeneration(true);
      if (file.charAt(0) != '_') {
        out = new LanguageWriterForPython
          (s_fileMan.createFile(symbol.getSymbolID(),
                                symbol.getSymbolType(), 
                                "PYTHONSRC", file));
      }
      else {
        out = new LanguageWriterForPython
          (s_fileMan.createFile(symbol.getSymbolID(),
                                symbol.getSymbolType(),
                                "PYTHONADMIN", file));
      }
    }
    finally {
      s_fileMan.setJavaStylePackageGeneration(saveState);
    }
    Comment comment = symbol.getComment();
    out.writeBanner(symbol, file, CodeConstants.C_IS_NOT_IMPL,
                    description);
    if (comment != null){
      String [] lines = comment.getComment();
      if (lines != null){
        // print the comment as a Python DOC string
        out.println();
        out.print("\"\"\"");
        for(int i = 0; i < lines.length; i++){
          out.println(LanguageWriterForPython.toPythonString(lines[i]));
        }
        out.println("\"\"\"");
      }
    }
    out.println();
    return out;
  }

   /**
    * Convert a SIDL symbol into the name of its associated set EPV
    * method, which is the symbol name appended with "__set_epv".
    */
   public static String getSetEPVName(SymbolID id) {
      return IOR.getSymbolName(id) + "__impl_set_epv";
   }

   /**
    * Convert a SIDL symbol into the name of its associated set static
    * EPV method, which is the symbol name appended with "__set_sepv".
    */
   public static String getSetSEPVName(SymbolID id) {
      return IOR.getSymbolName(id) + "__impl_set_sepv";
   }

  public static String getExtendableImport(Symbol symbol) {
    String symbolName = symbol.getFullName().replace('.', '_');
    StringBuffer buf = new StringBuffer(symbolName.length() + 8);
    buf.append(symbolName)
      .append("__import");
    return buf.toString();
  }

  public static String getExtendableWrapper(Symbol symbol) {
    String symbolName = symbol.getFullName();
    StringBuffer buf = new StringBuffer(symbolName.length() + 7);
    buf.append(symbolName.replace('.','_'))
      .append("__wrap");
    return buf.toString();
  }

  public static String getExtendableBorrow(Symbol symbol) {
    String symbolName = symbol.getFullName();
    StringBuffer buf = new StringBuffer(symbolName.length() + 9);
    buf.append(symbolName.replace('.','_'))
      .append("__weakRef");
    return buf.toString();
  }

  public static String getExceptionType(Symbol symbol) {
    String symbolName = symbol.getFullName();
    StringBuffer buf = new StringBuffer(symbolName.length() + 7);
    buf.append(symbolName.replace('.', '_'))
      .append("__type");
    return buf.toString();
  }

  public static String getExtendableConverter(Symbol symbol) {
    String symbolName = symbol.getFullName();
    StringBuffer buf = new StringBuffer(symbolName.length() + 7);
    buf.append(symbolName.replace('.','_'))
      .append("__convert");
    return buf.toString();
  }

  public static String getExtendableNewRef(Symbol symbol) {
    String symbolName = symbol.getFullName();
    StringBuffer buf = new StringBuffer(symbolName.length() + 7);
    buf.append(symbolName.replace('.','_'))
      .append("__newRef");
    return buf.toString();
  }

  public static String getExtendableAddRef(Symbol symbol) {
    String symbolName = symbol.getFullName();
    StringBuffer buf = new StringBuffer(symbolName.length() + 7);
    buf.append(symbolName.replace('.','_'))
      .append("__addRef");
    return buf.toString();
  }

  public static String getExtendableDeref(Symbol symbol) {
    return symbol.getFullName().replace('.','_') + "_deref";
  }

  public static String getBorrowArrayFromPython(Type arrayType) {
    if (arrayType.getType() != Type.SYMBOL) {
      return "SIDL_" + arrayType.getTypeString() + "__borrow_python_array";
    }
    else {
      return arrayType.getSymbolID().getFullName().replace('.', '_')
        + "__convert_python_array";
    }
  }

  public static String getBorrowArrayFromSIDL(Type arrayType) {
    if (arrayType.getType() != Type.SYMBOL) {
      return "SIDL_" + arrayType.getTypeString() + "__python_borrow_array";
    }
    else {
      return arrayType.getSymbolID().getFullName().replace('.', '_')
        + "__convert_sidl_array";
    }
  }

  public static String getCopyArrayFromPython(Type arrayType) {
    final Type elemType = arrayType.getArrayType();
    if (elemType.getType() != Type.SYMBOL) {
      if (arrayType.getArrayOrder() == Type.COLUMN_MAJOR) {
        return "SIDL_" + elemType.getTypeString() + 
          "__clone_python_array_column";
      }
      return "SIDL_" + elemType.getTypeString() + 
        "__clone_python_array_row";
    }
    else {
      return elemType.getSymbolID().getFullName().replace('.', '_')
        + "__convert_python_array";
    }
  }

  public static String getCopyArrayFromSIDL(Type arrayType) {
    if (arrayType.getType() != Type.SYMBOL) {
      return "SIDL_" + arrayType.getTypeString() + "__python_clone_array";
    }
    else {
      return arrayType.getSymbolID().getFullName().replace('.', '_')
        + "__convert_sidl_array";
    }
  }

  public static String getDestroyArray(Type arrayType) {
    if (arrayType.getType() != Type.SYMBOL) {
      return "SIDL_" + arrayType.getTypeString() + "__python_deleteRef_array";
    }
    else {
      return arrayType.getSymbolID().getFullName().replace('.', '_')
        + "__destroy_sidl_array";
    }
  }
}
