//
// File:        ParseSymbolXML.java
// Package:     gov.llnl.babel.parsers.xml
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: ParseSymbolXML.java,v 1.17 2003/09/18 14:46:47 epperly Exp $
// Description: convert SIDL symbol XML/DOM into a SIDL symbol object
//
// Copyright (c) 2000-2003, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.parsers.xml;

import gov.llnl.babel.parsers.xml.DTDManager;
import gov.llnl.babel.parsers.xml.ParseSymbolException;
import gov.llnl.babel.parsers.xml.StringXML;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Metadata;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Package;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolRedefinitionException;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import gov.llnl.babel.xml.ElementIterator;
import gov.llnl.babel.xml.XMLUtilities;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Iterator;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Class <code>ParseSymbolXML</code> converts an XML document into a SIDL
 * symbol.  Utility function <code>convert</code> takes either an input
 * stream or a validated DOM tree and generates a SIDL symbol.  Any errors
 * in format generate a <code>ParseSymbolException</code>.
 */
public class ParseSymbolXML {
   private static final String EOL = "\n"; // standard web end-of-line

   private Symbol d_symbol;

   /**
    * This is a convenience utility function that converts an XML input
    * stream into a SIDL symbol.  Any errors detected in the XML input
    * are converted into a <code>ParseSymbolException</code>.  Since this
    * method is static, it may be called without explicity creating an
    * object.
    */
   public static Symbol convert(InputSource is) throws ParseSymbolException {
      ParseSymbolXML xml2sym = new ParseSymbolXML(is);
      return xml2sym.getSymbol();
   }

   /**
    * This is a convenience utility function that converts a DOM document
    * into a SIDL symbol.  This method assumes that the DOM document is a
    * valid symbol representation.  Any errors detected in the DOM input
    * are converted into a <code>ParseSymbolException</code>.  Since this
    * method is static, it may be called without explicity creating an
    * object.
    */
   public static Symbol convert(Document doc) throws ParseSymbolException {
      ParseSymbolXML doc2sym = new ParseSymbolXML(doc);
      return doc2sym.getSymbol();
   }

   /**
    * Create an XML input stream to SIDL symbol converter object.  The
    * constructor parses the XML input stream and then converts the DOM
    * structure into a symbol.  The resulting symbol may be read by a call
    * to <code>getSymbol</code>.  Any errors will throw a parse symbol
    * exception.
    */
   public ParseSymbolXML(InputSource is) throws ParseSymbolException {
      Document document = null;
      try {
         document = XMLUtilities.parse(is, DTDManager.getInstance());
      } catch (IOException ex) {
         throw new ParseSymbolException(ex.getMessage());
      } catch (SAXException ex) {
         throw new ParseSymbolException(ex.getMessage());
      }
      parseSymbol(document.getDocumentElement());
   }

   /**
    * Create a DOM document to SIDL symbol converter object.  The DOM
    * document must be a valid representation of a SIDL symbol.  The
    * resulting symbol may be read by a call to <code>getSymbol</code>.
    * Any errors will throw a parse symbol exception.
    */
   public ParseSymbolXML(Document doc) throws ParseSymbolException {
      parseSymbol(doc.getDocumentElement());
   }

   /**
    * Return the SIDL symbol for the XML or DOM given in the constructor.
    */
   public Symbol getSymbol() {
      return d_symbol;
   }

   /**
    * Throw an exception of type <code>ParseSymbolException</code> based
    * on the argument message string.
    */
   private void error(String message) throws ParseSymbolException {
      throw new ParseSymbolException(message);
   }

   /**
    * Parse the DOM element (which must be an element of type "Symbol") and
    * create a <code>Symbol</code> object as a data member of the class.
    */
   private void parseSymbol(Element symbol) throws ParseSymbolException {
      if ((symbol == null) || !("Symbol".equals(symbol.getNodeName()))) {
         error("No <Symbol> root element found in XML/DOM document");
      }

      SymbolID id = parseSymbolID(getElement(symbol, "SymbolName"));
      Comment  cm = parseComment (getElement(symbol, "Comment"   ));
      Metadata md = parseMetadata(getElement(symbol, "Metadata"  ));

      if (hasChildElement(symbol, "Class")) {
         d_symbol = new Class(id, cm, md);
         parseClass(getElement(symbol, "Class"));

      } else if (hasChildElement(symbol, "Enumeration")) {
         d_symbol = new Enumeration(id, cm, md);
         parseEnum(getElement(symbol, "Enumeration"));
         
      } else if (hasChildElement(symbol, "Interface")) {
         d_symbol = new Interface(id, cm, md);
         parseInterface(getElement(symbol, "Interface"));
         
      } else if (hasChildElement(symbol, "Package")) {
         d_symbol = new Package(id, cm, md);
         parsePackage(getElement(symbol, "Package"));
         
      } else {
         error("Unknown SIDL symbol type in XML document");
      }
   }

   /**
    * Convert a DOM class element into a SIDL class symbol.  A class
    * description consists of an extends symbol, an implements block,
    * and a methods block.
    */
   private void parseClass(Element element) throws ParseSymbolException {
      validateSymbolName(element, "Class");
      Class cls = (Class) d_symbol;

      /*
       * Parse the single optional extends symbol (which must be a class).
       */
      Element extends_element = getElement(element, "Extends");
      Iterator parent = new ElementIterator(extends_element, "SymbolName");
      if (parent.hasNext()) {
         SymbolID id = parseSymbolID((Element) parent.next());
         try {
            Symbol sym = SymbolTable.getInstance().resolveSymbol(id);
            if (sym == null) {
               error("Class \"" + id.getSymbolName() + "\" not found");
            } else if (sym.getSymbolType() != Symbol.CLASS) {
               error("Symbol \"" + id.getSymbolName() + "\"  not a class");
            } else {
               cls.setParentClass((Class) sym);
            }
         } catch (SymbolRedefinitionException ex) {
            error(ex.getMessage());
         }
      }

      /*
       * Parse the implements block of interfaces.
       */
      Element implblock_element = getElement(element, "ImplementsBlock");
      Iterator parents = new ElementIterator(implblock_element, "SymbolName");
      while (parents.hasNext()) {
         SymbolID id = parseSymbolID((Element) parents.next());
         try {
            Symbol sym = SymbolTable.getInstance().resolveSymbol(id);
            if (sym == null) {
               error("Interface \"" + id.getSymbolName() + "\" not found");
            } else if (sym.getSymbolType() != Symbol.INTERFACE) {
               error("Symbol \"" + id.getSymbolName() + "\"  not interface");
            } else {
               cls.addParentInterface((Interface) sym);
            }
         } catch (SymbolRedefinitionException ex) {
            error(ex.getMessage());
         }
      }

      /*
       * Parse the methods block and add the methods to the class.
       */
      Element methodsblock_element = getElement(element, "MethodsBlock");
      Iterator methods = new ElementIterator(methodsblock_element, "Method");
      while (methods.hasNext()) {
         cls.addMethod(parseMethod((Element) methods.next()));
      }
   }

   /**
    * Convert a DOM enumeration element into a SIDL enumeration symbol.
    * Parse each of the enumerator elements and add the enumerator symbol
    * to the enumeration.
    */
   private void parseEnum(Element element) throws ParseSymbolException {
      validateSymbolName(element, "Enumeration");
      Enumeration enm = (Enumeration) d_symbol;

      Iterator enumerators = new ElementIterator(element, "Enumerator");
      while (enumerators.hasNext()) {
         Element entry = (Element) enumerators.next();
         String  name  = getAttribute(entry, "name");
         String  sval  = getAttribute(entry, "value");
         boolean user  = getAttribute(entry, "fromuser").equals("true");
         Comment cmt   = null;
         if (hasChildElement(entry, "Comment")) {
           cmt = parseComment(getElement(entry, "Comment"));
         }
         try {
            enm.addEnumerator(name, Integer.parseInt(sval), user, cmt);
         } catch (NumberFormatException ex) {
            error("Invalid integer value \""+sval+"\" for enumerated type");
         }
      }
   }

   /**
    * Convert a DOM interface element into a SIDL interface symbol.
    * An interface description consists of an extends block and a
    * methods block.
    */
   private void parseInterface(Element element) throws ParseSymbolException {
      validateSymbolName(element, "Interface");
      Interface ifc = (Interface) d_symbol;

      /*
       * Parse the extends block consisting of a collection of interfaces.
       */
      Element extends_block = getElement(element, "ExtendsBlock");
      Iterator parents = new ElementIterator(extends_block, "SymbolName");
      while (parents.hasNext()) {
         SymbolID id = parseSymbolID((Element) parents.next());
         try {
            Symbol sym = SymbolTable.getInstance().resolveSymbol(id);
            if (sym == null) {
               error("Interface \"" + id.getSymbolName() + "\" not found");
            } else if (sym.getSymbolType() != Symbol.INTERFACE) {
               error("Symbol \"" + id.getSymbolName() + "\"  not interface");
            } else {
               ifc.addParentInterface((Interface) sym);
            }
         } catch (SymbolRedefinitionException ex) {
            error(ex.getMessage());
         }
      }

      /*
       * Parse the methods block and add the methods to the interface.
       */
      Element methods_block = getElement(element, "MethodsBlock");
      Iterator methods = new ElementIterator(methods_block, "Method");
      while (methods.hasNext()) {
         ifc.addMethod(parseMethod((Element) methods.next()));
      }
   }

  /**
   * If the version attribute isn't specified, use the parent version.
   */
  static private Version chooseVersion(String version, Version parent)
  {
    if (version == null) return parent;
    return new Version(version);
  }

   /**
    * Convert a DOM package element into a SIDL package symbol.  For each
    * of the sub-elements of package <code>PackageSymbol</code>, parse the
    * name and type and create a package entry.
    */
   private void parsePackage(Element element) throws ParseSymbolException {
      validateSymbolName(element, "Package");
      Package p = (Package) d_symbol;
      Version pkgVersion = p.getSymbolID().getVersion();
      p.setFinal("true".equals(getAttribute(element, "final")));

      try {
        Iterator entries = new ElementIterator(element, "PackageSymbol");
        while (entries.hasNext()) {
          Element entry = (Element) entries.next();
          String name = getAttribute(entry, "name");
          String type = getAttribute(entry, "type");
          String version = getAttribute(entry, "version");

          p.addSymbol(new SymbolID
                      (p.getScopedName(name), 
                       chooseVersion(version, pkgVersion)),
                      StringXML.fromSymbolXML(type));
        }
      }
      catch (NumberFormatException nfe) {
        new ParseSymbolException("Bad version string: " + nfe.getMessage());
      }
   }

   /**
    * Convert a SIDL comment DOM element into a SIDL comment object.
    * Comments consist of a number of comment lines.  This algorithm
    * takes the comment DOM, prints it into a string, and then extracts
    * the comment lines, ignoring the first and last lines, which contain
    * comment tags.
    * 
    */
   private Comment parseComment(Element comment) throws ParseSymbolException {
      validateSymbolName(comment, "Comment");
      /* 
       * Note that it takes the XML encoded string and then encodes it once more
       * in printing it out.  It actually gets encoded a few more times (don't know where)
       * so I decode multiple times.
       */

      String s = XMLUtilities.decodeXMLString(
		   XMLUtilities.decodeXMLString(
                     XMLUtilities.decodeXMLString(
  		       XMLUtilities.decodeXMLString( XMLUtilities.getXMLString(comment) )
		       )
		     )
		   );

      /*
       * Count the number of newlines in the generated text.
       */
      int nlines = 0;
      int index  = s.indexOf(EOL);
      while(index > 0) {
         nlines++;
         index = s.indexOf(EOL, index+1);
      }

      /*
       * If there are comment lines, create a comment line array and extract
       * the comment subsequences from the generated text.
       */
      String[] lines = null;
      if (nlines > 1) {
         lines = new String[nlines-1];
         int start_index = s.indexOf(EOL);
         for (int n = 0; n < nlines-1; n++) {
            int end_index = s.indexOf(EOL, start_index+1);
            lines[n] = s.substring(start_index+1, end_index);
            start_index = end_index;
         }
      }

      return new Comment(lines);
   }
   
   /**
    * Recover the metadata object from its DOM representation.  The element
    * name is <code>Metadata</code> with attribute <code>date</code>.  There
    * may be sub-elements that are (key,value) pairs.
    */
   private Metadata parseMetadata(Element m) throws ParseSymbolException {
      validateSymbolName(m, "Metadata");

      Metadata metadata = null;
      String date = getAttribute(m, "date");
      try {
         metadata = new Metadata(date);
      } catch (ParseException ex) {
         error("Invalid date format \"" + date + "\" in metadata element");
      }

      Iterator entries = new ElementIterator(m, "MetadataEntry");
      while (entries.hasNext()) {
         Element entry = (Element) entries.next();
         String key = getAttribute(entry, "key");
         String val = getAttribute(entry, "value");
         metadata.addMetadata(key, val);
      }

      return metadata;
   }

   /**
    * Recover the method object from its DOM representation.  The element
    * name is <code>Method</code> with sub-elements for the argument list,
    * return type, and throws clause.
    */
   private Method parseMethod(Element e) throws ParseSymbolException {
      validateSymbolName(e, "Method");
      Method method = new Method();

      /*
       * Set method name, modifiers, return value, and comment
       */
      method.setMethodName(getAttribute(e, "shortname"), 
                           getAttribute(e, "extension"));
      method.setCommunicationModifier(
         StringXML.fromComXML(getAttribute(e, "communication")));
      method.setDefinitionModifier(
         StringXML.fromDefXML(getAttribute(e, "definition")));
      method.setReturnCopy(getAttribute(e, "copy").equals("true"));

      method.setComment(parseComment(getElement(e, "Comment")));
      method.setReturnType(parseType(getElement(e, "Type")));

      /*
       * Parse the argument list from the XML document
       */
      Element arglist = getElement(e, "ArgumentList");
      Iterator args = new ElementIterator(arglist, "Argument");
      while (args.hasNext()) {
         method.addArgument(parseArgument((Element) args.next()));
      }

      /*
       * Parse the throws clause from the XML document
       */
      Element throwslist = getElement(e, "ThrowsList");
      Iterator thrws = new ElementIterator(throwslist, "SymbolName");
      while (thrws.hasNext()) {
         method.addThrows(parseSymbolID((Element) thrws.next()));
      }

      return method;
   }

   /**
    * Convert a DOM argument element into a <code>Argument</code> object.
    * Throw a parse exception if there are any errors in format.
    */
   private Argument parseArgument(Element e) throws ParseSymbolException {
      validateSymbolName(e, "Argument");
      boolean copy = getAttribute(e, "copy").equals("true");
      int     mode = StringXML.fromModeXML(getAttribute(e, "mode"));
      String  name = getAttribute(e, "name");
      Type    type = parseType(getElement(e, "Type"));
      return new Argument(copy, mode, type, name);
   }

   /**
    * Convert a DOM type element into a <code>Type</code> object.  If
    * there are any format errors in the type element, then throw a parse
    * symbol exception.
    */
   private Type parseType(Element e) throws ParseSymbolException {
      validateSymbolName(e, "Type");
      int typeid = StringXML.fromTypeXML(getAttribute(e, "type"));

      Type type = null;
      if (typeid == Type.SYMBOL) {
         type = new Type(parseSymbolID(getElement(e, "SymbolName")));
      } else if (typeid == Type.ARRAY) {
         Element array = getElement(e, "Array");
         String sdim   = getAttribute(array, "dim");
         String sorder = getAttribute(array, "order");
         int dim       = 0;
         int order     = StringXML.fromOrderXML(sorder);
         try {
            dim = Integer.parseInt(sdim);
         } catch (NumberFormatException ex) {
            error("Invalid array dimension \"" + sdim + "\"");
         }
         type = new Type(parseType(getElement(array, "Type")), dim, order);
      } else {
         type = new Type(typeid);
      }

      return type;
   }

   /**
    * Convert a DOM symbol element into a <code>SymbolID</code> object.
    * If there are any format errors in the DOM element, then throw a
    * <code>ParseSymbolException</code>.
    */
   private SymbolID parseSymbolID(Element id) throws ParseSymbolException {
      validateSymbolName(id, "SymbolName");
      String name = getAttribute(id, "name");
      String vers = getAttribute(id, "version");

      SymbolID sid = null;
      try {
         sid = new SymbolID(name, new Version(vers));
      } catch (NumberFormatException ex) {
         error("Invalid version format \""+vers+"\" in symbol name");
      }
      return sid;
   }

   /**
    * Extract the specified element node from the parent element.  If the
    * child element does not exist, then throw a parse symbol exception.
    */
   private Element getElement(Element parent, String child)
         throws ParseSymbolException {
      Element element = XMLUtilities.lookupElement(parent, child);
      if (element == null) {
         error("Child element \"" + child + "\" not found in parent \""
             + parent.getTagName() + "\"");
      }
      return element;
   }

   /**
    * Determine whether the element has a child of the specified name.
    */
   private boolean hasChildElement(Element parent, String child) {
      return XMLUtilities.lookupElement(parent, child) != null;
   }

   /**
    * Validate the name of the element.  If the element name does not match
    * the specified name, then throw a <code>ParseSymbolException</code>.
    */
   private void validateSymbolName(Element e, String name)
         throws ParseSymbolException {
      String tag = e.getTagName();
      if (!name.equals(tag)) {
         error("Invalid element name \""+tag+"\" (expected \""+name+"\")");
      }
   }

   /**
    * Extract the specified attribute and throw a parse symbol exception if
    * the attribute is not defined (null).
    */
   private String getAttribute(Element e, String attr)
         throws ParseSymbolException {
      String value = e.getAttribute(attr);
      if (value == null) {
         error("Attribute \""+attr+"\" not found in \""+e.getTagName()+"\"");
      }
      return value;
   }

   /**
    * Extract the specified optional attribute.
    */
   private String getOptionalAttribute(Element e, String attr) {
      return e.getAttribute(attr);
   }
}
