/*
 * File:        SIDLf90array.c
 * Copyright:   (c) 2003 The Regents of the University of California
 * Release:     $Name: release-0-8-8 $
 * Revision:    @(#) $Revision: 1.3 $
 * Date:        $Date: 2003/10/27 21:34:47 $
 * Description: Functions to convert SIDL arrays into F90 derived types
 *
 */

#include "SIDLf90array.h"

#if defined(SIDL_MAX_F90_DESCRIPTOR) && !defined(FORTRAN90_DISABLED)

#include <stddef.h>
#include <stdio.h>
#include "CompilerCharacteristics.h"
#include "F90Compiler.h"

#ifndef included_SIDL_double_IOR_h
#include "SIDL_double_IOR.h"
#endif
#ifndef included_SIDL_dcomplex_IOR_h
#include "SIDL_dcomplex_IOR.h"
#endif
#ifndef included_SIDL_fcomplex_IOR_h
#include "SIDL_fcomplex_IOR.h"
#endif
#ifndef included_SIDL_float_IOR_h
#include "SIDL_float_IOR.h"
#endif
#ifndef included_SIDL_int_IOR_h
#include "SIDL_int_IOR.h"
#endif
#ifndef included_SIDL_long_IOR_h
#include "SIDL_long_IOR.h"
#endif

static F90_CompilerCharacteristics s_cc;

static int
getCompilerCharacteristics()
{
  static int s_notInitialized = 1;
  if (s_notInitialized) {
    s_notInitialized = F90_SetCompilerCharacteristics(&s_cc, FORTRAN_COMPILER);
    if (s_notInitialized) {
      fprintf(stderr,
              "Cannot determine F90 compiler characteristics for %s\n",
              FORTRAN_COMPILER);
    }
  }
  return s_notInitialized;
}

static int
genericConvert(void *ior,
               void *first,
               const int32_t lower[],
               const int32_t upper[],
               const int32_t stride[],
               const int dimen,
               const F90_ArrayDataType data_type,
               const size_t elem_size,
               struct SIDL_fortran_array *dest)
{
  long unsigned extent[7];
  long low[7], chasmStride[7];
  int i;
  if (getCompilerCharacteristics()) return 1;
  dest->d_ior = (ptrdiff_t)ior;
  for(i = 0; i < dimen; ++i){
    low[i] = lower[i];
    extent[i] = (upper[i] >= (lower[i]-1)) ? (1 + upper[i] - lower[i]) : 0;
    chasmStride[i] = stride[i]*elem_size;
  }
  return (s_cc.setArrayDesc)((void *)dest->d_descriptor, first, dimen,
                             F90_ArrayPointerInDerived,
                             data_type, elem_size,
                             low, extent, chasmStride);
}

static int 
genericNullify(const int                   dimen,
               F90_ArrayDataType           data_type,
               size_t                     elem_size,
               struct SIDL_fortran_array *dest)
{
  static const long lower[7] = { 1, 1, 1, 1, 1, 1, 1};
  static const unsigned long extent[7] =  {1, 1, 1, 1, 1, 1, 1};
  long stride[7];
  struct SIDL_dcomplex junk;
  int i;
  if (getCompilerCharacteristics()) return 1;
  dest->d_ior = 0;
  for(i = 0; i < dimen; ++i) {
    stride[i] = elem_size;
  }
  if ((s_cc.setArrayDesc)(dest->d_descriptor, (void *)&junk, 
                          dimen, F90_ArrayPointerInDerived,
                          data_type, elem_size,
                          lower, extent, stride)) return 1;
  (s_cc.nullifyArrayDesc)(dest->d_descriptor, dimen);
  return 0;
}

int
SIDL_dcomplex__array_convert2f90(const struct SIDL_dcomplex__array *src,
                                 const int src_dimen,
                                 struct SIDL_fortran_array *dest)
{
  return src 
    ? genericConvert((void *)src,
                     (void *)src->d_firstElement,
                     src->d_lower,
                     src->d_upper,
                     src->d_stride,
                     src->d_dimen,
                     F90_DComplex,
                     sizeof(struct SIDL_dcomplex),
                     dest)
    : genericNullify(src_dimen, F90_DComplex,
                     sizeof(struct SIDL_dcomplex), dest);
}

int
SIDL_double__array_convert2f90(const struct SIDL_double__array *src,
                               const int src_dimen,
                               struct SIDL_fortran_array *dest)
{
  return src 
    ? genericConvert((void *)src,
                     (void *)src->d_firstElement,
                     src->d_lower,
                     src->d_upper,
                     src->d_stride,
                     src_dimen,
                     F90_Double,
                     sizeof(double),
                     dest)
    : genericNullify(src_dimen, F90_Double,
                     sizeof(double), dest);
}

int
SIDL_fcomplex__array_convert2f90(const struct SIDL_fcomplex__array *src,
                                 const int src_dimen,
                                 struct SIDL_fortran_array *dest)
{
  return src 
    ? genericConvert((void *)src,
                     (void *)src->d_firstElement,
                     src->d_lower,
                     src->d_upper,
                     src->d_stride,
                     src_dimen,
                     F90_Complex,
                     sizeof(struct SIDL_fcomplex),
                     dest)
    : genericNullify(src_dimen, F90_Complex,
                     sizeof(struct SIDL_fcomplex), dest);
}

int
SIDL_float__array_convert2f90(const struct SIDL_float__array *src,
                              const int src_dimen,
                              struct SIDL_fortran_array *dest)
{
  return src 
    ? genericConvert((void *)src,
                     (void *)src->d_firstElement,
                     src->d_lower,
                     src->d_upper,
                     src->d_stride,
                     src_dimen,
                     F90_Real,
                     sizeof(float),
                     dest)
    : genericNullify(src_dimen, F90_Real,
                     sizeof(float), dest);
}

int
SIDL_int__array_convert2f90(const struct SIDL_int__array *src,
                            const int src_dimen,
                            struct SIDL_fortran_array *dest)
{
  return src 
    ? genericConvert((void *)src,
                     (void *)src->d_firstElement,
                     src->d_lower,
                     src->d_upper,
                     src->d_stride,
                     src_dimen,
                     F90_Integer4,
                     sizeof(int32_t),
                     dest)
    : genericNullify(src_dimen, F90_Integer4,
                     sizeof(int32_t), dest);
}


int
SIDL_long__array_convert2f90(const struct SIDL_long__array *src,
                             const int src_dimen,
                             struct SIDL_fortran_array *dest)
{
  return src 
    ? genericConvert((void *)src,
                     (void *)src->d_firstElement,
                     src->d_lower,
                     src->d_upper,
                     src->d_stride,
                     src_dimen,
                     F90_Integer8,
                     sizeof(int64_t),
                     dest)
    : genericNullify(src_dimen, F90_Integer8,
                     sizeof(int64_t), dest);
}
#endif /* defined(SIDL_MAX_F90_DESCRIPTOR) && !defined(FORTRAN90_DISABLED) */
