
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.component;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.table.*;
import org.gjt.btools.gui.Positioner;
import org.gjt.btools.gui.dialog.MessageBox;

/**
 * A component used to represent very long strings in table cells.
 * A standard label is inadequate for displaying very long strings
 * because there is no way to scroll across to see beyond the initial
 * portion that fits into the table cell.  This component provides a
 * button that brings up a dialog displaying the full cell contents.
 * <p>
 * This component is actually implemented both as a cell renderer and a
 * cell editor.  It cannot be used to edit table values, but because the
 * user must interact directly with this component, it must be
 * registered <b>both as a renderer and an editor</b> for the
 * appropriate table cells, and the table model must declare that
 * <b>such cells be editable</b>.  No change will ever be made to the
 * underlying table data.
 * <p>
 * The renderer is a simple label that will only display the leftmost
 * portion of the cell contents.  When the user clicks on such a cell,
 * the editor appears.  The editor consists of a read-only text edit field
 * that fills most of the table cell (this field can be scrolled and
 * selections can be made) and a button at the very right hand end that
 * brings up a dialog displaying the full cell contents.  Such dialogs
 * are not model and can be left on screen as long as desired (so
 * for instance multiple cell contents can be viewed and compared).  The
 * cell contents are copied to the dialog and so the dialog may remain
 * on screen well after the table has been destroyed.
 * <p>
 * The table cells may contain any objects; the method
 * <tt>Object.toString()</tt> will be used to convert them into strings
 * for display.
 * The method for choosing a title for the corresponding content viewer
 * dialog can be changed by overriding method <tt>getDialogTitle()</tt>.
 * The default implementation simply uses the corresponding column
 * heading in the table.
 *
 * @see #getDialogTitle
 */
public class LongStringRenderer extends AbstractCellEditor
        implements TableCellRenderer, TableCellEditor {
    /**
     * Border to be used for cells with no focus.
     */
    private static final Border noFocusBorder =
        new EmptyBorder(1, 2, 1, 2);

    /**
     * Number of clicks required to start editing.
     */
    private int clicksToEdit;

    /**
     * The specific string with which we are interacting.
     */
    private String value = null;
    /**
     * The title for the message box that will display the full cell
     * contents.
     */
    private String title = null;

    /**
     * The actual component used to render table cells.
     */
    private JLabel renderer;

    /**
     * The actual component used to pseudo-edit table cells.
     */
    private Box editor;
    /**
     * The field in which the string contents are displayed in the cell
     * editor.
     */
    private JTextField editorField;
    /**
     * The button in the cell editor that allows the full cell contents
     * to be displayed in a message box.
     */
    private JButton editorButton;

    /**
     * Creates a new long string renderer.
     * Cell editing will begin with a single mouse click.
     */
    public LongStringRenderer() {
        this(1);
    }

    /**
     * Creates a new long string renderer.
     *
     * @param clicksToEdit the number of mouse clicks required for
     * cell editing to begin.
     */
    public LongStringRenderer(int clicksToEdit) {
        this.clicksToEdit = clicksToEdit;

        renderer = new JLabel();
        renderer.setOpaque(true); // So backgroundcolours will show.

        editor = Box.createHorizontalBox();
        editorField = new JTextField();
        editorField.setEditable(false);
        editorButton = new JButton("...");
        editorButton.setBorder(BorderFactory.createRaisedBevelBorder());
        editorButton.setToolTipText("View entire cell");
        editorButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                new ContentViewer().show();
            }
        });
        editorButton.setMaximumSize(editorButton.getPreferredSize());
        editor.add(editorField);
        editor.add(editorButton);
    }

    /**
     * Returns the title to use for the dialog that will display the
     * full contents of the given cell.
     * The default implementation returns the name of the column in
     * which the given cell lies.
     *
     * @param table the table containing the given cell.
     * @param value the value stored in the given cell.
     * @param row the row of the given cell in the table.
     * @param column the column of the given cell in the table.
     * @return the title to use for the corresponding content viewer
     * dialog.
     */
    public String getDialogTitle(JTable table, Object value, int row,
            int column) {
        return table.getColumnName(column);
    }

    /**
     * Returns the component that does the rendering.
     *
     * @param table the table in which a cell is being rendered.
     * @param value the value of the table cell being rendered.
     * @param isSelected whether or not the table cell is currently
     * selected.
     * @param hasFocus whether or not the table cell currently has the
     * focus.
     * @param row the row of the table cell being rendered.
     * @param column the column of the table cell being rendered.
     * @return the component that does the rendering.
     */
    public Component getTableCellRendererComponent(JTable table,
            Object value, boolean isSelected, boolean hasFocus,
            int row, int column) {
        // Set the attributes of the component.
        renderer.setForeground(isSelected ? table.getSelectionForeground() :
            table.getForeground());
        renderer.setBackground(isSelected ? table.getSelectionBackground() :
            table.getBackground());
        renderer.setFont(table.getFont());

        // Set the border of the component.
        if (hasFocus) {
            renderer.setBorder(UIManager.getBorder(
                "Table.focusCellHighlightBorder"));
            renderer.setForeground(UIManager.getColor(
                "Table.focusCellForeground"));
            renderer.setBackground(UIManager.getColor(
                "Table.focusCellBackground"));
        } else
            renderer.setBorder(noFocusBorder);

        // Set the component's contents.
        renderer.setText(value.toString());
        
        return renderer;
    }

    /**
     * Returns the component that does the pseudo-editing.
     *
     * @param table the table in which a cell is being edited.
     * @param value the value of the table cell being edited.
     * @param isSelected whether or not the table cell is currently
     * selected.
     * @param row the row of the table cell being edited.
     * @param column the column of the table cell being edited.
     * @return the component that does the editing.
     */
    public Component getTableCellEditorComponent(JTable table, Object value,
            boolean isSelected, int row, int column) {
        this.value = value.toString();
        this.title = getDialogTitle(table, value, row, column);
        editorField.setText(this.value);
        return editor;
    }

    /**
     * Determines if the given event is cause to start editing the table
     * cell.
     *
     * @param e the event under consideration.
     * @return <tt>true</tt> if and only if the table cell should be
     * edited.
     */
    public boolean isCellEditable(EventObject e) {
        if (e instanceof MouseEvent)
            if (((MouseEvent)e).getClickCount() < clicksToEdit)
                return false;
        return true;
    }

    /**
     * Returns the value of the table cell being edited.
     *
     * @return the value of the table cell being edited.
     */
    public Object getCellEditorValue() {
        return value;
    }

    /**
     * Prepares the component for editing as requested by the given
     * event.
     *
     * @param e the event that caused editing to begin.
     * @return <tt>true</tt> if and only if the table cell being edited 
     * should be selected.
     */
    public boolean shouldSelectCell(EventObject e) {
        editor.requestFocus();
        return false;
    }

    /**
     * The dialog class used to display the full contents of a table
     * cell.
     */
    private class ContentViewer extends JDialog {
        /**
         * Creates a new content viewer dialog.
         */
        public ContentViewer() {
            super((Frame)null, title, false);
            init();
            pack();
            Positioner.centerOnScreen(this);
        }

        /**
         * Initialises the dialog components.
         */
        private void init() {
            GridBagLayout gb = new GridBagLayout();
            GridBagConstraints c = new GridBagConstraints();
            c.fill = c.NONE;
            c.gridx = 0;
            getContentPane().setLayout(gb);

            TextDisplayRegion contents = new TextDisplayRegion();
            contents.setPads(15, 15);
            contents.insertMessage(value, 80);

            c.weighty = 1;
            gb.setConstraints(contents, c);
            getContentPane().add(contents);

            JPanel buttonPanel = new JPanel();
            JButton ok = new JButton("OK");
            ok.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    dispose();
                }
            });
            buttonPanel.add(ok);

            c.weighty = 0;
            gb.setConstraints(buttonPanel, c);
            getContentPane().add(buttonPanel);

            setDefaultCloseOperation(DISPOSE_ON_CLOSE);
        }
    }
}

