/****************************************************************************
**
** Copyright (C) 2004-2006 Frank Hemer <frank@hemer.org>,
**                         Tilo Riemer <riemer@crossvc.com>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** CrossVC is available under two different licenses:
**
** If CrossVC is linked against the GPLed version of Qt 
** CrossVC is released under the terms of GPL also.
**
** If CrossVC is linked against a nonGPLed version of Qt 
** CrossVC is released under the terms of the 
** CrossVC License for non-Unix platforms (CLNU)
**
**
** CrossVC License for non-Unix platforms (CLNU):
**
** Redistribution and use in binary form, without modification, 
** are permitted provided that the following conditions are met:
**
** 1. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 2. It is not permitted to distribute the binary package under a name
**    different than CrossVC.
** 3. The name of the authors may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
** 4. The source code is the creative property of the authors.
**    Extensions and development under the terms of the Gnu Public License
**    are limited to the Unix platform. Any distribution or compilation of 
**    the source code against libraries licensed other than gpl requires 
**    the written permission of the authors.
**
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
**
**
** CrossVC License for Unix platforms:
**
** This program is free software; you can redistribute it and/or modify 
** it under the terms of the GNU General Public License as published by 
** the Free Software Foundation, version 2 of the License.
** This program is distributed in 
** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
** PARTICULAR PURPOSE.
**
** See the GNU General Public License version 2 for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "Watch.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"
#include "WatchersDialogImpl.h"

Watch * Watch::watchOnFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, ON);
}

Watch * Watch::watchOffFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, OFF);
}

Watch * Watch::watchAddFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, ADD);
}

Watch * Watch::watchRemoveFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, REMOVE);
}

Watch * Watch::showWatchersFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, SHOW);
}

Watch * Watch::watchOnDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, ON);
}

Watch * Watch::watchOffDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, OFF);
}

Watch * Watch::watchAddDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, ADD);
}

Watch * Watch::watchRemoveDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, REMOVE);
}

Watch * Watch::showWatchersDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, SHOW);
}

Watch * Watch::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir, int mode) {

   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;

   if (dir->getType() == DirBase::Cvs) {
      Watch * p = new Watch(whatsThisIconSet, parent, workBench, dir, isDir, mode);
      p->acceptCvs();
   } else {
      qDebug("Watch: Method not implemented");
   }
   return NULL;
}

Watch::Watch(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, bool isDir, int mode)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir),
      m_mode(mode)
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
   connect(this,SIGNAL(checkTree(DirBase *)),parent,SLOT(slot_checkStatusOfTree(DirBase *)));
}

Watch::~Watch() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void Watch::acceptCvs() {

   assert(m_dir);

   QString files = "";
   QStringList fileList;
   if (!m_isDir) {
      fileList = m_dir->getSelectedFiles();
      if (fileList.isEmpty()) {
         reject();
         return;
      }
      masqQuoteMarks(&fileList);
      files = masqWs(fileList.join("\" \""));
   }
   QString command = "watch ";
   int cmd = NOOP;
   switch(m_mode) {
      case ON: {
         command += "on";
         break;
      }
      case OFF: {
         command += "off";
         break;
      }
      case ADD: {
         command += "add";
         break;
      }
      case REMOVE: {
         command += "remove";
         break;
      }
      case SHOW: {
         command = "watchers";
         cmd = CVS_SHOW_WATCHERS_CMD;
         break;
      }
      default: {
         assert(false);
      }
   }

   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   
   callInteractive( topModule, dir, command,
	 files, cmd,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 NOROOT);
}

void Watch::reject() {
   emit deleteObject(this);
}

void Watch::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void Watch::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void Watch::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   cvsCallFinished();
   if (failed) {
      reject();
      return;
   }
   assert(m_dir);
   switch( cmd) {
      case NOOP: {
         emit checkTree(m_dir);
         break;
      }
      case CVS_SHOW_WATCHERS_CMD: {
         WatchersDialogImpl * dlg = new WatchersDialogImpl( m_whatsThisIconSet,
         m_parent, "WatchersDlg",
         LookAndFeel::g_nonModalF | WDestructiveClose);
         
         dlg->parseWatchers (buf, m_dir->repository());
         dlg->show();
         break;
      }
      default: {
         qDebug("Watch::afterCall: Unknown cmd");
         assert(false);
      }
   }
   reject();
}

