#!/bin/bash
#
# 	multistation-storage-refresh.sh
#
# 	Author:
# 		Jason Kim <jason@userful.com>
#
# 	Copyright (c) 2002-2007 by Userful Corporation
#
# 	DESCRIPTION
#		  This program regenerates /etc/multiuserdevices using HAL
#		on every bootup and keyboard assignment.
#
# 	Called from: multistation-storage-config.sh, multistation-storage-refresh.sh

logfile="/var/log/userful-start.log"
CONFIG_FILE="/etc/multiuserdevices"
ProgramName=`basename $0`

if [ ! -e $CONFIG_FILE ]; then
	touch $CONFIG_FILE
fi

# Remove all error line (not contain 5 items)
awk '{if (NF == 5) print $0}' $CONFIG_FILE > $CONFIG_FILE.old

# create empty /etc/multiuserdevices
rm -f $CONFIG_FILE
touch $CONFIG_FILE
chmod a+r $CONFIG_FILE

#--------------------------------------------
# Utilize HAL system to find storage devices.
# 
# SATA Hard Drive: 		storage.drive_type=disk 	storage.bus=scsi 		--> should be ignored
# IDE Hard Drive: 		storage.drive_type=disk 	storage.bus=ide 		--> should be ignored
# USB Flash Disk: 		storage.drive_type=disk 	storage.bus=usb
#
# Internal CDROM: 		storage.drive_type=cdrom 	storage.bus=ide 		--> shared by default
# USB CDROM: 	 		storage.drive_type=cdrom 	storage.bus=usb
# 
# Internal Floppy: 		storage.drive_type=floppy 	storage.bus=platform 	--> shared by default
# USB Floppy: 			storage.drive_type=floppy 	storage.bus=usb
#
#--------------------------------------------

# Found UDIs for every storage device.
STORAGE_UDIS=`hal-find-by-property --key "info.category" --string "storage"`

for UDI in $STORAGE_UDIS
do
	# storage drive type
	DRIVE_TYPE=`hal-get-property --udi "$UDI" --key "storage.drive_type"`

	###
	# Among the storage devices, internal hard drives should be ignored. 
	###
	if [ "$DRIVE_TYPE" == "disk" ]; then # Internal harddrive(IDE or SATA), USB flash disk drive.
		STORAGE_BUS=`hal-get-property --udi "$UDI" --key "storage.bus"`

		if [ "$STORAGE_BUS" != "usb" ]; then # Ignore internal harddrives.
			continue
		fi

	elif [ "$DRIVE_TYPE" == "cdrom" ]; then # Internal or USB CDROM drive
		STORAGE_BUS=`hal-get-property --udi "$UDI" --key "storage.bus"`

	elif [ "$DRIVE_TYPE" == "floppy" ]; then # Internal or USB Floppy drive
		STORAGE_BUS=`hal-get-property --udi "$UDI" --key "storage.bus"`

	else
		echo "Unsupported drive type. $DRIVE_TYPE"
		continue

	fi
	
	#
	# device vendor
	DEV_VENDOR=`hal-get-property --udi "$UDI" --key "info.vendor" 2> /dev/null`
	if [ "$DEV_VENDOR" == "" ]; then
		DEV_VENDOR="Unkown"
	else # remove blank
		DEV_VENDOR=`echo $DEV_VENDOR | tr "[[:blank:]]" "_"`
	fi

	#
	# device product name
	DEV_PRODUCT=`hal-get-property --udi "$UDI" --key "info.product" 2> /dev/null`
	if [ "$DEV_PRODUCT" == "" ]; then
		DEV_VENDOR="Unkown"
	else # remove blank
		DEV_PRODUCT=`echo $DEV_PRODUCT | tr "[[:blank:]]" "_"`
	fi

	#
	# block device name
	BLOCK_DEV=`hal-get-property --udi "$UDI" --key "block.device"`

	#
	# get sysfs_path
	PARENT_UDI=`hal-get-property --udi "$UDI" --key "info.parent"`

	TEMP_UDI=$UDI
	TEMP_PARENT_UDI=
	SUBSYSTEM=
	if [ "$STORAGE_BUS" == "usb" ]; then
		while [ "$SUBSYSTEM" != "usb" ]
		do
			TEMP_PARENT_UDI=`hal-get-property --udi "$TEMP_UDI" --key "info.parent"`
			SUBSYSTEM=`hal-get-property --udi "$TEMP_PARENT_UDI" --key "linux.subsystem"`
			TEMP_UDI=$TEMP_PARENT_UDI
		done

		DEV_PATH=`hal-get-property --udi "$TEMP_UDI" --key "linux.sysfs_path"`

	else
		DEV_PATH=`hal-get-property --udi "$PARENT_UDI" --key "linux.sysfs_path"`
	fi

	# Check if there is a manual configuration for the current device.
	# If there is no previous manual configuration, follow the screen number of a keyboard 
	# on the same USB path.

	PRE_ASSIGNED_SCREEN=`grep $DEV_PATH $CONFIG_FILE.old | awk '{print $3}'`
	PRE_ASSIGNED_FLAG=`grep $DEV_PATH $CONFIG_FILE.old | awk '{print $4}'`

	if [ "$PRE_ASSIGNED_SCREEN" != "" -a "$PRE_ASSIGNED_FLAG" == "W" ] ; then
		# If the current storage device was configured manually before, follow the old configuration.
		echo "`date +"%b %d %T"` $ProgramName: Found $DEV_PRODUCT($DEV_VENDOR) on $CONFIG_FILE."  >> $logfile
		ASSIGNED_SCREEN="$PRE_ASSIGNED_SCREEN"
		ASSIGNED_FLAG="W"

	else
		if [ "$STORAGE_BUS" == "usb" ]; then
			# Find screen number base on the current audio device path.
			SCREEN_NUM=`/opt/userful/bin/get-screen-num-from-devpath -T 12 -P $DEV_PATH 2>/dev/null`

			if [ "$SCREEN_NUM" == "-1" ]; then
                # Should decide what to do with unassigned device
                # This part can be a library code. (duplicated with multistation-storage-hotplug.sh)

                # Get current setting for the unassigned storage devices
                UNASSIGNED_DEV=`grep -m1 "^UNASSIGNED_DEVICE" /etc/userful/dm.conf  | awk -F'=' '{print $2}'`

                if [[ -z "$UNASSIGNED_DEV" ]]; then # No UNASSIGNED_DEVICE setting in dm.conf
                    echo "No setting for UNASSIGNED_DEVICE"
				    ASSIGNED_SCREEN="Shared_by_All_Stations"

                elif [[ "$UNASSIGNED_DEV" == "shared" ]]; then
                    echo "UNASSIGNED_DEVICE=shared"
				    ASSIGNED_SCREEN="Shared_by_All_Stations"

                elif [[ "$UNASSIGNED_DEV" == "disabled" ]]; then
                    echo "UNASSIGNED_DEVICE=disabled"
				    ASSIGNED_SCREEN="Disabled"

                elif [[ "$UNASSIGNED_DEV" == "onboard-sound" ]]; then
                    echo "UNASSIGNED_DEVICE=onboard-sound"

                    ONBOARD_SOUND=`grep -m1 "(pci)" /etc/usbaudio | awk '{print $3}'`

                    if [[ -n "$ONBOARD_SOUND" ]]; then
				        ASSIGNED_SCREEN=$ONBOARD_SOUND
                        SCREEN_NUM=`echo $ONBOARD_SOUND | awk -F':' '{print $2}' | awk -F'.' '{print $1}'`

                    else
				        ASSIGNED_SCREEN="Shared_by_All_Stations"
                    fi


                elif [[ ! $UNASSIGNED_DEV = *[^0-9]* ]]; then
                    echo "UNASSIGNED_DEVICE is number"
                    ASSIGNED_SCREEN="Srn:$UNASSIGNED_DEV.0"

                else
                    echo "Wrong value for UNASSIGNED_DEVICE"
				    ASSIGNED_SCREEN="Shared_by_All_Stations"
                fi

			else
				ASSIGNED_SCREEN="Srn:$SCREEN_NUM.0"
			fi

		else # share internal devices by default.
			ASSIGNED_SCREEN="Shared_by_All_Stations"
		fi

		ASSIGNED_FLAG="R/W"
	fi

	if [ "$ASSIGNED_SCREEN" == "Disabled" ]; then
		hal-set-property --udi "$UDI" --key "info.screen" --string "disabled"

	elif [ "$ASSIGNED_SCREEN" == "Shared_by_All_Stations" ]; then
		hal-set-property --udi "$UDI" --key "info.screen" --string "shared"

	else
		hal-set-property --udi "$UDI" --key "info.screen" --string "$SCREEN_NUM"
	fi

	# add current device to the config file /etc/multiuserdevices
	echo "`date +"%b %d %T"` $ProgramName: Assign storage device $DEV_PRODUCT($DEV_VENDOR) to $ASSIGNED_SCREEN." >> $logfile
	echo "$DEV_PRODUCT($DEV_VENDOR) $BLOCK_DEV $ASSIGNED_SCREEN $ASSIGNED_FLAG $DEV_PATH" >> $CONFIG_FILE

done
