/////////////////////////////////////////////////////////////////////////////
// Name:        VobListBox.cpp
// Purpose:     The list box to display information about streams in given VOB
// Author:      Alex Thuering
// Created:     03.05.2009
// RCS-ID:      $Id: VobListBox.cpp,v 1.16 2011/03/10 20:09:08 ntalex Exp $
// Copyright:   (c) Alex Thuering
// Licence:     GPL
/////////////////////////////////////////////////////////////////////////////

#include "VobListBox.h"
#include "Config.h"
#include "VideoPropDlg.h"
#include "SubtitlePropDlg.h"
#include <wx/filedlg.h>
#include <wx/arrimpl.cpp>
#include <wxSVG/mediadec_ffmpeg.h>
#include <wxVillaLib/utils.h>
#include <wxVillaLib/rc/video.png.h>
#include <wxVillaLib/rc/audio.png.h>
#include <wxVillaLib/rc/subtitle.png.h>

WX_DEFINE_OBJARRAY(RectList);
WX_DEFINE_OBJARRAY(RectListOfList);
WX_DEFINE_OBJARRAY(StringListOfList);

BEGIN_EVENT_TABLE(VobListBox, wxVListBox)
	EVT_LEFT_DCLICK(VobListBox::OnDoubleClick)
END_EVENT_TABLE()

const int ITEM_HEIGHT = 50;
#define ITEM_FONT wxSystemSettings::GetFont(wxSYS_DEFAULT_GUI_FONT)

VobListBox::VobListBox(wxWindow* parent, wxWindowID id, Vob* vob, PgcArray* pgcs, DVD* dvd): wxVListBox(parent, id) {
	m_vob = new Vob(*vob);
	m_pgcs = pgcs;
	m_dvd = dvd;
	m_aspectRatio = pgcs->GetVideo().GetAspect();
	m_videoImg = Scale(LoadFrame(m_vob->GetFilename()));
	m_audioImg = Scale(wxBITMAP_FROM_MEMORY(audio).ConvertToImage());
	m_subtitleImg = Scale(wxBITMAP_FROM_MEMORY(subtitle).ConvertToImage());
	RefreshInfo();
	SetMinSize(wxSize(-1, 300));
}

wxCoord VobListBox::OnMeasureItem(size_t n) const {
	if (n >= m_infoRect.size())
		return ITEM_HEIGHT;
	int h = m_infoRect[n][m_infoRect[n].size()-1].GetBottom();
	return h + 4 > ITEM_HEIGHT ? h + 4 : ITEM_HEIGHT;
}

void VobListBox::OnDrawItem(wxDC& dc, const wxRect& rect, size_t n) const {
	// image
	wxBitmap image = n == 0 ? m_videoImg : n > m_vob->GetAudioFilenames().size() ? m_subtitleImg : m_audioImg;
	dc.DrawBitmap(image, rect.x + 2, rect.y + 2);
	
	// info
	dc.SetFont(ITEM_FONT);
	dc.SetTextForeground((int)n == GetSelection() ? *wxWHITE : *wxBLACK);
	wxArrayString& info = m_info[n];
	RectList& infoRect = m_infoRect[n];
	for (unsigned int i=0; i<info.GetCount(); i++) {
		dc.DrawText(info[i], rect.x + infoRect[i].GetX(), rect.y + infoRect[i].GetY());
	}
}

void VobListBox::RefreshInfo() {
	SetItemCount(1 + m_vob->GetAudioFilenames().size() + m_vob->GetSubtitles().size());
	
	m_info.Clear();
	m_infoRect.Clear();
	
	int choiceIdx = 0;
	int itemY = 0;
	unsigned int audioIdx = 0;
	unsigned int subtitleIdx = 0;
	for (unsigned int n = 0; n < 1 + m_vob->GetAudioFilenames().size(); n++) {
		m_info.Add(wxArrayString());
		m_infoRect.Add(RectList());
		
		wxMemoryDC dc;
		dc.SetFont(ITEM_FONT);
		wxBitmap image = n == 0 ? m_videoImg : m_audioImg;
		int x = image.GetWidth() + 8;
		int y = 2;
		
		// filename
		y = AddInfo(n == 0 ? m_vob->GetFilename() : m_vob->GetAudioFilenames()[n-1], n, dc, x, y);
			
		// duration
		if (n == 0) {
			wxString s = _("Duration:") + wxString(wxT(" "));
			if (m_vob->GetDuration()>0) {
				int secs = (int) m_vob->GetDuration();
				int mins = secs / 60;
				secs %= 60;
				int hours = mins / 60;
				mins %= 60;
				s += wxString::Format(wxT("%02d:%02d:%02d"), hours, mins, secs);
			} else
				s += wxT("N/A");
			y = AddInfo(s, n, dc, x, y);
		}
		
		// stream info
		int stIdx = n == 0 ? 0 : (int)m_vob->GetStreams().GetCount() - m_vob->GetAudioFilenames().GetCount() + n - 1;
		int streamsCount = n == 0 ? (int)m_vob->GetStreams().GetCount() - m_vob->GetAudioFilenames().GetCount() : 1;
		for (int stN = 0; stN < streamsCount; stN++) {
			Stream* stream = m_vob->GetStreams()[stIdx + stN];
			wxString srcFormat = stream->GetSourceFormat();
			switch (stream->GetType()) {
			case stVIDEO: {
				y = AddInfo(_("Video:") + wxString(wxT(" ")) + srcFormat, n, dc, x, y);
				wxRect& rect = m_infoRect[n][m_infoRect[n].size()-1];
				y += AddChoiceCtrl(DVD::GetVideoFormatLabels(true), stream->GetVideoFormat() - 1, rect, itemY,
						choiceIdx, !m_vob->GetDoNotTranscode());
				break;
			}
			case stAUDIO: {
				y = AddInfo(_("Audio:") + wxString(wxT(" ")) + srcFormat, n, dc, x, y);
				wxRect& rect = m_infoRect[n][m_infoRect[n].size()-1];
				wxRect rect2 = rect; // copy before it will be changed
				y += AddChoiceCtrl(DVD::GetAudioFormatLabels(true, true), stream->GetAudioFormat(), rect, itemY,
						choiceIdx, !m_vob->GetDoNotTranscode());
				rect2.x += m_choiceList[choiceIdx-1]->GetSize().GetWidth() + 2;
				wxString langCode = s_config.GetDefAudioLanguage();
				if (stream->GetAudioFormat() != afNONE) {
					if (audioIdx < m_pgcs->GetAudioLangCodes().size())
						langCode = m_pgcs->GetAudioLangCodes()[audioIdx];
					audioIdx++;
				}
				int langIdx = DVD::GetAudioLanguageCodes().Index(langCode);
				AddChoiceCtrl(DVD::GetAudioLanguageCodes(), langIdx, rect2, itemY, choiceIdx, true);
				break;
			}
			case stSUBTITLE: {
				y = AddInfo(_("Subtitle:") + wxString(wxT(" ")) + srcFormat, n, dc, x, y);
				wxRect& rect = m_infoRect[n][m_infoRect[n].size()-1];
				wxRect rect2 = rect; // copy before it will be changed
				y += AddChoiceCtrl(DVD::GetSubtitleFormatLabels(true, true), stream->GetSubtitleFormat(), rect, itemY,
						choiceIdx, !m_vob->GetDoNotTranscode());
				rect2.x += m_choiceList[choiceIdx-1]->GetSize().GetWidth() + 2;
				wxString langCode = s_config.GetDefSubtitleLanguage();
				if (stream->GetSubtitleFormat() != sfNONE) {
					if (subtitleIdx < m_pgcs->GetSubPictures().size())
						langCode = m_pgcs->GetSubPictures()[subtitleIdx]->GetLangCode();
					subtitleIdx++;
				}
				int langIdx = DVD::GetAudioLanguageCodes().Index(langCode);
				AddChoiceCtrl(DVD::GetAudioLanguageCodes(), langIdx, rect2, itemY, choiceIdx, true);
				break;
			}
			default:
				break;
			}
		}
		itemY += y > ITEM_HEIGHT ? y + 1 : ITEM_HEIGHT + 1;
		
		if (n == 0)
			m_audioChoiceIdx = choiceIdx;
	}

	m_subtitleChoiceIdx = choiceIdx;
	for (unsigned int si = 0; si < m_vob->GetSubtitles().size(); si++) {
		m_info.Add(wxArrayString());
		m_infoRect.Add(RectList());
		int n = m_info.GetCount() - 1;
		
		wxMemoryDC dc;
		dc.SetFont(ITEM_FONT);
		int x = m_subtitleImg.GetWidth() + 8;
		int y = 2;
		
		// filename
		y = AddInfo(m_vob->GetSubtitles()[si]->GetFilename(), n, dc, x, y);
		y = AddInfo(_("Subtitle:") + wxString(wxT(" ")) + m_vob->GetSubtitles()[si]->GetFilename().AfterLast(wxT('.')),
				n, dc, x, y);
		
		wxRect& rect = m_infoRect[n][m_infoRect[n].size()-1];
		int langIdx = DVD::GetAudioLanguageCodes().Index(subtitleIdx < m_pgcs->GetSubPictures().size()
				? m_pgcs->GetSubPictures()[subtitleIdx]->GetLangCode() : s_config.GetDefSubtitleLanguage());
		subtitleIdx++;
		y += AddChoiceCtrl(DVD::GetAudioLanguageCodes(), langIdx, rect, itemY, choiceIdx, true);
		itemY += y > ITEM_HEIGHT ? y + 1 : ITEM_HEIGHT + 1;
	}
}

int VobListBox::AddInfo(const wxString& s, int n, wxDC& dc, int x, int y) {
	m_info[n].Add(s);
	wxCoord w;
	wxCoord h;
	dc.GetTextExtent(s, &w, &h);
	m_infoRect[n].Add(wxRect(x, y, w, h));
	return y + h + 2;
}

int VobListBox::AddChoiceCtrl(wxArrayString formats, int selection, wxRect& rect, int itemY, int& choiceIdx,
		bool enabled) {
	wxChoice* ctrl = NULL;
	if (choiceIdx >= (int) m_choiceList.size() || m_choiceList[choiceIdx] == NULL) {
		ctrl = new wxChoice(this, wxID_ANY, wxDefaultPosition, wxDefaultSize, formats);
		ctrl->SetSelection(selection);
		if (choiceIdx >= (int) m_choiceList.size())
			m_choiceList.push_back(ctrl);
		else
			m_choiceList[choiceIdx] = ctrl;
	} else
		ctrl = m_choiceList[choiceIdx];
	choiceIdx++;
	ctrl->Enable(enabled);
	ctrl->SetPosition(wxPoint(rect.GetRight() + 4, itemY + rect.GetTop()));
	if (ctrl->GetSize().GetHeight() > rect.GetHeight()) {
		int p = (ctrl->GetSize().GetHeight() - rect.GetHeight() + 1)/2;
		rect.SetY(rect.GetY() + p);
		rect.SetHeight(rect.GetHeight() + p);
		return 2*p;
	}
	return 0;
}

wxImage VobListBox::LoadFrame(const wxString& filename) const {
	wxImage image;
	wxFfmpegMediaDecoder decoder;
	if (decoder.Load(filename)) {
		double duration = decoder.GetDuration();
		if (duration > 0) {
			double dpos = duration * 0.05;
			decoder.SetPosition(dpos);
			for (int i = 0; i < 100; i++) {
				image = decoder.GetNextFrame();
				double pos = decoder.GetPosition();
				if (pos >= dpos || pos < 0)
					break;
			}
		} else
			image = decoder.GetNextFrame();
		decoder.Close();
	}
	return image.Ok() ? image : wxBITMAP_FROM_MEMORY(video).ConvertToImage();
}

wxBitmap VobListBox::Scale(wxImage image) {
	int h = ITEM_HEIGHT - 4;
	int w = image.GetWidth()*h/image.GetHeight();
	return wxBitmap(image.Scale(w, h));
}

void VobListBox::RemoveItem(int index) {
	if (index > (int)m_vob->GetAudioFilenames().size()) {
		// remove subtitle
		int subtitleIdx = index - 1 - m_vob->GetAudioFilenames().size();
		m_vob->GetSubtitles().RemoveAt(subtitleIdx);
		// choice index = count of video stream + count of audio streams * 2 + count of audio subtitle streams
		int choiceIdx = m_subtitleChoiceIdx + subtitleIdx;
		m_choiceList[choiceIdx]->Hide();
		m_choiceList.erase(m_choiceList.begin() + choiceIdx);
	} else {
		// removed audio file
		int audioIdx = index - 1;
		m_vob->RemoveAudioFile(audioIdx);
		// choice index = count of video stream + count of audio streams * 2
		int choiceIdx = m_audioChoiceIdx + audioIdx*2;
		m_choiceList[choiceIdx]->Hide();
		m_choiceList[choiceIdx + 1]->Hide();
		m_choiceList.erase(m_choiceList.begin() + choiceIdx);
		m_choiceList.erase(m_choiceList.begin() + choiceIdx);
	}
	RefreshInfo();
	RefreshAll();
}

void VobListBox::AddAudio(wxString filename) {
	m_vob->AddAudioFile(filename);
	m_vob->SetDoNotTranscode(false);
	// check if reencoding is needed
	Stream* stream = m_vob->GetStreams()[m_vob->GetStreams().GetCount() - 1];
	if (m_vob->GetStreams().GetCount() == 2 && stream->GetSourceAudioFormat() != m_dvd->GetAudioFormat())
		stream->SetDestinationFormat(m_dvd->GetAudioFormat());
	else if (stream->GetSourceAudioFormat() != afMP2 && stream->GetSourceAudioFormat() != afAC3)
		stream->SetDestinationFormat(m_dvd->GetAudioFormat());
	else if (stream->GetSourceSampleRate() != 48000)
		stream->SetDestinationFormat(stream->GetSourceAudioFormat());
	// add choice controls
	m_choiceList.insert(m_choiceList.begin() + m_subtitleChoiceIdx, NULL);
	m_choiceList.insert(m_choiceList.begin() + m_subtitleChoiceIdx, NULL);
	// update list box
	RefreshInfo();
	RefreshAll();
}

void VobListBox::AddSubtitle(wxString filename) {
	m_vob->GetSubtitles().Add(new TextSub(filename));
	// update list box
	RefreshInfo();
	RefreshAll();
}

void VobListBox::SetDoNotTranscode(bool value) {
	m_vob->SetDoNotTranscode(value);
	int choiceIdx = 0;
	for (unsigned int stIdx = 0; stIdx < m_vob->GetStreams().size(); stIdx++) {
		Stream* stream = m_vob->GetStreams()[stIdx];
		switch (stream->GetType()) {
		case stVIDEO:
			if (value)
				m_choiceList[choiceIdx]->SetSelection(vfCOPY-1); // vfNONE is not in the selection list
			m_choiceList[choiceIdx++]->Enable(!value);
			break;
		case stAUDIO:
			if (value)
				m_choiceList[choiceIdx]->SetSelection(afCOPY);
			m_choiceList[choiceIdx++]->Enable(!value);
			choiceIdx++;
			break;
		default:
			break;
		}
	}
}

/**
 * Returns true if "do not transcode active ist"
 */
bool VobListBox::GetDoNotTranscode() {
	return m_vob->GetDoNotTranscode();
}

bool VobListBox::HasAudioFiles() {
	return m_vob->GetAudioFilenames().size() > 0;
}

Vob* VobListBox::SetValues() {
	unsigned int choiceIdx = 0;
	unsigned int audioIdx = 0;
	unsigned int subtitleIdx = 0;
	for (unsigned int stIdx = 0; stIdx < m_vob->GetStreams().size(); stIdx++) {
		Stream* stream = m_vob->GetStreams()[stIdx];
		switch (stream->GetType()) {
		case stVIDEO:
			stream->SetDestinationFormat(m_choiceList[choiceIdx++]->GetSelection() + 1);
			break;
		case stAUDIO:
			stream->SetDestinationFormat(m_choiceList[choiceIdx++]->GetSelection());
			if (stream->GetAudioFormat() != afNONE) {
				if (audioIdx >= m_pgcs->GetAudioLangCodes().size())
					m_pgcs->GetAudioLangCodes().push_back(m_choiceList[choiceIdx]->GetStringSelection());
				else
					m_pgcs->GetAudioLangCodes()[audioIdx] = m_choiceList[choiceIdx]->GetStringSelection();
				audioIdx++;
			}
			choiceIdx++;
			break;
		case stSUBTITLE:
			stream->SetDestinationFormat(m_choiceList[choiceIdx++]->GetSelection());
			if (stream->GetSubtitleFormat() != sfNONE) {
				if (m_pgcs->GetSubPictures().size() <= subtitleIdx)
					m_pgcs->GetSubPictures().push_back(new SubPicture(m_choiceList[choiceIdx]->GetStringSelection()));
				else
					m_pgcs->GetSubPictures()[subtitleIdx]->SetLangCode(m_choiceList[choiceIdx]->GetStringSelection());
				subtitleIdx++;
			}
			choiceIdx++;
		default:
			break;
		}
	}
	for (unsigned int si = 0; si < m_vob->GetSubtitles().size(); si++) {
		if (m_pgcs->GetSubPictures().size() <= subtitleIdx)
			m_pgcs->GetSubPictures().push_back(new SubPicture(m_choiceList[choiceIdx++]->GetStringSelection()));
		else
			m_pgcs->GetSubPictures()[subtitleIdx]->SetLangCode(m_choiceList[choiceIdx++]->GetStringSelection());
		subtitleIdx++;
	}
	return m_vob;
}

/**
 * Get subtitle language code (for property dialog, ignores muxed subtitles)
 */
wxString VobListBox::GetSubtitleLangCode(int subtitleIndex) {
	return m_choiceList[m_subtitleChoiceIdx + subtitleIndex]->GetStringSelection();
}

/**
 * Sets subtitle language code (for property dialog, ignores muxed subtitles)
 */
void VobListBox::SetSubtitleLangCode(int subtitleIndex, wxString langCode) {
	m_choiceList[m_subtitleChoiceIdx + subtitleIndex]->SetSelection(DVD::GetAudioLanguageCodes().Index(langCode));
}

/**
 * Returns true if selected item has properties dialog
 */
bool VobListBox::HasPropDialog() {
	return GetSelection() == 0 || GetSelection() >= 1 + (int) m_vob->GetAudioFilenames().GetCount();
}

/**
 * Shows subtitle properties dialog  
 */
void VobListBox::ShowPropDialog() {
	if (GetSelection() == 0) {
		// video
		SetValues(); // update destination format
		VideoPropDlg dialog(this, m_vob, m_aspectRatio);
		if (dialog.ShowModal() == wxID_OK) {
			if (m_vob->GetStreams()[0]->GetDestinationFormat() > vfCOPY)
				SetDoNotTranscode(false);
			m_choiceList[0]->SetSelection(m_vob->GetStreams()[0]->GetDestinationFormat() - 1);
			m_aspectRatio = dialog.GetSelectedAspectRatio();
		}
	} else if (GetSelection() >= 1 + (int) m_vob->GetAudioFilenames().GetCount()) {
		// subtitle
		int idx = GetSelection() - 1 - m_vob->GetAudioFilenames().GetCount();
		TextSub* textsub = m_vob->GetSubtitles()[idx];
		SubtitlePropDlg dialog(this, textsub, GetSubtitleLangCode(idx));
		if (dialog.ShowModal() == wxID_OK) {
			SetSubtitleLangCode(idx, dialog.GetLangCode());
		}
	}
}

/**
 * Processes a double click event
 */
void VobListBox::OnDoubleClick(wxMouseEvent& evt) {
	ShowPropDialog();
}
