/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.client;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.SocketException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.team.internal.core.streams.PollingInputStream;
import org.eclipse.team.internal.core.streams.PollingOutputStream;
import org.eclipse.team.internal.core.streams.TimeoutOutputStream;
import org.eclipse.team.internal.ftp.FTPException;
import org.eclipse.team.internal.ftp.FTPPlugin;
import org.eclipse.team.internal.ftp.Policy;

abstract class DataConnection implements Runnable {
	private IProgressMonitor monitor;

	private Socket socket;
	private InputStream socketIn;
	private OutputStream socketOut;
	private FTPException pendingException;
	private Thread thread;
	private boolean isReady;
	
	protected DataConnection() {
		this.isReady = false;
	}
	
	/**
	 * Starts a background thread to open the socket, returns immediately.
	 * The progress monitor is passed to the worker if successful.
	 * @param monitor the progress monitor
	 */
	public void open(IProgressMonitor monitor) throws FTPException {
		this.monitor = Policy.monitorFor(monitor);
		thread = new Thread(this, Policy.bind("FTPClient.FTPDataTransferWorker")); //$NON-NLS-1$
		thread.start();
	}
	
	/**
	 * Stops the background thread and closes the socket, optionally blocks until completed.
	 * Uses the progress monitor specified in open after the worker exits.
	 * @param interrupt if true, interrupts the operation in progress, eats any exceptions and
	 *        returns immediately, else blocks until the operation completes
	 */
	public void close(boolean interrupt) throws FTPException {
		if (interrupt) {
			thread.interrupt();
		}
		if (pendingException != null) throw pendingException;
	}
	
	/**
	 * Checks the status of the receiver and yields to other threads if 
	 * nothing of interest is happened yet
	 */
	public boolean isReady() throws FTPException {
		// Throw any exceptions that happened while trying to connect
		if (pendingException != null) throw pendingException;
		if (! isReady) {
			Thread.yield();
		}
		return isReady;
	}

	/**
	 * Private.
	 */
	public void run() {
		isReady = false;
		if (Thread.interrupted()) return;
		try {
			try {
				socket = openSocket();
				if (Thread.interrupted()) return;
				try {
					socket.setKeepAlive(false);
				} catch (SocketException e) {
					throw new FTPCommunicationException(Policy.bind("FTPClient.CannotSetSocketOptions"), e); //$NON-NLS-1$
				}
				if (Thread.interrupted()) return;
				isReady = true;
			} finally {
				if (!isReady) {
					closeSocket();
				}
			}
		} catch (FTPException e) {
			pendingException = e;
		}
	}

	protected abstract Socket openSocket() throws FTPException;

	protected void closeSocket() throws FTPException {
		if (socket == null) return;
		// ensure the streams and the socket are closed properly even if multiple exceptions occur
		try {
			try {
				try {
					if (socketIn != null) socketIn.close();
				} finally {
					socketIn = null;
					if (socketOut != null) socketOut.close();
				}
			} finally {
				socketOut = null;
				socket.close();
			}
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.ErrorDisconnecting"), e); //$NON-NLS-1$
		} finally {
			socket = null;
		}
	}
	
	protected InputStream getSocketInputStream() throws FTPException {
		try {
			socket.shutdownOutput();
			socket.setSoTimeout(1000);
			socket.setReceiveBufferSize(FTPPlugin.getPlugin().getReceiveBufferSize());
			socketIn = new PollingInputStream(socket.getInputStream(),
				FTPPlugin.getPlugin().getTimeout(), monitor);
			return socketIn;
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("DataConnection.CannotOpenInput"), e); //$NON-NLS-1$
		}
	}
	
	protected OutputStream getSocketOutputStream() throws FTPException {
		try {
			socket.shutdownInput();
			socket.setSendBufferSize(FTPPlugin.getPlugin().getSendBufferSize());
			socketOut = new PollingOutputStream(new TimeoutOutputStream(socket.getOutputStream(),
				8192 /*bufferSize*/, 1000 /*writeTimeout*/, 1000 /*closeTimeout*/),
				FTPPlugin.getPlugin().getTimeout(), monitor);
			return socketOut;
		}catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("DataConnection.CannotOpenInput"), e); //$NON-NLS-1$
		}
	}
	
	protected InputStream getDataTransferInputStream() throws FTPException {
		InputStream in =  new DataTransferInputStream(this);
		return in;
	}
}
