/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.target;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.core.Assert;
import org.eclipse.team.internal.core.target.AuthenticatedSite;
import org.eclipse.team.internal.core.target.ITeamStatusConstants;
import org.eclipse.team.internal.core.target.TargetProvider;
import org.eclipse.team.internal.ftp.FTPPlugin;
import org.eclipse.team.internal.ftp.Policy;

public class FTPSite extends AuthenticatedSite {

	public final static String TYPE = "org.eclipse.team.ftp"; //$NON-NLS-1$
	public final static int DEFAULT_TIMEOUT = 60;

	protected URL proxy;
	protected int timeout = DEFAULT_TIMEOUT;
	protected boolean usePassive = false;
	
	public FTPSite(Properties configuration) {
		try {
			configure(configuration);
		} catch (TeamException e) {
			FTPPlugin.log(e.getStatus());
		}
	}
	
	public FTPSite(ObjectInputStream is) {
		//not sure yet where we get the bytes from		
		try {
			int verisonId = is.readInt();
			String propertiesString = is.readUTF();
			
			ByteArrayInputStream inputStream = new ByteArrayInputStream(propertiesString.getBytes());
			// Read the properties from the stream.
			Properties properties = new Properties();
			try {
				properties.load(inputStream);
			} catch (IOException e) {
				FTPPlugin.logError(Policy.bind("Config.error"), e); //$NON-NLS-1$
			}
			configure(properties);	
		} catch (IOException e) {
				FTPPlugin.logError(Policy.bind("Config.error"), e); //$NON-NLS-1$
		} catch (TeamException e) {
				FTPPlugin.logError(Policy.bind("Config.error"), e); //$NON-NLS-1$
		}
	}
	
	public Properties getConfiguration() {
		Properties properties = new Properties();
		properties.setProperty("location", getURL().toExternalForm()); //$NON-NLS-1$
		if (this.proxy != null) {
			properties.setProperty("proxyURL", this.proxy.toString()); //$NON-NLS-1$
		}
		properties.setProperty("connectionTimeout", Integer.toString(this.timeout)); //$NON-NLS-1$
		properties.setProperty("passive", this.usePassive ? "true" : "false"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		return properties;
	}
		
	private void configure(Properties configuration) throws TeamException {
		Assert.isNotNull(configuration, "configuration"); //$NON-NLS-1$

		// Set the root collection on the WebDAV server.
		// this is a full URL. i.e., http://www.foo.com/myfiles
		String stringUrl = configuration.getProperty("location"); //$NON-NLS-1$
		if (stringUrl == null) {
			throw new TeamException(ITeamStatusConstants.REQUIRED_CONFIGURATION_MISSING);
		}
		try {
			setURL(new URL(stringUrl));
		} catch (MalformedURLException e) {
			throw new TeamException(Policy.bind("FTPSite.Invalid_url_format_1") + stringUrl, e); //$NON-NLS-1$
		}

		// Authentication.
		if (configuration.getProperty("username")!=null) {
			//convert projects to use encrypted data.
			this.username = configuration.getProperty("username"); //$NON-NLS-1$
			this.password = configuration.getProperty("password"); //$NON-NLS-1$
			//Save username & pwd in encrypted form:
			Map authInfo=new HashMap(2);
			authInfo.put("name", username);
			authInfo.put("password", password);
			try {
				Platform.addAuthorizationInfo(getURL(), "", "", authInfo);
			} catch (CoreException e) {
				throw FTPPlugin.wrapException(e);
			} 
		} else {
			Map authInfo=Platform.getAuthorizationInfo(getURL(), "", "");
			if (authInfo!=null) {
				this.username=(String)authInfo.get("name");
				this.password=(String)authInfo.get("password");
			} else {
				this.username=this.password="";
			}
		}

		// Proxy server.
		String proxyString = configuration.getProperty("proxyURL"); //$NON-NLS-1$
		if (proxyString != null && proxyString.length() > 0) {
			try {
				this.proxy = new URL(proxyString);
			} catch (MalformedURLException exception) {
				// Setting the proxy fails if the URL is invalid.
				throw new TeamException(ITeamStatusConstants.INVALID_CONFIGURATION);
			}
		}

		// Timeouts.
		String timeoutString = configuration.getProperty("connectionTimeout"); //$NON-NLS-1$
		if (timeoutString != null && timeoutString.length() > 0) {
			try {
				timeout = Integer.parseInt(timeoutString);
			} catch(NumberFormatException ex) {
				throw new TeamException(Policy.bind("FTPSite.Exception_converting_timeout_value_2"), ex); //$NON-NLS-1$
			}
		}
		
		// usePassive.
		String usePassiveString = configuration.getProperty("passive"); //$NON-NLS-1$
		if (usePassiveString == null) {
			usePassive = true;
		} else {
			usePassive = usePassiveString.equals("true"); //$NON-NLS-1$
		}
	}
	
	/**
	 * @see Site#newProvider(IPath)
	 */
	public TargetProvider newProvider(IPath intrasitePath) throws TeamException {
		return new FTPTargetProvider(this, intrasitePath);
	}
	
	/**
	 * @see Site#getType()
	 */
	public String getType() {
		return TYPE;
	}

	/**
	 * Gets the proxy.
	 * @return Returns a String
	 */
	URL getProxy() {
		return proxy;
	}
	
	/**
	 * Gets the timeout in seconds.
	 * @return Returns a int
	 */
	int getTimeout() {
		return timeout;
	}
	
	/**
	 * Gets the passiveness.
	 * @return Returns a int
	 */
	boolean getUsePassive() {
		return usePassive;
	}
	/**
	 * @see Site#writeObject(ObjectOutputStream)
	 */
	public void writeObject(ObjectOutputStream os) {
		
		Properties configuration = getConfiguration();
			// Flatten the configuration to bytes.
		ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
		try {
			configuration.store(byteStream, null);
			byteStream.close();
	
			os.writeInt(FTPSiteFactory.CONFIG_FORMAT_VERSION);
			os.writeUTF(byteStream.toString());
			os.flush();
		} catch (IOException e) {
			FTPPlugin.logError(Policy.bind("FTPSite.Exception_writing_configuration_to_stream_3"), e); //$NON-NLS-1$
		}
	}
	
}
