/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.webdav.core;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.core.Assert;
import org.eclipse.team.internal.core.target.AuthenticatedSite;
import org.eclipse.team.internal.core.target.ITeamStatusConstants;
import org.eclipse.team.internal.core.target.TargetProvider;

/**
 * New locations are created with the HTTP client defiition required for accessing the server,
 * and the URL of the root collection for the resources, e.g.,
 * <pre>
 * 		Properties properties = new Properties();
 * 		// The location for the files is required.
 * 		properties.put("location", "http://myserver/myfiles/");
 * 		// The configuration of the http client is optional, but must be set up
 * 		// to enable access to the files location.
 * 		properties.put("httpClient.username", "myUsername");
 * 		properties.put("httpClient.password", "myPassword");
 * 		properties.put("httpClient.proxyURL", "http://myproxy:81");
 * 		properties.put("httpClient.connectionTimeout", "2"); 
 * </pre></p>
 */
public class WebDavSite extends AuthenticatedSite {
	public final static String TYPE = "org.eclipse.team.webdav"; //$NON-NLS-1$
	public final static int DEFAULT_TIMEOUT = 60; // seconds
	
	protected String proxy;
	protected int timeout = DEFAULT_TIMEOUT;
	
	public WebDavSite(Properties configuration) {
		try {
			configure(configuration);
		} catch (TeamException e) {
			TeamWebDavPlugin.handle(e);
		}
	}
	
	public WebDavSite(ObjectInputStream is) {
		//not sure yet where we get the bytes from		
		try {
			int verisonId = is.readInt();
			String propertiesString = is.readUTF();
			
			ByteArrayInputStream inputStream = new ByteArrayInputStream(propertiesString.getBytes());
			// Read the properties from the stream.
			Properties properties = new Properties();
			try {
				properties.load(inputStream);
			} catch (IOException e) {
				TeamWebDavPlugin.logError(Policy.bind("Config.error"), e); //$NON-NLS-1$
			}
			configure(properties);	
		} catch (IOException e) {
				TeamWebDavPlugin.logError(Policy.bind("Config.error"), e); //$NON-NLS-1$
		} catch (TeamException e) {
				TeamWebDavPlugin.logError(Policy.bind("Config.error"), e); //$NON-NLS-1$
		}
	}
	
	public void writeObject(ObjectOutputStream os) {
		
		Properties configuration = getConfiguration();
		// Flatten the configuration to bytes.
		ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
		try {
			configuration.store(byteStream, null);
			byteStream.close();
	
			os.writeInt(WebDavSiteFactory.CONFIG_FORMAT_VERSION);
			os.writeUTF(byteStream.toString());
			os.flush();
		} catch (IOException e) {
			TeamWebDavPlugin.logError(Policy.bind("Exception_writing_configuration_to_stream_2"), e); //$NON-NLS-1$
		}
	}

	public String getType() {
		return TYPE;
	}
	
	public Properties getConfiguration() {
		Properties properties = new Properties();
		properties.setProperty("location", getURL().toExternalForm()); //$NON-NLS-1$
		properties.setProperty("httpClient.proxyURL", this.proxy); //$NON-NLS-1$
		properties.setProperty("httpClient.connectionTimeout", Integer.toString(this.timeout)); //$NON-NLS-1$
		return properties;
	}
		
	private void configure(Properties configuration) throws TeamException {
		Assert.isNotNull(configuration, "configuration"); //$NON-NLS-1$

		// Set the root collection on the WebDAV server.
		// this is a full URL. i.e., http://www.foo.com/myfiles
		String stringUrl = configuration.getProperty("location"); //$NON-NLS-1$
		if (stringUrl == null) {
			throw new TeamException(ITeamStatusConstants.REQUIRED_CONFIGURATION_MISSING);
		}
		try {
			setURL(new URL(stringUrl));
		} catch (MalformedURLException e) {
			throw new TeamException(Policy.bind("Invalid_url_format_10") + stringUrl, e); //$NON-NLS-1$
		}

		// Authentication.
		if (configuration.getProperty("httpClient.username")!=null) {
			//convert projects to use encrypted data.
			this.username = configuration.getProperty("httpClient.username"); //$NON-NLS-1$
			this.password = configuration.getProperty("httpClient.password"); //$NON-NLS-1$
			if (username!=null) {
			//Save username & pwd in encrypted form:
				Map authInfo=new HashMap(2);
				authInfo.put("name", username);
				authInfo.put("password", password);
				try {
					Platform.addAuthorizationInfo(getURL(), "", "", authInfo);
				} catch (CoreException e) {
					throw TeamWebDavPlugin.wrapException(e);
				}
			}
		} else {
			Map authInfo=Platform.getAuthorizationInfo(getURL(), "", "");
			if (authInfo!=null) {
				this.username=(String)authInfo.get("name");
				this.password=(String)authInfo.get("password");
			} 
		}
		
		// Proxy server.
		this.proxy = configuration.getProperty("httpClient.proxyURL"); //$NON-NLS-1$

		// Timeouts.
		String timeoutString = configuration.getProperty("httpClient.connectionTimeout"); //$NON-NLS-1$
		if (timeoutString != null && timeoutString.length() > 0) {
			try {
				timeout = Integer.parseInt(timeoutString);
			} catch(NumberFormatException ex) {
				throw new TeamException(Policy.bind("Exception_converting_timeout_value_15"), ex); //$NON-NLS-1$
			}
		}
	}

	/**
	 * @see Site#newProvider()
	 */
	public TargetProvider newProvider(IPath intrasitePath) throws TeamException {
		return new WebDavProvider(this, intrasitePath);
	}
	
	/**
	 * Gets the proxy.
	 * @return Returns a String
	 */
	public String getProxy() {
		return proxy;
	}

	/**
	 * Gets the timeout in seconds.
	 * @return Returns a int
	 */
	public int getTimeout() {
		return timeout;
	}

}
