#if 1
#include <libxml/parserInternals.h>
#else
#include <parserInternals.h>
#endif

/*
   This is a SAX based parser for reading a single input file
   formatted in XML for the DTD specified in ggobi.dtd.

   SAX is an event based parser. As different elements of the document
   are encountered by the parser on its input stream, it invokes
   user-specified callbacks. For our purposes, we currently need only
   handle 3 event types:
      1) the start of a tag
      2) the end of a tag  
      3) regular text data within a tag

   As with all event driven approaches, we can specify a user-level
   data object that can is passed to each event handler/callback.
   This is used to parameterize the particular call. In our case,
   we pass an XMLParserData * object (defined in read_xml.h).
   This basically stores what state in the document we currently have
   and indicates how to interpret the future callbacks, and also
   the ggobid structure which we are attempting to fill in.
   Additionally, it maintains a list of default settings that are 
   to be applied to each row, but which can be set generically in the
   top-level element (ggobidata).
  

   When we encounter the start of tag, we are given the tag name, the
   list of attributes for the tag instance and a reference to the
   parser instance XMLParserData object we specified at the start of the
   parsing. This handler is startXMLElement.  Here, we dispatch
   a call to the appropriate method based on the tag name. Generally,
   this either allocates storage space in the ggobid structure being
   filled in, or stores partial information needed to do this.
   Additionally, for the tag ggobidata, the attributes are processed
   to store the default values for record attributes. These are
   attributes such as color, glyph (size and type). 

   As each tag is encountered, we convert its type to an enumerated
   type and store this as the pending state. This is used to determine
   how to interpret ASCII text within an element.

   When we encounter the end of a tag/element,  the routine
   endXMLElement is called. We use this to increment counters indicating
   which record, variable or edge is next in the stream. These are 
   quasi-global variables that are parser-instance specific and a
   necessary consquence of the event-driven style of parsing.

   Finally, when ASCII text within an XML element is discovered by the
   parser, the routine characters() is called. We call different
   routines (setRecordsValues() and setVariableName()) based on the
   state of the parsing. This was set at the start of each tag/element
   in startXMLElement. 
 
 */

#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <gtk/gtk.h>

#include "read_xml.h"

#include "externs.h"

#include "GGobiAPI.h"


void setColorScheme(const xmlChar **attrs, XMLParserData *data);

void startXMLElement(void *user_data, const xmlChar *name, const xmlChar **attrs);
void endXMLElement(void *user_data, const xmlChar *name);
void Characters(void *user_data, const xmlChar *ch, gint len);
void cumulateRecordData(XMLParserData *data, const xmlChar *ch, gint len);
void xmlSetMissingValue(XMLParserData *data, datad *d, vartabled *vt);
gint getAutoLevelIndex(const char * const label, XMLParserData *data, vartabled *el);
static gboolean setRecordValue(const char *tmp, datad *d, XMLParserData *data);
void resetRecordInfo(XMLParserData *data);
static void releaseCurrentDataInfo(XMLParserData *parserData);

const gchar *XMLSuffixes[] = {"", ".xml", ".xml.gz", ".xmlz"};


/*
 The different XML element names we recognize in the input format.

 countervariable is an automatic convenience  variable that generates
 a new variable in the dataset that does not have corresponding entries
 in the input file. For each record, we generate a value which is the
 index in the dataset of that record (starting at 1).

 Along the same lines, we now have randomuniformvariable. This
 is another automatic variable whose values are generated by the
 act of reading the dataset. 
 I am not certain that this belongs here. It is not reproducible.
 It doesn't generalize to different distributions, or even transformations
 of the U(0, 1) distribution. Perhaps having this feature implemented via
 a plugin which uses the new_datad event signal to add variables to a dataset.
 There is a disconnect between what the author of the dataset wants and
 what the plugin knows. So, having this may well be necessary. 
 One can of course use a scripting language (e.g. R, Python, ...) to 
 create the dataset externall.
 
 One can have any number of randomuniformvariable elements in a dataset,
 which is different from the countervariable. This is implemented by 
 introducing a new variable type (like categorical, real, ...)
 in the vartyped enumeration in vartable.h. This means that we have
 to update code to handle this element in the switch statements
 and this is error-prone. (We can easily overlook some of the switch statements
 and get odd behavior). We would rather have the  vartabled data structure as
 a class and have methods for it. Switch statements are simply not extensible.
 For this case, we only have to update vartable_nbook.c.

*/
const gchar * const xmlDataTagNames[] = {
  "ggobidata",
  "data",        /* DATASET */
  "description",
  "record",
  "records",
  "variables",
  "variable",
  "colormap",
  "color",
        /* variables */
  "realvariable",
  "categoricalvariable",
  "integervariable",
  "countervariable",
  "randomuniformvariable",
  "levels",
  "level",
        /* color scheme */
  "activeColorScheme",
        /* brushing attributes: color first; later, glyph type and size */
  "brush",
        /* data values */
  "real",
  "int",
  "string",
  "na",
  "quickHelp",
  "edges", 
  "edge",
  ""
  };

void
ggobi_XML_warning_handler(void *data, const gchar *msg, ...)
{
  va_list ap;
  xmlParserCtxtPtr p = (xmlParserCtxtPtr) ((XMLParserData*) data)->parser;

  va_start(ap, msg);
  fprintf(stderr, "Warning from XML parsing [%d, %d]: ",
    (int) p->input->line, (int) p->input->col);

  vfprintf(stderr, msg, ap); 

  fflush(stderr);  
}

void
ggobi_XML_error_handler(void *data, const gchar *msg, ...)
{
  va_list ap;
  xmlParserCtxtPtr p = (xmlParserCtxtPtr) ((XMLParserData*) data)->parser;

  fprintf(stderr, "Error in XML parsing [line %d, column %d]: ",
    (int) p->input->line, (int) p->input->col);

  va_start(ap, msg);
  vfprintf(stderr, msg, ap); 

  fflush(stderr);
}


/*
  We also need a version that takes a FILE*
  and reads from it. This is so that we can
  handle reading from standard input.

  The DOM style parsing can be initiated very simply.

  xmlDocPtr doc;
    doc = xmlParseFile(name);
 */



gboolean
data_xml_read (InputDescription *desc, ggobid *gg)
{
  xmlSAXHandlerPtr xmlParserHandler;
  xmlParserCtxtPtr ctx = (xmlParserCtxtPtr) g_malloc(sizeof(xmlParserCtxtPtr));
  XMLParserData data;
  gboolean ok = false;  
  gchar *name = g_strdup(desc->fileName); /* find_xml_file(desc->fileName, NULL, gg); */

  if (name == NULL)
    return (false);

  if (strcmp(name, desc->fileName) != 0) {
    g_printerr("Different input file name and resolved file name. Please report.\n");
  }

  xmlParserHandler = (xmlSAXHandlerPtr) g_malloc(sizeof(xmlSAXHandler));
  /* Make certain this is initialized so that we don't have any references
     to unwanted routines!
   */
  memset(xmlParserHandler, '\0', sizeof(xmlSAXHandler));

  xmlParserHandler->startElement = startXMLElement;
  xmlParserHandler->endElement = endXMLElement;
  xmlParserHandler->characters = Characters;

  xmlParserHandler->error = ggobi_XML_error_handler;
  xmlParserHandler->warning = ggobi_XML_warning_handler;

  initParserData(&data, xmlParserHandler, gg);

  ctx = xmlCreateFileParserCtxt(name);
  if(ctx == NULL) {
    xml_warning("File error:", name, "Can't open file ", &data);
    g_free (name);
    return(false);
  }

  ctx->validate = 1;

  ctx->userData = &data;
  data.parser = ctx;
  data.input = desc;
  ctx->sax = xmlParserHandler;

  xmlParseDocument(ctx);

  ctx->sax = NULL;
  xmlFreeParserCtxt(ctx);

  g_free(xmlParserHandler);
  g_free (name);

  {
    GSList *l;
    datad *d;
    ok = true;
    for (l = gg->d; l; l = l->next) {
      d = (datad *) l->data;
      /* ok &= (d->ncols > 0 && d->nrows > 0); */
      /*-- a simple edge set has no variables --*/
      ok &= (d->nrows > 0);
    }
  }
      
  return (ok);
}

void
initParserData(XMLParserData *data, xmlSAXHandlerPtr handler, ggobid *gg)
{
  data->gg = gg;
  data->current_record = 0;
  data->current_variable = 0;
  data->current_element = 0;
  data->current_data = NULL;

  data->current_color = 0;
  data->reading_colormap_file_p = false;
  data->state = UNKNOWN;
  data->terminateStrings_p = true;
  data->NA_identifier = NULL;

  data->idTable = NULL;
  data->handlers = handler;
  data->defaults.color = -1;
  data->defaults.glyphType = sessionOptions->info->glyph.type;
  data->defaults.glyphSize = sessionOptions->info->glyph.size;
  data->defaults.edgeWidth = -1;  /*-- this has no home in ggobi --*/
  data->defaults.hidden = false;

  data->recordLabelsVariable = -1;

  data->recordString = NULL;
  data->recordStringLength = 0;

  data->autoLevels = NULL;

  data->idNamesTable = NULL;
}

void 
startXMLElement(void *user_data, const xmlChar *name, const xmlChar **attrs)
{
  XMLParserData *data = (XMLParserData*)user_data;
  enum xmlDataState type = tagType(name, false);

  switch (type) {
    case COLORSCHEME:
      setColorScheme(attrs, data);
    break;
    case VARIABLES:
      allocVariables (attrs, data);
    break;
    case VARIABLE:
    case REAL_VARIABLE:
    case INTEGER_VARIABLE:
    case CATEGORICAL_VARIABLE:
    case UNIFORM_VARIABLE:
      newVariable (attrs, data, name);
    break;
    case COUNTER_VARIABLE:
      newVariable (attrs, data, name);
      data->counterVariableIndex = data->current_variable;
    break;

    case CATEGORICAL_LEVELS:
      categoricalLevels(attrs, data);
    break;
    case CATEGORICAL_LEVEL:
      setLevelIndex(attrs, data);
    break;

    case RECORDS:
      setDatasetInfo(attrs, data);
    break;
    case TOP:
      setGeneralInfo(attrs, data);
    break;
    case EDGE:
    case RECORD:
      newRecord(attrs, data);
    break;
    case COLORMAP:
      /*setColorMap(attrs, data);*/  /* requires some thought ... */
    break;
    case BRUSHSTYLE:
      setBrushStyle (attrs, data);
    break;
    case COLOR:
      /*setColormapEntry(attrs, data);*/  /* ditto .... */
    break;
    case DESCRIPTION:
      /* description text pending */
    break;
    case EDGES:
    case DATASET:
      setDataset(attrs, data, type);
    break;

    case REAL:   
    case INTEGER:   
    case STRING:   
    case NA:  
      if(data->recordString) { 
        setRecordValues(data, data->recordString, data->recordStringLength);
        if(type != NA && type != STRING)
          data->current_element++;
        resetRecordInfo(data);
      }
    break;
    case QUICK_HELP:
      break;
    default:
      fprintf(stderr, "Unrecognized XML state %s\n", name); fflush(stderr);    
    break;
  }

  data->state = type;
}

void
setColorScheme(const xmlChar **attrs, XMLParserData *data)
{
  const gchar *tmp;

  tmp = getAttribute(attrs, "file");
  if(tmp) {
     /* process this file to append its color schemes into the global list. */
     read_colorscheme((gchar *)tmp, &data->gg->colorSchemes); 
   }

   tmp = getAttribute(attrs, "name");
   if(tmp) {
       /* resolve the color scheme by name */

     colorschemed *scheme;
     scheme = findColorSchemeByName(data->gg->colorSchemes, tmp);
     if(scheme) {
       data->gg->activeColorScheme = scheme;
       colorscheme_init(scheme);
     } else
       ggobi_XML_error_handler(data,
         "Invalid colorscheme name %s. No such scheme.\n", tmp);
  } else {
     ggobi_XML_error_handler(data, "No colorscheme name specified\n");
  }
}

gint
setLevelIndex(const xmlChar **attrs, XMLParserData *data)
{
  const gchar *tmp = getAttribute(attrs, "value");
  gint itmp;
  datad *d = getCurrentXMLData(data);
  vartabled *el = vartable_element_get (data->current_variable, d);

  data->current_level++; /*-- current_level here ranges from 0 to nlevels-1 --*/

/*-- dfs: placeholder for proper debugging --*/
  if (data->current_level >=  el->nlevels) {
/*XXX Put in a more terminal error! */
    ggobi_XML_error_handler(data, "trouble: adding too many levels to %s\n", el->collab);
  }
/* */

  itmp = data->current_level;
  if (tmp != NULL) {
    itmp = strToInteger (tmp);
    if (itmp < 0) 
      g_printerr ("trouble: levels must be >= 0\n");
  }
  el->level_values[data->current_level] = itmp;
 
  return(data->current_level);
}

/*
 * If the user hasn't supplied level names and values, fill in
 * default values:  Level 0 .... Level (nlevels-1)
*/
void
completeCategoricalLevels (XMLParserData *data)
{
  datad *d = getCurrentXMLData(data);
  vartabled *el = vartable_element_get (data->current_variable, d);
  gint min = 1;

  if (data->current_level == -1) {
    gint k;

    if (el->lim_specified_p)
      min = (gint) el->lim_specified.min;

    /*-- Alert the user what we're about to do --*/
    g_print ("Supplying default level values for \"%s\" ranging from %d:%d\n",
      el->collab, min, min+el->nlevels-1);
    for (k=0; k<el->nlevels; k++) {
      el->level_values[k] = min + k;
/* XXX
 * To be really correct, we should probably free any level_names
 * that may have been already created ...
*/
      el->level_names[k] = g_strdup_printf ("L%d", k+1);
    }
  }
}

void
categoricalLevels(const xmlChar **attrs, XMLParserData *data)
{
  datad *d = getCurrentXMLData(data);
  vartabled *el = vartable_element_get (data->current_variable, d);
  gint i;

  const gchar *tmp = getAttribute(attrs, "count");

  if (tmp != NULL) {
    el->nlevels = strToInteger(tmp);
    if(el->nlevels > 0) {
      el->level_values = (gint *) g_malloc(el->nlevels * sizeof(gint)); 
      el->level_counts = (gint *) g_malloc(el->nlevels * sizeof(gint)); 
      el->level_names = (gchar **) g_malloc(el->nlevels * sizeof(gchar *)); 
      for (i=0; i<el->nlevels; i++)
        el->level_counts[i] = 0;
    } else {
      el->level_values = NULL;
      el->level_counts = NULL;
      el->level_names  = NULL;
    }
  }

  data->current_level = -1; /* We'll increment the first one. */

  if(el->nlevels < 1) {
    fprintf(stderr, "Levels for %s mis-specified\n", el->collab);
    fflush(stderr); 
  }
}

void
addLevel(XMLParserData *data, const gchar *c, gint len)
{
  datad *d = getCurrentXMLData(data);
  vartabled *el = vartable_element_get (data->current_variable, d);

  gchar *val = g_strdup(c);

/*-- dfs: placeholder for proper debugging --*/
  if (data->current_level >= el->nlevels)
    g_printerr ("trouble: adding too many levels to %s\n", el->collab);

/*XXX check not off by one! If so, probably increment data->current_level. */
  el->level_names[data->current_level] = g_strdup(val);

  g_free (val);
}

void
resetRecordInfo(XMLParserData *data)
{
  if(data->recordString)
    g_free(data->recordString);

  data->recordString = NULL;
  data->recordStringLength = 0;
}

void
resolveEdgeIds(XMLParserData *parserData)
{

}

void 
resolveAllEdgeIds(XMLParserData *parserData)
{

}

gint
edgecompare (const void *val1, const void *val2)
{
  const SortableEndpoints *e1 = (const SortableEndpoints *) val1;
  const SortableEndpoints *e2 = (const SortableEndpoints *) val2;
  gint acomp, bcomp;

  acomp = strcmp(e1->a, e2->a);
  bcomp = strcmp(e1->b, e2->b);

  if (acomp < 0 || (acomp == 0 && bcomp < 0))
    return (-1);
  else if (acomp == 0 && bcomp == 0)
    return (0);
  else
    return (1);
}

void 
setEdgePartners (XMLParserData *parserData)
{
  datad *e = getCurrentXMLData(parserData);
  SortableEndpoints *ep;
  gint i, k, n;

  if (e->edge.n <= 0)
    return;

  n = 2 * e->edge.n;
  ep = (SortableEndpoints *) g_malloc (n * sizeof (SortableEndpoints));

  /*-- I'm assuming that we have no edges from i to i --*/
  for (i=0, k=0; i<e->edge.n; i++) {
    ep[k].a = g_strdup ((gchar *) e->edge.sym_endpoints[i].a);
    ep[k].b = g_strdup ((gchar *) e->edge.sym_endpoints[i].b);
    ep[k].jcase = i;
    k++;
    ep[k].a = g_strdup ((gchar *) e->edge.sym_endpoints[i].b);
    ep[k].b = g_strdup ((gchar *) e->edge.sym_endpoints[i].a);
    ep[k].jcase = i;
    k++;
  }
  qsort ((gchar *) ep, n, sizeof (SortableEndpoints), edgecompare);

  for (i=1; i<n; i++) {
    k = i-1;
    if (strcmp(ep[i].a, ep[k].a) == 0 && strcmp(ep[i].b, ep[k].b) == 0) {
      e->edge.sym_endpoints[ ep[i].jcase ].jpartner = ep[k].jcase;
      e->edge.sym_endpoints[ ep[k].jcase ].jpartner = ep[i].jcase;
    }
  }

  for (i=0; i<n; i++) {
    g_free (ep[i].a);
    g_free (ep[i].b);
  }
  g_free (ep);
}


void endXMLElement(void *user_data, const xmlChar *name)
{
  XMLParserData *data = (XMLParserData*)user_data;
  enum xmlDataState type = tagType(name, true);

  switch(type) {
    case EDGES:
      resolveEdgeIds(data);
    case DATASET:
    {
      datad *d = getCurrentXMLData(data);
      setEdgePartners(data);
      releaseCurrentDataInfo(data);
      if (data->current_record < d->nrows) {
        g_printerr ("There are fewer records than declared for '%s'; exiting.\n",
          d->name);
        exit(101);
      }
    }
    break;

    case EDGE:
      data->current_record++;
    break;
    case RECORD:
      setRecordValues(data, data->recordString, data->recordStringLength);
      data->current_record++;
      resetRecordInfo(data);
    break;
    case NA:
      xmlSetMissingValue(data, getCurrentXMLData(data), NULL);
      data->current_element++; 
    break;
    case REAL:
    case INTEGER:
      setRecordValues(data, data->recordString, data->recordStringLength);
    break;
    case STRING:
    /* This is the individual setRecordValue(), i.e. not with an 's' at the end.
    */
      setRecordValue((const char *) data->recordString, data->current_data, data);
      data->current_element++;
    break;
    case VARIABLE:
    case REAL_VARIABLE:
    case CATEGORICAL_VARIABLE:
    case COUNTER_VARIABLE:
    case INTEGER_VARIABLE:
    case UNIFORM_VARIABLE:
      data->current_variable++;
    break;
    case COLOR:
      data->current_color++;
    break;
    case COLORMAP:
       /* Only set this if we are reading from the main file 
          and not a secondary colormap file.
        */
      if(data->reading_colormap_file_p == false)
        GGOBI(registerColorMap)(data->gg);
    break;
    case CATEGORICAL_LEVELS:
      completeCategoricalLevels (data);
    break;
    case CATEGORICAL_LEVEL:
    break;
    case TOP:
         /* resolve all the edges */
      resolveAllEdgeIds(data);      
      break;
    default:
      data = NULL; /* just any code so we can stop.*/
    break;
  }

  if(data) {
    resetRecordInfo(data);
    data->state = UNKNOWN; 
  }
}


XmlTagType
tagType(const xmlChar *name, gboolean endTag)
{
  gint n = sizeof(xmlDataTagNames)/sizeof(xmlDataTagNames)[0] - 1; 
  gint i;
  const gchar *tmp = (const gchar *)name;
  /*
  if(endTag) {
    tmp++;
  }
  */

  for(i = 0; i < n; i++) {
    if(strcmp(tmp, xmlDataTagNames[i]) == 0) {
     return((enum xmlDataState) i);
    }
  }

  return(UNKNOWN);
}


/*
  Called when data within an element is found by the parser.
  Note that the parser does not have to give us all the data
  in one go (although it usually does). Instead,
  it can make several calls to this routine for the same
  element instances. E.g 
    <text>
      a few words
     and another sentence
    </text>
   may result in two calls - one for each sentence.

   This is important to handle as it means that we must be prepared
   to read values within a record across different calls and remember
   which variable/column we last completed for this record.
   This is the current_element field in XMLParserData. 
   Additionally, if a variable name is split across multiple calls
   we must append subsequent calls to the initial value.
  

   Additionally, new lines and leading and trailing white space 
   are not removed. Hence, we must do this ourselves. We use
   skipWhiteSpace to move to first non-whitespace character in the
   string.

   Also, the text is given to us as a a sequence of bytes rather than
   a non-terminated string. Thus, we are told the number of bytes
   rather than being able to use strlen to compute the length of the
   string.
   If the flag terminateStrings is set in the XMLParserData instance,
   this routine takes care of copying the data into a regularly
   NULL-terminated string so that the routines to which the data
   is passed can work with it more easily.
 */
void 
Characters(void *user_data, const xmlChar *ch, gint len)
{
  gchar *tmp = NULL;
  gint dlen = len;
  const xmlChar *c;
  XMLParserData *data = (XMLParserData*)user_data;

  c = (const xmlChar *) skipWhiteSpace(ch, &dlen);
  if(dlen < 1 || c[0] == '\n')
    return;

  if(data->terminateStrings_p) {
    tmp = (gchar *) g_malloc(sizeof(gchar)*(dlen+1));

    memcpy(tmp, c, dlen);
    memset(tmp+dlen, '\0', 1);

    c = (const xmlChar *) tmp;
  }

  switch(data->state) {
    case NA:
    case RECORD:
    case REAL:
    case STRING:
    case INTEGER:
    case UNKNOWN:
        /* Now we call
            setRecordValues (data, c, dlen); 
           after gathering the entire string for the record so that we
           don't get bizarre splits such as  "1.3 1.4 1"  followed by ".4..."
         */
        cumulateRecordData(data, c, dlen);
    break;
    case VARIABLE:
    case CATEGORICAL_VARIABLE:
    case REAL_VARIABLE:
    case UNIFORM_VARIABLE:
      setVariableName (data, c, dlen);
    break;
    case COLOR:
      setColorValue (data, c, dlen);
    break;
    case CATEGORICAL_LEVEL:
      addLevel(data, (const gchar *) c, dlen);
    break;
    default:
    break;
  }

  if(data->terminateStrings_p) {
    g_free(tmp);
  }
}

const xmlChar *
skipWhiteSpace(const xmlChar *ch, gint *len)
{
  const xmlChar *tmp = ch;
  while(*len >= 0) {
    if(*len == 0 || (tmp[0] != ' ' && tmp[0] != '\t' && tmp[0] != '\n'))
      break;
    tmp++;
    (*len)--;
  }

  return(tmp);
}

void
cumulateRecordData(XMLParserData *data, const xmlChar *ch, gint len)
{
  data->recordString = (xmlChar *) g_realloc(data->recordString,
    (len + data->recordStringLength + 1) * sizeof(xmlChar));
  memcpy(data->recordString + data->recordStringLength, ch,
    len * sizeof(xmlChar));
  data->recordStringLength += len;
  data->recordString[data->recordStringLength] = '\0';

  return;
}



/**
  Called in response to a ggobidata tag which contains
  the different datasets. The count element here is the
  number of datasets to expect.
 */
gboolean
setGeneralInfo (const xmlChar **attrs, XMLParserData *data)
{
  const gchar *tmp = getAttribute(attrs, "count");

  if (tmp != NULL) {
    data->expectedDatasetCount = strToInteger(tmp);
  }

  tmp = getAttribute(attrs, "ids");

  return(true);
}


gboolean
setDatasetInfo (const xmlChar **attrs, XMLParserData *data)
{
  const gchar *tmp = getAttribute(attrs, "count");
  datad *d = getCurrentXMLData(data);

  if (tmp == NULL) {
    g_printerr ("No count attribute\n");
    exit(101);
  }

  d->nrows = strToInteger(tmp);
  d->nrows_in_plot = d->nrows;  /*-- for now --*/

  rowlabels_alloc (d);
  br_glyph_ids_alloc (d);
  br_glyph_ids_init (d, data->gg);

  d->edge.n = 0;

  br_color_ids_alloc (d, data->gg);
  br_color_ids_init (d, data->gg);

  setDefaultDatasetValues(attrs, data);

  if (tmp) {
    arrayf_alloc (&d->raw, d->nrows, d->ncols);
    br_hidden_alloc (d);
    br_hidden_init (d);
  }

  data->current_variable = 0;
  data->current_record = 0;
  data->current_variable = 0;
  data->current_element = 0;

/*-- dfs: this seems to be needed; are there more? --*/
  data->idTable = NULL;

  return (true);
}

gboolean
setDefaultDatasetValues(const xmlChar **attrs, XMLParserData *data)
{
  const gchar * tmp = getAttribute(attrs, "missingValue");
  if(tmp != NULL) {
    data->NA_identifier = g_strdup(tmp);
  }

  setGlyph (attrs, data, -1);  
  setColor (attrs, data, -1);
  setHidden (attrs, data, -1);
  return(true);
}

gint
strToInteger(const gchar *tmp)
{
  gint value;

  value = atoi(tmp);

  return(value);
}


const gchar *
getAttribute(const xmlChar **attrs, gchar *name)
{
  const xmlChar **tmp = attrs;
  while(tmp && tmp[0]) {
    if(strcmp(name, (const gchar *)tmp[0]) == 0)
      return((const gchar *)tmp[1]);
    tmp += 2;
  }

  return(NULL);
}

gboolean 
newRecord(const xmlChar **attrs, XMLParserData *data)
{
  datad *d = getCurrentXMLData(data);

  d->readXMLRecord(attrs, data);

  return(true);
}

gboolean
setHidden(const xmlChar **attrs, XMLParserData *data, gint i)
{
  const gchar *tmp;
  datad *d = getCurrentXMLData(data);

  tmp = getAttribute(attrs, "hidden");
  if(tmp) {
    gboolean hidden = asLogical(tmp);

    if (i < 0) {
      data->defaults.hidden = hidden;
    } else
      d->hidden.els[i] = d->hidden_now.els[i] = d->hidden_prev.els[i] = hidden;
  }

  return(tmp != NULL);
}

gboolean
asLogical(const gchar *sval)
{
  guint i;
  gboolean val = false;
  const gchar *const trues[] = {"T","true", "True","1"};
  for(i = 0; i < sizeof(trues)/sizeof(trues[0]); i++) {
    if(strcmp(sval, trues[i]) == 0)
      return(true);
  }

  return(val);
}

gboolean
setColor(const xmlChar **attrs, XMLParserData *data, gint i)
{
  const gchar *tmp;
  gint value = data->defaults.color;
  datad *d = getCurrentXMLData(data);
  colorschemed *scheme = data->gg->activeColorScheme;

  tmp = getAttribute(attrs, "color");
  if(tmp) {
    value = strToInteger(tmp);
  }

/*
 * this is testing against MAXNCOLORS, but it should test against the
 * current scheme
*/
  /*if(value < 0 || value > MAXNCOLORS) {*/
  if(value < 0 || value >= scheme->n) {
    if(tmp)
      xml_warning("color", tmp, "Out of range", data);
  } else {
    if(i < 0)
     data->defaults.color = value;
    else 
     d->color.els[i] = d->color_now.els[i] = d->color_prev.els[i] = value;
  }

  return (value != -1);
}

gboolean
setGlyph(const xmlChar **attrs, XMLParserData *data, gint i)
{
  const gchar *tmp;
  gint value;
  datad *d = getCurrentXMLData(data);

/*
 * glyphSize  0:7
*/
  value = data->defaults.glyphSize;
  tmp = getAttribute(attrs, "glyphSize");
  if (tmp) {
    value = strToInteger(tmp);
  }

  if (value < 0 || value >= NGLYPHSIZES) {
    if (tmp)
      xml_warning ("glyphSize", tmp, "Out of range", data);
  } else {
    if (i < 0) {
      data->defaults.glyphSize = value;
    } else {
      /*
       * note that even if defaults.glyphSize was set below, during
       * the 'glyph' section, the values for record i are assigned here.
      */
      d->glyph.els[i].size = d->glyph_now.els[i].size 
            = d->glyph_prev.els[i].size = value;
    }
  }

/*
 * glyphType  0:6
*/
  value = data->defaults.glyphType;
  tmp = getAttribute(attrs, "glyphType");
  if (tmp) {
    /*
     * make sure this attribute is an integer; if someone puts a
     * string here, like "plus" or "fc", value = 0 and the mistake
     * isn't caught later when value is tested.
    */
    value = mapGlyphName(tmp);
    if (value == UNKNOWN_GLYPH) {
     if(tmp[0] < '0' || tmp[0] > '6') {
       g_printerr ("%s is an illegal value for glyphType; it must be on [0,6]\n", tmp);
       exit(101);
     }

     value = strToInteger(tmp);
    }
  }
  if(value < 0 || value >= NGLYPHTYPES) {
    if(tmp)
      xml_warning("glyphType", tmp, "Out of range", data);
  } else {
    if(i < 0) {
      data->defaults.glyphType = value;
    } else {
      /*
       * note that even if defaults.glyphType was set below, during
       * the 'glyph' section, the values for record i are assigned here.
      */
      d->glyph.els[i].type = d->glyph_now.els[i].type = 
           d->glyph_prev.els[i].type = value;
    }
  }

/*
 * glyph:  strings like "plus 3" or "."
*/
  tmp = getAttribute(attrs, "glyph");
  if(tmp != NULL) {
    const gchar *next;
    gint j;
    next = tmp;
    next = strtok((gchar *)tmp, " ");
    j = 0;
    while(next) {
      if(j == 0) {  /* type */
        value = mapGlyphName(next);
        if(i < 0) {
          data->defaults.glyphType = value;
        } else {
          d->glyph.els[i].type = d->glyph_now.els[i].type =
              d->glyph_prev.els[i].type = value;       
        }
      } else {  /* size */
        value = strToInteger(next);
        if(i < 0) {
          if (value >= 0 && value < NGLYPHTYPES) {
            data->defaults.glyphSize = value;
          } else {
            xml_warning("File error:", next, "glyph improperly specified",
              data);
          }
        } else {
          d->glyph.els[i].size = d->glyph_now.els[i].size =
            d->glyph_prev.els[i].size = value;     
        }
      }
      j++;
      next = strtok(NULL, " ");
    }
  }

  return (value != -1);
}


void
xml_warning(const gchar *attribute, const gchar *value, const gchar *msg,
  XMLParserData *data)
{
  g_printerr ("Incorrect data (record %d)\n",  data->current_record);
  g_printerr ("\t%s %s: value = %s\n",  attribute, msg, value);
}

void
setMissingValue(int i, int j, datad *d, vartabled *vt)
{
  if (d->nmissing == 0) {
    arrays_alloc (&d->missing, d->nrows, d->ncols);
    arrays_zero (&d->missing);
  }
  d->missing.vals[i][j] = 1;
  if(vt == NULL)
    vt = vartable_element_get (j, d);
  vt->nmissing++;
  d->raw.vals[i][j] = 0;
  d->nmissing++;
}

void
xmlSetMissingValue(XMLParserData *data, datad *d, vartabled *vt)
{
  setMissingValue(data->current_record, data->current_element, d, vt);
}


static vartabled *
applyRandomUniforms(datad *d, XMLParserData *data)
{
  vartabled *vt = NULL;
  while(data->current_element < d->raw.ncols 
           && (vt =  vartable_element_get (data->current_element, d)) && vt->vartype == uniform) {
    d->raw.vals[data->current_record][data->current_element] = randvalue();
    vt = vartable_element_get (++(data->current_element), d);
  }  
 
  return(vt);
}

static gboolean
setRecordValue(const char *tmp, datad *d, XMLParserData *data)
{
  gdouble value;
  vartabled *vt;

  /* If we have a counter variable in this dataset, check whether the
     cursor is at that. */
  if(data->counterVariableIndex > -1 &&
     data->current_element == data->counterVariableIndex)
  {
    d->raw.vals[data->current_record][data->current_element] =
      data->current_record + 1;      
    data->current_element++;
  }

  /* If reading past the last column or row, stop */
  if (data->current_record >= d->raw.nrows ||
      data->current_element >= d->raw.ncols)
  {
     g_printerr ("Row %d (counting from 1) has too many elements\n",
       data->current_record+1);
     data->current_element = 0;
     return(false);
  }

/*  vt = vartable_element_get (data->current_element, d); */
  vt = applyRandomUniforms(d, data);
  if(!vt)
    return(true);

  /*
   * this is a missing value if 
   *  1. the file does not specify a string and this string is either
   *     "na", "NA", or ".", or
   *  2. the file specifies a string for NA and this is that string
  */
  if ((data->NA_identifier == NULL &&
        (strcmp (tmp, "na") == 0 ||
         strcmp (tmp, "NA") == 0 ||
         strcmp (tmp, ".") == 0)) ||
      (data->NA_identifier && strcmp (tmp, data->NA_identifier) == 0))
  {
    xmlSetMissingValue(data, d, vt);
  } else {

    value = asNumber (tmp);
    
    if(vt->vartype == categorical) {
      if(data->autoLevels && data->autoLevels[data->current_element]) {
        value = getAutoLevelIndex(tmp, data, vt);
        vt->level_counts[(gint)value]++;
      } else {
        gint level = checkLevelValue(vt, value);
        if (level == -1) {
          ggobi_XML_error_handler(data, 
           "incorrect level in record %d, variable `%s', dataset `%s' in the XML input file\n", 
           (int) data->current_record + 1, vt->collab,
           data->current_data->name ? data->current_data->name : "");
        } else {
          vt->level_counts[level]++;
        }
      }
    } else if(data->state == STRING) {
      ggobi_XML_error_handler(data,
        "<string> element for non categorical variable (%s) in record %d\n",
        vt->collab, (int) data->current_record + 1);
      value = 0;
    }

    d->raw.vals[data->current_record][data->current_element] = value;
  }

  /* If the dataset is using one of the variables as the row labels,
     then resolve the name.
   */
  if(data->recordLabelsVariable == data->current_element) {
    gchar *tmp1;
    /* If this is a categorical, lookup the level id. */
    gchar buf[100];
    if(d->missing.vals &&
       d->missing.vals[data->current_record][data->current_element])
    {
      /* sprintf(buf, "%s", "NA"); */
      tmp1 = g_strdup("NA");
    }
    else {
      if(vt && vt->vartype == categorical) {
        /* To be correct, we need to match the level_values and find the
           corresponding entry. */
        tmp1 = (gchar *) GGobi_getLevelName(vt, value);
        if(tmp1)
          tmp1 = g_strdup(tmp1);
      } else {
        sprintf(buf, "%f", value);
        tmp1 = g_strdup(buf);
      }
    }
    g_array_insert_val(d->rowlab, data->current_record, tmp1);
    g_free (tmp1);
  }

  return(true);
}

/*
  Read the values for this record from free-formatted text. The entries
  are white-space delimited. They should not have quotes or anything
  that needs to be escaped.
*/
gboolean
setRecordValues (XMLParserData *data, const xmlChar *line, gint len)
{
  const gchar *tmp;
  datad *d = getCurrentXMLData(data);

  if(!line) {
     applyRandomUniforms(d, data);
     return(false);
  }

  tmp = strtok((gchar*) line, " \t\n");

  while (tmp && (tmp < (gchar *) (line + len))) {
    if(setRecordValue(tmp, d, data) == false)
        return(false);
    data->current_element++;
    tmp = strtok (NULL, " \t\n");
  }

  applyRandomUniforms(d, data);
  return (true);
}

/*
  Convert the specified string to a numeric value.
 */
gdouble
asNumber(const char *sval)
{
  return(atof(sval));
}




/*
   Read the declaration of a variable, gathering its information
   from the specified attributes.
   This includes its name, transformation name, etc.

   Called in response to a <variable>, <realvariable> or
   <categoricalvariable> tag.
 */

gboolean
newVariable(const xmlChar **attrs, XMLParserData *data, const xmlChar *tagName)
{
  const gchar *tmp, *tmp1;
  datad *d = getCurrentXMLData(data);
  vartabled *el;

  if(data->current_variable >= d->ncols) {
     g_printerr("Too many variables (%d) given number given in the <variables count='%d'> element for dataset %s\n", 
                   data->current_variable, d->raw.ncols, d->name);
    return(false);
  }

  el = vartable_element_get (data->current_variable, d);

  data->variable_transform_name_as_attribute = false;
/*  we don't know what to do with this information yet
  tmp = getAttribute(attrs, "transformName");
  if (tmp) {
    data->variable_transform_name_as_attribute = true;
    el->collab_tform = g_strdup(tmp);
  }
*/

  tmp = getAttribute(attrs, "name");
  /*-- invent a variable name here if the user didn't supply one  */
  if (tmp == NULL)
    tmp = g_strdup_printf ("Var %d", data->current_variable);
  el->collab = g_strdup(tmp);
  if (data->variable_transform_name_as_attribute == false)
    el->collab_tform = g_strdup(tmp);

  tmp = getAttribute(attrs, "nickname");
  if (tmp != NULL) {
    el->nickname = g_strdup(tmp);
    /*-- no nickname_tform; defeats the purpose of having a 2-letter name --*/
    /*-- if word is shorter than 2 characters, g_strndup pads with nulls --*/
  } else el->nickname = g_strndup (el->collab, 2);

  tmp = getAttribute(attrs, "recordLabel");
  if (tmp != NULL) {
    data->recordLabelsVariable = data->current_variable;
  }

  tmp = getAttribute(attrs, "min");
  tmp1 = getAttribute(attrs, "max");
  if(tmp && tmp1) {
    gdouble mn, mx;
    mn = asNumber(tmp);
    mx = asNumber(tmp1);
    el->lim_specified.min = mn < mx ? mn : mx;
    el->lim_specified.max = mn > mx ? mn : mx;
    /* ? */
    el->lim_specified_tform.min = el->lim_specified.min;
    el->lim_specified_tform.max = el->lim_specified.max;

    if(mn > mx) {
      g_printerr ("Minimum is greater than maximum for variable %s\n",
        el->collab);
    }
    el->lim_specified_p = true;
  }


  if (strcmp((const char *)tagName, "categoricalvariable") == 0) {
    el->vartype = categorical;

      /* Mark this as being a variable for which we must compute the levels. */
    if( (tmp = getAttribute(attrs, "levels")) && strcmp(tmp, "auto") == 0) {
      if(data->autoLevels == NULL) {
         data->autoLevels = (GHashTable **)
           g_malloc(sizeof(GHashTable*) * data->current_data->ncols);
         memset(data->autoLevels, '\0',
           sizeof(gboolean) * data->current_data->ncols);
      }
       /* glib-2.0 provides a g_hash_table_new_full with which we can
          specify the `free' routine for elements. This should simplify
          (slightly) releaseCurrentDataInfo(). */
      data->autoLevels[data->current_variable] = g_hash_table_new(g_str_hash,
        g_str_equal);
    }
  } else if (strcmp((const char *)tagName, "integervariable") == 0) {
    el->vartype = integer;
  } else if (strcmp((const char *)tagName, "countervariable") == 0) {
    el->vartype = counter;
  } else if (strcmp((const char *)tagName, "randomuniformvariable") == 0) {
    el->vartype = uniform;
  } /* real by default */

  return (true);
}


/*
   Reads the number of variables in the dataset from the attributes
   and allocates space for them in the ggobid structure.
   At this point, we have the number of records and variables
   and can initialize the data areas of the ggobid structure.

    Called in response to a <variables> tag. (Note the plural.)
 */
gboolean 
allocVariables (const xmlChar **attrs, XMLParserData *data)
{
  const gchar *tmp = getAttribute (attrs, "count");
  datad *d = getCurrentXMLData(data);

  if(tmp == NULL) {
    g_printerr ("No count for variables attribute\n");
    exit(101);
  }

  d->ncols = strToInteger(tmp);


  /* The following is unlikely to happen as we have not got the count
     of the number of rows at this stage. But we leave it here just
     in case there are datasets for which the input gives the count
     in the top-level tag.*/
  if(d->nrows > 0 && d->ncols > 0) {
    arrayf_alloc (&d->raw, d->nrows, d->ncols);
      br_hidden_alloc (d);
  }


  vartable_alloc (d);
  vartable_init (d);


  return (true);
}


/*
  Reads the text in name and assigns it as the name of the
  variable currently being read within the 
  <variable> tag. The index for the variable is stored in 
  data->current_variable.

   Called when parsing free-formatted text within a <variable> tag.
 */
gboolean
setVariableName(XMLParserData *data, const xmlChar *name, gint len)
{
  gchar *tmp = (gchar *) g_malloc (sizeof(gchar) * (len+1));
  datad *d = getCurrentXMLData(data);
  vartabled *el = vartable_element_get (data->current_variable, d);
  gchar *lbl = g_strdup_printf ("Var %d", data->current_variable);

  tmp[len] = '\0';
  memcpy (tmp, name, len);

  /* Handle the case where we have multiple calls to the characters
     handler for the same variable because the data is split
   */
  if (el->collab != NULL) {
    /* need to append tmp to the existing value.*/
  }

  el->collab = tmp;

  /* Note that if we do have multiple calls to this for the same
     variable then we cannot handle the case where the 
     user does not specify the transformation variable
     unless we use a flag in XMLParserData. This is
     variable_transform_name_as_attribute.
   */
  /* problem: this is never NULL, it's either 'Var n' or something specified */
  /*if (el->collab_tform == NULL) {*/
  if (strcmp (el->collab_tform, lbl) == 0) {
    el->collab_tform = g_strdup (tmp);
  }

  g_free (lbl);
  g_free (tmp);

  return (true);
}

/*----------------------------------------------------------------------*/


/*
 Prints the attributes.
 For debugging.
*/
void
showAttributes (const xmlChar **attrs)
{
  const xmlChar **tmp;
  tmp = attrs;
  while (tmp && tmp[0]) {
    g_printerr ("\t %s=%s\n", tmp[0], tmp[1]);
    tmp += 2;
  }
}

/* Finds the directory associated with the specified file.
   Strips away the basename by looking for the last 
   directory separator.
 */
gchar *
getFileDirectory(const gchar *filename)
{
  gchar *tmp;
  tmp =  strrchr(filename, G_DIR_SEPARATOR);
  if(tmp) {
    gint n = tmp - filename + 2;
    tmp = (gchar*) g_malloc(n*sizeof(gchar));
    memcpy(tmp, filename, n);
    tmp[n-1] = '\0';
  } else
   tmp = g_strdup("./");
 
  return(tmp);
}

/*
  Checks for files with different extensions.

  The directory is needed so that we can resolve
  files relative to the location of the original 
  file from which it was referenced,
   e.g. colormap file reference from flea.xml
   should be relative to that one.
 */

gchar *
find_xml_file(const gchar *filename, const gchar *dir, ggobid *gg)
{
  gint i;
  gchar* name = NULL;
  FILE *f;
  gint dirlen = 0;
  const gchar **suffixes = XMLSuffixes;
  gint nsuffixes = sizeof(suffixes)/sizeof(suffixes[0]);

  if(dir)
    dirlen = strlen(dir);

    /* If filename starts with a /, so it is an absolute name,
       then ignore the directory argument.
     */
  if(filename[0] == G_DIR_SEPARATOR)
    dirlen = 0;

  for(i = 0; i < nsuffixes;i++) {
    name = (gchar*) g_malloc(sizeof(gchar) *
      (dirlen + strlen(filename)+strlen(suffixes[i]) + 2));
    sprintf(name,"%s/%s%s", dirlen ? dir : "", filename, suffixes[i]);
    if((f = fopen(name,"r")) != NULL) {
      fclose(f);
      break;
    }
    if(name) {
      g_free(name);
      name = NULL;
    }
  }

  if(name == NULL) {
    /* If we can't find the file, then we should return the filename
       as it might be an http or ftp prefix. Could check this. Later,
       when we know more about the possibilities to expect.
     */
    name = g_strdup(filename);
  }

 return(name);
}

/*
  This is reentrant.  
  First we check the size attribute. Then we check the
  for the specification of an external file.
 */
gboolean
setColorMap(const xmlChar **attrs, XMLParserData *data)
{
  const gchar *tmp, *file; 
  gint size = 0;
  tmp = getAttribute(attrs, "size");
  file = getAttribute(attrs, "file");

  if(tmp)
    size = strToInteger(tmp);
  else {
   if(file == NULL)
     return(false);
  }

  if(file) {
    const gchar *type = getAttribute(attrs, "type");
    if(type != NULL) {
      if(strcmp("xml", type) == 0)
        xmlParseColorMap(file, size, data);
      else
        asciiParseColorMap(file, size, data);
    } else {
       xmlParseColorMap(file, size, data);
    }
  }

 /*
  * This appends the colors, but I don't want to allow more than
  * MAXNCOLORS colors no matter what the user asks for, so I'll be
  * ignoring some of the colors in the set.  -- dfs
  if(size > 0 || file) {
    ggobid *gg = data->gg;
    if(file) {
      gg->ncolors += size;
      gg->ncolors = MIN (gg->ncolors, MAXNCOLORS);
      gg->color_table = (GdkColor *)
        g_realloc (gg->color_table, gg->ncolors * sizeof (GdkColor));
      gg->colorNames = (gchar **)
        g_realloc (gg->colorNames, gg->ncolors * sizeof (gchar *));
      memset(gg->colorNames + (gg->ncolors-size), '\0', size*sizeof(gchar *));
    } else {
      gg->ncolors = size;
      gg->ncolors = MIN (gg->ncolors, MAXNCOLORS);
      gg->color_table = (GdkColor *) g_malloc (size * sizeof (GdkColor));
      gg->colorNames = (gchar **) g_malloc (size * sizeof (gchar *));
      memset(gg->colorNames, '\0', size * sizeof (gchar *));
    }
  }
 */

  return(true);
}

gboolean
setColormapEntry(const xmlChar **attrs, XMLParserData *data)
{
  const gchar * const names[] = {"r", "g", "b"};
  gdouble vals[3] = {-1., -1. , -1.};
  const gchar *tmp;
  gboolean ok = true;
  gint which = data->current_color, i;
  GdkColor *color;
  GdkColormap *cmap = gdk_colormap_get_system ();

  tmp = getAttribute(attrs, "id");

  if(tmp) {
    if(strcmp("bg",tmp) == 0) {
      which = -1;
      /*color = &data->gg->bg_color;*/  /* replaced using activeColorScheme*/
    }
    else  if(strcmp("fg",tmp) == 0) {
      which = -1;
      /*color = &data->gg->bg_color;*/   /* ditto */
    }
    else {
       /* Note that we set the current color to this index.
          Thus we can skip values, etc.
        */
      which = data->current_color = strToInteger(tmp) - 1;
      /*color = data->gg->color_table + which;*/   /* ditto */
    }
  } else {
    /*color = data->gg->color_table + data->current_color;*/ /* ditto */
  }

  for(i = 0; i < 3; i++) {
    const gchar *tmp1 = getAttribute(attrs, (gchar *) names[i]);
    if(tmp1) {
      vals[i] = asNumber(tmp1);
    } else {
      ok = false;
      break;
    }
  }

  if(which > -1 &&
     /*which < data->gg->ncolors &&*/ /* ditto */
     (tmp = getAttribute(attrs, "name") ) )
  {
    /*data->gg->colorNames[which] = g_strdup(tmp);*/ /* ditto*/
  }

  if(ok) {
    setColorValues(color, vals);

   /* If this is a foreground or background setting, then get the color.
      Otherwise, wait until we have finished the entire 
    */
   if(which < 0)
     gdk_colormap_alloc_color(cmap, color, false, true);
  }

  return(ok);
}

/*
  An RGB value in simple text form.
 */
gboolean
setColorValue(XMLParserData *data, const xmlChar *line, gint len)
{
/*
  gdouble values[3] = {-1, -1, -1};
  gint which = 0;
  const gchar *tmp = strtok((gchar*) line, " \t\n");

  GdkColor *color = data->gg->color_table + data->current_color;
 
  while(tmp) {
    values[which++] = asNumber(tmp);
    tmp = strtok(NULL, " \t\n");
  }

  if(which == 3)
    setColorValues(color, values);

*/
  return(true); 
}


void
setColorValues(GdkColor *color, gdouble *vals)
{
  color->red = (guint16) (vals[0]*65535.0);
  color->green = (guint16) (vals[1]*65535.0);
  color->blue = (guint16) (vals[2]*65535.0);
}


/*
 The colormap file will have its own size.
 */

gboolean
xmlParseColorMap(const gchar *fileName, gint size, XMLParserData *data)
{
  xmlParserCtxtPtr ctx;
  gchar *tmp, *tmp1;

  tmp = g_strdup(data->input->dirName);   /* getFileDirectory(data->input->filename); */
  tmp1 = find_xml_file(fileName, tmp, data->gg);

  if(tmp1) {
    ctx  = xmlCreateFileParserCtxt(tmp1);

    if(ctx == NULL)
      return(false);

    ctx->userData = data;
    ctx->sax = data->handlers;
    data->reading_colormap_file_p = true;

    xmlParseDocument(ctx);

    ctx->sax = NULL;
    xmlFreeParserCtxt(ctx);

    data->reading_colormap_file_p = false;

    addInputFile(data->input, tmp1);
    g_free(tmp1);
  }

  g_free(tmp);

/* doesn't use schemes */
/*
  return(size == data->gg->ncolors);
*/
  return (size);
}


/*
  Reads color map entries from an ASCII file as a rectangular array
  of size, at most, size by 3 rows.

  Doesn't do anything at the moment.
 */

gboolean
asciiParseColorMap(const gchar *fileName, gint size, XMLParserData *data)
{
  return(false);
}



/* 
  glib-2.0 defines this as a void routine, i.e. no return value.
  glib-1.2 however expects a return value of type gboolean.
 */
void
freeLevelHashEntry(gpointer key, gpointer value, gpointer data)
{
  g_free(value);
  if(data)
    g_free(key);
/*  return(true); */
}

static void
releaseCurrentDataInfo(XMLParserData *parserData)
{
   if(!parserData->current_data) 
     return;

/*
   if(parserData->idTable && parserData->usesStringIds == false) {
      g_hash_table_foreach(parserData->idTable,
        (GHFunc) freeLevelHashEntry, parserData);
      g_hash_table_destroy(parserData->idTable); 
   }
*/

   if(parserData->autoLevels) {
      int i;
      for(i = 0; i < parserData->current_data->ncols ; i++) {
         if(parserData->autoLevels[i]) {
                 /* don't free the keys (so pass NULL as third argument) 
                    since these are used in the level_names array.*/
          g_hash_table_foreach(parserData->autoLevels[i],
            (GHFunc) freeLevelHashEntry, NULL);
           g_hash_table_destroy(parserData->autoLevels[i]); 
     }
      }
      parserData->autoLevels = NULL;
   }
}


gboolean
setDataset(const xmlChar **attrs, XMLParserData *parserData, enum xmlDataState type) 
{
  datad *data;
  gchar *name;
  const gchar *tmp;

  data = gtk_ggobi_data_new(parserData->gg);

  data->readXMLRecord = readXMLRecord;
  parserData->counterVariableIndex = -1;

  tmp = getAttribute(attrs, (gchar *) "name");
  if(tmp == NULL) {
    name = (gchar *) malloc(sizeof(gchar)*8);
    sprintf(name, "data%d", (gint) g_slist_length(parserData->gg->d));
  } else
    name = g_strdup(tmp);
  data->name = name;

  tmp = getAttribute(attrs, "nickname");
  if (tmp != NULL) {
    data->nickname = g_strdup(tmp);
    /*-- if word is shorter than 5 characters, g_strndup pads with nulls --*/
  } else data->nickname = g_strndup (data->name, 5);

  parserData->current_data = data;

  if(type == EDGES) {
    setDatasetInfo(attrs, parserData);
  }

 return(true);
}


datad *
getCurrentXMLData(XMLParserData* parserData)
{
  datad *data = parserData->current_data;
  if(data == NULL) {
    data = datad_new(NULL, parserData->gg);
    parserData->current_data = data;
  }
  if(data->input == NULL)
    data->input = parserData->input;
  return(data);
}


void
setEdge(gint start, gint end, gint i, datad *d)
{
  /*-- if encountering the first edge, allocate endpoints array --*/
  if (d->edge.n == 0) 
    edges_alloc (d->nrows, d);
}

gboolean
readXMLRecord(const xmlChar **attrs, XMLParserData *data)
{
  datad *d = getCurrentXMLData(data);
  const gchar *tmp;
  gchar *stmp;
  gint i = data->current_record;

  if (i == d->nrows) {
    g_printerr ("There are more records than declared for '%s'; exiting.\n",
      d->name);
    exit(101);
  }

  data->current_element = 0;

  tmp = getAttribute(attrs, "label");
  if(!tmp) {
    if(data->recordLabelsVariable > -1) {
      /* Wait until we have read the specific values! */
    } else {
      /* Length is to hold the current record number as a string. */
      stmp = g_malloc(sizeof(gchar) * 10);
      g_snprintf(stmp, 9, "%d", i);
    }
  } else
    stmp = g_strdup (tmp);

  g_array_insert_val (d->rowlab, data->current_record, stmp);

  setColor(attrs, data, i);
  setGlyph(attrs, data, i);
  setHidden(attrs, data, i);

/*
 * Probably something's missing here:  if any record has an
 * id, then does every record need one?  I think so.  -- dfs
 */
 
  tmp = getAttribute(attrs, "id");
  if(tmp) {
    guint *ptr;
    /*int value;*/
    gchar *dupTmp;
    /* No need to check since this will either be the first and hence
       NULL or already created, so can use an else for this condition. */
    if(data->idTable == NULL) {
      data->idTable = g_hash_table_new(g_str_hash, g_str_equal);
      d->idTable = data->idTable;
      d->rowIds = (gchar **) g_malloc(sizeof(gchar *) * d->nrows);
      memset(d->rowIds, '\0', sizeof(gchar *) * d->nrows);
    } else {
      if(g_hash_table_lookup(data->idTable, tmp))
         ggobi_XML_error_handler(data,
           "duplicated id in record %d of dataset %s\n", 
           data->current_record + 1, data->current_data->name);
    }

    ptr = (guint *) g_malloc(sizeof(guint));
    ptr[0] = i;
    g_hash_table_insert(data->idTable, dupTmp = intern(data, tmp), ptr);
    d->rowIds[i] = dupTmp;
  }

/*
 * Probably something's missing here:  if edges should be
 * present, then every record should have a source and an
 * endpoint, and there's no validation going on now. --dfs
*/

  /* Read the edge source and destination pair if, present. */
  tmp = getAttribute(attrs, "source");   
  if (tmp != (const gchar *) NULL) {
   const gchar *dest;
   dest = getAttribute(attrs, "destination");
   if(!dest) {
      xml_warning("edge specification error:", tmp, "source but no destination attribute for record.", data);
      return(true);
   }
   if (strcmp (tmp, dest) == 0) {
      xml_warning("edge specification error:", tmp, "source is the same as destination for record.", data);
      return(true);
   }

   if(d->edge.sym_endpoints == NULL) {
     d->edge.n = d->nrows;
     d->edge.sym_endpoints = (SymbolicEndpoints *)
       g_malloc(sizeof(SymbolicEndpoints) * d->edge.n);
   }
     
   d->edge.sym_endpoints[data->current_record].a = intern(data, tmp);
   d->edge.sym_endpoints[data->current_record].b = intern(data, dest);
   d->edge.sym_endpoints[data->current_record].jpartner = -1;
  }

  return(true);
}

char * const
intern(XMLParserData *data, const char * const el)
{
 char * ans;

 if(data->idNamesTable == NULL) {
   data->idNamesTable = g_hash_table_new(g_str_hash, g_str_equal);  
 }

 ans = g_hash_table_lookup(data->idNamesTable, el);
 if(!ans) {
   ans = g_strdup(el);
   g_hash_table_insert(data->idNamesTable, ans, ans);
 }
 return(ans);
}


gint
checkLevelValue(vartabled *vt, double value)
{
  gint i;
  for(i = 0; i < vt->nlevels; i++) {
    if(vt->level_values[i] == (int) value)
      return(i);
  }

  return(-1);
}


gboolean
setBrushStyle(const xmlChar ** attrs, XMLParserData * data)
{
  const gchar *tmp;
  gint value;
  gboolean retval = true;

  tmp = getAttribute(attrs, "color");
  if (tmp != NULL) {
    colorschemed *scheme = data->gg->activeColorScheme;
    value = strToInteger(tmp);
    if (value >=0 && value < scheme->n) {
      data->gg->color_id = value;
    } else {
      xml_warning("File error:", tmp, "brushing color improperly specified",
        data);
      return false;
    } 
  }

/*
 * glyphType  0:6
*/
  tmp = getAttribute(attrs, "glyphType");
  if (tmp != NULL) {
    /*
     * make sure this attribute is an integer; if someone puts a
     * string here, like "plus" or "fc", value = 0 and the mistake
     * isn't caught later when value is tested.
    */
    if (tmp[0] < '0' || tmp[0] > '6') {
      xml_warning ("brushing glyphType", tmp, "Must be on [0,6]\n", data);
      return false;
    }
    value = mapGlyphName(tmp);
    if(value == UNKNOWN_GLYPH)
      value = strToInteger(tmp);

    if(value < 0 || value >= NGLYPHTYPES) {
      xml_warning("glyphType", tmp, "Out of range", data);
      return false;
    } else {
      data->gg->glyph_id.type = value;
    }
  }

/*
 * glyphSize  0:7
*/
  tmp = getAttribute(attrs, "glyphSize");
  if (tmp != NULL) {
    value = strToInteger(tmp);

    if (value < 0 || value >= NGLYPHSIZES) {
      xml_warning ("glyphSize", tmp, "Out of range", data);
    } else {
      data->gg->glyph_id.size = value;
    }
  }


  tmp = getAttribute(attrs, "glyph");
  if (tmp != NULL) {
    const gchar *next;
    gint j;
    next = tmp;
    next = strtok((gchar *)tmp, " ");
    j = 0;
    while(next) {
      if(j == 0) {  /* type */
        value = mapGlyphName(next);
        data->gg->glyph_id.type = value;
      } else {     /* size */
        value = strToInteger(next);
        if (value >= 0 && value < NGLYPHTYPES) {
          data->gg->glyph_id.size = value;
        } else {
          xml_warning("File error:", next,
           "brushing glyph improperly specified", data);
          retval = false;
        } 
      }
      j++;
      next = strtok(NULL, " ");
    }
  }

  return retval;
}


gint
getAutoLevelIndex(const char * const label, XMLParserData *data, vartabled *el)
{
  GHashTable *tbl = data->autoLevels[data->current_element];
  gpointer val;
  gint index = -1;
  gint i;
  val = g_hash_table_lookup(tbl, (gconstpointer) label);

  if(!val) {
    gint *itmp;
    gint n =   el->nlevels + 1;
    if(n == 1) {
      el->level_values = (gint *) g_malloc(sizeof(gint) * n);
      el->level_counts = (gint *) g_malloc(sizeof(gint) * n);
      el->level_names = (gchar **) g_malloc(sizeof(gchar *) * n);
      for (i=0; i<el->nlevels; i++)
        el->level_counts[i] = 0;
    } else {
      el->level_values = (gint *) g_realloc(el->level_values, sizeof(gint) * n);
      el->level_counts = (gint *) g_realloc(el->level_counts, sizeof(gint) * n);
      el->level_names = (gchar **) g_realloc(el->level_names, sizeof(gchar *) * n);
    }
    el->level_counts[n-1] = 0;
    el->level_values[n-1] = n-1;
    el->level_names[n-1] = g_strdup(label);

    itmp = (gint *) g_malloc(sizeof(gint));
    *itmp = index = n - 1;
    g_hash_table_insert(tbl, el->level_names[n-1], itmp);
    el->nlevels++;
  } else 
    index = * ((gint *) val);

  return(index);
}

/* Routines for walking the hash table and getting all the row labels
   into an array. */

static void
getLabel(gpointer key, gpointer val, gchar **labels)
{
  if(val) {
    labels[*(guint *)val] = (gchar *) key;
  }
}

gchar **
getRowLabsFromTable(GHashTable *tbl, gchar **names)
{
  if(!names)
     names = (gchar **) g_malloc(sizeof(gchar *) * g_hash_table_size(tbl));

  g_hash_table_foreach(tbl, (GHFunc) getLabel, names);

  return(names);
}

