/*
 * $Id$
 */

/*
 ** Copyright (C) 2001-2003 Dirk-Jan C. Binnema <dirk-jan@djcbsoftware.nl>
 **  
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **  
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; woithout even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **  
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **  
 */
#ifdef HAVE_CONFIG
#include <config.h>
#endif /*HAVE_CONFIG*/

#include <stdio.h>
#include <unistd.h>
#include <libgnuvd/libgnuvd.h>


#ifndef VERSION
#define VERSION "?.?"
#endif /*VERSION*/


static int  show_results     (Buffer *results);
static char get_special_char (char *str);
static void help             ();

int
main (int argc, char* argv[])
{
	int result, c;
	int show_stress =  0;

	char *word = NULL;
	
	VDQuery *vd_query  = NULL;
	VDError vd_ex;

	while (1) {

		int c = getopt (argc, argv, "sh");
		switch  (c) {
		case -1:
			break;
		case 'h':
			help();
			return 0;
		default:
			help();
			return 1;
		}

		if (optind >= argc) {
			help();
			return 1;
		}  else
			break;
	}
			
	word  = argv[optind];
	vd_ex = VD_ERR_OK;
	
	vd_query = vd_query_new (word, &vd_ex);
	if (!vd_query) {
		if (vd_ex != VD_ERR_OK)
			fprintf (stderr, "error: %s\n", vd_error_string(vd_ex));
		else 	/* should not happen */	
			fprintf (stderr, "an unknown error occured\n");
		return 1;
	}
			
	vd_query_perform (vd_query, &vd_ex);
	if (vd_ex == VD_ERR_NOT_FOUND) {
		fprintf (stderr, "'%s' was not found in the dictionary\n", word); 
		vd_query_destroy (vd_query);
		return 2;	
	} else if (vd_ex != VD_ERR_OK) {
		fprintf  (stderr, "error: %s\n", vd_error_string (vd_ex));
		vd_query_destroy (vd_query);
		return 1;	
	} 

	show_results (vd_query->_result_buffer);
	vd_query_destroy (vd_query);
	
	return 0;
}


static int
show_results (Buffer* buffer)
{
	BufferIterator i;
	int j;
	char *data;
	char k;

	const int s_kars_max = 10;
	char s_kars[s_kars_max+2];

	int skip_spc     = 1;
	char *stress_pos = NULL;

	/* strip superfluous space after */
	for (i = buffer_end (buffer) - 1; i >= buffer_begin(buffer) + 3; --i) 
		if ((buffer_at (buffer, i-3) == 'D') &&
		    (buffer_at (buffer, i-2) == 'L') &&
		    (buffer_at (buffer, i-1) == '>'))
			break;
	buffer_erase (buffer, i, buffer_end(buffer) - i - 1);
	data = (char*) buffer_data (buffer);

	for (j = buffer_begin (buffer); j < buffer_end(buffer); ++j) {
		
		/* DL->\n */
		if (buffer_end(buffer) - j > 5 && strncmp(&data[j], "<DL>", 4) == 0) {
			j += 4;
			continue;
		}
	
		/* DD->\n */
		if (buffer_end(buffer) - j > 8 && strncmp(&data[j], "<DD><B>", 7) == 0) {
			j += 6;
			printf ("\n");
			continue;
		}

		/* 
		 *  Insert a ' at the start of a stressed syllable
		 *  contributed by Berteun <berteun@dds.nl> 
		 */
		if (buffer_end(buffer) - j > 6) { 
			if ((strncmp (&data[j], "<BIG>", 5) == 0) /* Start of word */
			    || (strncmp (&data[j], "&#183;", 6) == 0)) { /* Middle of word */
                               stress_pos = strstr(&data[j], "<U>"); /* Next stress */
                               if(stress_pos && stress_pos < strstr(&data[j+1], "<DD>")
				  && (stress_pos < strstr(&data[j+1], "&#183;")
				      /* happens at the last syllable of the last word */
				      || strstr(&data[j+1], "&#183;") == NULL) ) {
                                       printf ("'");
                                       j += (strncmp(&data[j], "<BIG>", 5) == 0) ? 4 : 5;
                                       continue;
                               }
                       }
		}

		/* remove tags */
		if (data[j] == '<' || data[j] == '\t') {
			while (j < buffer_end(buffer) && data[j] != '>')
				++j;
			continue;
		}
		
		/* replace special chars */
		if (data[j] == '&') {
			int c = 0;
			while (j < buffer_end(buffer) && data[j] != ';' && c < s_kars_max)
				s_kars [c++] = data[j++];
			if (data[j] == ';') {
				s_kars [c++] = ';';
				s_kars [c]   = '\0';

				k = get_special_char (s_kars);
				if (k)
					printf ("%c", k);
			}
			continue;
		}
		
		/* skip superfluous space at beginning */
		if (skip_spc) 
			if (isspace(data[j]))
				continue;
			else
				skip_spc = 0;
		
		printf ("%c", data[j]);
	}
}


static char
get_special_char (char *str)
{
	int i;
	
	struct map_entry {
		char *entity;
		char ascii;
	};
	
	static struct map_entry symbol_map[]  = {
		/* special chars */
				
		{ "&#36;",     '$'},
		{ "&#37;",     '%'},
		{ "&#38;",     '&'},
		{ "&#39;",     '\''},
		{ "&#64;",     '@'},
		{ "&#126;",    '~'},
		{ "&#167;",    ''},
		{ "&#176;",    ''},
		{ "&#178;",    ''},
		{ "&#180;",     0 },
		{ "&#183;",     '-' },
		{ "&#229;",     ''},
		{ "&#956;",     ''},
		

		/* A/a */
		{ "&agrave;",  ''},
		{ "&acute;",   ''},
		{ "&acirc;",   ''},
		{ "&auml;",    ''},
		{ "&aring;",   ''},
		{ "&aelig;",   ''}, 
		
		{ "&Agrave;",  ''},
		{ "&Acute;",   ''},
		{ "&Acirc;",   ''},
		{ "&Auml;",    ''},
		{ "&Aring;",   ''},
		{ "&Aelig;",   ''}, 

		/* C/c */
		{ "&ccedil;",  ''},
		{ "&Ccdeil;",  ''},
		
		/* E/e */
		{ "&egrave;",  ''},
		{ "&ecute;",   ''},
		{ "&ecirc;",   ''},
		{ "&euml;",    ''},
		{ "&eacute;",  ''}, /* weird -> 'sao tom_' */ 
			
		{ "&Egrave;",  ''},
		{ "&Ecute;",   ''},
		{ "&Ecirc;",   ''},
		{ "&Euml;",    ''},
	
		/* I/i */
		{ "&igrave;",  ''},
		{ "&icute;",   ''},
		{ "&icirc;",   ''},
		{ "&iuml;",    ''},
			
		{ "&Igrave;",  ''},
		{ "&Icute;",   ''},
		{ "&Icirc;",   ''},
		{ "&Iuml;",    ''},

		/* n */
		{ "&ntilde;",   ''},
		{ "&Ntilde;",   ''},
		
		
		/* O/o */
		{ "&ograve;",  ''},
		{ "&ocute;",   ''},
		{ "&ocirc;",   ''},
		{ "&ouml;",    ''},
		{ "&oslash;",  ''},
		{ "&otilde;",  ''},
		
		{ "&Ograve;",  ''},
		{ "&Ocute;",   ''},
		{ "&Ocirc;",   ''},
		{ "&Ouml;",    ''},
		{ "&Oslash;",  ''},
		{ "&Otilde;",  ''},
		
		/* U/u */
		{ "&ugrave;",  ''},
		{ "&ucute;",   ''},
		{ "&ucirc;",   ''},
		{ "&uuml;",    ''},
			
		{ "&Ugrave;",  ''},
		{ "&Ucute;",   ''},
		{ "&Ucirc;",   ''},
		{ "&Uuml;",    ''},
		
		/* ij */
		{ "&yuml;",    ''}};
	

	for (i = 0; i != sizeof (symbol_map) / sizeof (struct map_entry); ++i)
		if (strcmp (str, symbol_map[i].entity) == 0)
			return symbol_map[i].ascii;
	return '?';
}



static void
help()
{
	printf ("gnuvd version " VERSION  ",\n"
		"(c) 2001-2003 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>\n\n"
		"usage: gnuvd [options] word\n"
		"where [options] are:\n"
		"\t-h: show this help message\n");
}


