/*********************************************************************
 *
 *      Copyright (C) 2002-2003 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: IdentifierNode.java 1086 2003-12-04 08:41:02Z nfiedler $
 *
 ********************************************************************/

package com.bluemarsh.jswat.expr;

import com.bluemarsh.jswat.parser.java.node.Token;
import com.bluemarsh.jswat.util.VariableValue;
import com.bluemarsh.jswat.util.Variables;
import com.sun.jdi.ThreadReference;

/**
 * Class IdentifierNode represents an identifier. It may refer to a
 * variable (including arrays) or a method name.
 *
 * @author  Nathan Fiedler
 */
class IdentifierNode extends AbstractNode {
    /** Identifier name of this identifier node. */
    private String identifierName;

    /**
     * Constructs a IdentifierNode with the given identifier name.
     *
     * @param  node  lexical token.
     * @param  name  identifier name.
     */
    public IdentifierNode(Token node, String name) {
        super(node);
        identifierName = name;
    } // IdentifierNode

    /**
     * Append the given string to the end of the identifier name.
     *
     * @param  s  suffix to append.
     */
    public void append(String s) {
        identifierName += s;
    } // append

    /**
     * Returns the value of this node.
     *
     * @param  context  evaluation context.
     * @return  value.
     * @throws  EvaluationException
     *          if there was an evaluation error.
     */
    protected Object eval(EvaluationContext context)
        throws EvaluationException {

        // Evaluate the name as a variable reference.
        ThreadReference th = context.getThread();
        if (th == null) {
            throw new EvaluationException(
                Bundle.getString("error.ident.thread"));
        }
        try {
            return Variables.getValue(identifierName, th, context.getFrame());
        } catch (Exception e) {
            throw new EvaluationException(
                Bundle.getString("error.ident.failed"), e);
        }
    } // eval

    /**
     * Simply returns the name of the identifier, without evaluation.
     *
     * @return  identifier name.
     */
    public String getName() {
        return identifierName;
    } // getName

    /**
     * Returns the signature of the type this node represents. If the
     * type is void, or otherwise unrecognizable, an exception is
     * thrown.
     *
     * @param  context  evaluation context.
     * @return  type signature, or null if value is null.
     * @throws  EvaluationException
     *          if an error occurred during evaluation.
     */
    protected String type(EvaluationContext context)
        throws EvaluationException {

        // Evaluate the name as a variable reference.
        ThreadReference th = context.getThread();
        if (th == null) {
            throw new EvaluationException(
                Bundle.getString("error.ident.thread"));
        }
        try {
            VariableValue fav = Variables.getField(
                identifierName, th, context.getFrame());
            return fav.type().signature();
        } catch (Exception e) {
            throw new EvaluationException(
                Bundle.getString("error.ident.failed"), e);
        }
    } // type
} // IdentifierNode
