/**
 * @file   adin_oss.c
 * 
 * <EN>
 * @brief  A reference sample of A/D-in plugin
 *
 * This file describes the specifications of plugin functions to be
 * defined to make an A/D-in plugin.  An A/D-in plugin will extend a
 * new audio sream input into Julius by addin a new choice to the
 * "-input" option.
 *
 * The recording format should be 16 bit (signed short), and sampling
 * rate should be set to the given value at adin_standby().
 * 
 * </EN>
 * 
 * <JA>
 * @brief  ǥϥץ饰ΤҤʷ
 *
 * Υեϡǥϥץ饰ݤ٤
 * 饰ؿˤĤƲ⤷Ƥ롥ǥϥץ饰ϡ
 * Julius ˿ʲϥǥХɲä롥"-input" ˿
 * ɲä졤¹Ի Julius ФƤꤹ뤳ȤǡΥץ
 * ͳǲǧ뤳ȤǤ롥
 *
 * ǥϥץ饰Ǽ٤ǡΥեޥåȤ 16bit 
 * 뤳ȡˡץ󥰥졼Ȥ adin_standby() ƤӽФ
 * Ϳ졼Ȥ˹碌뤳ȡ
 * 
 * </JA>
 *
 * Common functions that can be defined in any type of plugin:
 *   - get_plugin_info()
 *   - initialize()
 * 
 * A/D-in plugin functions:
 * 
 * Required:
 *   - adin_get_optname()
 *   - adin_get_configuration()
 *   - adin_standby()
 *   - adin_open()
 *   - adin_read()
 *   - adin_close()
 *
 * Optional:
 *   - adin_terminate()
 *   - adin_pause()
 *   - adin_resume()
 * 
 * 
 * @author Akinobu Lee
 * @date   Thu Aug  7 14:28:37 2008
 * 
 * $Revision: 1.1 $
 * 
 */

/***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "plugin_defs.h"

/**
 * <EN>
 * Description string of this plugin file.
 * </EN>
 * <JA>
 * Υץ饰եʸ
 * </JA>
 * 
 */
#define PLUGIN_TITLE "A/D-in plugin for Julius"

/**
 * <EN>
 * string to be specified at "-input" option at Julius to use this plugin
 * as input module.
 * </EN>
 * <JA>
 * Υץ饰ѤƲϤԤݤˡJulius "-input" ץ
 * Ϳ٤ʸ.
 * </JA>
 * 
 */
#define INPUT_OPT "myadin"

/** 
 * <EN>
 * @brief  Initialization at loading time (optional)
 * 
 * If defined, this will be called just before this plugin is loaded to Julius.
 * if this returns -1, the whole functions in this file will not be loaded.
 *
 * This function is OPTIONAL.
 * </EN>
 * <JA>
 * @brief  ɤ߹߻νǤա
 *
 * ưJulius Υץ饰ɤ߹ݤ˺ǽ˸ƤФ롥
 * -1 ֤ȡΥץ饰Τɤ߹ޤʤʤ롥
 * ¹ԲǽΥå˻Ȥ롥
 *
 * </JA>
 * 
 * 
 * @return 0 on success, -1 on failure.
 * 
 */
int
initialize()
{
  return 0;
}

/** 
 * <EN>
 * @brief  Get information of this plugin (required)
 *
 * This function should return informations of this plugin file.
 * The required info will be specified by opcode:
 *  - 0: return description string of this file into buf
 *
 * This will be called just after Julius find this file and after
 * initialize().
 *
 * @param opcode [in] requested operation code
 * @param buf [out] buffer to store the return string
 * @param buflen [in] maximum length of buf
 *
 * @return 0 on success, -1 on failure.  On failure, Julius will ignore this
 * plugin.
 * 
 * </EN>
 * <JA>
 * @brief  ץ饰ɬܡ
 *
 * Υץ饰˴ؤ֤Ϳ줿 opcode ˤäư롥
 *  - 0 ξ硤Υץ饰ե̾ΤͿ줿Хåե˳Ǽ
 *
 * δؿϡJulius Υץ饰ɤ߹ľ˸ƤФ롥
 * 
 * @param opcode [in] ׵ư ( 0 Τ߼)
 * @param buf [out] ͤǼХåե
 * @param buflen [in] buf κĹ
 * 
 * @return 顼 -1,  0 ֤顼Ȥ -1 ֤硤
 * Υץ饰Τɤ߹ޤʤ
 * </JA>
 * 
 */
int
get_plugin_info(int opcode, char *buf, int buflen)
{
  switch(opcode) {
  case 0:
    /* plugin description string */
    strncpy(buf, PLUGIN_TITLE, buflen);
    break;
  }

  return 0;
}

/************************************************************************/
/************************************************************************/
/* A/D-in plugin functions */

/** 
 * <EN>
 * @brief  Return option string to select at option. (required)
 *
 * This function should return option string which should be specified
 * as an argument "-input" option, to be used on Julius.  The returning
 * string should not be the same with any existing value.
 *
 * This function will be called several times at option parsing at startup.
 *
 * @param buf [out] buffer to store the return string
 * @param buflen [in] maximum length of buf
 * </EN>
 * <JA>
 * @brief  ѤΥץʸ֤ɬܡ
 *
 * Υץ饰ϤȤ򤹤ݤˡ"-input" ץǻꤹ
 * ٤ʸǼ֤֤ʸϡƥˤǤˤΤ䡤
 * ¾Υץ饰󤬻ѤƤΤƱǤʤȡ
 * ʤ⤷Ʊä祷ƥ¦ͥ褵
 *
 * δؿϡưΥץϻ˲٤ƤФ롥
 *
 * @param buf [out] ͤǼ֤Хåե
 * @param buflen [in] buf κĹ
 * </JA>
 * 
 */
void
adin_get_optname(char *buf, int buflen)
{
  strncpy(buf, INPUT_OPT, buflen);
}

/** 
 * <EN>
 * @brief  Return decoder parameter values related to this adin plugin (required)
 * 
 * This function should return configuration values about how to set up
 * decoder to use this adin plugin.  The return value is dependent on
 * the given opcode, as described below:
 *
 * opcode = 0:  return whether real-time processing of 1st pass
 * should be enabled by default.
 *
 * if returns 0 (disabled) , Julius will do buffered input, spooling
 * the incoming input till EOF or silence cut segmentation, extract
 * feature vector, then recognize the whole.  If returns 1 (enabled),
 * on-the-fly decoding will be performed, reading input and decoding
 * it concurrently.
 *
 * A real-time decoding uses some approximation on feature extraction
 * related to sentence-based normalization i.e. CMN or energy normalization.  
 * This value is typically 0 on off-line recognition, and 1 for on-line
 * recognition.
 *
 * This value is device-dependent default value, and can be overridden by
 * user option "-realtime" and "-norealtime".
 *
 * opcode = 1: return whether silence cut segmentation should be
 * enabled by default
 *
 * return 0 to disable, 1 to enable.
 * 
 * On file input, you can choose whether silence detection and
 * segmentation should be performed before recognition.  On live input
 * like microphone, where input stream is infinite, you would perfer
 * choose 1 to enable it.
 * 
 * This value is device-dependent default value, and can be overridden by
 * user option "-cutsilence" and "-nocutsilence".
 *
 * opcode = 2: return whether input threading is necessary or not.
 * 
 * On Unix, when set to 1, Julius forks a separate thread for A/D-in
 * input.  It can be useful when recognition is slow and some audio
 * inputs are dropped.  Note that this should be enabled only for
 * infinite input like microphone or line input, since EOF handling on
 * threaded mode is not supported yet.  Recommended value is 1 for
 * microphone input, 0 for file and network (tcp/ip) input.
 * Ignored on Win32.
 * 
 * @param opcode [in] requested operation code
 * 
 * @return values required for the opcode as described.
 * </EN>
 * 
 * <JA>
 * @brief  Ϥΰ˴ؤѥ᡼֤ɬܡ
 *
 * Julius ϥץ饰ɤ٤ˤĤơѥ᡼
 * ֤Ϳ줿ʲ opcode Ȥˡ֤ͤ
 *
 * opcode = 0: ꥢ륿ǧԤɤΥǥե
 *
 * 1 ֤ȡJulius ϤФħФǧʿԤƹԤ
 * ꥢ륿ǧԤ0 ξ硤äϤüʤ뤤϶ڤ
 * ޤǼäƤ顤ħФԤθǧ򳫻Ϥ롥
 * ꥢ륿ǤϡCMN 䥨ͥ륮ʿѤʤɡȯΤѤ
 * ħ̤롥
 *
 * ̾ޥϤʤɥꥢ륿ʷ̤ߤ 1
 * եϤʤɥե饤ǧξ 0 ֤Ȥ¿
 *
 * ʤͤϡϤꤹǥեͤǤꡤ
 * Juliusμ¹Իץ "-realtime", "-norealtime" ǤѹǤ롥
 * ץ󤬻ꤵ줿Ϥλ꤬ͥ褵롥
 * 
 * opcode = 1: ̵ָФˤ϶ڤΥǥե
 *
 * Julius ϲФƿ򺹤ˤȽԤ
 * ٥ʲʬ򥹥åפǶڤäϤȤ뤳ȤǤ
 * 롥̵ǤμưڤΥǥեȤ 1 ͭ 0 ̵
 * ȤǤ롥
 *
 * ̾ޥʤɤľϤǤ 1ȯäȤβեǤ 0 
 * ֤Ȥ¿
 *
 * ʤͤϡϤꤹǥեͤǤꡤ
 * Juliusμ¹Իץ "-cutsilence", "-nocutsilence" ǤѹǤ롥
 * ץ󤬻ꤵ줿Ϥλ꤬ͥ褵롥
 *
 * opcode = 2: Ϥ򥹥åɲ뤫Υǥե
 *
 * ϼ̥åɤˤ뤫ɤ򤹤롥
 * ǧ®٤٤ǡμꤳܤȯͭǤ롥
 * ߤJuliusǤϡEOF ˤǧλʤΤǡ
 * ޥϤʤɤĹͭ¤ǤʤϤˤĤƤΤߥåɲͭ
 * ٤Ǥ롥
 *
 * ̾ޥ UDP ʤɤǤ 1 ˤե TCP/IP åȤǤ
 * 0 ˤ롥
 *
 * @param opcode [in] ׵ư ( 0 Τ߼)
 * 
 * @return opcode Ȥ׵ᤵ줿֤ͤ
 * </JA>
 */
int
adin_get_configuration(int opcode)
{
  /* For your convenience, UNCOMMENT ONE OF THEM BELOW that match your needs */

  /* typical values for live microphone/line input */
  switch(opcode) {
  case 0:	
    return 1;
  case 1:
    return 1;
  case 2:
    return 1;
  }
  /* typical values for offline file input */
  /* 
   * switch(opcode) {
   * case 0:	   
   *   return 0;
   * case 1:
   *   return 0;
   * case 2:
   *   return 0;
   * }
   */
  /* typical setting for tcpip input */
  /* assuming speech to be segmented at sender */
  /* 
   * switch(opcode) {
   * case 0:	   
   *   return 1;
   * case 1:
   *   return 0;
   * case 2:
   *   return 0;
   * }
   */
  /* typical setting for tcpip input */
  /* assuming receiving continous speech stream and segmented
     should be done at Julius side */
  /* 
   * switch(opcode) {
   * case 0:	   
   *   return 1;
   * case 1:
   *   return 1;
   * case 2:
   *   return 0;
   * }
   */
}
 

/************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>
static int audio_fd;
static int freq;

/**
 * <EN>
 * @brief  Initialize input device (required)
 *
 * This will be called only once at start up of Julius.  You can
 * initialize the device, check if the device exists or prepare a socket
 * for connection.
 *
 * If this function returns FALSE, Julius will exit.
 * 
 * JuliusLib: this function will be called at j_adin_init().
 *
 * @param sfreq [in] required sampling frequency
 * @param dummy [in] a dummy data
 * 
 * @return TRUE on success, FALSE on failure.
 * </EN>
 * <JA>
 * @brief  ǥХɬܡ
 *
 * δؿϵư˰ƤФ롥ǤϥǥХΥå
 * åȤѰդȤäϤΤνԤΤ˻Ȥ
 *
 * FALSE ֤硤Julius Ͻλ롥
 * 
 * JuliusLib: δؿ j_adin_init() ǸƤФ롥
 *
 * @param sfreq [in] ץ󥰥졼
 * @param dummy [in] ߡǡ̤ѡ
 * 
 * @return  TRUEԻ FALSE ֤
 * </JA>
 */
boolean
adin_standby(int sfreq, void *dummy)
{
  /* store the frequency */
  freq = sfreq;
  return TRUE;
}
 
/**
 * <EN>
 * @brief  Open an input stream (required)
 *
 * This function should open a new audio stream for input.
 * You may open a capture device, open an audio file, or wait for
 * connection with other network client at this function.
 *
 * If this function returns FALSE, Julius will exit recognition loop.
 * 
 * JuliusLib: this will be called at j_open_stream().
 * 
 * @return TRUE on success, FALSE on failure.
 * </EN>
 * <JA>
 * @brief  ϲȥ꡼򳫤ɬܡ
 *
 * ϲȥ꡼򿷵˳̾ǥХեΥץ
 * ͥåȥ饤Ȥ³ʤɤ򤳤ǹԤ
 *
 * FALSE ֤ȤJulius ǧ롼פȴ롥
 * 
 * JuliusLib: δؿ j_open_stream() ǸƤФ롥
 * 
 * @return  TRUEԻ FALSE ֤
 * </JA>
 */
boolean
adin_open()
{
  /* do open the device */
  int fmt;
  int stereo;
  int ret;
  int s;
  char buf[2];

  if ((audio_fd = open("/dev/dsp", O_RDONLY)) == -1) {
    printf("Error: cannot open /dev/dsp\n");
    return FALSE;
  }
  fmt = AFMT_S16_LE;               /* 16bit signed (little endian) */
  if (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &fmt) == -1) {
    printf("Error: failed set format to 16bit signed\n");
    return FALSE;
  }
  stereo = 0;			/* mono */
  ret = ioctl(audio_fd, SNDCTL_DSP_STEREO, &stereo);
  if (ret == -1 || stereo != 0) {
    stereo = 1;
    ret = ioctl(audio_fd, SNDCTL_DSP_CHANNELS, &stereo);
    if (ret == -1 || stereo != 1) {
      printf("Error: failed to set monoral channel\n");
      return FALSE;
    }
  }
  s = freq;
  if (ioctl(audio_fd, SNDCTL_DSP_SPEED, &s) == -1) {
    printf("Erorr: failed to set sample rate to %dHz\n", freq);
    return FALSE;
  }

  /* start recording... */
  read(audio_fd, buf, 2);

  return(TRUE);
}

/**
 * <EN>
 * @brief  Read samples from device (required)
 *
 * This function is for reading samples to be recognized from input stream.
 * This will be called repeatedly at each time the read samples are fully
 * processed.
 *
 * The sampling format should be 16bit, 1 channel.
 *
 * @a sampnum is the maximum number of samples that can be read into @a buf.
 * The actual number of read samples should be returned.
 *
 * Impotant notes about I/O blocking:
 *  - Do not wait until all the @a sampnum samples are read.
 *    Blocking inside this function will block the whole recognition process.
 *    If device allows, it is better to read only the available data
 *    in the stream and return immediately.
 *  - Avoid returning value of 0 when no data is available, wait for some
 *    data to come inside this function.  When you are using non-blocking
 *    operation, you may want to return 0 when no data is available.
 *    However, returning 0 will cause Julius to call again this function
 *    immediately, and cause busy loop to make CPU load to reach 100%.
 *
 * So the ideal operation will be first wait_for_some_data_to_come, and
 * if any data becomes available, read them at most @a sampnum samples
 * and return the number of read samples.
 *
 * Positive return value should be the number of read samples, or one
 * of ADIN_EOF, ADIN_SEGMENT or ADIN_ERROR.  Return value of ADIN_EOF
 * tells end of stream, which causes Julius to finish current
 * recognition and close stream.  ADIN_SEGMENT requests Julius to
 * segment the current input.  The current recognition will be stopped
 * at this point, recognition result will be output, and then Julius
 * continues to the next input.  The behavior of ADIN_SEGMENT is
 * similar to ADIN_EOF except that ADIN_SEGMENT does not close/open
 * stream, but just stop and restart the recognition.  At last, return
 * value should be ADIN_ERROR on error, in which Julius exits itself
 * immediately.
 * 
 * @param buf [out] output buffer to store samples obtained.
 * @param sampnum [in] maximum number of samples that can be stored in @a buf.
 * 
 * @return actural number of read samples, ADIN_EOF on end of stream,
 * ADIN_SEGMENT to request segmentation to Julius, or ADIN_ERROR on error.
 * </EN>
 * <JA>
 * @brief  ǥХ饵ץɤ߹ɬܡ
 *
 * δؿϥȥ꡼फ鲻ץɤ߹ࡥ
 *
 * Хåե˳Ǽ֤ǡη 16bit, 1 ͥǤ뤳ȡ
 * 
 * @a sampnum  @a buf ˳Ǽ뤳ȤΤǤΥץǤ롥
 * ֤ͤȤơºݤɤ߹ޤ줿ץ뤤ϰʲ
 * 顼ɤ֤
 * 
 * δؿǧ˲٤ƤФ졤ɤޤ줿ǡ Julius ˤ
 *  ǧ롥ɤ߹ʬνλȡϤɤ߹
 * िˤδؿٸƤФ롥
 * 
 * δؿǤ I/O blocking ˤĤƤϰʲդɬפǤ롧
 * 
 *  - Ĺ֤Υ֥å򤱤뤳ȡ@a sampnum ׵᥵ץǤϤ
 *  @a buf ˳ǼǽʺǤˡδؿǥ֥åǧ
 *  Τ֥å롥ɤ߹ߤĹ֥֥åʤ褦ɴ
 *  ץ٤ɤ߹֤뤤Ϻǽ˥Хåեˤ
 *  åɤ߹߲ǽʥǡץʬɤ
 *  褦ˤΤ褤
 *    
 *  - non-blocking ⡼ɤѤ硤 0 ֤ʤȡ
 *    Хåե˥ǡ¸ߤʤȤ0 ֤ Julius ϥץ
 *    ̵Τޤ¨¤ˤδؿƤӽФ줬ӥȤ
 *    ȯCPUɤäƤޤХåե˥ǡ̵Ȥ
 *    ¨¤ 0 ֤msec Ǥ褤ΤǤδؿԤ
 *    Ȥ˾ޤ
 *
 * ֤ͤϡºݤɤ߹ץͤȤ֤뤤
 * ADIN_EOF, ADIN_SEGMENT, ADIN_ERROR Τɤ줫֤ADIN_EOF ϥ
 * ꡼बüޤãȤɽ֤ȡJulius ϸߤǧ
 * λȥ꡼Ĥ롥ADIN_ERROR Ϥδؿǿ
 * 顼֤줬֤줿硤Julius Ϥξǰ۾
 * λ롥
 *
 * ADIN_SEGMENT ֤ȤǡJulius ˸ߤǧ򸽻Ƕڤ뤳
 * ׵᤹뤳ȤǤ롥ߤǧϤλǤäڤ졤
 * ޤǤǧ̤ꡦϤ줿ȡǧϤޤꤳ
 * ؿƤФ롥ADIN_SEGMENT  ADIN_EOF ưƤ뤬
 * ADIN_EOF  adin_close(), adin_open() Ƥǥȥ꡼λ
 * ΤФơADIN_SEGMENT ϤƤФϤ³Ԥ롥ε
 * ǽϡȤХͥåȥͳǲǡƤȤˡ
 * ¦鲻ǧON/OFFVAD򥳥ȥ뤷ʤɤ
 * ȤȤǤ롥
 * 
 * @param buf [out] 줿ץǼХåե
 * @param sampnum [in] @a buf ˳ǼǤ祵ץ
 * 
 * @return ºݤɤ߹ޤ줿ץ뤤 end of stream  ADIN_EOF,
 * Julius ˶ڤ׵ФȤˤ ADIN_SEGMENT, 顼ADIN_ERROR 
 * ֤
 * </JA>
 */
int
adin_read(SP16 *buf, int sampnum)
{
  audio_buf_info info;
  int size, cnt;

  /* get sample num that can be read without blocking */
  if (ioctl(audio_fd, SNDCTL_DSP_GETISPACE, &info) == -1) {
    printf("Error: adin_oss: failed to get number of samples in the buffer\n");
    return(ADIN_ERROR);
  }
  /* get them as much as possible */
  size = sampnum * sizeof(SP16);
  if (size > info.bytes) size = info.bytes;
  size &= ~ 1;		/* Force 16bit alignment */
  cnt = read(audio_fd, buf, size);
  if ( cnt < 0 ) {
    printf("Error: adin_oss: failed to read samples\n");
    return (ADIN_ERROR);
  }
  cnt /= sizeof(short);
    
  return(cnt);
}

/**
 * <EN>
 * @brief  Close the current input stream (required)
 *
 * This function will be called when the input stream has reached
 * end of file (i.e. the last call of adin_read() returns ADIN_EOF)
 *       
 * You may close a capture device, close an audio file, or
 * disconnect network client.
 *
 * If this function returns TRUE, Julius will go again to adin_open()
 * to open another stream.  If returns FALSE, Julius will exit
 * the recognition loop.
 * 
 * JuliusLib: This will be called at the end of j_recognize_stream().
 * 
 * @return TRUE on success, FALSE on failure.
 * </EN>
 * <JA>
 * @brief  ϥȥ꡼Ĥɬܡ
 *
 * ߤΥȥ꡼Ĥ롥δؿϡϥȥ꡼बüEOF
 * ãȤʤʤ adin_read()  ADIN_EOF ֤Ȥˤ
 * ƤФ롥ǥХĤ롤եĤ롤뤤ϥͥåȥ³
 * ǤΤ˻ȤȤǤ롥
 *
 * ｪλȤTRUE֤ȤJulius  adin_open() ä
 * ¾Υȥ꡼򳫤Ȥ롥 FALSE ֤ȤϡJulius 
 * ǧ롼פȴ롥
 * 
 * JuliusLib: δؿ j_recognize_stream() κǸǸƤФ롥
 * 
 * @return  TRUEԻ FALSE ֤
 * </JA>
 */
boolean
adin_close()
{
  close(audio_fd);
  return TRUE;
}

/************************************************************************/

/**
 * <EN>
 * @brief  A hook for Termination request (optional)
 *
 * This function will be called when Julius receives a Termination
 * request to stop running.  This can be used to synchronize input
 * facility with Julius's running status.
 * 
 * Termination will occur when Julius is running on module mode and
 * received TERMINATE command from client, or j_request_terminate()
 * is called inside application.  On termination, Julius will stop
 * recognition immediately (discard current input if in process),
 * and wait until received RESUME command or call of j_request_resume().
 *
 * This hook function will be called just after a Termination request.
 * Please note that this will be called when Julius receives request,
 * not on actual termination.
 * 
 * @return TRUE on success, FALSE on failure.
 * </EN>
 * <JA>
 * @brief  ׵ѥեåǤա
 *
 * δؿȡJulius ׵äݤˤδؿƤӽФ
 * ȤäơJulius ǡƳƱƱ뤳Ȥ
 * Ǥ롥㡧Ф׵Фʤɡ
 *
 * ׵ϡJulius ץꥱ䥯饤Ȥ
 * ǧǤ׵Ǥ롥ŪˤϡJulius ⥸塼⡼ɤư
 *  TERMINATE ޥɤ򥯥饤ȤäȤ䡤
 * JuliusLibȤ߹ץꥱ j_request_terminate() 
 * ƤȤȯ롥
 *
 * ׵ȡJulius ϸߤǧǤ롥
 * ǧǤä硤Ϥ˴¨Ǥ롥
 * κƳϡRESUME ޥɤ j_request_resume() θƤӽФǹԤ롥
 *
 * δؿ׵ Julius äǸƤФ롥
 * ºݤ˽ǤǸƤФΤǤϤʤȤդ줿
 * 
 * @return  TRUE, 顼 FALSE ֤
 * </JA>
 * 
 */
boolean
adin_terminate()
{
  printf("terminate request\n");
  return TRUE;
}

/**
 * <EN>
 * @brief  A hook for Pause request (optional)
 *
 * This function will be called when Julius receives a Pause request
 * to stop running.  This can be used to synchronize input facility
 * with Julius's running status.
 * 
 * Pause will occur when Julius is running on module mode and
 * received PAUSE command from client, or j_request_pause()
 * is called inside application.  On pausing, Julius will 
 * stop recognition and then wait until it receives RESUME command
 * or j_request_resume() is called.  When pausing occurs while recognition is
 * running, Julius will process it to the end before stops.
 *
 * This hook function will be called just after a Pause request.
 * Please note that this will be called when Julius receives request,
 * not on actual pause.
 *
 * @return TRUE on success, FALSE on failure.
 * </EN>
 * <JA>
 * @brief  ׵ѥեåǤա
 *
 * δؿȡJulius ׵äݤˤδؿƤӽФ
 * ȤäơJulius ǡƳƱƱ뤳Ȥ
 * Ǥ롥㡧Ф׵Фʤɡ
 *
 * ׵ϡJulius ץꥱ䥯饤Ȥ롤
 * ǧΰߤ׵Ǥ롥ŪˤϡJulius ⥸塼⡼ɤư
 *  PAUSE ޥɤ򥯥饤ȤäȤ䡤
 * JuliusLibȤ߹ץꥱ j_request_pause() 
 * ƤȤȯ롥
 *
 * ׵ȡJulius ϸߤǧǤ롥
 * ǧǤä硤ǧޤԤäƤǤ롥
 * κƳϡRESUME ޥɤ j_request_resume() θƤӽФǹԤ롥
 * 
 * ׵ (adin_terminate) Ȥΰ㤤ϡǧ׵Ȥư
 * ۤʤ롥׵Ǥ϶Ǥ뤬׵ǤϤǧޤ
 * ԤäƤߤ롥
 *
 * δؿ׵ Julius äǸƤФ롥
 * ºݤ˽ߤǸƤФΤǤϤʤȤդ줿
 * 
 * @return  TRUE, 顼 FALSE ֤
 * </JA>
 * 
 */
boolean
adin_pause()
{
  printf("pause request\n");
  return TRUE;
}

/**
 * <EN>
 * @brief  A hook for Resume request (optional)
 *
 * This function will be called when Julius received a resume request
 * to recover from pause/termination status.
 * 
 * Resume will occur when Julius has been stopped by receiving RESUME
 * command from client on module mode, or j_request_resume() is called
 * inside application.
 *
 * This hook function will be called just after a resume request.
 * This can be used to make this A/D-in plugin cooperate with the
 * pause/resume status, for example to tell audio client to restart
 * audio streaming.
 *
 * This function is totally optional.
 * 
 * @return TRUE on success, FALSE on failure.
 * </EN>
 * <JA>
 * @brief  ǧƳ׵ѥեåǤա
 *
 * δؿȡJulius ߾֤ǧƳ׵κݤ
 * δؿƤӽФ
 *
 * ǧƳ׵ϡJulius ⥸塼⡼ɤư RESUME ޥɤ
 * 饤ȤäȤ䡤JuliusLibȤ߹ץꥱ
 *  j_request_resume() ƤȤȯ롥κƳ׵᤬ȯ
 * ȡJulius ߤƤǧƳ롥
 *
 * աδؿϡºݤߤȤ˸ƤФΤǤϤʤJulius 
 * ׵äǡΤӤ˸ƤФ롥ʣƤФ뤳Ȥ䡤
 * ǤưǤˤˤΥޥɤäȤˤƤ
 * 뤳Ȥ뤳Ȥդ줿
 * 
 * @return  TRUE, 顼 FALSE ֤
 * </JA>
 * 
 */
boolean
adin_resume()
{
  printf("resume request\n");
  return TRUE;
}

/* end of file */
