package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.gui.library.LibraryMediator;
import com.limegroup.gnutella.gui.notify.*;
import com.limegroup.gnutella.gui.options.OptionsMediator;
import com.limegroup.gnutella.gui.statistics.StatisticsMediator;
import com.limegroup.gnutella.gui.menu.MenuMediator;
import com.limegroup.gnutella.gui.download.DownloadMediator;
import com.limegroup.gnutella.gui.search.SearchMediator;
import com.limegroup.gnutella.gui.upload.UploadMediator;
import com.limegroup.gnutella.gui.connection.ConnectionMediator;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.Launcher;
import com.limegroup.gnutella.settings.BooleanSetting;
import com.limegroup.gnutella.settings.IntSetting;
import com.limegroup.gnutella.settings.ApplicationSettings;

import com.sun.java.util.collections.*;
import java.io.*;
import java.awt.*;
import javax.swing.*;
import java.net.*;
import java.lang.reflect.*;

/**
 * This class acts as a central point of access for all gui components, a sort
 * of "hub" for the frontend.  This should be the only common class that all
 * frontend components have access to, reducing the overall dependencies and
 * therefore increasing the modularity of the code.
 *
 * <p>Any functions or services that should be accessible to multiple classes
 * should be added to this class.  These currently include such functions as
 * easily displaying standardly-formatted messages to the user, obtaining
 * locale-specific strings, and obtaining image resources, among others.
 *
 * <p>All of the methods in this class should be called from the event-
 * dispatch (Swing) thread.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class GUIMediator {

	/**
	 * Singleton for easy access to the mediator.
	 */
	private static GUIMediator _instance = null;

	/**
	 * Constant for the index of the search tab in the main application
	 * window.
	 */
	public static final int SEARCH_INDEX = 0;

	/**
	 * Constant for the index of the monitor tab in the main application
	 * window.
	 */
	public static final int MONITOR_INDEX = 1;

	/**
	 * Constant for the index of the connections tab in the main application
	 * window.
	 */
	public static final int CONNECTIONS_INDEX = 2;

	/**
	 * Constant for the index of the library tab in the main application
	 * window.
	 */
	public static final int LIBRARY_INDEX = 3;

	/**
	 * Constant for the index of the shopping tab in the main application
	 * window.
	 */
	public static final int SHOPPING_INDEX = 4;

	/**
	 * Constant specifying whether or not the user has donated to the LimeWire
	 * project.
	 */
	private static boolean HAS_DONATED = true;

	/**
	 * The main <tt>JFrame</tt> for the application.
	 */
	private static final JFrame FRAME = new JFrame();

	/**
	 * <tt>List</tt> of <tt>RefreshListener</tt> classes to notify of UI
	 * refresh events.
	 */
	private static final com.sun.java.util.collections.List REFRESH_LIST =
		new ArrayList();

	/**
	 * <tt>List</tt> of <tt>ThemeObserver</tt> classes to notify of
	 * ui components of theme changes.
	 */
	private static final com.sun.java.util.collections.List THEME_OBSERVERS =
		new LinkedList();
		
	/**
	 * Constant <tt>Component</tt> for use as a standard horizontal
	 * separator.
	 */
	private static final Component HORIZONTAL_SEPARATOR =
		Box.createRigidArea(new Dimension(6,0));

	/**
	 * Constant <tt>Component</tt> for use as a standard horizontal separator.
	 */
	private static final Component VERTICAL_SEPARATOR = 
		Box.createRigidArea(new Dimension(0,6));

	/**
	 * String to be displayed in title bar of LW client.
	 */
	private static final String APP_TITLE = 
		GUIMediator.getStringResource("APP_TITLE");

	/**
	 * Constant for when the user selects the yes button
	 * in a message giving the user a yes and a no option.
	 */
	public static final int YES_OPTION = MessageService.YES_OPTION;

	/**
	 * Constant for when the user selects the no button
	 * in a message giving the user a yes and a no option.
	 */
	public static final int NO_OPTION = MessageService.NO_OPTION;

	/**
	 * Handle to the <tt>OptionsMediator</tt> class that is responsible for
	 * displaying customizable options to the user.
	 */
	private static OptionsMediator _optionsMediator;

	/**
	 * Constant handle to the <tt>MainFrame</tt> instance that handles
	 * constructing all of the primary gui components.
	 */
	private final MainFrame MAIN_FRAME = new MainFrame(FRAME);

	/**
	 * Constant handle to the <tt>DownloadMediator</tt> class that is
	 * responsible for displaying active downloads to the user.
	 */
	private final DownloadMediator DOWNLOAD_MEDIATOR =
		MAIN_FRAME.getDownloadMediator();

	/**
	 * Constant handle to the <tt>UploadMediator</tt> class that is
	 * responsible for displaying active uploads to the user.
	 */
	private final UploadMediator UPLOAD_MEDIATOR =
		MAIN_FRAME.getUploadMediator();

	/**
	 * Constant handle to the <tt>ConnectionMediator</tt> class that is
	 * responsible for displaying current connections to the user.
	 */
	private final ConnectionMediator CONNECTION_MEDIATOR =
		MAIN_FRAME.getConnectionMediator();

	/**
	 * Constant handle to the <tt>ShoppingView</tt> class.
	 */
	private final ShoppingView SHOPPING_VIEW =
		MAIN_FRAME.getShoppingView();

	/**
	 * Constant handle to the <tt>LibraryMediator</tt> class that is
	 * responsible for displaying files in the user's repository.
	 */
	private final LibraryMediator LIBRARY_MEDIATOR = 
		MAIN_FRAME.getLibraryMediator();

	/**
	 * Constant handle to the <tt>MenuMediator</tt> class that is responsible
	 * for displaying and controlling the main menu bar of the program.
	 */
	private final MenuMediator MENU_MEDIATOR =
		MAIN_FRAME.getMenuMediator();

	/**
	 * Constant handle to the <tt>DownloadView</tt> class that is responsible
	 * for displaying the status of the network and connectivity to the user.
	 */
	private final StatusLine STATUS_LINE =
		MAIN_FRAME.getStatusLine();

	/**
	 * Constant handle to the <tt>StatisticsMediator</tt> class that is
	 * responsible for displaying statistics to the user.
	 */
	private final StatisticsMediator STATISTICS_MEDIATOR =
		MAIN_FRAME.getStatisticsMediator();

	/**
	 * Handle to <tt>RouterService</tt> to give frontend classes access to the
	 * backend.
	 */
	private RouterService _routerService;

	/**
	 * Private constructor to ensure that this class cannot be constructed
	 * from another class.
	 */
	private GUIMediator() {
		FRAME.setTitle(APP_TITLE);
		_optionsMediator = MAIN_FRAME.getOptionsMediator();
		addRefreshListener(STATISTICS_MEDIATOR);
	}

	/**
	 * Singleton accessor for this class.
	 *
	 * @return the <tt>GUIMediator</tt> instance
	 */
	public static synchronized GUIMediator instance() {
		if (_instance == null) {
			_instance = new GUIMediator();
		}
		return _instance;
	}
	
	/**
	 * Accessor for whether or not the GUIMediator has been constructed yet.
	 */
	public static synchronized boolean isConstructed() {
	    return _instance != null;
	}

	/**
	 * The host catcher and the statistics view need the backend to be
	 * initialized for these methods to be called.
	 */
	public final void startTimer() {
		RefreshTimer timer = new RefreshTimer();
		timer.startTimer();
	}

	/**
	 * Returns a boolean specifying whether or not the wrapped
	 * <tt>JFrame</tt> is visible or not.
	 *
	 * @return <tt>true</tt> if the <tt>JFram</tt> is visible,
	 *  <tt>false</tt> otherwise
	 */
	public static final boolean getAppVisible() {
		return FRAME.isShowing();
	}

	/**
	 * Specifies whether or not the main application window should be visible
	 * or not.
	 *
	 * @param visible specifies whether or not the application should be
	 *                made visible or not
	 */
	public static final void setAppVisible(boolean visible) {
		FRAME.setVisible(visible);
	}

	/**
	 * Call this method when you want to update the stats being shown in the
	 * title bar.
	 *
	 * @param stats the new statistics to display
	 */
	private final void setStatistics(String stats) {
		if (RouterService.isShieldedLeaf()) 
			FRAME.setTitle(APP_TITLE);
		else
			FRAME.setTitle(APP_TITLE + "  (" + stats + ")");
	}

	/**
	 * Returns a <tt>Dimension</tt> instance containing the dimensions of the
	 * wrapped JFrame.
	 *
	 * @return a <tt>Dimension</tt> instance containing the width and height
	 *         of the wrapped JFrame
	 */
	public static final Dimension getAppSize() {
		return FRAME.getSize();
	}

	/**
	 * Returns a <tt>Point</tt> instance containing the x, y position of the
	 * wrapped <ttJFrame</tt> on the screen.
	 *
	 * @return a <tt>Point</tt> instance containting the x, y position of the
	 *         wrapped JFrame
	 */
	public static final Point getAppLocation() {
		return FRAME.getLocation();
	}

	/**
	 * Returns the <tt>MainFrame</tt> instance.  <tt>MainFrame</tt> maintains
	 * handles to all of the major gui classes.
	 *
	 * @return the <tt>MainFrame</tt> instance
	 */
	public final MainFrame getMainFrame() {
		return MAIN_FRAME;
	} 

	/**
	 * Call the ShoppingView component to relayout itself.
	 */
	public void wakeupShoppingView() {
		getMainFrame().getShoppingView().wakeupComponent();
	}

	/**
	 * Returns the main application <tt>JFrame</tt> instance.
	 *
	 * @return the main application <tt>JFrame</tt> instance
	 */
	public static final JFrame getAppFrame() {
		return FRAME;
	}

	/**
	 * Returns the router service variable for other classes to access.
	 *
	 * @return the <tt>RouterService</tt> instance
	 */
	public final RouterService getRouter() {
		return _routerService;
	}

	/**
	 * Sets the router service variable for other classes to access. 
	 *
	 * @param routerService the <tt>RouterService</tt> instance for other
	 *                      classes to access
	 */
	public final void setRouterService(RouterService routerService) {
		_routerService = routerService;
	}

	/**
	 * Refreshes the various gui components that require refreshing.
	 */
	public final void refreshGUI() {
		try {
			for (int i = 0; i < REFRESH_LIST.size(); i++) {
				((RefreshListener)REFRESH_LIST.get(i)).refresh();
			}

        		// update the status panel
        		int  numHosts    = (int)RouterService.getNumHosts();
        		long numFiles    = RouterService.getNumFiles();
        		long totSize     = RouterService.getTotalFileSize();
        		int  sharedFiles = RouterService.getNumSharedFiles();
        		int  pendingShare= RouterService.getNumPendingShared();
        		String toDisplay = STATUS_LINE.setStatistics(
				numHosts, numFiles,
				totSize, sharedFiles,
				pendingShare);
			setStatistics(toDisplay);

        		RouterService.updateHorizon();
        		this.setConnectedUI(RouterService.isConnected());
		} catch (Throwable e) {
			GUIMediator.showInternalError(e);
		}
	}

	/**
	 * Sets the visibility state of the options window.
	 *
	 * @param visible the visibility state to set the window to
	 */
	public void setOptionsVisible(boolean visible) {
		if (_optionsMediator == null) return;
		LIBRARY_MEDIATOR.cancelEditing();
		_optionsMediator.setOptionsVisible(visible);
	}

	/**
	 * Returns whether or not the options window is visible
	 *
	 * @return <tt>true</tt> if the options window is visible,
	 *  <tt>false</tt> otherwise
	 */
	public static boolean isOptionsVisible() {
		if (_optionsMediator == null) return false;
		return _optionsMediator.isOptionsVisible();
	}

	/**
	 * Gets a handle to the options window main <tt>JComponent</tt> instance.
	 *
	 * @return the options window main <tt>JComponent</tt>, or <tt>null</tt>
	 *  if the options window has not yet been constructed (the window is 
	 *  guaranteed to be constructed if it is visible)
	 */
	public static Component getMainOptionsComponent() {
		if (_optionsMediator == null) return null;
		return _optionsMediator.getMainOptionsComponent();
	}

	/**
	 * Sets the visibility of the statistics window.
	 *
	 * @param visible the visibility state to set the window to
	 */
	public final void setStatisticsVisible(boolean visible) {
		STATISTICS_MEDIATOR.setStatisticsVisible(visible);		
	}

	/**
	 * Displays the passed password dialog
	 * @param dialog The dialog box to be displayed
	 */
	public final int showPasswordDialog(PasswordDialog dialog){
		return dialog.showDialog();
	}

	/**
	 * Notifies any necessary components that the specified tab has been
	 * selected.
	 *
	 * @param SELECTED_TAB the index of the tab that has been selected
	 */
	public final void handleTabSelection(final int SELECTED_TAB) {
		switch (SELECTED_TAB) {
		case SEARCH_INDEX:
			MENU_MEDIATOR.handleStandardSelection();
			break;
		case MONITOR_INDEX:
			MENU_MEDIATOR.handleStandardSelection();
			break;
		case CONNECTIONS_INDEX:
			MENU_MEDIATOR.handleStandardSelection();
			break;
		case LIBRARY_INDEX:
			MENU_MEDIATOR.handleLibrarySelection();
			break;
		case SHOPPING_INDEX:
			SHOPPING_VIEW.prepareComponent();
			break;
		}
	}

	/**
	 * Sets the tab pane to display the given tab. 
	 *
	 * @param index the index of the tab to display
	 */
	public void setWindow(int index) {
		MAIN_FRAME.setSelectedIndex(index);
	}

	/**
	 * Updates the icon at the specified tab index.
	 *
	 * @param index the fixed index of the tab to update
	 */
	public void updateTabIcon(int index) {
		MAIN_FRAME.updateTabIcon(index);
	}

	/**
	 * Clear the connections in the connection view.
	 */
	public void clearConnections() {
		CONNECTION_MEDIATOR.clearConnections();
	}

	/**
	 * Sets the "keep alive" value displayed to the user in the Connections
	 * tab.
	 *
	 * @param keepAlive a <tt>String</tt> representing the keep
	 *                  alive to display
	 */
	public void setKeepAliveText(String keepAlive) {
		CONNECTION_MEDIATOR.setKeepAliveText(keepAlive);
	}

	/**
	 * Trigger a search based on a string.
	 *
	 * @param query the query <tt>String</tt>
	 * @return the GUID of the query sent to the network.
	 *         Used mainly for testing
	 */
	public byte[] triggerSearch(String query) {
		MAIN_FRAME.setSelectedIndex(SEARCH_INDEX);
		return SearchMediator.triggerSearch(query);		
	}

	/**
	 * Sets the connected/disconnected visual status of the client.
	 *
	 * @param connected the connected/disconnected status of the client
	 */
	private void setConnectedUI(final boolean connected) {
		SearchMediator.setConnected(connected);
		MENU_MEDIATOR.setConnected(connected);
		STATUS_LINE.setConnected(connected);
		if (connected == false) {
			this.setSearching(false);
		}
	}

  	/**
  	 * Returns the total number of uploads for this session. 
	 *
	 * @return the total number of uploads for this session
  	 */
  	public int getTotalUploads() {
  		return UPLOAD_MEDIATOR.getTotalUploads();
  	}

  	/**
  	 * Returns the total number of currently active uploads.
	 *
	 * @return the total number of currently active uploads
  	 */
  	public int getCurrentUploads() {
  		return UPLOAD_MEDIATOR.getCurrentUploads();
  	}

  	/**
  	 * Returns the total number of downloads for this session. 
	 *
	 * @return the total number of downloads for this session
  	 */
  	public final int getTotalDownloads() {
  		return DOWNLOAD_MEDIATOR.getTotalDownloads();
  	}

  	/**
  	 * Returns the total number of currently active downloads.
	 *
	 * @return the total number of currently active downloads
  	 */
  	public final int getCurrentDownloads() {
  		return DOWNLOAD_MEDIATOR.getCurrentDownloads();
  	}

	/**
	 * Tells the library to add a new top-level (shared) folder.
	 */
	public final void addSharedLibraryFolder() {
		LIBRARY_MEDIATOR.addSharedLibraryFolder();
	}

	/**
	 * Tells the library to add a new folder to the currently selected folder.
	 */
	public final void addNewLibraryFolder() {
		LIBRARY_MEDIATOR.addNewLibraryFolder();
	}

	/**
	 * Tells the library to unshare the currently selected folder.
	 */
	public final void unshareLibraryFolder() {
		LIBRARY_MEDIATOR.unshareLibraryFolder();
	}

	/**
	 * Tells the library to rename the currently selected folder.
	 */
	public final void renameLibraryFolder() {
		LIBRARY_MEDIATOR.renameLibraryFolder();
	}

	/**
	 * Tells the library to launch the currently selected file(s).
	 */
	public final void launchLibraryFile() {
		LIBRARY_MEDIATOR.launch();
	}

	/**
	 * Tells the library to delete the selected file(s).
	 */
	public final void deleteLibraryFile() {
		LIBRARY_MEDIATOR.deleteLibraryFile();
	}

	/**
	 * Tells the library to refresh its components.
	 */
	public final void refreshLibrary() {
		LIBRARY_MEDIATOR.refresh();
	}

	/**
	 * Tells the library to update the shared directories that it displays.
	 */
	public final void clearLibrary() {
		LIBRARY_MEDIATOR.clearLibrary();
	}

	/**
	 * Hides the GUI by either sending it to the System Tray or
	 * minimizing the window.  Mimimize behavior occurs on platforms
	 * which do not support the System Tray.
	 * @see restoreView
	 */
	private static void hideView() {
		// Frame must be visible for setState to work.  Make invisible
		// after iconifying
		if (CommonUtils.isJava118()) {
			GUIMediator.setAppVisible(false);
			return;
		}

		// get the frame states through reflection
		final int ICONIFIED_FRAME_STATE = GUIMediator.getIconifiedFrameState();
		final int FRAME_STATE           = GUIMediator.getFrameState();

		// if the constants could not be loaded through reflection, just
		// set the application to not be visible and return
		if ((ICONIFIED_FRAME_STATE == -1) || (FRAME_STATE == -1)) {
			GUIMediator.setAppVisible(false);
		} else {
			try {
				// otherwise, set the frame state to be iconified
				GUIMediator.setFrameState(ICONIFIED_FRAME_STATE);
			} catch (IOException e) {
				GUIMediator.setAppVisible(false);
			}
		}

		if (CommonUtils.supportsTray()) {
			GUIMediator.setAppVisible(false);
			GUIMediator.addNotify();
		}
	}

	/**
	 * Makes the GUI visible by either restoring it from the System Tray or
	 * the task bar.
	 * @see hideView
	 */
	public static void restoreView() {
		// Frame must be visible for setState to work.  Make visible
		// before restoring.
		if (CommonUtils.supportsTray()) {
			GUIMediator.setAppVisible(true);
			GUIMediator.hideNotify();
		} 

		// If shutdown sequence was initiated, cancel it.  Auto shutdown is
		// disabled when the GUI is visible.
		Finalizer.cancelShutdown();

		if (CommonUtils.isJava118()) {
			GUIMediator.setAppVisible(true);
			return;
		}

		// get the frame states through reflection
		final int NORMAL_FRAME_STATE = GUIMediator.getNormalFrameState();
		final int FRAME_STATE        = GUIMediator.getFrameState();

		// if the constants could not be loaded through reflection, just
		// set the application to be visible and return
		if ((NORMAL_FRAME_STATE == -1) || (FRAME_STATE == -1)) {
			GUIMediator.setAppVisible(true);
			return;
		}

		// otherwise, set the frame state to normal if it's not already
		if (FRAME_STATE != NORMAL_FRAME_STATE) {
			try {
				GUIMediator.setFrameState(NORMAL_FRAME_STATE);
			} catch (IOException e) {
				// if there's a problem, just set the app visible
				GUIMediator.setAppVisible(true);
				return;
			}
		}
	}

	/**
	 * Accesses the "NORMAL" field in the java.awt.Frame class through
	 * reflection, as it's only available on 1.2 and above.  If the member of
	 * <tt>Frame</tt> could not be found, this returns <tt>-1</tt>.
	 *
	 * @return the constant <tt>int</tt> for the Frame.NORMAL frame state,
	 *  looked up through reflection because it may not be available,
	 *  depending on the JVM, or -1 if it could not be loaded for any reason
	 */
	private static int getNormalFrameState() {
		try {
			Field normaleStateField = Frame.class.getField("NORMAL");
			return normaleStateField.getInt(Frame.class);
		} catch (Exception e) {
			// just catching any exception, as we don't really care
			// which one it is of the bunch
			return -1;
		}
	}

	/**
	 * Accesses the "ICONIFIED" field in the java.awt.Frame class through
	 * reflection, as it's only available on 1.2 and above.  If the member of
	 * <tt>Frame</tt> could not be found, this returns <tt>-1</tt>.
	 *
	 * @return the constant <tt>int</tt> for the Frame.ICONIFIED frame state,
	 *  looked up through reflection because it may not be available,
	 *  depending on the JVM, or -1 if it could not be loaded for any reason
	 */
	private static int getIconifiedFrameState() {
		try {
			Field iconifiedeStateField = Frame.class.getField("ICONIFIED");
			return iconifiedeStateField.getInt(Frame.class);
		} catch (Exception e) {
			// just catching any exception, as we don't really care
			// which one it is of the bunch
			return -1;
		}
	}

	/**
	 * Returns the current state of the <tt>Frame</tt> instance for the
	 * application, using reflection because the getState method was not
	 * available before 1.2.
	 *
	 * @return the state of the <tt>Frame</tt> instance, or -1 if the state
	 *  could not be accessed for any reason
	 */
	private static int getFrameState() {
		try {
			Method getStateMethod = Frame.class.getDeclaredMethod(
				"getState", new Class[] {});
			return ((Integer)getStateMethod.invoke(FRAME,
				new Object[] {})).intValue();
		} catch (Exception e) {
			// just catching any exception, as we don't really care
			// which one it is of the bunch
			return -1;
		}
	}

	/**
	 * Sets the current state of the <tt>Frame</tt> instance for the
	 * application, using reflection because the setState method was
	 * not available before 1.2.
	 *
	 * @param state the state of the <tt>Frame</tt> instance to set
	 * @throws <tt>IOException</tt> if there was any problem setting
	 *  the state of the frame
	 */
	private static void setFrameState(int state) throws IOException {
		try {
			Method setStateMethod = Frame.class.getDeclaredMethod(
				"setState", new Class[] {int.class});
			setStateMethod.invoke(FRAME,
				new Object[] {new Integer(state)});
		} catch (Exception e) {
			// just catching any exception, as we don't really care
			// which one it is of the bunch
		}
	}

	/**
	 * Determines the appropriate shutdown behavior based on user settings.
	 * This implementation decides between exiting the application immediately,
	 * or exiting after all file transfers in progress are complete.
	 */
	public static void close() {
		if (ApplicationSettings.MINIMIZE_TO_TRAY.getValue())
			hideView();
		else if (ApplicationSettings.SHUTDOWN_AFTER_TRANSFERS.getValue() &&
				 !CommonUtils.isMacClassic())
			GUIMediator.shutdownAfterTransfers();
		else shutdown();
	}

	/**
	 * Shutdown the program cleanly.
	 */
	public static void shutdown() {
		Finalizer.shutdown();
	}

	/**
	 * Shutdown the program cleanly after all transfers in progress are
	 * complete.  Calling this method causes the GUI to be hidden while the
	 * application waits to shutdown.
	 * @see hideView
	 */
	public static void shutdownAfterTransfers() {
		Finalizer.shutdownAfterTransfers();
		GUIMediator.hideView();
	}

	/**
	 * Shows the "About" menu with more information about the program.
	 */
	public static final void showAboutWindow() {
		AboutWindow about = new AboutWindow();
		about.showDialog();
	}

	/**
	 * Updates the icon associated with the user notification area, such as
	 * the system tray on Windows.
	 *
	 * @param imageFileName the name of the new image file to use,
	 *  preferably a file in and relative to the current directory,
	 *  as in "LimeWire.ico"
	 */
	public static void updateNotifyImage(final String imageFileName) {
		NotifyUserProxy.instance().updateImage(imageFileName);
	}

	/**
	 * Updates the text associated with the user notification area, such as
	 * the system tray on Windows.
	 *
	 * @param text the new text to use for the user notification area for
	 *  such things as the system tray tooltip on Windows
	 */
	public static void updateNotifyText(final String text) {
		NotifyUserProxy.instance().updateDesc(text);
	}

	/**
	 * Updates the icon and the text associated with the user notification
	 * area, such as the system tray on Windows.
	 *
	 * @param imageFileName the name of the new image file to use, preferably
	 *  a file in and relative to the current directory, as in "LimeWire.ico"
	 * @param text the new text to use for the user notification area for
	 *  such things as the system tray tooltip on Windows
	 */
	public static void updateNotify(
			final String imageFileName,
			final String text) {
		NotifyUserProxy.instance().updateNotify(imageFileName, text);
	}

	/**
	 * Hides the user notification area.  If this method is called,
	 * the caller must ensure that the user always has a way to exit
	 * the application even though the notification area is hidden, or
	 * to make the notification area visible again.
	 */
	public static void hideNotify() {
		NotifyUserProxy.instance().hideNotify();
	}

	/**
	 * Shows the user notification area.  The user notification icon and
	 * tooltip created by the NotifyUser object are not modified.
	 */
	public static void addNotify() {
		NotifyUserProxy.instance().addNotify();
	}

	/**
	 * Serves as a single point of access for any icons used in the program.
	 *
	 * @param imageName the name of the icon to return without path
	 *                  information, as in "plug.gif"
	 * @return the <tt>ImageIcon</tt> object specified in the param string
	 */
	//public static final ImageIcon getImageResource(final String imageName) {
	//	return ResourceManager.getImageResource(imageName);
	//}

	public static final ImageIcon getThemeImage(final String name) {
		return ResourceManager.getThemeImage(name);
	}

	/**
	 * Returns a new <tt>URL</tt> instance for the specified file name.
	 * The file must be located in the com/limegroup/gnutella/gui/resources
	 * directory, or this will return <tt>null</tt>.
	 *
	 * @param FILE_NAME the name of the file to return a url for without path
	 *  information, as in "about.html"
	 * @return the <tt>URL</tt> instance for the specified file, or
	 * <tt>null</tt> if the <tt>URL</tt> could not be loaded
	 */
	public static URL getURLResource(final String FILE_NAME) {
		return ResourceManager.getURLResource(FILE_NAME);
	}

	/**
	 * Returns the locale-specific String from the resource manager.
	 *
	 * @return an internationalized <tt>String</tt> instance
	 *         corresponding with the <tt>resourceKey</tt>
	 */
	public static final String getStringResource(
			final String resourceKey) {
		return ResourceManager.getStringResource(resourceKey);
	}

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user in the form of a yes or no
	 * question.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @return an integer indicating a yes or a no response from the user
	 */
	public static final int showYesNoMessage(
			final String messageKey) {
		return MessageService.instance().showYesNoMessage(
			getStringResource(messageKey));
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user in the form of a yes or no
	 * question.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param defaultValue the IntSetting to store/retrieve the default value
	 * @return an integer indicating a yes or a no response from the user
	 */
	public static final int showYesNoMessage(
			final String messageKey, final IntSetting defaultValue) {
		return MessageService.instance().showYesNoMessage(
			getStringResource(messageKey), defaultValue);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user with a hard-coded <tt>String</tt>
	 * appended to it.  This is in the form of a yes or no question.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @return  an integer indicating a yes or a no response from the user
	 */
	public static final int showYesNoMessage(
			final String messageKey, 
			final String message) {
		return MessageService.instance().showYesNoMessage(
			getStringResource(messageKey) + " " +
			message);
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user with a hard-coded <tt>String</tt>
	 * appended to it.  This is in the form of a yes or no question.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
     * @param defaultValue the IntSetting to store/retrieve the defaultValue
	 * @return  an integer indicating a yes or a no response from the user
	 */
	public static final int showYesNoMessage(
			final String messageKey, 
			final String message,
			final IntSetting defaultValue) {
		return MessageService.instance().showYesNoMessage(
			getStringResource(messageKey) + " " +
			message, defaultValue);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user with a hard-coded <tt>String</tt>
	 * in the middle of the message between two locale-specific
	 * <tt>String</tt> values.  This is in the form of a yes or no
	 * question.<p>
	 *
	 * The <tt>messageStartKey</tt> and the <tt>messageEndKey</tt>
	 * parameters must be keys for locale-specific message <tt>String</tt>s
	 * and not a hard-coded values.
	 *
	 * @param messageStartKey the key for the locale-specific message to
	 *                        display at the beginning of the message
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @param messageEndKey the key for the locale-specific message to
	 *                      display at the end of the message
	 * @return  an integer indicating a yes or a no response from the user.
	 */
	public static final int showYesNoMessage(
			final String messageStartKey,
			final String message, 
			final String messageEndKey) {
		return MessageService.instance().showYesNoMessage(
			getStringResource(messageStartKey) + " " +
			message + " " +
			getStringResource(messageEndKey));
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user with a hard-coded <tt>String</tt>
	 * in the middle of the message between two locale-specific
	 * <tt>String</tt> values.  This is in the form of a yes or no
	 * question.<p>
	 *
	 * The <tt>messageStartKey</tt> and the <tt>messageEndKey</tt>
	 * parameters must be keys for locale-specific message <tt>String</tt>s
	 * and not a hard-coded values.
	 *
	 * @param messageStartKey the key for the locale-specific message to
	 *                        display at the beginning of the message
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @param messageEndKey the key for the locale-specific message to
	 *                      display at the end of the message
	 * @param defaultValue the IntSetting that stores/retrieves the defaultValue
	 * @return  an integer indicating a yes or a no response from the user.
	 */
	public static final int showYesNoMessage(
			final String messageStartKey,
			final String message, 
			final String messageEndKey,
			final IntSetting defaultValue) {
		return MessageService.instance().showYesNoMessage(
			getStringResource(messageStartKey) + " " +
			message + " " +
			getStringResource(messageEndKey),
			defaultValue);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 */
	public static final void showMessage(
			final String messageKey) {
		MessageService.instance().showMessage(
			getStringResource(messageKey));
	}

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param ignore the BooleanSetting that stores/retrieves whether or
	 *  not to display this message.
	 */
	public static final void showMessage(
			final String messageKey, final BooleanSetting ignore) {
		MessageService.instance().showMessage(
			getStringResource(messageKey), ignore);
	}
	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays
	 * a message to the user with a locale-specific message with a
	 * hard-coded message appended to it.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 */
	public static final void showMessage(
			final String messageKey, 
			final String message) {
		MessageService.instance().showMessage(
			getStringResource(messageKey) + " " +
			message);
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays
	 * a message to the user with a locale-specific message with a
	 * hard-coded message appended to it.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
     * @param ignore the BooleanSetting that stores/retrieves whether or
     *  not to display this message.
	 */
	public static final void showMessage(
			final String messageKey, 
			final String message,
			final BooleanSetting ignore) {
		MessageService.instance().showMessage(
			getStringResource(messageKey) + " " +
			message, ignore);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays
	 * a message to the user with a locale-specific message with a
	 * hard-coded message appended to it.<p>
	 *
	 * The <tt>messageStartKey</tt> and the <tt>messageEndKey</tt>
	 * parameters must be keys for locale-specific message
	 * <tt>String</tt>s and not a hard-coded values.
	 *
	 * @param messageStartKey the key for the locale-specific message to 
	 *                        display at the beginning of the message
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @param messageEndKey the key for the locale-specific message to 
	 *                      display at the end of the message
	 */
	public static final void showMessage(
			final String messageStartKey, 
			final String message,
			final String messageEndKey) {
		MessageService.instance().showMessage(
			getStringResource(messageStartKey) + " " +
			message + " " +
			getStringResource(messageEndKey));
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays
	 * a message to the user with a locale-specific message with a
	 * hard-coded message appended to it.<p>
	 *
	 * The <tt>messageStartKey</tt> and the <tt>messageEndKey</tt>
	 * parameters must be keys for locale-specific message
	 * <tt>String</tt>s and not a hard-coded values.
	 *
	 * @param messageStartKey the key for the locale-specific message to 
	 *                        display at the beginning of the message
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @param messageEndKey the key for the locale-specific message to 
	 *                      display at the end of the message
     * @param ignore the BooleanSetting for that stores/retrieves whether
     *  or not to display this message.	 
	 */
	public static final void showMessage(
			final String messageStartKey, 
			final String message,
			final String messageEndKey,
			final BooleanSetting ignore) {
		MessageService.instance().showMessage(
			getStringResource(messageStartKey) + " " +
			message + " " +
			getStringResource(messageEndKey), ignore);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays a
	 * confirmation message to the user.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 */	
	public static final void showConfirmMessage(
			final String messageKey) {
		MessageService.instance().showConfirmMessage(
			getStringResource(messageKey));
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays a
	 * confirmation message to the user.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
     * @param ignore the BooleanSetting for that stores/retrieves whether
     *  or not to display this message.	 
	 */	
	public static final void showConfirmMessage(
			final String messageKey,
			final BooleanSetting ignore) {
		MessageService.instance().showConfirmMessage(
			getStringResource(messageKey), ignore);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays
	 * a confirmation message to the user.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 */	
	public static final void showConfirmMessage(
			final String messageKey,
			final String message) {
		MessageService.instance().showConfirmMessage(
			getStringResource(messageKey) + " " +
			message);
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays
	 * a confirmation message to the user.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
     * @param ignore the BooleanSetting for that stores/retrieves whether
     *  or not to display this message.	 	 
	 */	
	public static final void showConfirmMessage(
			final String messageKey,
			final String message,
			final BooleanSetting ignore) {
		MessageService.instance().showConfirmMessage(
			getStringResource(messageKey) + " " +
			message, ignore);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays a
	 * confirmation message to the user.<p>
	 *
	 * The <tt>messageStartKey</tt> and the <tt>messageEndKey</tt>
	 * parameters must be keys for locale-specific message <tt>String</tt>s
	 * and not a hard-coded values.
	 *
	 * @param messageStartKey the key for the locale-specific message to
	 *                        display at the beginning of the message
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @param messageEndKey the key for the locale-specific message to
	 *                      display at the end of the message
	 */	
	public static final void showConfirmMessage(
			final String messageStartKey,
			final String message,
			final String messageEndKey) {
		MessageService.instance().showConfirmMessage(
			getStringResource(messageStartKey) + " " +
			message + " " +
			getStringResource(messageEndKey));
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays a
	 * confirmation message to the user.<p>
	 *
	 * The <tt>messageStartKey</tt> and the <tt>messageEndKey</tt>
	 * parameters must be keys for locale-specific message <tt>String</tt>s
	 * and not a hard-coded values.
	 *
	 * @param messageStartKey the key for the locale-specific message to
	 *                        display at the beginning of the message
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @param messageEndKey the key for the locale-specific message to
	 *                      display at the end of the message
     * @param ignore the BooleanSetting for that stores/retrieves whether
     *  or not to display this message.	 	 	 
	 */	
	public static final void showConfirmMessage(
			final String messageStartKey,
			final String message,
			final String messageEndKey,
			final BooleanSetting ignore) {
		MessageService.instance().showConfirmMessage(
			getStringResource(messageStartKey) + " " +
			message + " " +
			getStringResource(messageEndKey), ignore);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display.
	 */
	public static final void showError(
			final String messageKey) {
		MessageService.instance().showError(
			getStringResource(messageKey));
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display.
     * @param ignore the BooleanSetting for that stores/retrieves whether
     *  or not to display this message.	 
	 */
	public static final void showError(
			final String messageKey,
			final BooleanSetting ignore) {
		MessageService.instance().showError(
			getStringResource(messageKey), ignore);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user with a hard-coded message
	 * appended to it at the end.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 */
	public static final void showError(
	        final String messageKey, 
		    final String message) {
		MessageService.instance().showError(
			getStringResource(messageKey) + " " +
			message);
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.  Displays a
	 * locale-specific message to the user with a hard-coded message
	 * appended to it at the end.<p>
	 *
	 * The <tt>messageKey</tt> parameter must be the key for a locale-
	 * specific message <tt>String</tt> and not a hard-coded value.
	 *
	 * @param messageKey the key for the locale-specific message to display
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
     * @param ignore the BooleanSetting for that stores/retrieves whether
     *  or not to display this message.		 
	 */
	public static final void showError(
	        final String messageKey, 
		    final String message,
		    final BooleanSetting ignore) {
		MessageService.instance().showError(
			getStringResource(messageKey) + " " +
			message, ignore);
	}	

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays
	 * a message to the user with a locale-specific message with a
	 * hard-coded message appended to it.<p>
	 *
	 * The <tt>messageStartKey</tt> and the <tt>messageEndKey</tt>
	 * parameters must be keys for locale-specific message
	 * <tt>String</tt>s and not a hard-coded values.
	 *
	 * @param messageStartKey the key for the locale-specific message to
	 *                        display at the beginning of the message
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @param messageEndKey the key for the locale-specific message to
	 *                      display at the end of the message
	 */
	public static final void showError(
	 		final String messageStartKey,
			final String message,
			final String messageEndKey) {
		MessageService.instance().showError(
			getStringResource(messageStartKey) + " " +
			message + " " +
			getStringResource(messageEndKey));
	}

	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class. Displays
	 * a message to the user with a locale-specific message with a
	 * hard-coded message appended to it.<p>
	 *
	 * The <tt>messageStartKey</tt> and the <tt>messageEndKey</tt>
	 * parameters must be keys for locale-specific message
	 * <tt>String</tt>s and not a hard-coded values.
	 *
	 * @param messageStartKey the key for the locale-specific message to
	 *                        display at the beginning of the message
	 * @param message a second, non-locale-specific message to display, such
	 *                as a filename
	 * @param messageEndKey the key for the locale-specific message to
	 *                      display at the end of the message
     * @param ignore the BooleanSetting for that stores/retrieves whether
     *  or not to display this message.		 
	 */
	public static final void showError(
	 		final String messageStartKey,
			final String message,
			final String messageEndKey,
			final BooleanSetting ignore) {
		MessageService.instance().showError(
			getStringResource(messageStartKey) + " " +
			message + " " +
			getStringResource(messageEndKey), ignore);
	}
	
	/**
	 * Acts as a proxy for the <tt>MessageService</tt> class.
	 * @param t a Throwable object for displaying more information to the user
	 * @param detail A detailed message to display with the error
	 * @param curThread the thread the error occured in.
	 */	
	public static final void showInternalError(Throwable t, String detail,
      Thread curThread) {
		MessageService.instance().showInternalError(
		    getStringResource("ERROR_INTERNAL"),
            t, 
            detail,
            curThread);
	}

	/**
	 * Stub for calling showInternalError(t, null, Thread.currentThread())
	 *
	 * @param t a Throwable object for displaying more information to the user
	 */	
	public static final void showInternalError(Throwable t) {
        showInternalError(t, null, Thread.currentThread());
	}

	/**
	 * Stub for calling showInternalError(t, detail, Thread.currentThread())
	 *
	 * @param t a Throwable object for displaying more information to the user
	 * @param detail A detailed error message to display to the user.
	 */	
	public static final void showInternalError(Throwable t, String detail) {
        showInternalError(t, detail, Thread.currentThread());
	}
	
	/**
	 * Stub for calling showInternalError(t, null, curThread)
	 *
	 * @param t a Throwable object for displaying more information to the user
	 * @param curThread the thread the error occured in
	 */	
	public static final void showInternalError(Throwable t, Thread curThread) {
        showInternalError(t, null, curThread);
	}	

	/**
	 * Acts as a proxy for the Launcher class so that other classes only need
	 * to know about this mediator class.
	 *
	 * <p>Opens the specified url in a browser.
	 *
	 * @param url the url to open
	 * @return an int indicating the success of the browser launch
	 * @throws IOException if the url cannot be loaded do to an IO problem
	 */	
	public static final int openURL(String url) throws IOException {
		return Launcher.openURL(url);
	}

	/**
	 * Acts as a proxy for the Launcher class so that other classes only need
	 * to know about this mediator class.
	 *
	 * <p>Launches the file specified in its associated application.
	 *
	 * @param file a <tt>File</tt> instance denoting the abstract pathname
	 *             of the file to launch
	 * @return an int indicating the success of the file launch
	 * @throws IOException if the file cannot be launched do to an IO problem
	 */
	public static final int launchFile(File file) throws IOException {
		try {
			return Launcher.launchFile(file);
		} catch (SecurityException se) {
			showError("MESSAGE_FILE_LAUNCHING_SECURITY_MESSAGE");
		}
		return -1;
	}

	/**
	 * Returns a <tt>Component</tt> standardly sized for horizontal separators.
	 *
	 * @return the constant <tt>Component</tt> used as a standard horizontal
	 *         separator
	 */
	public static final Component getHorizontalSeparator() {
		return HORIZONTAL_SEPARATOR;
	}

	/**
	 * Returns a <tt>Component</tt> standardly sized for vertical separators.
	 *
	 * @return the constant <tt>Component</tt> used as a standard vertical
	 *         separator
	 */
	public static final Component getVerticalSeparator() {
		return VERTICAL_SEPARATOR;
	}

	/**
	 * Disconnects the user from the network.
	 */
	public void connect() {
		RouterService.connect();
		this.setConnectedUI(true);
	}

	/**
	 * Connects the user to the network.
	 */
	public void disconnect() {
		RouterService.disconnect();
		this.setConnectedUI(false);
	}

	/**
	 * Returns a <tt>boolean</tt> specifying whether or not the user has
	 * donated to the LimeWire project.
	 *
	 * @return <tt>true</tt> if the user has donated, <tt>false</tt> otherwise
	 */
	public static boolean hasDonated() {
		return HAS_DONATED;
	}

	/**
	 * Sets the visible/invisible state of the tab associated with the
	 * specified index.  The indeces correspond to the order of the tabs
	 * whether or not they are visible, as specified by the tab indices in
	 * this class.
	 *
	 * @param TAB_INDEX the index of the tab to make visible or invisible
	 * @param VISIBLE the visible/invisible state to set the tab to
	 */
	public void setTabVisible(final int TAB_INDEX, final boolean VISIBLE) {
		MAIN_FRAME.setTabVisible(TAB_INDEX, VISIBLE);
	}

	/**
	 * Modifies the text displayed to the user in the splash screen to
	 * provide application loading information.
	 *
	 * @param text the text to display
	 */
	public static void setSplashScreenString(String text) {
		SplashWindow.setStatusText(text);
	}

	/**
	 * Returns the point for the placing the specified component on the
	 * center of the screen.
	 *
	 * @param comp the <tt>Component</tt> to use for getting the relative
	 *             center point
	 * @return the <tt>Point</tt> for centering the specified
	 *         <tt>Component</tt> on the screen
	 */
	public static Point getScreenCenterPoint(Component comp) {
		final Dimension COMPONENT_DIMENSION = comp.getSize();
		Dimension screenSize =
			Toolkit.getDefaultToolkit().getScreenSize();
		int appWidth = Math.min(screenSize.width,
		                        COMPONENT_DIMENSION.width);
		// compare against a little bit less than the screen size,
		// as the screen size includes the taskbar
		int appHeight = Math.min(screenSize.height - 40,
		                         COMPONENT_DIMENSION.height);
		return new Point((screenSize.width - appWidth) / 2,
		                 (screenSize.height - appHeight) / 2);
	}

	/**
	 * Adds the <tt>FinalizeListener</tt> class to the list of classes that
	 * should be notified of finalize events.
	 *
	 * @param fin the <tt>FinalizeListener</tt> class that should be notified
	 */
	public static void addFinalizeListener(FinalizeListener fin) {
	    Finalizer.addFinalizeListener(fin);
	}

	/**
	 * Sets the searching or not searching status of the application.
	 *
	 * @param searching the searching status of the application
	 */
	public void setSearching(boolean searching) {
		MAIN_FRAME.setSearching(searching);
	}

	/**
	 * Adds the specified <tt>RefreshListener</tt> instance to the list of
	 * listeners to be notified when a UI refresh event occurs.
	 *
	 * @param the new <tt>RefreshListener</tt> to add
	 */
	public static void addRefreshListener(RefreshListener listener) {
		REFRESH_LIST.add(listener);
	}

	/**
	 * Returns whether or not the client is currently connected to the
	 * Gnutella network.
	 *
	 * @return <tt>true</tt> if the client is connected to Gnutella,
	 *         <tt>false</tt> otherwise
	 */
	public boolean isConnected() {
		return RouterService.isConnected();
	}

	/**
	 * Returns the <tt>Locale</tt> instance currently in use.
	 *
	 * @return the <tt>Locale</tt> instance currently in use
	 */
	static java.util.Locale getLocale() {
		return ResourceManager.getLocale();
	}

	/**
	 * Returns the look and feel for when the application first loaded.
	 *
	 * @return  the look and feel for when the application first loaded
	 */
	public static LookAndFeel getInitialLookAndFeel() {
		return ResourceManager.getInitialLookAndFeel();
	}
	/**
	 * Launches the specified audio file in the player.
	 *
	 * @param file the <tt>File</tt> instance to launch
	 */
	public void launchAudio(File file) {
		STATUS_LINE.launchAudio(file);
	}

	/**
	 * Notification that an audio file has been double-clicked.
	 */
	public void audioFileDoubleClicked() {
		STATUS_LINE.audioFileDoubleClicked();
	}

	/**
	 * Returns whether or not the browser page has been successfully loaded.
	 *
	 * @return <tt>true</tt> if the browser page is successfully loaded,
	 *  otherwise <tt>false</tt>
	 */
	public static boolean isBrowserPageLoaded() {
		return ResourceManager.isBrowserPageLoaded();
	}

	/**
	 * Adds the specified <tt>ThemeObserver</tt> instance to the list of
	 * <tt>ThemeObserver</tt>s that should be notified whenever the theme
	 * changes.
	 *
	 * @param observer the <tt>ThemeObserver</tt> to add to the notification
	 *  list
	 */
	public static void addThemeObserver(ThemeObserver observer) {
		THEME_OBSERVERS.add(observer);
	}
	
	/**
	 * Removes the specified <tt>ThemeObserver</tt> instance from the list
	 * of <tt>ThemeObserver</tt>s.  This is necessary to allow the removed
	 * component to be garbage-collected.
	 *
	 * @param observer the <tt>ThemeObserver</tt> to remove from the notification
	 *  list
	 */
    public static void removeThemeObserver(ThemeObserver observer) {
        THEME_OBSERVERS.remove(observer);
    }

	/**
	 * Sets the current theme pack in use by LimeWire.
	 *
	 * @param theme the <tt>File</tt> containing theme data
	 */
	public void setCurrentTheme(File theme) {
	    ResourceManager.instance().themeChanged();
	    LimeButtonUI.themeChanged();

		SwingUtilities.updateComponentTreeUI(
			GUIMediator.getMainOptionsComponent());
		SwingUtilities.updateComponentTreeUI(
			STATISTICS_MEDIATOR.getMainStatisticsComponent());
		SwingUtilities.updateComponentTreeUI(FRAME);
		
        Iterator iter = THEME_OBSERVERS.iterator();
		while (iter.hasNext()) {
			ThemeObserver curObserver = (ThemeObserver)iter.next();
			curObserver.updateTheme();
		}
		
        GUIMediator.getMainOptionsComponent().validate();
        STATISTICS_MEDIATOR.getMainStatisticsComponent().validate();
        FRAME.validate();

	}

}
