package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.settings.BooleanSetting;
import com.limegroup.gnutella.settings.IntSetting;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.*;

import java.awt.*;
import javax.swing.*;
import java.io.*;
import java.awt.event.*;
import java.util.Properties;
import com.sun.java.util.collections.Map;
import com.sun.java.util.collections.HashMap;
import com.sun.java.util.collections.Iterator;

/**
 * This class handles displaying messages to the user.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class MessageService {

	/**
	 * Constant for when the user selects the yes button in a message 
	 * giving the user a yes and a no option.
	 */
	public static final int YES_OPTION = 101;

	/**
	 * Constant for when the user selects the no button in a message giving 
	 * the user a yes and a no option.
	 */
	public static final int NO_OPTION  = 102;
	
	/**
	 * Constant for when the 'Always use this answer' checkbox wants to 
	 * remember the answer.
	 */
	public static final int REMEMBER_ANSWER = 1;
	
	/**
	 * Constant for when the 'Always use this answer' checkbox does not
	 * want to remember the answer.
	 */
	public static final int FORGET_ANSWER = 0;

	/**
	 * <tt>MessageService</tt> instance, following singleton.
	 */
	private static final MessageService INSTANCE = new MessageService();

	/**
	 * Instance accessor for the <tt>MessageService</tt>.
	 */
	public static MessageService instance() {
		return INSTANCE;
	}

	/**
	 * Initializes all of the necessary messaging components.
	 */
	MessageService() {
		GUIMediator.setSplashScreenString(
	        GUIMediator.getStringResource("SPLASH_STATUS_MESSAGE_SERVICE"));
	}


	/**
	 * Display a standardly formatted error message with
	 * the specified String. 
	 *
	 * @param message the message to display to the user
	 */
	final void showError(String message) {
    	JOptionPane.showMessageDialog(getParentComponent(), 
                    new MultiLineLabel(message),
				  GUIMediator.getStringResource("MESSAGE_ERROR_CAPTION"),
				  JOptionPane.ERROR_MESSAGE);
	}
	
	/**
	 * Display a standardly formatted error message with
	 * the specified String. 
	 *
	 * @param message the message to display to the user
	 * @param ignore the Boolean setting to store/retrieve whether or not to
	 *  ignore this message in the future.	 
	 */
	final void showError(String message, BooleanSetting ignore) {
	    if ( !ignore.getValue() ) {
		    JOptionPane.showMessageDialog(getParentComponent(), 
                      doNotDisplayAgainLabel(message, ignore),
					  GUIMediator.getStringResource("MESSAGE_ERROR_CAPTION"),
					  JOptionPane.ERROR_MESSAGE);
        }
	}	

	/**
	 * Display a standardly formatted information message with
	 * the specified String. 
	 *
	 * @param message the message to display to the user
	 */	
	final void showMessage(String message) {
		JOptionPane.showMessageDialog(getParentComponent(), 
				  new MultiLineLabel(message),
				  GUIMediator.getStringResource("MESSAGE_CAPTION"),
				  JOptionPane.INFORMATION_MESSAGE);
	}
	
	/**
	 * Display a standardly formatted information message with
	 * the specified String.  Store whether or not to display message
	 * again in the BooleanSetting ignore.
	 *
	 * @param message the message to display to the user
	 * @param ignore the Boolean setting to store/retrieve whether or not to
	 *  ignore this message in the future.
	 */	
	final void showMessage(String message, BooleanSetting ignore) {
	    if ( !ignore.getValue() ) {
    		JOptionPane.showMessageDialog(getParentComponent(),
                      doNotDisplayAgainLabel(message, ignore),
					  GUIMediator.getStringResource("MESSAGE_CAPTION"),
					  JOptionPane.INFORMATION_MESSAGE);
        }
	}	

	/**
	 * Display a standardly formatted confirmation message with
	 * the specified String. 
	 *
	 * @param message the message to display to the user
	 */	
	final void showConfirmMessage(String message) {
		JOptionPane.showConfirmDialog(getParentComponent(), 
					  new MultiLineLabel(message),
					  GUIMediator.getStringResource("MESSAGE_CAPTION"),
					  JOptionPane.INFORMATION_MESSAGE);
	}
	
	/**
	 * Display a standardly formatted confirmation message with
	 * the specified String.  Store whether or not to display
	 * the message again in the BooleanSetting ignore.
	 *
	 * @param message the message to display to the user
	 * @param ignore the Boolean setting to store/retrieve whether or not to
	 *  ignore this message in the future.	 
	 */	
	final void showConfirmMessage(String message, BooleanSetting ignore) {
	    if ( !ignore.getValue() ) {
    		JOptionPane.showConfirmDialog(getParentComponent(), 
						  doNotDisplayAgainLabel(message, ignore),
						  GUIMediator.getStringResource("MESSAGE_CAPTION"),
						  JOptionPane.INFORMATION_MESSAGE);
        }
	}	

	/**
	 * Displays a message to the user and returns 
	 * MessageService.YES_OPTION if the user selects yes and
	 * MessageService.NO_OPTION if the user selects no.
	 *
	 * @param message the message to display to the user
	 */ 
	final int showYesNoMessage(String message) {
		int option = 
		    JOptionPane.showConfirmDialog(getParentComponent(), 
							  new MultiLineLabel(message), 
							  GUIMediator.getStringResource("MESSAGE_CAPTION"),
							  JOptionPane.YES_NO_OPTION);
		if(option == JOptionPane.YES_OPTION) return MessageService.YES_OPTION;
		return MessageService.NO_OPTION;
	}
	
	/**
	 * Displays a message to the user and returns 
	 * MessageService.YES_OPTION if the user selects yes and
	 * MessageService.NO_OPTION if the user selects no.  Stores
	 * the default response in IntSetting default.
	 *
	 * @param message the message to display to the user
	 * @param defValue the IntSetting to store/retrieve the the default
	 *  value for this question.
	 */ 
	final int showYesNoMessage(String message, IntSetting defValue) {
	    // if default has a valid value, use it.
	    if(defValue.getValue() == YES_OPTION || defValue.getValue() == NO_OPTION)
	        return defValue.getValue();
	        
        // We only get here if the default didn't have a valid value.	        
		int option = 
		    JOptionPane.showConfirmDialog(getParentComponent(),
                alwaysUseThisAnswerLabel(message, defValue),
                GUIMediator.getStringResource("MESSAGE_CAPTION"),
                JOptionPane.YES_NO_OPTION);
                
        int ret;                
		if(option == JOptionPane.YES_OPTION)
		    ret = MessageService.YES_OPTION;
        else
            ret = MessageService.NO_OPTION;
            
        // If we wanted to remember the answer, remember it.            
        if ( defValue.getValue() == REMEMBER_ANSWER )
            defValue.setValue(ret);
        else
            defValue.setValue(FORGET_ANSWER);
            
        return ret;
	}	

	/**
	 * Convenience method for determining which window should be the parent
	 * of message windows.
	 *
	 * @return the <tt>Component</tt> that should act as the parent of message
	 *  windows
	 */
	private static Component getParentComponent() {
		if(GUIMediator.isOptionsVisible()) 
			return GUIMediator.getMainOptionsComponent();
		return GUIMediator.getAppFrame();
	}

	/**
	 * Display a standardly formatted internal error message
	 * coming from the backend.
	 *
	 * @param message the message to display to the user
	 * @param t the <tt>Throwable</tt> object containing information 
	 *  about the error
	 * @param detail the detailed message to display after the stack trace.
     * @param curThread the current thread giving the error
	 */	
	final void showInternalError(String message, Throwable t, String detail,
                                 Thread curThread) {
        // ThreadDeath must NOT be caught, or a thread will be left zombied.
        if( t instanceof ThreadDeath) {
            throw (ThreadDeath)t;
        }
                                    
		t.printStackTrace();
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.print("LimeWire version ");
		pw.println(CommonUtils.getLimeWireVersion());            
		pw.print("Java version ");
		pw.print(CommonUtils.getJavaVersion());
		pw.print(" from ");
		pw.println(prop("java.vendor"));
		pw.print(CommonUtils.getOS());
		pw.print(" v. ");
		pw.print(prop("os.version"));
		pw.print(" on ");
		pw.println(prop("os.arch"));
		Runtime runtime = Runtime.getRuntime();
		pw.println("Free/total memory: "
				   +runtime.freeMemory()+"/"+runtime.totalMemory());
		pw.println();
		
		t.printStackTrace(pw);

		pw.println();
		
		if( detail != null ) {
		    pw.println("Detail: " + detail);
		    pw.println();
        }
        
	    if( t instanceof ExceptionInInitializerError ) {
	        ExceptionInInitializerError err = (ExceptionInInitializerError)t;
	        if( err.getException() != null ) {
    	        pw.println("Underlying error: ");
    	        err.getException().printStackTrace(pw);
    	        pw.println();
            }
        }		    

		addDebugInformation(pw, curThread);

		File propsFile = new File(CommonUtils.getUserSettingsDir(),
								  "limewire.props");
		Properties props = new Properties();
		try {
			FileInputStream fis = new FileInputStream(propsFile);
			props.load(fis);
			fis.close();
			// list the properties in the PrintWriter.
			props.list(pw);
		} catch(FileNotFoundException fnfe) {
		} catch(IOException ioe) {
		}

        pw.println("");
        pw.println("");
        pw.println("");
		pw.println("FILES IN CURRENT DIRECTORY:");
        File curDir = CommonUtils.getCurrentDirectory();
        String[] files = curDir.list();
        for(int i=0; i<files.length; i++) {
            File curFile = new File(curDir, files[i]);
            pw.println(curFile.toString());
            pw.println("LAST MODIFIED: "+curFile.lastModified());
            pw.println("SIZE: "+curFile.length());
            pw.println();
        }

		pw.flush();
		
		displayError(message, sw.toString());
	}
	
	private final void addDebugInformation(PrintWriter pw, Thread curThread) {
	    pw.println("-- listing session information --");
	    
        pw.println("Current thread: " +
            curThread.getName() );
        pw.println("Active Threads: " +
            Thread.currentThread().getThreadGroup().activeCount() );

        // Do not continue if the GUI has not been constructed.
        if( !GUIMediator.isConstructed() ) {
            pw.println("GUI not constructed.");
            pw.println();
            return;
        } 
   
	    if( !RouterService.isStarted() ) {
	        pw.println("ROUTER SERVICE NOT STARTED");
	        pw.println();
	        return;
	    }
	    
	    try {
	        pw.println("Uptime: " +
	            GUIUtils.seconds2time(
	                (int)(RouterService.getCurrentUptime()/1000)) );
    	    pw.println("Is connected: " +
    	        RouterService.isConnected() );
    	    pw.println("Number of Ultrapeer -> Ultrapeer Connections: " +
    	        RouterService.getNumUltrapeerToUltrapeerConnections() );
    	    pw.println("Number of Ultrapeer -> Leaf Connections: " +
    	        RouterService.getNumUltrapeerToLeafConnections() );
    	    pw.println("Number of Leaf -> Ultrapeer Connections: " +
    	        RouterService.getNumLeafToUltrapeerConnections() );
    	    pw.println("Number of Old Connections: " +
    	        RouterService.getNumOldConnections() );
    	    pw.println("Acting as Ultrapeer: " +
    	        RouterService.isSupernode() );
    	    pw.println("Acting as Shielded Leaf: " +
    	        RouterService.isShieldedLeaf() );
    	    pw.println("Number of Active Uploads: " +
    	        RouterService.getNumUploads() );
    	    pw.println("Number of Queued Uploads: " +
    	        RouterService.getNumQueuedUploads() );
    	    pw.println("Number of Active Managed Downloads: " +
    	        RouterService.getNumActiveDownloads() );
    	    pw.println("Number of Active HTTP Downloaders: " +
    	        RouterService.getNumIndividualDownloaders() );
            pw.println("Number of Waiting Downloads: " +
                RouterService.getNumWaitingDownloads() );
    	    pw.println("Received incoming this session: " +
    	        RouterService.acceptedIncomingConnection() );
    	    pw.println("Number of Shared Files: " +
    	        RouterService.getNumSharedFiles() );
        } catch(Throwable t) {
            pw.println("UNABLE TO RETRIEVE INFORMATION.");
            t.printStackTrace(pw);
        }
        
        pw.println();
	    pw.println("-- listing threads --");
	    try {
	        Thread[] allThreads = new Thread[Thread.activeCount()];
	        int copied = Thread.enumerate(allThreads);
	        Map threads = new HashMap();
	        for(int i = 0; i < copied; i++) {
	            String name = allThreads[i].getName();
	            Object val = threads.get(name);
	            if(val == null)
	                threads.put(name, new Integer(1));
	            else {
	                int num = ((Integer)val).intValue()+1;
	                threads.put(name,new Integer(num));
                }
	        }
	        for(Iterator i = threads.entrySet().iterator(); i.hasNext();) {
	            Map.Entry info = (Map.Entry)i.next();
	            pw.println( info.getKey() + ": " + info.getValue());
	        }
	    } catch(Throwable t) {
	        pw.println("Unable to list thread counts.");
	        t.printStackTrace(pw);
	    }        
        
        pw.println();	        
    }

    /** 
	 * Returns the System property with the given name, or
     * "?" if it is unknown. 
	 */
    private final String prop(String name) {
        String value = System.getProperty(name);
        if (value == null) return "?";
        else return value;
    }


	/**
	 * Displays an internal error with specialized formatting.
	 *
	 * @param header the message to display to the user regarding the error.
	 * @param error the string containing the stack trace, properties, etc. 
     *  for the error
	 */
	private final void displayError(String header, String error) {
		final JDialog DIALOG = new JDialog(GUIMediator.getAppFrame(), true);
		final Dimension DIALOG_DIMENSION = new Dimension(260, 120);
		final Dimension INNER_SIZE = new Dimension(220, 100);
		DIALOG.setSize(DIALOG_DIMENSION);

        JPanel mainPanel = new JPanel();
        mainPanel.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));
		mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.Y_AXIS));



        MultiLineLabel label = new MultiLineLabel(header, 400);
		JPanel labelPanel = new JPanel();
		JPanel innerLabelPanel = new JPanel();
		labelPanel.setLayout(new BoxLayout(labelPanel, BoxLayout.X_AXIS));		
		innerLabelPanel.setLayout(new BoxLayout(innerLabelPanel, 
                                                BoxLayout.Y_AXIS));
		innerLabelPanel.add(label);
		innerLabelPanel.add(Box.createVerticalStrut(6));
		labelPanel.add(innerLabelPanel);
		labelPanel.add(Box.createHorizontalGlue());

        final JTextArea textArea = new JTextArea(error);
        textArea.selectAll();
        textArea.copy();
        textArea.setColumns(50);
        textArea.setEditable(false);
        JScrollPane scroller = new JScrollPane(textArea);
        scroller.setBorder(BorderFactory.createEtchedBorder());
		scroller.setPreferredSize(INNER_SIZE);


        JPanel buttonPanel = new JPanel();
        JButton copyButton = new JButton("Copy Selection");
        copyButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				textArea.copy();
			}
		});
        JButton quitButton = new JButton("Ok");
        quitButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e){
				DIALOG.dispose();
			}
		});
        buttonPanel.add(copyButton);
        buttonPanel.add(quitButton);

        mainPanel.add(labelPanel);
        mainPanel.add(scroller);
        mainPanel.add(buttonPanel);

        DIALOG.getContentPane().add(mainPanel);

		if(GUIMediator.getAppVisible())
			DIALOG.setLocationRelativeTo(getParentComponent());
		
		else {

			Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
			DIALOG.setLocation((screenSize.width - DIALOG_DIMENSION.width)/2,
							   (screenSize.height - DIALOG_DIMENSION.height)/2);

		}

		DIALOG.pack();
		DIALOG.show();
    }
    
    private final JComponent doNotDisplayAgainLabel(
      String message, final BooleanSetting setting) {
        JPanel thePanel = new JPanel( new BorderLayout(0, 15) ); 
        JCheckBox option = new JCheckBox(
            GUIMediator.getStringResource("OPTIONS_DO_NOT_ASK_AGAIN")
        );
        MultiLineLabel lbl = new MultiLineLabel(message);
        thePanel.add( lbl, BorderLayout.NORTH );
        thePanel.add( option, BorderLayout.WEST );
        option.addItemListener( new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                setting.setValue( e.getStateChange() == ItemEvent.SELECTED );
            }
        });
        return thePanel;
    }
    
    private final JComponent alwaysUseThisAnswerLabel(
      String message, final IntSetting setting) {
        JPanel thePanel = new JPanel( new BorderLayout(0, 15) ); 
        JCheckBox option = new JCheckBox(
            GUIMediator.getStringResource("OPTIONS_ALWAYS_USE_ANSWER")
        );
        MultiLineLabel lbl = new MultiLineLabel(message);
        thePanel.add( lbl, BorderLayout.NORTH );
        thePanel.add( option, BorderLayout.WEST );
        option.addItemListener( new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                if ( e.getStateChange() == ItemEvent.SELECTED )
                    setting.setValue( REMEMBER_ANSWER );
                else
                    setting.setValue( FORGET_ANSWER );
            }
        });
        return thePanel;
    }    
}
