package com.limegroup.gnutella.gui.tables;

import com.sun.java.util.collections.*;
import java.awt.event.MouseEvent;
import java.awt.*;
import javax.swing.*;
import javax.swing.table.*;

import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.JMultilineToolTip;
import com.limegroup.gnutella.settings.ThemeFileHandler;
import com.limegroup.gnutella.settings.UISettings;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.DataUtils;

/**
 * A specialized JTable for use with special Lime functions.
 * 1) Allows the user to easily
 *    set a column as visible or invisible, rather than
 *    having to remove/add columns.
 *    It internally will remember where the column was and
 *    add/remove it as needed.
 * 2) It remembers which column is sorted and whether the sort
 *    is ascending or descending.
 *    For use with adding arrows to the tableHeader.
 * 3) Shows special tooltips for each row.
 * @author Sam Berlin
 */
public final class LimeJTable extends JTable implements JSortTable {

    protected Map _hiddenColumns = new HashMap();
    protected int sortedColumnIndex = -1;
    protected boolean sortedColumnAscending = true;
    protected int pressedColumnIndex = -1;
    private String[] tips;
    private static LimeTableColumn _lastRemoved;
    protected ColumnPreferenceHandler columnPreferences;
    
    private static final boolean IS_JAVA_13 =
        CommonUtils.isJava13OrLater() && !CommonUtils.isJava14OrLater();

    /**
     * Same as JTable().
     * The table MUST have setModel called with a DataLineModel in order
     * for this class to function properly.
     */
    public LimeJTable() {
        super();
        setToolTipText("");
    }

    /**
     * Same as JTable(DataLineModel)
     */
    public LimeJTable(DataLineModel dm) {
        super(dm);
        setToolTipText("");
    }

    /**
     * Access the ColumnPreferenceHandler.
     */
    public ColumnPreferenceHandler getColumnPreferenceHandler() {
        return columnPreferences;
    }

    /**
     * Set the ColumnPreferenceHandler
     */
    public void setColumnPreferenceHandler(ColumnPreferenceHandler handl) {
        columnPreferences = handl;
    }

    /**
     * set the pressed header column.
     * @param col The MODEL index of the column
     */
    public void setPressedColumnIndex(int col) {
        pressedColumnIndex = col;
    }

    /**
     * get the pressed header column
     * @return the VIEW index of the pressed column.
     */
    public int getPressedColumnIndex() {
        return convertColumnIndexToView(pressedColumnIndex);
    }

    /**
     * @return the VIEW index of the sorted column.
     */
    public int getSortedColumnIndex() {
        return convertColumnIndexToView(sortedColumnIndex);
    }

    /**
     * accessor function
     */
    public boolean isSortedColumnAscending() { return sortedColumnAscending; }

    /**
     * Tells the table that 'X' column is sorted.
     * @param index the MODEL index of the column
     * @param ascending whether or not the sort is ascending
     */
    public void sortColumn(int index, boolean ascending) {
        sortedColumnIndex = index;
        sortedColumnAscending = ascending;
    }

    /**
     * Simple function that tucks away hidden columns for use later.
     * And it uses them later!
     */
    public void setColumnVisible(Object columnId, boolean visible)
        throws LastColumnException {
        if ( !visible ) {
            TableColumnModel model = getColumnModel();
            // don't allow the last column to be removed.
            if ( model.getColumnCount() == 1 ) throw new LastColumnException();
            TableColumn column = model.getColumn( model.getColumnIndex(columnId) );
            _hiddenColumns.put( columnId, column );
            _lastRemoved = (LimeTableColumn)column;
            removeColumn(column);
        } else {
            TableColumn column = (TableColumn)_hiddenColumns.get(columnId);
            _hiddenColumns.remove( columnId );
            addColumn(column);
        }
    }

    /**
     * Returns an iterator of the removed columns.
     */
    public Iterator getHiddenColumns() {
        return Collections.unmodifiableCollection(
                        _hiddenColumns.values()).iterator();
    }

    /**
     * Returns the last removed column.
     */
    public LimeTableColumn getLastRemovedColumn() {
        return _lastRemoved;
    }

    /**
     * Determines whether or not a column is visible in this table.
     */
    public boolean isColumnVisible(Object columnId) {
        return !_hiddenColumns.containsKey(columnId);
    }

    /**
     * Sets the internal tooltip text for use with the next
     * createToolTip.
     */
    public String getToolTipText(MouseEvent e) {
        Point p = e.getPoint();
        int row = rowAtPoint(p);
        if ( row > -1 ) {
            DataLineModel dlm = (DataLineModel)dataModel;
            //set the internal tips for later use with createToolTip
            tips = dlm.getToolTipArray(row);
            // NOTE: the below return triggers the tooltip manager to
            // create a tooltip.
            // If it is null, one won't be created.
            // If two different rows return the same tip, the manager
            // won't be triggered to create a 'new' tip.
            // Rather than return the actual row#, which could stay the same
            // if sorting is enabled & the DataLine moves,
            // return the string representation
            // of the dataline, so if the row moves out from under the mouse,
            // the tooltip will auto change when the mouse
            // moves around the new DataLine (same row)
            if (tips == null)
                return null;
            else
                return dlm.get(row).toString();
        }
        tips = DataUtils.EMPTY_STRING_ARRAY;
        return null;
    }


    /**
     *@return The JToolTip returned is actually a JMultilineToolTip
     */
    public JToolTip createToolTip() {
        JMultilineToolTip ret = JMultilineToolTip.instance();
        ret.setToolTipArray( tips );
        tips = DataUtils.EMPTY_STRING_ARRAY;
        return ret;
    }

    /**
     * Overrides JTable's default implementation in order to add
     * LimeTableColumn columns.
     */
    public void createDefaultColumnsFromModel() {
        DataLineModel dlm = (DataLineModel)dataModel;
        if (dlm != null) {
            // Remove any current columns
            TableColumnModel cm = getColumnModel();
            while (cm.getColumnCount() > 0) {
                cm.removeColumn(cm.getColumn(0));
            }
        
            // Create new columns from the data model info
            for (int i = 0; i < dlm.getColumnCount(); i++) {
                TableColumn newColumn = dlm.getTableColumn(i);
                addColumn(newColumn);
            }
        }
    }
    
    /**
     * Returns the color that a specific row will be.
     */
    public Color getBackgroundForRow(int row) {
        if(row % 2 == 0 || !UISettings.ROW_STRIPE_ENABLED.getValue())
            return getBackground();
        else
            return ThemeFileHandler.TABLE_ALTERNATE_COLOR.getValue();
    }
        
    
    /**
     * This overrides JTable.prepareRenderer so that we can stripe the
     * rows as needed.
     */    
    public Component prepareRenderer(TableCellRenderer renderer,
      int row, int column) {
        Object value = getValueAt(row, column);
    	boolean isSelected = isCellSelected(row, column);
    	boolean rowIsAnchor = (selectionModel.getAnchorSelectionIndex()==row);
    	boolean colIsAnchor =
    	  (columnModel.getSelectionModel().getAnchorSelectionIndex()==column);
    	boolean hasFocus = (rowIsAnchor && colIsAnchor) && hasFocus();
    
    	Component r = renderer.getTableCellRendererComponent(this, value,
    	                                              isSelected, hasFocus,
    	                                              row, column);
	                      
        Color  odd = ThemeFileHandler.TABLE_ALTERNATE_COLOR.getValue();
        Color even = ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue();
        
        if ( isSelected ) {
            // do nothing if selected.
        } else if (hasFocus && isCellEditable(row, column)) {
            // do nothing if we're focused & editting.
        } else if (even.equals(odd)) {
            // do nothing if backgrounds are the same.
        } else if (!UISettings.ROW_STRIPE_ENABLED.getValue()) {
            // if the renderer's background isn't already the normal one,
            // change it.  (needed for real-time changing of the option)
            if( r != null && !r.equals(even) )
                r.setBackground(even);
        } else if ( row % 2 != 0 ) {
            r.setBackground(odd);
        } else {
            r.setBackground(even);
        }
        
        // This is necessary to work around the 'optimization' of
        // DefaultTableCellRenderer in 1.3
        // See: http://developer.java.sun.com/developer/bugParade/bugs/4382860.html
        if( IS_JAVA_13 ) {
        	Color back = r.getBackground();
            boolean colorMatch = (back != null) && ( back.equals(even) );
            ((JComponent)r).setOpaque(!colorMatch);
        }
        
        return r;
    }
}
