/****************************************************************************
**
** $Id: FileTableImpl.cpp,v 1.23 2004/01/11 22:11:44 riemer Exp $
**
** Copyright (C) 2001-2004 Frank Hemer <frank@hemer.org>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** LinCVS is available under two different licenses:
**
** If LinCVS is linked against the GPLed version of Qt 
** LinCVS is released under the terms of GPL also.
**
** If LinCVS is linked against a nonGPLed version of Qt 
** LinCVS is released under the terms of the 
** LinCVS License for non-Unix platforms (LLNU)
**
**
** LinCVS License for non-Unix platforms (LLNU):
**
** Redistribution and use in binary form, without modification, 
** are permitted provided that the following conditions are met:
**
** 1. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 2. It is not permitted to distribute the binary package under a name
**    different than LinCVS.
** 3. The name of the authors may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
** 4. The source code is the creative property of the authors.
**    Extensions and development under the terms of the Gnu Public License
**    are limited to the Unix platform. Any distribution or compilation of 
**    the source code against libraries licensed other than gpl requires 
**    the written permission of the authors.
**
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
**
**
** LinCVS License for Unix platforms:
**
** This program is free software; you can redistribute it and/or modify 
** it under the terms of the GNU General Public License as published by 
** the Free Software Foundation; either version 2 of the License, or 
** (at your  option) any later version.  This program is distributed in 
** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
** PARTICULAR PURPOSE.
**
** See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include <FileTableImpl.h>

#include <qapplication.h>
#include <qtable.h>
#include <qtextedit.h>
#include <qlineedit.h>
#include <qregexp.h>
#include <qstyle.h>
#include <qpainter.h>

// QString history = "";

class FileTableItem : public QTableItem {

public:

  FileTableItem(QTable*,QTableItem::EditType,QString,const QColor&,const QColor&,bool unchangeable=false,int id=0);
  virtual ~FileTableItem();

  void setColorGroup(QColorGroup&);
  void paint( QPainter*, const QColorGroup&, const QRect&, bool);
  void toggle();
  bool isSelected() { return m_state;}
  void setSelected(bool b) { if (m_state != b) toggle();}
  bool changeable() { return !m_unchangeable;}
  int id() { return m_id;}

  QColor m_selectedColor;
  QColor m_unselectedColor;
  QColorGroup m_colorGroup;
  bool m_state;
  bool m_unchangeable;
  int m_id;
};

FileTableItem::FileTableItem(QTable* tbl,
			     QTableItem::EditType type,
			     QString txt,
			     const QColor& selectedColor,
			     const QColor& unselectedColor,
			     bool unchangeable,
			     int id)
  : QTableItem(tbl,type,txt),
    m_selectedColor(selectedColor),
    m_unselectedColor(unselectedColor),
    m_colorGroup(QColorGroup()),
    m_unchangeable(unchangeable),
    m_id(id) {

  m_state = false;
  m_colorGroup.setColor(QColorGroup::Base,unselectedColor);
}

FileTableItem::~FileTableItem() {}

void FileTableItem::paint( QPainter* p, const QColorGroup&, const QRect& cr, bool) {
  QTableItem::paint( p, m_colorGroup, cr, false);
}

void FileTableItem::setColorGroup(QColorGroup& grp) {
  m_colorGroup = QColorGroup(grp);
}

void FileTableItem::toggle() {
  if (m_unchangeable) return;
  if (m_state^=true) {
//     qDebug("toggle on row:"+QString::number(row())+", col: "+QString::number(col()));
    m_colorGroup.setColor(QColorGroup::Base,m_selectedColor);
  } else {
//     qDebug("toggle off row:"+QString::number(row())+", col: "+QString::number(col()));
    m_colorGroup.setColor(QColorGroup::Base,m_unselectedColor);
  }
}

//============================================FileTableImpl============================================

FileTableImpl::FileTableImpl( QWidget* parent,
			      const char* name,
			      WFlags fl )
  : FileTable( parent, name, fl)
{
  Table->horizontalHeader()->hide();
  Table->verticalHeader()->hide();
  Table->setTopMargin(0);
  Table->setLeftMargin(0);
  Table->setSelectionMode(QTable::MultiRow);
  Table->setFocusPolicy(QWidget::NoFocus);
  Table->setHScrollBarMode(QScrollView::AlwaysOn);
  Table->setVScrollBarMode(QScrollView::AlwaysOn);

  m_stop = false;
  m_rows = 0;
  m_lines = 0;
  m_blocks = 0;
  m_selectedRow = -1;
  m_focusedRowPos = -1;
  m_wasBlockSelect = false;
  m_lastMarked = -1;
  m_Partner = 0;
  m_maxWidth = 0;
  fm = new QFontMetrics(font());
  m_lineSpacing = fm->lineSpacing();
  m_ScrollBarMinPos = -1;
//   history = "";

  Table->setCornerWidget(new QWidget(this));
  Table->cornerWidget()->setBackgroundColor("gray");

  connect( Table->horizontalScrollBar(), SIGNAL(valueChanged(int)),
	   this,SLOT(xScrollPosChanged(int)) );
  connect( Table->horizontalScrollBar(), SIGNAL(sliderMoved(int)),
	   this,SLOT(xScrollPosChanged(int)) );
  connect( Table->verticalScrollBar(), SIGNAL(valueChanged(int)),
	   this,SLOT(yScrollPosChanged(int)) );
  connect( Table->verticalScrollBar(), SIGNAL(sliderMoved(int)),
	   this,SLOT(yScrollPosChanged(int)) );

}

FileTableImpl::~FileTableImpl() {
}

void FileTableImpl::setFont ( const QFont & font) {
  if (fm) delete fm;
  fm = new QFontMetrics(font);
  m_lineSpacing = fm->lineSpacing();
  FileTable::setFont(font);
}

void FileTableImpl::cancelRead() {
  m_stop = true;
}

void FileTableImpl::setID( int id) {
  m_ID = id;
}

void FileTableImpl::xScrollPosChanged(int val) {
  emit xScrollValueChanged(val);
}

void FileTableImpl::yScrollPosChanged(int val) {
  emit yScrollValueChanged(val);
}

void FileTableImpl::setXScrollPos(int val) {
  Table->horizontalScrollBar()->setValue(val);
}

void FileTableImpl::setYScrollPos(int val) {
  Table->verticalScrollBar()->setValue(val);
}

void FileTableImpl::adjustSize(int size) {
  QString hlp;
  hlp.setNum(m_lines);
  hlp += ": ";
  int col0width = fm->width(hlp);
  Table->setColumnWidth(0,col0width);
  int col1width = col0width+size;
  m_maxWidth = col1width;
  int tableWidth = Table->horizontalScrollBar()->width()-Table->cornerWidget()->width();
  if (tableWidth>col1width) {
    col1width = tableWidth;
  }
  Table->setColumnWidth(1,col1width);
}

void FileTableImpl::focusInEvent( QFocusEvent*) {
  Table->cornerWidget()->setBackgroundColor("red");
}

void FileTableImpl::focusOutEvent( QFocusEvent*) {
  Table->cornerWidget()->setBackgroundColor("gray");
}

void FileTableImpl::keyPressEvent( QKeyEvent* e) {

  FileTable::keyPressEvent( e );

  switch (e->key()) {

    case Qt::Key_Delete: {
      emit clear( m_ID, m_selectedRow);
      break;
    }
    case Qt::Key_Down: {
      if (m_wasBlockSelect) {
	m_wasBlockSelect = false;
	m_lastMarked = m_selectedRow;
      }
      if (e->state() & Qt::ControlButton || !Table->verticalScrollBar()->isVisible()) {
	if (m_selectedRow==-1) setFocusRow(0);
	else if (m_selectedRow<m_rows-1) {
	  int i = 0;
	  FileTableItem* item = (FileTableItem*)Table->item(m_selectedRow,1);
	  if (item) {
	    int id = item->id();
	    for (i = m_selectedRow+1;i<m_rows;i++) {
	      item = (FileTableItem*)Table->item(i,1);
	      if (item && item->id()!=id) {
		setFocusRow(i);
		return;
	      }
	    }
	    setFocusRow(m_rows-1);
	  }
	}
      } else {
	if (m_selectedRow==-1) setFocusRow(0);
	else setFocusPos(m_focusedRowPos+m_lineSpacing);
      }
      break;
    }
    case Qt::Key_Up: {
      if (m_wasBlockSelect) {
	m_wasBlockSelect = false;
	m_lastMarked = m_selectedRow;
      }
      if (e->state() & Qt::ControlButton || !Table->verticalScrollBar()->isVisible()) {
	if (m_selectedRow==-1) setFocusRow(0);
	else if (m_selectedRow>0) {
	  int i = 0;
	  FileTableItem* item = (FileTableItem*)Table->item(m_selectedRow,1);
	  if (item) {
	    int id = item->id();
	    for (i = m_selectedRow-1;i>=0;i--) {
	      item = (FileTableItem*)Table->item(i,1);
	      if (item && item->id()!=id) {
		setFocusRow(i);
		return;
	      }
	    }
	    setFocusRow(0);
	  }
	}
      } else {
	if (m_selectedRow==-1) setFocusRow(0);
	else setFocusPos(m_focusedRowPos-m_lineSpacing);
      }
      break;
    }
    case Qt::Key_Space: {
      if (m_selectedRow==-1) setFocusRow(0);
      if (e->state() & Qt::ShiftButton) {

	FileTableItem*item = (FileTableItem*)Table->item(m_selectedRow,1);
	if (item && !item->changeable()) return;

	item = (FileTableItem*)Table->item(m_lastMarked,1);
	bool state = false;
	if (item) {
	  state = item->isSelected();
	  if (m_wasBlockSelect && (item = (FileTableItem*)Table->item(m_selectedRow,1)) && item) state = !item->isSelected();
	  else if (state == ((FileTableItem*)Table->item(m_selectedRow,1))->isSelected()) return;//start and end marker need different state
	}
	if (m_selectedRow<m_lastMarked) {
	  int i = 0;
	  m_wasBlockSelect = true;
	  int to = setBlockState(m_selectedRow+1-1,m_lastMarked-1,state);
	  for (i=m_selectedRow+1;i<=to;i++) {
	    setSelected(i,state);
	  }
	  if (to != m_lastMarked-1) return;//different block, don't select
	} else if (m_lastMarked<m_selectedRow) {
	  int i = 0;
	  m_wasBlockSelect = true;
	  int to = setBlockState(m_lastMarked+1,m_selectedRow-1+1,state);
	  for (i=m_lastMarked+1;i<to;i++) {
	    setSelected(i,state);
	  }
	  if (to != m_selectedRow-1+1) return;//different block, don't select
	}

      } else {
	m_wasBlockSelect = false;
      }
      FileTableItem*item = (FileTableItem*)Table->item(m_selectedRow,1);
      if (item) {
	item->toggle();
	Table->updateCell(m_selectedRow,1);
	if (!m_wasBlockSelect) {
	  m_lastMarked = m_selectedRow;
	}
	if (!item->changeable()) return;
	if (!m_wasBlockSelect) {
	  if (item->isSelected()) {
	    emit selectionChanged(m_ID,m_selectedRow,item->text());
	  } else {
	    emit unselected(m_ID,m_selectedRow);
	  }
	}
      }
      break;
    }
    case Qt::Key_Prior: {
      if (m_wasBlockSelect) {
	m_wasBlockSelect = false;
	m_lastMarked = m_selectedRow;
      }
      setFocusPos(m_focusedRowPos-(Table->visibleHeight()/2));
      break;
    }
    case Qt::Key_Next: {
      if (m_wasBlockSelect) {
	m_wasBlockSelect = false;
	m_lastMarked = m_selectedRow;
      }
      setFocusPos(m_focusedRowPos+(Table->visibleHeight()/2));
      break;
    }
    case Qt::Key_Home: {
      if (m_wasBlockSelect) {
	m_wasBlockSelect = false;
	m_lastMarked = m_selectedRow;
      }
      setFocusPos(0);
      break;
    }
    case Qt::Key_End: {
      if (m_wasBlockSelect) {
	m_wasBlockSelect = false;
	m_lastMarked = m_selectedRow;
      }
      setFocusPos(Table->contentsHeight());
      break;
    }
  }
}

void FileTableImpl::setBackground(QColor color) {
  QPalette pal = Table->palette();
  setPaletteBackgroundColor(color);
  pal.setColor(QColorGroup::Highlight,color);
  Table->setPalette(pal);
  Table->setPaletteBackgroundColor(color);
}

void FileTableImpl::tableClicked(int row, int col, int button, const QPoint&) {

//   switch(button) {
//     case 1: {
//       history += "\nL,"+QString::number(row)+","+((col==0) ? "Full" : "Line");
//       break;
//     }
//     case 2: {
//       history += "\nR,"+QString::number(row)+","+((col==0) ? "Full" : "Line");
//       break;
//     }
//   }
//   qDebug("###############################"+history+"\n------------------------------------");

  emit clicked(m_ID,row);

  FileTableItem* item = NULL;
  if (row != m_selectedRow) {//new row not jet focused
    item = (FileTableItem*)Table->item(row,0);
    if (item) {//set focus pointer to new row
      item->setSelected(true);
      Table->updateCell(row,0);
      emit focused(m_ID,row);
    }
    else return;
    item = (FileTableItem*)Table->item(m_selectedRow,0);
    if (item) {//reset old focus
      item->setSelected(false);
      Table->updateCell(m_selectedRow,0);
    }
    if (m_wasBlockSelect) {
      m_wasBlockSelect = false;
      m_lastMarked = m_selectedRow;
    }
    m_selectedRow = row;
    m_focusedRowPos = Table->rowPos(m_selectedRow);
  }

  if (col == 1) {

    if (button == 2) {
      
      if (m_lastMarked == -1) return;

      item = (FileTableItem*)Table->item(m_selectedRow,1);
      if (item && !item->changeable()) return;

      item = (FileTableItem*)Table->item(m_lastMarked,1);
      bool state = false;
      if (item) {
	state = item->isSelected();
	if (m_wasBlockSelect && (item = (FileTableItem*)Table->item(m_selectedRow,1)) && item) state = !item->isSelected();
	else if (state == ((FileTableItem*)Table->item(m_selectedRow,1))->isSelected()) return;//start and end marker need different state
      }
      if (m_selectedRow<m_lastMarked) {
	int i = 0;
	m_wasBlockSelect = true;
	int to = setBlockState(m_selectedRow+1-1,m_lastMarked-1,state);
	for (i=m_selectedRow+1;i<=to;i++) {
	  setSelected(i,state);
	}
	if (to != m_lastMarked-1) return;//different block, don't select
      } else if (m_lastMarked<m_selectedRow) {
	int i = 0;
	m_wasBlockSelect = true;
	int to = setBlockState(m_lastMarked+1,m_selectedRow-1+1,state);
	for (i=m_lastMarked+1;i<to;i++) {
	  setSelected(i,state);
	}
	if (to != m_selectedRow-1+1) return;//different block, don't select
      }

    } else {
      m_wasBlockSelect = false;
    }

    item = (FileTableItem*)Table->item(row,1);
    if (item) item->toggle();
    Table->updateCell(row,1);
    if (!m_wasBlockSelect) {
      m_lastMarked = m_selectedRow;
    }
    if (item) {
      if (!item->changeable()) return;
      if (!m_wasBlockSelect) {
	if (item->isSelected()) {
	  emit selectionChanged(m_ID,row,item->text());
	} else {
	  emit unselected(m_ID,row);
	}
      }
    }

  } else if (col == 0) {

    item = (FileTableItem*)Table->item(row,1);
    if (item && item->changeable()) {
      setBlock(item->id(),!item->isSelected());
    }
  }
}

void FileTableImpl::setBlock( int id, bool state) {

  int pos = m_selectedRow;
  int i = 0;
  FileTableItem* item = NULL;
  for (i = pos-1;i>=0;i--) {
    item = (FileTableItem*)Table->item(i,1);
    if (!item) return;
    if (item->id()!=id) break;
  }

  int from = i+1;
  for (i=i+1;i<pos;i++) {
    item = (FileTableItem*)Table->item(i,1);
    if (item->isSelected() == state) {
      if (from != i) {
	if (state) {
	  emit selectedBlock(m_ID,from,i-1);
	} else {
	  emit unSelectedBlock(m_ID,from,i-1);
	}
      }
      from = i+1;
    }
    setSelected(i,state);
  }
  for (i = pos;i<m_rows;i++) {
    item = (FileTableItem*)Table->item(i,1);
    if (item && item->id()==id) {
      if (item->isSelected() == state) {
	if (from != i) {
	  if (state) {
	    emit selectedBlock(m_ID,from,i-1);
	  } else {
	    emit unSelectedBlock(m_ID,from,i-1);
	  }
	}
	from = i+1;
      }
      setSelected(i,state);
    } else {
      if (from <= i-1) {
	if (state) {
	  emit selectedBlock(m_ID,from,i-1);
	} else {
	  emit unSelectedBlock(m_ID,from,i-1);
	}
      }
      break;
    }
  }
}

int FileTableImpl::setBlockState( int from, int to, bool state) {

  FileTableItem* item = NULL;
  int i;
  int end = from;
  for (i = from;i<=to;i++) {
    item = (FileTableItem*)Table->item(i,1);
    if (!item->changeable()) return end;//return end of allowed state change range (don't leave block)
    if (item->isSelected() == state) {
      if (from != i) {
	end = i-1;
	if (state) {
	  emit selectedBlock(m_ID,from,i-1);
	} else {
	  emit unSelectedBlock(m_ID,from,i-1);
	}
      }
      from = i+1;
    }
  }

  if (from <= to) {
    if (state) {
      emit selectedBlock(m_ID,from,to);
    } else {
      emit unSelectedBlock(m_ID,from,to);
    }
  }
  return to;
}

void FileTableImpl::setFocusRow(int row) {

  FileTableItem* item = NULL;
  if (row != m_selectedRow) {
    item = (FileTableItem*)Table->item(row,0);
    if (item) {
      item->setSelected(true);
      Table->updateCell(row,0);
      emit focused(m_ID,row);
    }
    else return;
    item = (FileTableItem*)Table->item(m_selectedRow,0);
    if (item) {
      item->setSelected(false);
      Table->updateCell(m_selectedRow,0);
    }
    m_selectedRow = row;
    Table->setContentsPos(0,Table->rowPos(m_selectedRow)-(Table->visibleHeight()/2)+(m_lineSpacing/2));
    m_focusedRowPos = Table->rowPos(m_selectedRow);
  }
}

void FileTableImpl::setFocusPos(int pos) {

  if (pos<0) pos = 0;
  if (pos>=Table->contentsHeight()) pos = Table->contentsHeight()-1;
  m_focusedRowPos = pos;

  int row = Table->rowAt(pos);
  FileTableItem* item = NULL;
  if (row != m_selectedRow) {
    item = (FileTableItem*)Table->item(row,0);
    if (item) {
      item->setSelected(true);
      Table->updateCell(row,0);
      emit focused(m_ID,row);
    }
    else return;
    item = (FileTableItem*)Table->item(m_selectedRow,0);
    if (item) {
      item->setSelected(false);
      Table->updateCell(m_selectedRow,0);
    }
    m_selectedRow = row;
  }
  Table->setContentsPos(0,pos-(Table->visibleHeight()/2)+(m_lineSpacing/2));
}

void FileTableImpl::setFocus(int,int row) {

  FileTableItem* item = NULL;
  if (row != m_selectedRow) {
    item = (FileTableItem*)Table->item(row,0);
    if (item) {
      item->setSelected(true);
      Table->updateCell(row,0);
    }
    else return;
    item = (FileTableItem*)Table->item(m_selectedRow,0);
    if (item) {
      item->setSelected(false);
      Table->updateCell(m_selectedRow,0);
    }
    m_selectedRow = row;
    m_focusedRowPos = Table->rowPos(m_selectedRow);
  }

}

void FileTableImpl::setSelected(int row,bool state) {

  FileTableItem* item =(FileTableItem*)Table->item(row,1);
  if (item && (item->isSelected() != state)) {
    item->setSelected(state);
    Table->updateCell(row,1);
    if (!item->changeable()) return;
//     if (state) {
//       emit selectionChanged(m_ID,row,item->text());
//     } else {
//       emit unselected(m_ID,row);
//     }
  }
}

void FileTableImpl::move(int from, int offset, int count) {
  int i;
  for (i = 0; i<count; i++) {
    Table->swapCells(from+i,1,from+offset+i,1);
  }
  Table->updateContents();
}

void FileTableImpl::reserveLines(int count) {
  Table->insertRows( m_rows,count);
}

void FileTableImpl::addLine(QString txt,
			    bool unchangeable,
			    const QColor selectedNumColor,
			    const QColor unselectedNumColor,
			    const QColor selectedColor) {
  addLine(txt,unchangeable,selectedNumColor,unselectedNumColor,selectedColor,paletteBackgroundColor());
}

void FileTableImpl::addLine(QString txt,
			    bool unchangeable,
			    const QColor selectedNumColor,
			    const QColor unselectedNumColor,
			    const QColor selectedColor,
			    const QColor unselectedColor) {

  int from = 0;
  int to = 0;
  int loop = 100;
  to=txt.find("\n",to);
  do {
    if (loop == 0) {
      qApp->processEvents(1);//ProgressDialog needs processing
      loop = 100;
    }
    QString tmpTxt = txt.mid((unsigned int)from,(unsigned int)(to-from));

    QString numString = QString::number(m_lines+1)+":";
    FileTableItem* numItem = new FileTableItem(Table,QTableItem::Never,numString,selectedNumColor,unselectedNumColor,false,m_blocks);
    FileTableItem* txtItem = new FileTableItem(Table,QTableItem::Never,tmpTxt,selectedColor,unselectedColor,unchangeable,m_blocks);

    Table->setRowHeight(m_rows, m_lineSpacing);
    Table->setItem( m_rows,0,numItem);
    Table->setItem( m_rows,1,txtItem);

    m_rows++;
    m_lines++;
    from = to+1;
    loop--;
  } while ( ((to=txt.find("\n",from))>-1) && (!m_stop));

  m_blocks++;
}

void FileTableImpl::addBlock(QString txt,
			    bool unchangeable,
			    const QColor selectedNumColor,
			    const QColor unselectedNumColor,
			    const QColor selectedColor) {
  addLine(txt,unchangeable,selectedNumColor,unselectedNumColor,selectedColor,paletteBackgroundColor());
}

void FileTableImpl::addBlock(QString txt,
			    bool unchangeable,
			    const QColor selectedNumColor,
			    const QColor unselectedNumColor,
			    const QColor selectedColor,
			    const QColor unselectedColor) {

  int from = 0;
  int to = 0;
  int count = txt.contains("\n");
  int loop = 100;
  if (count == 0) return;
  Table->insertRows( m_rows,count);
  to=txt.find("\n",to);
  do {
    if (loop == 0) {
      qApp->processEvents(1);//ProgressDialog needs processing
      loop = 100;
    }
    QString tmpTxt = txt.mid((unsigned int)from,(unsigned int)(to-from));

    QString numString = QString::number(m_lines+1)+":";
    FileTableItem* numItem = new FileTableItem(Table,QTableItem::Never,numString,selectedNumColor,unselectedNumColor,false,m_blocks);
    FileTableItem* txtItem = new FileTableItem(Table,QTableItem::Never,tmpTxt,selectedColor,unselectedColor,unchangeable,m_blocks);

    Table->setRowHeight(m_rows, m_lineSpacing);
    Table->setItem( m_rows,0,numItem);
    Table->setItem( m_rows,1,txtItem);

    m_rows++;
    m_lines++;
    from = to+1;
    loop--;
  } while ( ((to=txt.find("\n",from))>-1) && (!m_stop));

  m_blocks++;
}

void FileTableImpl::addParagraph(QString txt,
				 bool unchangeable,
				 const QColor selectedNumColor,
				 const QColor unselectedNumColor,
				 const QColor selectedColor) {
  addParagraph(txt,unchangeable,selectedNumColor,unselectedNumColor,selectedColor,paletteBackgroundColor());
}

void FileTableImpl::addParagraph(QString txt,
				 bool unchangeable,
				 const QColor selectedNumColor,
				 const QColor unselectedNumColor,
				 const QColor selectedColor,
				 const QColor unselectedColor) {

  QString numString = QString::number(m_lines+1)+":";
  int i=0;
  int addLinesCount = txt.contains('\n') + 1;
  int from = m_lines + 2;
  int to = m_lines + addLinesCount;
  for (i=from;i<=to;i++) {
    numString += "\n"+QString::number(i)+":";
  }
  FileTableItem* numItem = new FileTableItem(Table,QTableItem::Never,numString,selectedNumColor,unselectedNumColor,false,m_blocks);
  FileTableItem* txtItem = new FileTableItem(Table,QTableItem::Never,txt,selectedColor,unselectedColor,unchangeable,m_blocks);

  Table->insertRows( m_rows,1);
  Table->setRowHeight(m_rows, addLinesCount*m_lineSpacing);
  Table->setItem( m_rows,0,numItem);
  Table->setItem( m_rows,1,txtItem);

  m_rows++;
  m_lines += addLinesCount;
  m_blocks++;
}

void FileTableImpl::setText(int row, const QString& txt) {
  Table->setText(row,1,txt);
}

const QString FileTableImpl::getText(int row) {
  if (row<m_lines) {
    return Table->text(row,1);
  } else {
    return QString::null;
  }
}

int FileTableImpl::getTopLine() {
  return Table->rowAt(Table->contentsY());
}

int FileTableImpl::getFocusedLine() {
  return m_selectedRow;
}

void FileTableImpl::resizeEvent( QResizeEvent *e) {
  FileTable::resizeEvent(e);
  Table->updateScrollBars();
  if (Table->verticalScrollBar()->isVisible()) {
    QRect rect = Table->verticalScrollBar()->style().querySubControlMetrics(QStyle::CC_ScrollBar,
									    Table->verticalScrollBar(),
									    QStyle::SC_ScrollBarGroove);
    m_ScrollBarMinPos = rect.top();
    m_ScrollBarMaxPos = rect.height();
  }

  int tableWidth = Table->horizontalScrollBar()->width()-Table->cornerWidget()->width();
  if ( (Table->columnWidth(1) < tableWidth) || (tableWidth > m_maxWidth)) {
    Table->setColumnWidth(1,tableWidth);
  } else if (Table->columnWidth(1) != m_maxWidth) {
    Table->setColumnWidth(1,m_maxWidth);
  }
}

void FileTableImpl::adjustScrollBarOffsets(int &from, int &to) {
  Table->updateScrollBars();

  if (Table->verticalScrollBar()->isVisible()) {
    if (m_ScrollBarMinPos == -1) {
      QRect rect = Table->verticalScrollBar()->style().querySubControlMetrics(QStyle::CC_ScrollBar,
									      Table->verticalScrollBar(),
									      QStyle::SC_ScrollBarGroove);
      m_ScrollBarMinPos = rect.top();
      m_ScrollBarMaxPos = rect.height();
      from += m_ScrollBarMinPos;
      to = m_ScrollBarMaxPos;
    } else {
      from += m_ScrollBarMinPos;
      to = m_ScrollBarMaxPos;
    }
  }
}
